// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A7__F206

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_16_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1200)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xd4ecc3abu, 0xf9d2f6afu, 0, 6 }, // d4ecc3abf9d2f6af = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0x18364015u, 0x0bb791efu, 0, 63 }, // 183640150bb791ef = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x57368856u, 0xd32e90bfu, 23, 29 }, // 57368856d32e90bf = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1200
    { 0xb17e23a1u, 0x7b17f377u, 23, 6 }, // b17e23a17b17f377 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1200
    { 0x241ab1eeu, 0x3d212281u, 23, 46 }, // 241ab1ee3d212281 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1200
    { 0x48e63961u, 0x882df2bcu, 23, 63 }, // 48e63961882df2bc = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1200
    { 0x7d5c3380u, 0xb1fa6e0eu, 103, 109 }, // 7d5c3380b1fa6e0e = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0xea602a32u, 0x2116a6f4u, 103, 126 }, // ea602a322116a6f4 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0xdf1f9e26u, 0x969a377au, 103, 86 }, // df1f9e26969a377a = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x054841ffu, 0xddb5e42du, 103, 143 }, // 054841ffddb5e42d = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0xab036a0bu, 0x5042f6c1u, 160, 126 }, // ab036a0b5042f6c1 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0x36b5f746u, 0x6ad72ad3u, 160, 86 }, // 36b5f7466ad72ad3 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0xf14c0ab4u, 0x335097e7u, 160, 143 }, // f14c0ab4335097e7 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0x6268871cu, 0x17b83f31u, 166, 109 }, // 6268871c17b83f31 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_64__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0xc7a9cf8au, 0x1966ae15u, 166, 126 }, // c7a9cf8a1966ae15 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_64__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0x23e3740au, 0xd96a17b1u, 166, 86 }, // 23e3740ad96a17b1 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_64__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0xa88098f4u, 0xe3e640b1u, 166, 143 }, // a88098f4e3e640b1 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_16_3_T_T_0___gfx120x__P__64_64__CO__wave4_warp4_stg1--Arch_gfx1200
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1200_mod0
{{ 1, 1, 1, 1, 1, 0, 1, 1, 5, 1},
 { 5, 2, 2, 3, 4, 5, 2, 2, 3, 2},
 { 7, 9, 2, 4, 3, 4, 5, 5, 5, 5},
 { 7, 6, 4, 3, 2, 2, 2, 3, 5, 5},
 { 6, 6, 5, 3, 4, 5, 4, 5, 5, 4},
 { 6, 8, 2, 5, 2, 5, 4, 5, 3, 3},
 { 6, 9, 3, 2, 3, 3, 2, 3, 2, 7},
 { 8, 7,12,14,15,15,15,13,11, 5},
 { 9, 9,11,15,15,15,13, 4, 2, 5},
 { 8, 8,15,15,15,15,10, 4,16,14}}
// End of GPU gfx1200_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A7__F206 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

