
#include "dawn/wire/WireCmd_autogen.h"

#include "dawn/common/Assert.h"
#include "dawn/common/Log.h"
#include "dawn/common/Numeric.h"
#include "dawn/wire/BufferConsumer_impl.h"
#include "dawn/wire/Wire.h"

#include <algorithm>
#include <cstring>
#include <limits>

#if defined(__GNUC__) || defined(__clang__)
// error: 'offsetof' within non-standard-layout type 'wgpu::XXX' is conditionally-supported
#pragma GCC diagnostic ignored "-Winvalid-offsetof"
#endif
namespace dawn::wire {
namespace {

// Allocates enough space from allocator to countain T[count] and return it in out.
// Return FatalError if the allocator couldn't allocate the memory.
// Always writes to |out| on success.
template <typename T, typename N>
WireResult GetSpace(DeserializeAllocator* allocator, N count, T** out) {
    // Because we use this function extensively when `count` == 1, we can optimize the
    // size computations a bit more for those cases via constexpr version of the
    // alignment computation.
    constexpr size_t kSizeofT = WireAlignSizeof<T>();
    size_t size = 0;
    if (count == 1) {
      size = kSizeofT;
    } else {
      auto sizeN = WireAlignSizeofN<T>(count);
      // A size of 0 indicates an overflow, so return an error.
      if (!sizeN) {
        return WireResult::FatalError;
      }
      size = *sizeN;
    }

    *out = static_cast<T*>(allocator->GetSpace(size));
    if (*out == nullptr) {
        return WireResult::FatalError;
    }

    return WireResult::Success;
}

struct WGPUChainedStructTransfer {
    WGPUSType sType;
    bool hasNext;
};

size_t GetChainedStructExtraRequiredSize(WGPUChainedStruct* chainedStruct);
[[nodiscard]] WireResult SerializeChainedStruct(WGPUChainedStruct* chainedStruct,
                                                SerializeBuffer* buffer,
                                                const ObjectIdProvider& provider);
WireResult DeserializeChainedStruct(WGPUChainedStruct** outChainNext,
                                    DeserializeBuffer* deserializeBuffer,
                                    DeserializeAllocator* allocator,
                                    const ObjectIdResolver& resolver);

// Manually define serialization and deserialization for WGPUStringView because
// it has a special encoding where:
//  { .data = nullptr, .length = WGPU_STRLEN }  --> nil
//  { .data = non-null, .length = WGPU_STRLEN } --> null-terminated, use strlen
//  { .data = ..., .length = 0 }             --> ""
//  { .data = ..., .length > 0 }             --> string of size `length`
struct WGPUStringViewTransfer {
    bool has_data;
    uint64_t length;
};

size_t WGPUStringViewGetExtraRequiredSize(const WGPUStringView& record) {
    size_t size = record.length;
    if (size == WGPU_STRLEN) {
        // This is a null-terminated string, or it's nil.
        size = record.data ? std::strlen(record.data) : 0;
    }
    return Align(size, kWireBufferAlignment);
}

WireResult WGPUStringViewSerialize(
    const WGPUStringView& record,
    WGPUStringViewTransfer* transfer,
    SerializeBuffer* buffer) {

    bool has_data = record.data != nullptr;
    uint64_t length = record.length;
    transfer->has_data = has_data;

    if (!has_data) {
        transfer->length = length;
        return WireResult::Success;
    }
    if (length == WGPU_STRLEN) {
        length = std::strlen(record.data);
    }
    if (length > 0) {
        char* memberBuffer;
        WIRE_TRY(buffer->NextN(length, &memberBuffer));
        memcpy(memberBuffer, record.data, length);
    }
    transfer->length = length;
    return WireResult::Success;
}

WireResult WGPUStringViewDeserialize(
    WGPUStringView* record,
    const volatile WGPUStringViewTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator) {

    bool has_data = transfer->has_data;
    uint64_t length = transfer->length;

    if (length > WGPU_STRLEN) {
        return WireResult::FatalError;
    }
    if (!has_data) {
        record->data = nullptr;
        if (length != 0 && length != WGPU_STRLEN) {
            // Invalid string.
            return WireResult::FatalError;
        }
        record->length = static_cast<size_t>(length);
        return WireResult::Success;
    }
    if (length == 0) {
        record->data = "";
        record->length = 0;
        return WireResult::Success;
    }

    size_t stringLength = static_cast<size_t>(length);
    const volatile char* stringInBuffer;
    WIRE_TRY(deserializeBuffer->ReadN(stringLength, &stringInBuffer));

    char* copiedString;
    WIRE_TRY(GetSpace(allocator, stringLength, &copiedString));
    memcpy(copiedString, const_cast<const char*>(stringInBuffer), stringLength);

    record->data = copiedString;
    record->length = stringLength;
    return WireResult::Success;
}


struct WGPUAdapterPropertiesD3DTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t shaderModel;
};

static_assert(offsetof(WGPUAdapterPropertiesD3DTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUAdapterPropertiesD3DGetExtraRequiredSize([[maybe_unused]] const WGPUAdapterPropertiesD3D& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesD3DGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesD3DSerialize(
    const WGPUAdapterPropertiesD3D& record,
    WGPUAdapterPropertiesD3DTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_AdapterPropertiesD3D);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->shaderModel = record.shaderModel;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesD3DSerialize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesD3DDeserialize(
    WGPUAdapterPropertiesD3D* record,
    const volatile WGPUAdapterPropertiesD3DTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_AdapterPropertiesD3D);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->shaderModel) >= sizeof(transfer->shaderModel), "Deserialize assignment may not narrow.");
    record->shaderModel = transfer->shaderModel;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesD3DDeserialize);

struct WGPUAdapterPropertiesSubgroupsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t subgroupMinSize;
    uint32_t subgroupMaxSize;
};

static_assert(offsetof(WGPUAdapterPropertiesSubgroupsTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUAdapterPropertiesSubgroupsGetExtraRequiredSize([[maybe_unused]] const WGPUAdapterPropertiesSubgroups& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesSubgroupsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesSubgroupsSerialize(
    const WGPUAdapterPropertiesSubgroups& record,
    WGPUAdapterPropertiesSubgroupsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_AdapterPropertiesSubgroups);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->subgroupMinSize = record.subgroupMinSize;
    transfer->subgroupMaxSize = record.subgroupMaxSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesSubgroupsSerialize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesSubgroupsDeserialize(
    WGPUAdapterPropertiesSubgroups* record,
    const volatile WGPUAdapterPropertiesSubgroupsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_AdapterPropertiesSubgroups);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->subgroupMinSize) >= sizeof(transfer->subgroupMinSize), "Deserialize assignment may not narrow.");
    record->subgroupMinSize = transfer->subgroupMinSize;
    static_assert(sizeof(record->subgroupMaxSize) >= sizeof(transfer->subgroupMaxSize), "Deserialize assignment may not narrow.");
    record->subgroupMaxSize = transfer->subgroupMaxSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesSubgroupsDeserialize);

struct WGPUAdapterPropertiesVkTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t driverVersion;
};

static_assert(offsetof(WGPUAdapterPropertiesVkTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUAdapterPropertiesVkGetExtraRequiredSize([[maybe_unused]] const WGPUAdapterPropertiesVk& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesVkGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesVkSerialize(
    const WGPUAdapterPropertiesVk& record,
    WGPUAdapterPropertiesVkTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_AdapterPropertiesVk);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->driverVersion = record.driverVersion;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesVkSerialize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesVkDeserialize(
    WGPUAdapterPropertiesVk* record,
    const volatile WGPUAdapterPropertiesVkTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_AdapterPropertiesVk);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->driverVersion) >= sizeof(transfer->driverVersion), "Deserialize assignment may not narrow.");
    record->driverVersion = transfer->driverVersion;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesVkDeserialize);

struct WGPUBindGroupEntryTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    uint32_t binding;
    ObjectId buffer;
    uint64_t offset;
    uint64_t size;
    ObjectId sampler;
    ObjectId textureView;
};


[[maybe_unused]] size_t WGPUBindGroupEntryGetExtraRequiredSize([[maybe_unused]] const WGPUBindGroupEntry& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBindGroupEntryGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBindGroupEntrySerialize(
    const WGPUBindGroupEntry& record,
    WGPUBindGroupEntryTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->binding = record.binding;
    WIRE_TRY(provider.GetOptionalId(record.buffer, &transfer->buffer));
    transfer->offset = record.offset;
    transfer->size = record.size;
    WIRE_TRY(provider.GetOptionalId(record.sampler, &transfer->sampler));
    WIRE_TRY(provider.GetOptionalId(record.textureView, &transfer->textureView));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupEntrySerialize);

[[maybe_unused]] WireResult WGPUBindGroupEntryDeserialize(
    WGPUBindGroupEntry* record,
    const volatile WGPUBindGroupEntryTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->binding) >= sizeof(transfer->binding), "Deserialize assignment may not narrow.");
    record->binding = transfer->binding;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->buffer, &record->buffer));
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->sampler, &record->sampler));
    WIRE_TRY(resolver.GetOptionalFromId(transfer->textureView, &record->textureView));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupEntryDeserialize);

struct WGPUBlendComponentTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    WGPUBlendOperation operation;
    WGPUBlendFactor srcFactor;
    WGPUBlendFactor dstFactor;
};


[[maybe_unused]] size_t WGPUBlendComponentGetExtraRequiredSize([[maybe_unused]] const WGPUBlendComponent& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBlendComponentGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBlendComponentSerialize(
    const WGPUBlendComponent& record,
    WGPUBlendComponentTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->operation = record.operation;
    transfer->srcFactor = record.srcFactor;
    transfer->dstFactor = record.dstFactor;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBlendComponentSerialize);

[[maybe_unused]] WireResult WGPUBlendComponentDeserialize(
    WGPUBlendComponent* record,
    const volatile WGPUBlendComponentTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->operation) >= sizeof(transfer->operation), "Deserialize assignment may not narrow.");
    record->operation = transfer->operation;
    static_assert(sizeof(record->srcFactor) >= sizeof(transfer->srcFactor), "Deserialize assignment may not narrow.");
    record->srcFactor = transfer->srcFactor;
    static_assert(sizeof(record->dstFactor) >= sizeof(transfer->dstFactor), "Deserialize assignment may not narrow.");
    record->dstFactor = transfer->dstFactor;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBlendComponentDeserialize);

struct WGPUBufferBindingLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUBufferBindingType type;
    WGPUBool hasDynamicOffset;
    uint64_t minBindingSize;
};


[[maybe_unused]] size_t WGPUBufferBindingLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUBufferBindingLayout& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBufferBindingLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBufferBindingLayoutSerialize(
    const WGPUBufferBindingLayout& record,
    WGPUBufferBindingLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->type = record.type;
    transfer->hasDynamicOffset = record.hasDynamicOffset;
    transfer->minBindingSize = record.minBindingSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBufferBindingLayoutSerialize);

[[maybe_unused]] WireResult WGPUBufferBindingLayoutDeserialize(
    WGPUBufferBindingLayout* record,
    const volatile WGPUBufferBindingLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    static_assert(sizeof(record->hasDynamicOffset) >= sizeof(transfer->hasDynamicOffset), "Deserialize assignment may not narrow.");
    record->hasDynamicOffset = transfer->hasDynamicOffset;
    static_assert(sizeof(record->minBindingSize) >= sizeof(transfer->minBindingSize), "Deserialize assignment may not narrow.");
    record->minBindingSize = transfer->minBindingSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBufferBindingLayoutDeserialize);

struct WGPUBufferHostMappedPointerTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

};

static_assert(offsetof(WGPUBufferHostMappedPointerTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUBufferHostMappedPointerGetExtraRequiredSize([[maybe_unused]] const WGPUBufferHostMappedPointer& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBufferHostMappedPointerGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBufferHostMappedPointerSerialize(
    const WGPUBufferHostMappedPointer& record,
    WGPUBufferHostMappedPointerTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_BufferHostMappedPointer);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    if (record.pointer != nullptr) return WireResult::FatalError;
    if (record.disposeCallback != nullptr) return WireResult::FatalError;
    if (record.userdata != nullptr) return WireResult::FatalError;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBufferHostMappedPointerSerialize);

[[maybe_unused]] WireResult WGPUBufferHostMappedPointerDeserialize(
    WGPUBufferHostMappedPointer* record,
    const volatile WGPUBufferHostMappedPointerTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_BufferHostMappedPointer);
    DAWN_ASSERT(record->chain.next == nullptr);

    record->pointer = nullptr;
    record->disposeCallback = nullptr;
    record->userdata = nullptr;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBufferHostMappedPointerDeserialize);

struct WGPUColorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    double r;
    double g;
    double b;
    double a;
};


[[maybe_unused]] size_t WGPUColorGetExtraRequiredSize([[maybe_unused]] const WGPUColor& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUColorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUColorSerialize(
    const WGPUColor& record,
    WGPUColorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->r = record.r;
    transfer->g = record.g;
    transfer->b = record.b;
    transfer->a = record.a;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUColorSerialize);

[[maybe_unused]] WireResult WGPUColorDeserialize(
    WGPUColor* record,
    const volatile WGPUColorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->r) >= sizeof(transfer->r), "Deserialize assignment may not narrow.");
    record->r = transfer->r;
    static_assert(sizeof(record->g) >= sizeof(transfer->g), "Deserialize assignment may not narrow.");
    record->g = transfer->g;
    static_assert(sizeof(record->b) >= sizeof(transfer->b), "Deserialize assignment may not narrow.");
    record->b = transfer->b;
    static_assert(sizeof(record->a) >= sizeof(transfer->a), "Deserialize assignment may not narrow.");
    record->a = transfer->a;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUColorDeserialize);

struct WGPUColorTargetStateExpandResolveTextureDawnTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUBool enabled;
};

static_assert(offsetof(WGPUColorTargetStateExpandResolveTextureDawnTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUColorTargetStateExpandResolveTextureDawnGetExtraRequiredSize([[maybe_unused]] const WGPUColorTargetStateExpandResolveTextureDawn& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUColorTargetStateExpandResolveTextureDawnGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUColorTargetStateExpandResolveTextureDawnSerialize(
    const WGPUColorTargetStateExpandResolveTextureDawn& record,
    WGPUColorTargetStateExpandResolveTextureDawnTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_ColorTargetStateExpandResolveTextureDawn);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->enabled = record.enabled;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUColorTargetStateExpandResolveTextureDawnSerialize);

[[maybe_unused]] WireResult WGPUColorTargetStateExpandResolveTextureDawnDeserialize(
    WGPUColorTargetStateExpandResolveTextureDawn* record,
    const volatile WGPUColorTargetStateExpandResolveTextureDawnTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_ColorTargetStateExpandResolveTextureDawn);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->enabled) >= sizeof(transfer->enabled), "Deserialize assignment may not narrow.");
    record->enabled = transfer->enabled;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUColorTargetStateExpandResolveTextureDawnDeserialize);

struct WGPUCopyTextureForBrowserOptionsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUBool flipY;
    WGPUBool needsColorSpaceConversion;
    WGPUAlphaMode srcAlphaMode;
    WGPUBool has_srcTransferFunctionParameters;
    WGPUBool has_conversionMatrix;
    WGPUBool has_dstTransferFunctionParameters;
    WGPUAlphaMode dstAlphaMode;
    WGPUBool internalUsage;
};


[[maybe_unused]] size_t WGPUCopyTextureForBrowserOptionsGetExtraRequiredSize([[maybe_unused]] const WGPUCopyTextureForBrowserOptions& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    if (record.srcTransferFunctionParameters != nullptr)
    {
        auto memberLength = 7u;
        auto size = WireAlignSizeofN<float>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    if (record.conversionMatrix != nullptr)
    {
        auto memberLength = 9u;
        auto size = WireAlignSizeofN<float>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    if (record.dstTransferFunctionParameters != nullptr)
    {
        auto memberLength = 7u;
        auto size = WireAlignSizeofN<float>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUCopyTextureForBrowserOptionsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUCopyTextureForBrowserOptionsSerialize(
    const WGPUCopyTextureForBrowserOptions& record,
    WGPUCopyTextureForBrowserOptionsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->flipY = record.flipY;
    transfer->needsColorSpaceConversion = record.needsColorSpaceConversion;
    transfer->srcAlphaMode = record.srcAlphaMode;
    transfer->dstAlphaMode = record.dstAlphaMode;
    transfer->internalUsage = record.internalUsage;
    bool has_srcTransferFunctionParameters = record.srcTransferFunctionParameters != nullptr;
    transfer->has_srcTransferFunctionParameters = has_srcTransferFunctionParameters;
    if (has_srcTransferFunctionParameters) {
        auto memberLength = 7u;

        float* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.srcTransferFunctionParameters,
                sizeof(float) * memberLength);
        }
    }
    bool has_conversionMatrix = record.conversionMatrix != nullptr;
    transfer->has_conversionMatrix = has_conversionMatrix;
    if (has_conversionMatrix) {
        auto memberLength = 9u;

        float* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.conversionMatrix,
                sizeof(float) * memberLength);
        }
    }
    bool has_dstTransferFunctionParameters = record.dstTransferFunctionParameters != nullptr;
    transfer->has_dstTransferFunctionParameters = has_dstTransferFunctionParameters;
    if (has_dstTransferFunctionParameters) {
        auto memberLength = 7u;

        float* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.dstTransferFunctionParameters,
                sizeof(float) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCopyTextureForBrowserOptionsSerialize);

[[maybe_unused]] WireResult WGPUCopyTextureForBrowserOptionsDeserialize(
    WGPUCopyTextureForBrowserOptions* record,
    const volatile WGPUCopyTextureForBrowserOptionsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->flipY) >= sizeof(transfer->flipY), "Deserialize assignment may not narrow.");
    record->flipY = transfer->flipY;
    static_assert(sizeof(record->needsColorSpaceConversion) >= sizeof(transfer->needsColorSpaceConversion), "Deserialize assignment may not narrow.");
    record->needsColorSpaceConversion = transfer->needsColorSpaceConversion;
    static_assert(sizeof(record->srcAlphaMode) >= sizeof(transfer->srcAlphaMode), "Deserialize assignment may not narrow.");
    record->srcAlphaMode = transfer->srcAlphaMode;
    static_assert(sizeof(record->dstAlphaMode) >= sizeof(transfer->dstAlphaMode), "Deserialize assignment may not narrow.");
    record->dstAlphaMode = transfer->dstAlphaMode;
    static_assert(sizeof(record->internalUsage) >= sizeof(transfer->internalUsage), "Deserialize assignment may not narrow.");
    record->internalUsage = transfer->internalUsage;
    bool has_srcTransferFunctionParameters = transfer->has_srcTransferFunctionParameters;
    record->srcTransferFunctionParameters = nullptr;
    if (has_srcTransferFunctionParameters) {
        auto memberLength = 7u;
        const volatile float* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        float* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->srcTransferFunctionParameters = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const float*>(memberBuffer),
              sizeof(float) * memberLength);
        }
    }
    bool has_conversionMatrix = transfer->has_conversionMatrix;
    record->conversionMatrix = nullptr;
    if (has_conversionMatrix) {
        auto memberLength = 9u;
        const volatile float* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        float* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->conversionMatrix = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const float*>(memberBuffer),
              sizeof(float) * memberLength);
        }
    }
    bool has_dstTransferFunctionParameters = transfer->has_dstTransferFunctionParameters;
    record->dstTransferFunctionParameters = nullptr;
    if (has_dstTransferFunctionParameters) {
        auto memberLength = 7u;
        const volatile float* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        float* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->dstTransferFunctionParameters = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const float*>(memberBuffer),
              sizeof(float) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCopyTextureForBrowserOptionsDeserialize);

struct WGPUDawnAdapterPropertiesPowerPreferenceTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUPowerPreference powerPreference;
};

static_assert(offsetof(WGPUDawnAdapterPropertiesPowerPreferenceTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnAdapterPropertiesPowerPreferenceGetExtraRequiredSize([[maybe_unused]] const WGPUDawnAdapterPropertiesPowerPreference& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnAdapterPropertiesPowerPreferenceGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnAdapterPropertiesPowerPreferenceSerialize(
    const WGPUDawnAdapterPropertiesPowerPreference& record,
    WGPUDawnAdapterPropertiesPowerPreferenceTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnAdapterPropertiesPowerPreference);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->powerPreference = record.powerPreference;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnAdapterPropertiesPowerPreferenceSerialize);

[[maybe_unused]] WireResult WGPUDawnAdapterPropertiesPowerPreferenceDeserialize(
    WGPUDawnAdapterPropertiesPowerPreference* record,
    const volatile WGPUDawnAdapterPropertiesPowerPreferenceTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnAdapterPropertiesPowerPreference);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->powerPreference) >= sizeof(transfer->powerPreference), "Deserialize assignment may not narrow.");
    record->powerPreference = transfer->powerPreference;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnAdapterPropertiesPowerPreferenceDeserialize);

struct WGPUDawnBufferDescriptorErrorInfoFromWireClientTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUBool outOfMemory;
};

static_assert(offsetof(WGPUDawnBufferDescriptorErrorInfoFromWireClientTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnBufferDescriptorErrorInfoFromWireClientGetExtraRequiredSize([[maybe_unused]] const WGPUDawnBufferDescriptorErrorInfoFromWireClient& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnBufferDescriptorErrorInfoFromWireClientGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnBufferDescriptorErrorInfoFromWireClientSerialize(
    const WGPUDawnBufferDescriptorErrorInfoFromWireClient& record,
    WGPUDawnBufferDescriptorErrorInfoFromWireClientTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnBufferDescriptorErrorInfoFromWireClient);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->outOfMemory = record.outOfMemory;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnBufferDescriptorErrorInfoFromWireClientSerialize);

[[maybe_unused]] WireResult WGPUDawnBufferDescriptorErrorInfoFromWireClientDeserialize(
    WGPUDawnBufferDescriptorErrorInfoFromWireClient* record,
    const volatile WGPUDawnBufferDescriptorErrorInfoFromWireClientTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnBufferDescriptorErrorInfoFromWireClient);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->outOfMemory) >= sizeof(transfer->outOfMemory), "Deserialize assignment may not narrow.");
    record->outOfMemory = transfer->outOfMemory;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnBufferDescriptorErrorInfoFromWireClientDeserialize);

struct WGPUDawnDrmFormatPropertiesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint64_t modifier;
    uint32_t modifierPlaneCount;
};


[[maybe_unused]] size_t WGPUDawnDrmFormatPropertiesGetExtraRequiredSize([[maybe_unused]] const WGPUDawnDrmFormatProperties& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnDrmFormatPropertiesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnDrmFormatPropertiesSerialize(
    const WGPUDawnDrmFormatProperties& record,
    WGPUDawnDrmFormatPropertiesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->modifier = record.modifier;
    transfer->modifierPlaneCount = record.modifierPlaneCount;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnDrmFormatPropertiesSerialize);

[[maybe_unused]] WireResult WGPUDawnDrmFormatPropertiesDeserialize(
    WGPUDawnDrmFormatProperties* record,
    const volatile WGPUDawnDrmFormatPropertiesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->modifier) >= sizeof(transfer->modifier), "Deserialize assignment may not narrow.");
    record->modifier = transfer->modifier;
    static_assert(sizeof(record->modifierPlaneCount) >= sizeof(transfer->modifierPlaneCount), "Deserialize assignment may not narrow.");
    record->modifierPlaneCount = transfer->modifierPlaneCount;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnDrmFormatPropertiesDeserialize);

struct WGPUDawnEncoderInternalUsageDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUBool useInternalUsages;
};

static_assert(offsetof(WGPUDawnEncoderInternalUsageDescriptorTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnEncoderInternalUsageDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUDawnEncoderInternalUsageDescriptor& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnEncoderInternalUsageDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnEncoderInternalUsageDescriptorSerialize(
    const WGPUDawnEncoderInternalUsageDescriptor& record,
    WGPUDawnEncoderInternalUsageDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnEncoderInternalUsageDescriptor);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->useInternalUsages = record.useInternalUsages;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnEncoderInternalUsageDescriptorSerialize);

[[maybe_unused]] WireResult WGPUDawnEncoderInternalUsageDescriptorDeserialize(
    WGPUDawnEncoderInternalUsageDescriptor* record,
    const volatile WGPUDawnEncoderInternalUsageDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnEncoderInternalUsageDescriptor);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->useInternalUsages) >= sizeof(transfer->useInternalUsages), "Deserialize assignment may not narrow.");
    record->useInternalUsages = transfer->useInternalUsages;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnEncoderInternalUsageDescriptorDeserialize);

struct WGPUDawnExperimentalImmediateDataLimitsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t maxImmediateDataRangeByteSize;
};

static_assert(offsetof(WGPUDawnExperimentalImmediateDataLimitsTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnExperimentalImmediateDataLimitsGetExtraRequiredSize([[maybe_unused]] const WGPUDawnExperimentalImmediateDataLimits& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnExperimentalImmediateDataLimitsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnExperimentalImmediateDataLimitsSerialize(
    const WGPUDawnExperimentalImmediateDataLimits& record,
    WGPUDawnExperimentalImmediateDataLimitsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnExperimentalImmediateDataLimits);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->maxImmediateDataRangeByteSize = record.maxImmediateDataRangeByteSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnExperimentalImmediateDataLimitsSerialize);

[[maybe_unused]] WireResult WGPUDawnExperimentalImmediateDataLimitsDeserialize(
    WGPUDawnExperimentalImmediateDataLimits* record,
    const volatile WGPUDawnExperimentalImmediateDataLimitsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnExperimentalImmediateDataLimits);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->maxImmediateDataRangeByteSize) >= sizeof(transfer->maxImmediateDataRangeByteSize), "Deserialize assignment may not narrow.");
    record->maxImmediateDataRangeByteSize = transfer->maxImmediateDataRangeByteSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnExperimentalImmediateDataLimitsDeserialize);

struct WGPUDawnExperimentalSubgroupLimitsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t minSubgroupSize;
    uint32_t maxSubgroupSize;
};

static_assert(offsetof(WGPUDawnExperimentalSubgroupLimitsTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnExperimentalSubgroupLimitsGetExtraRequiredSize([[maybe_unused]] const WGPUDawnExperimentalSubgroupLimits& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnExperimentalSubgroupLimitsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnExperimentalSubgroupLimitsSerialize(
    const WGPUDawnExperimentalSubgroupLimits& record,
    WGPUDawnExperimentalSubgroupLimitsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnExperimentalSubgroupLimits);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->minSubgroupSize = record.minSubgroupSize;
    transfer->maxSubgroupSize = record.maxSubgroupSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnExperimentalSubgroupLimitsSerialize);

[[maybe_unused]] WireResult WGPUDawnExperimentalSubgroupLimitsDeserialize(
    WGPUDawnExperimentalSubgroupLimits* record,
    const volatile WGPUDawnExperimentalSubgroupLimitsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnExperimentalSubgroupLimits);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->minSubgroupSize) >= sizeof(transfer->minSubgroupSize), "Deserialize assignment may not narrow.");
    record->minSubgroupSize = transfer->minSubgroupSize;
    static_assert(sizeof(record->maxSubgroupSize) >= sizeof(transfer->maxSubgroupSize), "Deserialize assignment may not narrow.");
    record->maxSubgroupSize = transfer->maxSubgroupSize;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnExperimentalSubgroupLimitsDeserialize);

struct WGPUDawnFormatCapabilitiesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

};


[[maybe_unused]] size_t WGPUDawnFormatCapabilitiesGetExtraRequiredSize([[maybe_unused]] const WGPUDawnFormatCapabilities& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnFormatCapabilitiesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnFormatCapabilitiesSerialize(
    const WGPUDawnFormatCapabilities& record,
    WGPUDawnFormatCapabilitiesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }


    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnFormatCapabilitiesSerialize);

[[maybe_unused]] WireResult WGPUDawnFormatCapabilitiesDeserialize(
    WGPUDawnFormatCapabilities* record,
    const volatile WGPUDawnFormatCapabilitiesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }


    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnFormatCapabilitiesDeserialize);

struct WGPUDawnRenderPassColorAttachmentRenderToSingleSampledTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t implicitSampleCount;
};

static_assert(offsetof(WGPUDawnRenderPassColorAttachmentRenderToSingleSampledTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnRenderPassColorAttachmentRenderToSingleSampledGetExtraRequiredSize([[maybe_unused]] const WGPUDawnRenderPassColorAttachmentRenderToSingleSampled& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnRenderPassColorAttachmentRenderToSingleSampledGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnRenderPassColorAttachmentRenderToSingleSampledSerialize(
    const WGPUDawnRenderPassColorAttachmentRenderToSingleSampled& record,
    WGPUDawnRenderPassColorAttachmentRenderToSingleSampledTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnRenderPassColorAttachmentRenderToSingleSampled);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->implicitSampleCount = record.implicitSampleCount;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnRenderPassColorAttachmentRenderToSingleSampledSerialize);

[[maybe_unused]] WireResult WGPUDawnRenderPassColorAttachmentRenderToSingleSampledDeserialize(
    WGPUDawnRenderPassColorAttachmentRenderToSingleSampled* record,
    const volatile WGPUDawnRenderPassColorAttachmentRenderToSingleSampledTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnRenderPassColorAttachmentRenderToSingleSampled);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->implicitSampleCount) >= sizeof(transfer->implicitSampleCount), "Deserialize assignment may not narrow.");
    record->implicitSampleCount = transfer->implicitSampleCount;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnRenderPassColorAttachmentRenderToSingleSampledDeserialize);

struct WGPUDawnShaderModuleSPIRVOptionsDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUBool allowNonUniformDerivatives;
};

static_assert(offsetof(WGPUDawnShaderModuleSPIRVOptionsDescriptorTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnShaderModuleSPIRVOptionsDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUDawnShaderModuleSPIRVOptionsDescriptor& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnShaderModuleSPIRVOptionsDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnShaderModuleSPIRVOptionsDescriptorSerialize(
    const WGPUDawnShaderModuleSPIRVOptionsDescriptor& record,
    WGPUDawnShaderModuleSPIRVOptionsDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnShaderModuleSPIRVOptionsDescriptor);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->allowNonUniformDerivatives = record.allowNonUniformDerivatives;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnShaderModuleSPIRVOptionsDescriptorSerialize);

[[maybe_unused]] WireResult WGPUDawnShaderModuleSPIRVOptionsDescriptorDeserialize(
    WGPUDawnShaderModuleSPIRVOptionsDescriptor* record,
    const volatile WGPUDawnShaderModuleSPIRVOptionsDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnShaderModuleSPIRVOptionsDescriptor);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->allowNonUniformDerivatives) >= sizeof(transfer->allowNonUniformDerivatives), "Deserialize assignment may not narrow.");
    record->allowNonUniformDerivatives = transfer->allowNonUniformDerivatives;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnShaderModuleSPIRVOptionsDescriptorDeserialize);

struct WGPUDawnTexelCopyBufferRowAlignmentLimitsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t minTexelCopyBufferRowAlignment;
};

static_assert(offsetof(WGPUDawnTexelCopyBufferRowAlignmentLimitsTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnTexelCopyBufferRowAlignmentLimitsGetExtraRequiredSize([[maybe_unused]] const WGPUDawnTexelCopyBufferRowAlignmentLimits& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnTexelCopyBufferRowAlignmentLimitsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnTexelCopyBufferRowAlignmentLimitsSerialize(
    const WGPUDawnTexelCopyBufferRowAlignmentLimits& record,
    WGPUDawnTexelCopyBufferRowAlignmentLimitsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnTexelCopyBufferRowAlignmentLimits);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->minTexelCopyBufferRowAlignment = record.minTexelCopyBufferRowAlignment;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnTexelCopyBufferRowAlignmentLimitsSerialize);

[[maybe_unused]] WireResult WGPUDawnTexelCopyBufferRowAlignmentLimitsDeserialize(
    WGPUDawnTexelCopyBufferRowAlignmentLimits* record,
    const volatile WGPUDawnTexelCopyBufferRowAlignmentLimitsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnTexelCopyBufferRowAlignmentLimits);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->minTexelCopyBufferRowAlignment) >= sizeof(transfer->minTexelCopyBufferRowAlignment), "Deserialize assignment may not narrow.");
    record->minTexelCopyBufferRowAlignment = transfer->minTexelCopyBufferRowAlignment;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnTexelCopyBufferRowAlignmentLimitsDeserialize);

struct WGPUDawnTextureInternalUsageDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUTextureUsage internalUsage;
};

static_assert(offsetof(WGPUDawnTextureInternalUsageDescriptorTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnTextureInternalUsageDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUDawnTextureInternalUsageDescriptor& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnTextureInternalUsageDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnTextureInternalUsageDescriptorSerialize(
    const WGPUDawnTextureInternalUsageDescriptor& record,
    WGPUDawnTextureInternalUsageDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnTextureInternalUsageDescriptor);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->internalUsage = record.internalUsage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnTextureInternalUsageDescriptorSerialize);

[[maybe_unused]] WireResult WGPUDawnTextureInternalUsageDescriptorDeserialize(
    WGPUDawnTextureInternalUsageDescriptor* record,
    const volatile WGPUDawnTextureInternalUsageDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnTextureInternalUsageDescriptor);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->internalUsage) >= sizeof(transfer->internalUsage), "Deserialize assignment may not narrow.");
    record->internalUsage = transfer->internalUsage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnTextureInternalUsageDescriptorDeserialize);

struct WGPUDawnWireWGSLControlTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUBool enableExperimental;
    WGPUBool enableUnsafe;
    WGPUBool enableTesting;
};

static_assert(offsetof(WGPUDawnWireWGSLControlTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnWireWGSLControlGetExtraRequiredSize([[maybe_unused]] const WGPUDawnWireWGSLControl& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnWireWGSLControlGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnWireWGSLControlSerialize(
    const WGPUDawnWireWGSLControl& record,
    WGPUDawnWireWGSLControlTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnWireWGSLControl);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->enableExperimental = record.enableExperimental;
    transfer->enableUnsafe = record.enableUnsafe;
    transfer->enableTesting = record.enableTesting;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnWireWGSLControlSerialize);

[[maybe_unused]] WireResult WGPUDawnWireWGSLControlDeserialize(
    WGPUDawnWireWGSLControl* record,
    const volatile WGPUDawnWireWGSLControlTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnWireWGSLControl);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->enableExperimental) >= sizeof(transfer->enableExperimental), "Deserialize assignment may not narrow.");
    record->enableExperimental = transfer->enableExperimental;
    static_assert(sizeof(record->enableUnsafe) >= sizeof(transfer->enableUnsafe), "Deserialize assignment may not narrow.");
    record->enableUnsafe = transfer->enableUnsafe;
    static_assert(sizeof(record->enableTesting) >= sizeof(transfer->enableTesting), "Deserialize assignment may not narrow.");
    record->enableTesting = transfer->enableTesting;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnWireWGSLControlDeserialize);

struct WGPUExtent2DTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint32_t width;
    uint32_t height;
};


[[maybe_unused]] size_t WGPUExtent2DGetExtraRequiredSize([[maybe_unused]] const WGPUExtent2D& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUExtent2DGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUExtent2DSerialize(
    const WGPUExtent2D& record,
    WGPUExtent2DTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->width = record.width;
    transfer->height = record.height;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExtent2DSerialize);

[[maybe_unused]] WireResult WGPUExtent2DDeserialize(
    WGPUExtent2D* record,
    const volatile WGPUExtent2DTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->width) >= sizeof(transfer->width), "Deserialize assignment may not narrow.");
    record->width = transfer->width;
    static_assert(sizeof(record->height) >= sizeof(transfer->height), "Deserialize assignment may not narrow.");
    record->height = transfer->height;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExtent2DDeserialize);

struct WGPUExtent3DTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint32_t width;
    uint32_t height;
    uint32_t depthOrArrayLayers;
};


[[maybe_unused]] size_t WGPUExtent3DGetExtraRequiredSize([[maybe_unused]] const WGPUExtent3D& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUExtent3DGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUExtent3DSerialize(
    const WGPUExtent3D& record,
    WGPUExtent3DTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->width = record.width;
    transfer->height = record.height;
    transfer->depthOrArrayLayers = record.depthOrArrayLayers;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExtent3DSerialize);

[[maybe_unused]] WireResult WGPUExtent3DDeserialize(
    WGPUExtent3D* record,
    const volatile WGPUExtent3DTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->width) >= sizeof(transfer->width), "Deserialize assignment may not narrow.");
    record->width = transfer->width;
    static_assert(sizeof(record->height) >= sizeof(transfer->height), "Deserialize assignment may not narrow.");
    record->height = transfer->height;
    static_assert(sizeof(record->depthOrArrayLayers) >= sizeof(transfer->depthOrArrayLayers), "Deserialize assignment may not narrow.");
    record->depthOrArrayLayers = transfer->depthOrArrayLayers;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExtent3DDeserialize);

struct WGPUExternalTextureBindingEntryTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    ObjectId externalTexture;
};

static_assert(offsetof(WGPUExternalTextureBindingEntryTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUExternalTextureBindingEntryGetExtraRequiredSize([[maybe_unused]] const WGPUExternalTextureBindingEntry& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUExternalTextureBindingEntryGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUExternalTextureBindingEntrySerialize(
    const WGPUExternalTextureBindingEntry& record,
    WGPUExternalTextureBindingEntryTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_ExternalTextureBindingEntry);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    WIRE_TRY(provider.GetId(record.externalTexture, &transfer->externalTexture));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExternalTextureBindingEntrySerialize);

[[maybe_unused]] WireResult WGPUExternalTextureBindingEntryDeserialize(
    WGPUExternalTextureBindingEntry* record,
    const volatile WGPUExternalTextureBindingEntryTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_ExternalTextureBindingEntry);
    DAWN_ASSERT(record->chain.next == nullptr);

    WIRE_TRY(resolver.GetFromId(transfer->externalTexture, &record->externalTexture));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExternalTextureBindingEntryDeserialize);

struct WGPUExternalTextureBindingLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

};

static_assert(offsetof(WGPUExternalTextureBindingLayoutTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUExternalTextureBindingLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUExternalTextureBindingLayout& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUExternalTextureBindingLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUExternalTextureBindingLayoutSerialize(
    const WGPUExternalTextureBindingLayout& record,
    WGPUExternalTextureBindingLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_ExternalTextureBindingLayout);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));


    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExternalTextureBindingLayoutSerialize);

[[maybe_unused]] WireResult WGPUExternalTextureBindingLayoutDeserialize(
    WGPUExternalTextureBindingLayout* record,
    const volatile WGPUExternalTextureBindingLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_ExternalTextureBindingLayout);
    DAWN_ASSERT(record->chain.next == nullptr);


    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExternalTextureBindingLayoutDeserialize);

struct WGPUFutureTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint64_t id;
};


[[maybe_unused]] size_t WGPUFutureGetExtraRequiredSize([[maybe_unused]] const WGPUFuture& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUFutureGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUFutureSerialize(
    const WGPUFuture& record,
    WGPUFutureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->id = record.id;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUFutureSerialize);

[[maybe_unused]] WireResult WGPUFutureDeserialize(
    WGPUFuture* record,
    const volatile WGPUFutureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->id) >= sizeof(transfer->id), "Deserialize assignment may not narrow.");
    record->id = transfer->id;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUFutureDeserialize);

struct WGPUInstanceCapabilitiesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUBool timedWaitAnyEnable;
    uint64_t timedWaitAnyMaxCount;
};


[[maybe_unused]] size_t WGPUInstanceCapabilitiesGetExtraRequiredSize([[maybe_unused]] const WGPUInstanceCapabilities& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUInstanceCapabilitiesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUInstanceCapabilitiesSerialize(
    const WGPUInstanceCapabilities& record,
    WGPUInstanceCapabilitiesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->timedWaitAnyEnable = record.timedWaitAnyEnable;
    transfer->timedWaitAnyMaxCount = record.timedWaitAnyMaxCount;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUInstanceCapabilitiesSerialize);

[[maybe_unused]] WireResult WGPUInstanceCapabilitiesDeserialize(
    WGPUInstanceCapabilities* record,
    const volatile WGPUInstanceCapabilitiesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->timedWaitAnyEnable) >= sizeof(transfer->timedWaitAnyEnable), "Deserialize assignment may not narrow.");
    record->timedWaitAnyEnable = transfer->timedWaitAnyEnable;
    if (transfer->timedWaitAnyMaxCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->timedWaitAnyMaxCount = checked_cast<size_t>(transfer->timedWaitAnyMaxCount);

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUInstanceCapabilitiesDeserialize);

struct WGPULimitsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint32_t maxTextureDimension1D;
    uint32_t maxTextureDimension2D;
    uint32_t maxTextureDimension3D;
    uint32_t maxTextureArrayLayers;
    uint32_t maxBindGroups;
    uint32_t maxBindGroupsPlusVertexBuffers;
    uint32_t maxBindingsPerBindGroup;
    uint32_t maxDynamicUniformBuffersPerPipelineLayout;
    uint32_t maxDynamicStorageBuffersPerPipelineLayout;
    uint32_t maxSampledTexturesPerShaderStage;
    uint32_t maxSamplersPerShaderStage;
    uint32_t maxStorageBuffersPerShaderStage;
    uint32_t maxStorageTexturesPerShaderStage;
    uint32_t maxUniformBuffersPerShaderStage;
    uint64_t maxUniformBufferBindingSize;
    uint64_t maxStorageBufferBindingSize;
    uint32_t minUniformBufferOffsetAlignment;
    uint32_t minStorageBufferOffsetAlignment;
    uint32_t maxVertexBuffers;
    uint64_t maxBufferSize;
    uint32_t maxVertexAttributes;
    uint32_t maxVertexBufferArrayStride;
    uint32_t maxInterStageShaderComponents;
    uint32_t maxInterStageShaderVariables;
    uint32_t maxColorAttachments;
    uint32_t maxColorAttachmentBytesPerSample;
    uint32_t maxComputeWorkgroupStorageSize;
    uint32_t maxComputeInvocationsPerWorkgroup;
    uint32_t maxComputeWorkgroupSizeX;
    uint32_t maxComputeWorkgroupSizeY;
    uint32_t maxComputeWorkgroupSizeZ;
    uint32_t maxComputeWorkgroupsPerDimension;
    uint32_t maxStorageBuffersInVertexStage;
    uint32_t maxStorageTexturesInVertexStage;
    uint32_t maxStorageBuffersInFragmentStage;
    uint32_t maxStorageTexturesInFragmentStage;
};


[[maybe_unused]] size_t WGPULimitsGetExtraRequiredSize([[maybe_unused]] const WGPULimits& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPULimitsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPULimitsSerialize(
    const WGPULimits& record,
    WGPULimitsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->maxTextureDimension1D = record.maxTextureDimension1D;
    transfer->maxTextureDimension2D = record.maxTextureDimension2D;
    transfer->maxTextureDimension3D = record.maxTextureDimension3D;
    transfer->maxTextureArrayLayers = record.maxTextureArrayLayers;
    transfer->maxBindGroups = record.maxBindGroups;
    transfer->maxBindGroupsPlusVertexBuffers = record.maxBindGroupsPlusVertexBuffers;
    transfer->maxBindingsPerBindGroup = record.maxBindingsPerBindGroup;
    transfer->maxDynamicUniformBuffersPerPipelineLayout = record.maxDynamicUniformBuffersPerPipelineLayout;
    transfer->maxDynamicStorageBuffersPerPipelineLayout = record.maxDynamicStorageBuffersPerPipelineLayout;
    transfer->maxSampledTexturesPerShaderStage = record.maxSampledTexturesPerShaderStage;
    transfer->maxSamplersPerShaderStage = record.maxSamplersPerShaderStage;
    transfer->maxStorageBuffersPerShaderStage = record.maxStorageBuffersPerShaderStage;
    transfer->maxStorageTexturesPerShaderStage = record.maxStorageTexturesPerShaderStage;
    transfer->maxUniformBuffersPerShaderStage = record.maxUniformBuffersPerShaderStage;
    transfer->maxUniformBufferBindingSize = record.maxUniformBufferBindingSize;
    transfer->maxStorageBufferBindingSize = record.maxStorageBufferBindingSize;
    transfer->minUniformBufferOffsetAlignment = record.minUniformBufferOffsetAlignment;
    transfer->minStorageBufferOffsetAlignment = record.minStorageBufferOffsetAlignment;
    transfer->maxVertexBuffers = record.maxVertexBuffers;
    transfer->maxBufferSize = record.maxBufferSize;
    transfer->maxVertexAttributes = record.maxVertexAttributes;
    transfer->maxVertexBufferArrayStride = record.maxVertexBufferArrayStride;
    transfer->maxInterStageShaderComponents = record.maxInterStageShaderComponents;
    transfer->maxInterStageShaderVariables = record.maxInterStageShaderVariables;
    transfer->maxColorAttachments = record.maxColorAttachments;
    transfer->maxColorAttachmentBytesPerSample = record.maxColorAttachmentBytesPerSample;
    transfer->maxComputeWorkgroupStorageSize = record.maxComputeWorkgroupStorageSize;
    transfer->maxComputeInvocationsPerWorkgroup = record.maxComputeInvocationsPerWorkgroup;
    transfer->maxComputeWorkgroupSizeX = record.maxComputeWorkgroupSizeX;
    transfer->maxComputeWorkgroupSizeY = record.maxComputeWorkgroupSizeY;
    transfer->maxComputeWorkgroupSizeZ = record.maxComputeWorkgroupSizeZ;
    transfer->maxComputeWorkgroupsPerDimension = record.maxComputeWorkgroupsPerDimension;
    transfer->maxStorageBuffersInVertexStage = record.maxStorageBuffersInVertexStage;
    transfer->maxStorageTexturesInVertexStage = record.maxStorageTexturesInVertexStage;
    transfer->maxStorageBuffersInFragmentStage = record.maxStorageBuffersInFragmentStage;
    transfer->maxStorageTexturesInFragmentStage = record.maxStorageTexturesInFragmentStage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPULimitsSerialize);

[[maybe_unused]] WireResult WGPULimitsDeserialize(
    WGPULimits* record,
    const volatile WGPULimitsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->maxTextureDimension1D) >= sizeof(transfer->maxTextureDimension1D), "Deserialize assignment may not narrow.");
    record->maxTextureDimension1D = transfer->maxTextureDimension1D;
    static_assert(sizeof(record->maxTextureDimension2D) >= sizeof(transfer->maxTextureDimension2D), "Deserialize assignment may not narrow.");
    record->maxTextureDimension2D = transfer->maxTextureDimension2D;
    static_assert(sizeof(record->maxTextureDimension3D) >= sizeof(transfer->maxTextureDimension3D), "Deserialize assignment may not narrow.");
    record->maxTextureDimension3D = transfer->maxTextureDimension3D;
    static_assert(sizeof(record->maxTextureArrayLayers) >= sizeof(transfer->maxTextureArrayLayers), "Deserialize assignment may not narrow.");
    record->maxTextureArrayLayers = transfer->maxTextureArrayLayers;
    static_assert(sizeof(record->maxBindGroups) >= sizeof(transfer->maxBindGroups), "Deserialize assignment may not narrow.");
    record->maxBindGroups = transfer->maxBindGroups;
    static_assert(sizeof(record->maxBindGroupsPlusVertexBuffers) >= sizeof(transfer->maxBindGroupsPlusVertexBuffers), "Deserialize assignment may not narrow.");
    record->maxBindGroupsPlusVertexBuffers = transfer->maxBindGroupsPlusVertexBuffers;
    static_assert(sizeof(record->maxBindingsPerBindGroup) >= sizeof(transfer->maxBindingsPerBindGroup), "Deserialize assignment may not narrow.");
    record->maxBindingsPerBindGroup = transfer->maxBindingsPerBindGroup;
    static_assert(sizeof(record->maxDynamicUniformBuffersPerPipelineLayout) >= sizeof(transfer->maxDynamicUniformBuffersPerPipelineLayout), "Deserialize assignment may not narrow.");
    record->maxDynamicUniformBuffersPerPipelineLayout = transfer->maxDynamicUniformBuffersPerPipelineLayout;
    static_assert(sizeof(record->maxDynamicStorageBuffersPerPipelineLayout) >= sizeof(transfer->maxDynamicStorageBuffersPerPipelineLayout), "Deserialize assignment may not narrow.");
    record->maxDynamicStorageBuffersPerPipelineLayout = transfer->maxDynamicStorageBuffersPerPipelineLayout;
    static_assert(sizeof(record->maxSampledTexturesPerShaderStage) >= sizeof(transfer->maxSampledTexturesPerShaderStage), "Deserialize assignment may not narrow.");
    record->maxSampledTexturesPerShaderStage = transfer->maxSampledTexturesPerShaderStage;
    static_assert(sizeof(record->maxSamplersPerShaderStage) >= sizeof(transfer->maxSamplersPerShaderStage), "Deserialize assignment may not narrow.");
    record->maxSamplersPerShaderStage = transfer->maxSamplersPerShaderStage;
    static_assert(sizeof(record->maxStorageBuffersPerShaderStage) >= sizeof(transfer->maxStorageBuffersPerShaderStage), "Deserialize assignment may not narrow.");
    record->maxStorageBuffersPerShaderStage = transfer->maxStorageBuffersPerShaderStage;
    static_assert(sizeof(record->maxStorageTexturesPerShaderStage) >= sizeof(transfer->maxStorageTexturesPerShaderStage), "Deserialize assignment may not narrow.");
    record->maxStorageTexturesPerShaderStage = transfer->maxStorageTexturesPerShaderStage;
    static_assert(sizeof(record->maxUniformBuffersPerShaderStage) >= sizeof(transfer->maxUniformBuffersPerShaderStage), "Deserialize assignment may not narrow.");
    record->maxUniformBuffersPerShaderStage = transfer->maxUniformBuffersPerShaderStage;
    static_assert(sizeof(record->maxUniformBufferBindingSize) >= sizeof(transfer->maxUniformBufferBindingSize), "Deserialize assignment may not narrow.");
    record->maxUniformBufferBindingSize = transfer->maxUniformBufferBindingSize;
    static_assert(sizeof(record->maxStorageBufferBindingSize) >= sizeof(transfer->maxStorageBufferBindingSize), "Deserialize assignment may not narrow.");
    record->maxStorageBufferBindingSize = transfer->maxStorageBufferBindingSize;
    static_assert(sizeof(record->minUniformBufferOffsetAlignment) >= sizeof(transfer->minUniformBufferOffsetAlignment), "Deserialize assignment may not narrow.");
    record->minUniformBufferOffsetAlignment = transfer->minUniformBufferOffsetAlignment;
    static_assert(sizeof(record->minStorageBufferOffsetAlignment) >= sizeof(transfer->minStorageBufferOffsetAlignment), "Deserialize assignment may not narrow.");
    record->minStorageBufferOffsetAlignment = transfer->minStorageBufferOffsetAlignment;
    static_assert(sizeof(record->maxVertexBuffers) >= sizeof(transfer->maxVertexBuffers), "Deserialize assignment may not narrow.");
    record->maxVertexBuffers = transfer->maxVertexBuffers;
    static_assert(sizeof(record->maxBufferSize) >= sizeof(transfer->maxBufferSize), "Deserialize assignment may not narrow.");
    record->maxBufferSize = transfer->maxBufferSize;
    static_assert(sizeof(record->maxVertexAttributes) >= sizeof(transfer->maxVertexAttributes), "Deserialize assignment may not narrow.");
    record->maxVertexAttributes = transfer->maxVertexAttributes;
    static_assert(sizeof(record->maxVertexBufferArrayStride) >= sizeof(transfer->maxVertexBufferArrayStride), "Deserialize assignment may not narrow.");
    record->maxVertexBufferArrayStride = transfer->maxVertexBufferArrayStride;
    static_assert(sizeof(record->maxInterStageShaderComponents) >= sizeof(transfer->maxInterStageShaderComponents), "Deserialize assignment may not narrow.");
    record->maxInterStageShaderComponents = transfer->maxInterStageShaderComponents;
    static_assert(sizeof(record->maxInterStageShaderVariables) >= sizeof(transfer->maxInterStageShaderVariables), "Deserialize assignment may not narrow.");
    record->maxInterStageShaderVariables = transfer->maxInterStageShaderVariables;
    static_assert(sizeof(record->maxColorAttachments) >= sizeof(transfer->maxColorAttachments), "Deserialize assignment may not narrow.");
    record->maxColorAttachments = transfer->maxColorAttachments;
    static_assert(sizeof(record->maxColorAttachmentBytesPerSample) >= sizeof(transfer->maxColorAttachmentBytesPerSample), "Deserialize assignment may not narrow.");
    record->maxColorAttachmentBytesPerSample = transfer->maxColorAttachmentBytesPerSample;
    static_assert(sizeof(record->maxComputeWorkgroupStorageSize) >= sizeof(transfer->maxComputeWorkgroupStorageSize), "Deserialize assignment may not narrow.");
    record->maxComputeWorkgroupStorageSize = transfer->maxComputeWorkgroupStorageSize;
    static_assert(sizeof(record->maxComputeInvocationsPerWorkgroup) >= sizeof(transfer->maxComputeInvocationsPerWorkgroup), "Deserialize assignment may not narrow.");
    record->maxComputeInvocationsPerWorkgroup = transfer->maxComputeInvocationsPerWorkgroup;
    static_assert(sizeof(record->maxComputeWorkgroupSizeX) >= sizeof(transfer->maxComputeWorkgroupSizeX), "Deserialize assignment may not narrow.");
    record->maxComputeWorkgroupSizeX = transfer->maxComputeWorkgroupSizeX;
    static_assert(sizeof(record->maxComputeWorkgroupSizeY) >= sizeof(transfer->maxComputeWorkgroupSizeY), "Deserialize assignment may not narrow.");
    record->maxComputeWorkgroupSizeY = transfer->maxComputeWorkgroupSizeY;
    static_assert(sizeof(record->maxComputeWorkgroupSizeZ) >= sizeof(transfer->maxComputeWorkgroupSizeZ), "Deserialize assignment may not narrow.");
    record->maxComputeWorkgroupSizeZ = transfer->maxComputeWorkgroupSizeZ;
    static_assert(sizeof(record->maxComputeWorkgroupsPerDimension) >= sizeof(transfer->maxComputeWorkgroupsPerDimension), "Deserialize assignment may not narrow.");
    record->maxComputeWorkgroupsPerDimension = transfer->maxComputeWorkgroupsPerDimension;
    static_assert(sizeof(record->maxStorageBuffersInVertexStage) >= sizeof(transfer->maxStorageBuffersInVertexStage), "Deserialize assignment may not narrow.");
    record->maxStorageBuffersInVertexStage = transfer->maxStorageBuffersInVertexStage;
    static_assert(sizeof(record->maxStorageTexturesInVertexStage) >= sizeof(transfer->maxStorageTexturesInVertexStage), "Deserialize assignment may not narrow.");
    record->maxStorageTexturesInVertexStage = transfer->maxStorageTexturesInVertexStage;
    static_assert(sizeof(record->maxStorageBuffersInFragmentStage) >= sizeof(transfer->maxStorageBuffersInFragmentStage), "Deserialize assignment may not narrow.");
    record->maxStorageBuffersInFragmentStage = transfer->maxStorageBuffersInFragmentStage;
    static_assert(sizeof(record->maxStorageTexturesInFragmentStage) >= sizeof(transfer->maxStorageTexturesInFragmentStage), "Deserialize assignment may not narrow.");
    record->maxStorageTexturesInFragmentStage = transfer->maxStorageTexturesInFragmentStage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPULimitsDeserialize);

struct WGPUMemoryHeapInfoTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    WGPUHeapProperty properties;
    uint64_t size;
};


[[maybe_unused]] size_t WGPUMemoryHeapInfoGetExtraRequiredSize([[maybe_unused]] const WGPUMemoryHeapInfo& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUMemoryHeapInfoGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUMemoryHeapInfoSerialize(
    const WGPUMemoryHeapInfo& record,
    WGPUMemoryHeapInfoTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->properties = record.properties;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUMemoryHeapInfoSerialize);

[[maybe_unused]] WireResult WGPUMemoryHeapInfoDeserialize(
    WGPUMemoryHeapInfo* record,
    const volatile WGPUMemoryHeapInfoTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->properties) >= sizeof(transfer->properties), "Deserialize assignment may not narrow.");
    record->properties = transfer->properties;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUMemoryHeapInfoDeserialize);

struct WGPUMultisampleStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    uint32_t count;
    uint32_t mask;
    WGPUBool alphaToCoverageEnabled;
};


[[maybe_unused]] size_t WGPUMultisampleStateGetExtraRequiredSize([[maybe_unused]] const WGPUMultisampleState& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUMultisampleStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUMultisampleStateSerialize(
    const WGPUMultisampleState& record,
    WGPUMultisampleStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->count = record.count;
    transfer->mask = record.mask;
    transfer->alphaToCoverageEnabled = record.alphaToCoverageEnabled;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUMultisampleStateSerialize);

[[maybe_unused]] WireResult WGPUMultisampleStateDeserialize(
    WGPUMultisampleState* record,
    const volatile WGPUMultisampleStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->count) >= sizeof(transfer->count), "Deserialize assignment may not narrow.");
    record->count = transfer->count;
    static_assert(sizeof(record->mask) >= sizeof(transfer->mask), "Deserialize assignment may not narrow.");
    record->mask = transfer->mask;
    static_assert(sizeof(record->alphaToCoverageEnabled) >= sizeof(transfer->alphaToCoverageEnabled), "Deserialize assignment may not narrow.");
    record->alphaToCoverageEnabled = transfer->alphaToCoverageEnabled;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUMultisampleStateDeserialize);

struct WGPUOrigin2DTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint32_t x;
    uint32_t y;
};


[[maybe_unused]] size_t WGPUOrigin2DGetExtraRequiredSize([[maybe_unused]] const WGPUOrigin2D& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUOrigin2DGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUOrigin2DSerialize(
    const WGPUOrigin2D& record,
    WGPUOrigin2DTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->x = record.x;
    transfer->y = record.y;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUOrigin2DSerialize);

[[maybe_unused]] WireResult WGPUOrigin2DDeserialize(
    WGPUOrigin2D* record,
    const volatile WGPUOrigin2DTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->x) >= sizeof(transfer->x), "Deserialize assignment may not narrow.");
    record->x = transfer->x;
    static_assert(sizeof(record->y) >= sizeof(transfer->y), "Deserialize assignment may not narrow.");
    record->y = transfer->y;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUOrigin2DDeserialize);

struct WGPUOrigin3DTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint32_t x;
    uint32_t y;
    uint32_t z;
};


[[maybe_unused]] size_t WGPUOrigin3DGetExtraRequiredSize([[maybe_unused]] const WGPUOrigin3D& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUOrigin3DGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUOrigin3DSerialize(
    const WGPUOrigin3D& record,
    WGPUOrigin3DTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->x = record.x;
    transfer->y = record.y;
    transfer->z = record.z;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUOrigin3DSerialize);

[[maybe_unused]] WireResult WGPUOrigin3DDeserialize(
    WGPUOrigin3D* record,
    const volatile WGPUOrigin3DTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->x) >= sizeof(transfer->x), "Deserialize assignment may not narrow.");
    record->x = transfer->x;
    static_assert(sizeof(record->y) >= sizeof(transfer->y), "Deserialize assignment may not narrow.");
    record->y = transfer->y;
    static_assert(sizeof(record->z) >= sizeof(transfer->z), "Deserialize assignment may not narrow.");
    record->z = transfer->z;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUOrigin3DDeserialize);

struct WGPUPassTimestampWritesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId querySet;
    uint32_t beginningOfPassWriteIndex;
    uint32_t endOfPassWriteIndex;
};


[[maybe_unused]] size_t WGPUPassTimestampWritesGetExtraRequiredSize([[maybe_unused]] const WGPUPassTimestampWrites& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUPassTimestampWritesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUPassTimestampWritesSerialize(
    const WGPUPassTimestampWrites& record,
    WGPUPassTimestampWritesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetId(record.querySet, &transfer->querySet));
    transfer->beginningOfPassWriteIndex = record.beginningOfPassWriteIndex;
    transfer->endOfPassWriteIndex = record.endOfPassWriteIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPassTimestampWritesSerialize);

[[maybe_unused]] WireResult WGPUPassTimestampWritesDeserialize(
    WGPUPassTimestampWrites* record,
    const volatile WGPUPassTimestampWritesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetFromId(transfer->querySet, &record->querySet));
    static_assert(sizeof(record->beginningOfPassWriteIndex) >= sizeof(transfer->beginningOfPassWriteIndex), "Deserialize assignment may not narrow.");
    record->beginningOfPassWriteIndex = transfer->beginningOfPassWriteIndex;
    static_assert(sizeof(record->endOfPassWriteIndex) >= sizeof(transfer->endOfPassWriteIndex), "Deserialize assignment may not narrow.");
    record->endOfPassWriteIndex = transfer->endOfPassWriteIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPassTimestampWritesDeserialize);

struct WGPUPipelineLayoutStorageAttachmentTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint64_t offset;
    WGPUTextureFormat format;
};


[[maybe_unused]] size_t WGPUPipelineLayoutStorageAttachmentGetExtraRequiredSize([[maybe_unused]] const WGPUPipelineLayoutStorageAttachment& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUPipelineLayoutStorageAttachmentGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUPipelineLayoutStorageAttachmentSerialize(
    const WGPUPipelineLayoutStorageAttachment& record,
    WGPUPipelineLayoutStorageAttachmentTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->offset = record.offset;
    transfer->format = record.format;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPipelineLayoutStorageAttachmentSerialize);

[[maybe_unused]] WireResult WGPUPipelineLayoutStorageAttachmentDeserialize(
    WGPUPipelineLayoutStorageAttachment* record,
    const volatile WGPUPipelineLayoutStorageAttachmentTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPipelineLayoutStorageAttachmentDeserialize);

struct WGPUPrimitiveStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUPrimitiveTopology topology;
    WGPUIndexFormat stripIndexFormat;
    WGPUFrontFace frontFace;
    WGPUCullMode cullMode;
    WGPUBool unclippedDepth;
};


[[maybe_unused]] size_t WGPUPrimitiveStateGetExtraRequiredSize([[maybe_unused]] const WGPUPrimitiveState& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUPrimitiveStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUPrimitiveStateSerialize(
    const WGPUPrimitiveState& record,
    WGPUPrimitiveStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->topology = record.topology;
    transfer->stripIndexFormat = record.stripIndexFormat;
    transfer->frontFace = record.frontFace;
    transfer->cullMode = record.cullMode;
    transfer->unclippedDepth = record.unclippedDepth;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPrimitiveStateSerialize);

[[maybe_unused]] WireResult WGPUPrimitiveStateDeserialize(
    WGPUPrimitiveState* record,
    const volatile WGPUPrimitiveStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->topology) >= sizeof(transfer->topology), "Deserialize assignment may not narrow.");
    record->topology = transfer->topology;
    static_assert(sizeof(record->stripIndexFormat) >= sizeof(transfer->stripIndexFormat), "Deserialize assignment may not narrow.");
    record->stripIndexFormat = transfer->stripIndexFormat;
    static_assert(sizeof(record->frontFace) >= sizeof(transfer->frontFace), "Deserialize assignment may not narrow.");
    record->frontFace = transfer->frontFace;
    static_assert(sizeof(record->cullMode) >= sizeof(transfer->cullMode), "Deserialize assignment may not narrow.");
    record->cullMode = transfer->cullMode;
    static_assert(sizeof(record->unclippedDepth) >= sizeof(transfer->unclippedDepth), "Deserialize assignment may not narrow.");
    record->unclippedDepth = transfer->unclippedDepth;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPrimitiveStateDeserialize);

struct WGPURenderPassDepthStencilAttachmentTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId view;
    WGPULoadOp depthLoadOp;
    WGPUStoreOp depthStoreOp;
    float depthClearValue;
    WGPUBool depthReadOnly;
    WGPULoadOp stencilLoadOp;
    WGPUStoreOp stencilStoreOp;
    uint32_t stencilClearValue;
    WGPUBool stencilReadOnly;
};


[[maybe_unused]] size_t WGPURenderPassDepthStencilAttachmentGetExtraRequiredSize([[maybe_unused]] const WGPURenderPassDepthStencilAttachment& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPassDepthStencilAttachmentGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPassDepthStencilAttachmentSerialize(
    const WGPURenderPassDepthStencilAttachment& record,
    WGPURenderPassDepthStencilAttachmentTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetId(record.view, &transfer->view));
    transfer->depthLoadOp = record.depthLoadOp;
    transfer->depthStoreOp = record.depthStoreOp;
    transfer->depthClearValue = record.depthClearValue;
    transfer->depthReadOnly = record.depthReadOnly;
    transfer->stencilLoadOp = record.stencilLoadOp;
    transfer->stencilStoreOp = record.stencilStoreOp;
    transfer->stencilClearValue = record.stencilClearValue;
    transfer->stencilReadOnly = record.stencilReadOnly;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassDepthStencilAttachmentSerialize);

[[maybe_unused]] WireResult WGPURenderPassDepthStencilAttachmentDeserialize(
    WGPURenderPassDepthStencilAttachment* record,
    const volatile WGPURenderPassDepthStencilAttachmentTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetFromId(transfer->view, &record->view));
    static_assert(sizeof(record->depthLoadOp) >= sizeof(transfer->depthLoadOp), "Deserialize assignment may not narrow.");
    record->depthLoadOp = transfer->depthLoadOp;
    static_assert(sizeof(record->depthStoreOp) >= sizeof(transfer->depthStoreOp), "Deserialize assignment may not narrow.");
    record->depthStoreOp = transfer->depthStoreOp;
    static_assert(sizeof(record->depthClearValue) >= sizeof(transfer->depthClearValue), "Deserialize assignment may not narrow.");
    record->depthClearValue = transfer->depthClearValue;
    static_assert(sizeof(record->depthReadOnly) >= sizeof(transfer->depthReadOnly), "Deserialize assignment may not narrow.");
    record->depthReadOnly = transfer->depthReadOnly;
    static_assert(sizeof(record->stencilLoadOp) >= sizeof(transfer->stencilLoadOp), "Deserialize assignment may not narrow.");
    record->stencilLoadOp = transfer->stencilLoadOp;
    static_assert(sizeof(record->stencilStoreOp) >= sizeof(transfer->stencilStoreOp), "Deserialize assignment may not narrow.");
    record->stencilStoreOp = transfer->stencilStoreOp;
    static_assert(sizeof(record->stencilClearValue) >= sizeof(transfer->stencilClearValue), "Deserialize assignment may not narrow.");
    record->stencilClearValue = transfer->stencilClearValue;
    static_assert(sizeof(record->stencilReadOnly) >= sizeof(transfer->stencilReadOnly), "Deserialize assignment may not narrow.");
    record->stencilReadOnly = transfer->stencilReadOnly;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassDepthStencilAttachmentDeserialize);

struct WGPURenderPassDescriptorExpandResolveRectTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t x;
    uint32_t y;
    uint32_t width;
    uint32_t height;
};

static_assert(offsetof(WGPURenderPassDescriptorExpandResolveRectTransfer, chain) == 0);

[[maybe_unused]] size_t WGPURenderPassDescriptorExpandResolveRectGetExtraRequiredSize([[maybe_unused]] const WGPURenderPassDescriptorExpandResolveRect& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPassDescriptorExpandResolveRectGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPassDescriptorExpandResolveRectSerialize(
    const WGPURenderPassDescriptorExpandResolveRect& record,
    WGPURenderPassDescriptorExpandResolveRectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_RenderPassDescriptorExpandResolveRect);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->x = record.x;
    transfer->y = record.y;
    transfer->width = record.width;
    transfer->height = record.height;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassDescriptorExpandResolveRectSerialize);

[[maybe_unused]] WireResult WGPURenderPassDescriptorExpandResolveRectDeserialize(
    WGPURenderPassDescriptorExpandResolveRect* record,
    const volatile WGPURenderPassDescriptorExpandResolveRectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_RenderPassDescriptorExpandResolveRect);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->x) >= sizeof(transfer->x), "Deserialize assignment may not narrow.");
    record->x = transfer->x;
    static_assert(sizeof(record->y) >= sizeof(transfer->y), "Deserialize assignment may not narrow.");
    record->y = transfer->y;
    static_assert(sizeof(record->width) >= sizeof(transfer->width), "Deserialize assignment may not narrow.");
    record->width = transfer->width;
    static_assert(sizeof(record->height) >= sizeof(transfer->height), "Deserialize assignment may not narrow.");
    record->height = transfer->height;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassDescriptorExpandResolveRectDeserialize);

struct WGPURenderPassMaxDrawCountTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint64_t maxDrawCount;
};

static_assert(offsetof(WGPURenderPassMaxDrawCountTransfer, chain) == 0);

[[maybe_unused]] size_t WGPURenderPassMaxDrawCountGetExtraRequiredSize([[maybe_unused]] const WGPURenderPassMaxDrawCount& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPassMaxDrawCountGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPassMaxDrawCountSerialize(
    const WGPURenderPassMaxDrawCount& record,
    WGPURenderPassMaxDrawCountTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_RenderPassMaxDrawCount);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->maxDrawCount = record.maxDrawCount;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassMaxDrawCountSerialize);

[[maybe_unused]] WireResult WGPURenderPassMaxDrawCountDeserialize(
    WGPURenderPassMaxDrawCount* record,
    const volatile WGPURenderPassMaxDrawCountTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_RenderPassMaxDrawCount);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->maxDrawCount) >= sizeof(transfer->maxDrawCount), "Deserialize assignment may not narrow.");
    record->maxDrawCount = transfer->maxDrawCount;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassMaxDrawCountDeserialize);

struct WGPURequestAdapterOptionsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId compatibleSurface;
    WGPUFeatureLevel featureLevel;
    WGPUPowerPreference powerPreference;
    WGPUBackendType backendType;
    WGPUBool forceFallbackAdapter;
};


[[maybe_unused]] size_t WGPURequestAdapterOptionsGetExtraRequiredSize([[maybe_unused]] const WGPURequestAdapterOptions& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURequestAdapterOptionsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURequestAdapterOptionsSerialize(
    const WGPURequestAdapterOptions& record,
    WGPURequestAdapterOptionsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetOptionalId(record.compatibleSurface, &transfer->compatibleSurface));
    transfer->featureLevel = record.featureLevel;
    transfer->powerPreference = record.powerPreference;
    transfer->backendType = record.backendType;
    transfer->forceFallbackAdapter = record.forceFallbackAdapter;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURequestAdapterOptionsSerialize);

[[maybe_unused]] WireResult WGPURequestAdapterOptionsDeserialize(
    WGPURequestAdapterOptions* record,
    const volatile WGPURequestAdapterOptionsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetOptionalFromId(transfer->compatibleSurface, &record->compatibleSurface));
    static_assert(sizeof(record->featureLevel) >= sizeof(transfer->featureLevel), "Deserialize assignment may not narrow.");
    record->featureLevel = transfer->featureLevel;
    static_assert(sizeof(record->powerPreference) >= sizeof(transfer->powerPreference), "Deserialize assignment may not narrow.");
    record->powerPreference = transfer->powerPreference;
    static_assert(sizeof(record->backendType) >= sizeof(transfer->backendType), "Deserialize assignment may not narrow.");
    record->backendType = transfer->backendType;
    static_assert(sizeof(record->forceFallbackAdapter) >= sizeof(transfer->forceFallbackAdapter), "Deserialize assignment may not narrow.");
    record->forceFallbackAdapter = transfer->forceFallbackAdapter;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURequestAdapterOptionsDeserialize);

struct WGPUSamplerBindingLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUSamplerBindingType type;
};


[[maybe_unused]] size_t WGPUSamplerBindingLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUSamplerBindingLayout& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSamplerBindingLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSamplerBindingLayoutSerialize(
    const WGPUSamplerBindingLayout& record,
    WGPUSamplerBindingLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->type = record.type;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSamplerBindingLayoutSerialize);

[[maybe_unused]] WireResult WGPUSamplerBindingLayoutDeserialize(
    WGPUSamplerBindingLayout* record,
    const volatile WGPUSamplerBindingLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSamplerBindingLayoutDeserialize);

struct WGPUShaderModuleCompilationOptionsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUBool strictMath;
};

static_assert(offsetof(WGPUShaderModuleCompilationOptionsTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUShaderModuleCompilationOptionsGetExtraRequiredSize([[maybe_unused]] const WGPUShaderModuleCompilationOptions& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUShaderModuleCompilationOptionsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUShaderModuleCompilationOptionsSerialize(
    const WGPUShaderModuleCompilationOptions& record,
    WGPUShaderModuleCompilationOptionsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_ShaderModuleCompilationOptions);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->strictMath = record.strictMath;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderModuleCompilationOptionsSerialize);

[[maybe_unused]] WireResult WGPUShaderModuleCompilationOptionsDeserialize(
    WGPUShaderModuleCompilationOptions* record,
    const volatile WGPUShaderModuleCompilationOptionsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_ShaderModuleCompilationOptions);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->strictMath) >= sizeof(transfer->strictMath), "Deserialize assignment may not narrow.");
    record->strictMath = transfer->strictMath;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderModuleCompilationOptionsDeserialize);

struct WGPUShaderSourceSPIRVTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t codeSize;
};

static_assert(offsetof(WGPUShaderSourceSPIRVTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUShaderSourceSPIRVGetExtraRequiredSize([[maybe_unused]] const WGPUShaderSourceSPIRV& record) {
    size_t result = 0;

    {
        auto memberLength = record.codeSize;
        auto size = WireAlignSizeofN<uint32_t>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUShaderSourceSPIRVGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUShaderSourceSPIRVSerialize(
    const WGPUShaderSourceSPIRV& record,
    WGPUShaderSourceSPIRVTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_ShaderSourceSPIRV);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->codeSize = record.codeSize;
    {
        auto memberLength = record.codeSize;

        uint32_t* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.code,
                sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderSourceSPIRVSerialize);

[[maybe_unused]] WireResult WGPUShaderSourceSPIRVDeserialize(
    WGPUShaderSourceSPIRV* record,
    const volatile WGPUShaderSourceSPIRVTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_ShaderSourceSPIRV);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->codeSize) >= sizeof(transfer->codeSize), "Deserialize assignment may not narrow.");
    record->codeSize = transfer->codeSize;
    {
        auto memberLength = record->codeSize;
        const volatile uint32_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint32_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->code = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint32_t*>(memberBuffer),
              sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderSourceSPIRVDeserialize);

struct WGPUStaticSamplerBindingLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    ObjectId sampler;
    uint32_t sampledTextureBinding;
};

static_assert(offsetof(WGPUStaticSamplerBindingLayoutTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUStaticSamplerBindingLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUStaticSamplerBindingLayout& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUStaticSamplerBindingLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUStaticSamplerBindingLayoutSerialize(
    const WGPUStaticSamplerBindingLayout& record,
    WGPUStaticSamplerBindingLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_StaticSamplerBindingLayout);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    WIRE_TRY(provider.GetId(record.sampler, &transfer->sampler));
    transfer->sampledTextureBinding = record.sampledTextureBinding;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUStaticSamplerBindingLayoutSerialize);

[[maybe_unused]] WireResult WGPUStaticSamplerBindingLayoutDeserialize(
    WGPUStaticSamplerBindingLayout* record,
    const volatile WGPUStaticSamplerBindingLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_StaticSamplerBindingLayout);
    DAWN_ASSERT(record->chain.next == nullptr);

    WIRE_TRY(resolver.GetFromId(transfer->sampler, &record->sampler));
    static_assert(sizeof(record->sampledTextureBinding) >= sizeof(transfer->sampledTextureBinding), "Deserialize assignment may not narrow.");
    record->sampledTextureBinding = transfer->sampledTextureBinding;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUStaticSamplerBindingLayoutDeserialize);

struct WGPUStencilFaceStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    WGPUCompareFunction compare;
    WGPUStencilOperation failOp;
    WGPUStencilOperation depthFailOp;
    WGPUStencilOperation passOp;
};


[[maybe_unused]] size_t WGPUStencilFaceStateGetExtraRequiredSize([[maybe_unused]] const WGPUStencilFaceState& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUStencilFaceStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUStencilFaceStateSerialize(
    const WGPUStencilFaceState& record,
    WGPUStencilFaceStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->compare = record.compare;
    transfer->failOp = record.failOp;
    transfer->depthFailOp = record.depthFailOp;
    transfer->passOp = record.passOp;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUStencilFaceStateSerialize);

[[maybe_unused]] WireResult WGPUStencilFaceStateDeserialize(
    WGPUStencilFaceState* record,
    const volatile WGPUStencilFaceStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->compare) >= sizeof(transfer->compare), "Deserialize assignment may not narrow.");
    record->compare = transfer->compare;
    static_assert(sizeof(record->failOp) >= sizeof(transfer->failOp), "Deserialize assignment may not narrow.");
    record->failOp = transfer->failOp;
    static_assert(sizeof(record->depthFailOp) >= sizeof(transfer->depthFailOp), "Deserialize assignment may not narrow.");
    record->depthFailOp = transfer->depthFailOp;
    static_assert(sizeof(record->passOp) >= sizeof(transfer->passOp), "Deserialize assignment may not narrow.");
    record->passOp = transfer->passOp;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUStencilFaceStateDeserialize);

struct WGPUStorageTextureBindingLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStorageTextureAccess access;
    WGPUTextureFormat format;
    WGPUTextureViewDimension viewDimension;
};


[[maybe_unused]] size_t WGPUStorageTextureBindingLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUStorageTextureBindingLayout& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUStorageTextureBindingLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUStorageTextureBindingLayoutSerialize(
    const WGPUStorageTextureBindingLayout& record,
    WGPUStorageTextureBindingLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->access = record.access;
    transfer->format = record.format;
    transfer->viewDimension = record.viewDimension;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUStorageTextureBindingLayoutSerialize);

[[maybe_unused]] WireResult WGPUStorageTextureBindingLayoutDeserialize(
    WGPUStorageTextureBindingLayout* record,
    const volatile WGPUStorageTextureBindingLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->access) >= sizeof(transfer->access), "Deserialize assignment may not narrow.");
    record->access = transfer->access;
    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->viewDimension) >= sizeof(transfer->viewDimension), "Deserialize assignment may not narrow.");
    record->viewDimension = transfer->viewDimension;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUStorageTextureBindingLayoutDeserialize);

struct WGPUSupportedWGSLLanguageFeaturesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint64_t featureCount;
};


[[maybe_unused]] size_t WGPUSupportedWGSLLanguageFeaturesGetExtraRequiredSize([[maybe_unused]] const WGPUSupportedWGSLLanguageFeatures& record) {
    size_t result = 0;

    {
        auto memberLength = record.featureCount;
        auto size = WireAlignSizeofN<WGPUWGSLLanguageFeatureName>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSupportedWGSLLanguageFeaturesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSupportedWGSLLanguageFeaturesSerialize(
    const WGPUSupportedWGSLLanguageFeatures& record,
    WGPUSupportedWGSLLanguageFeaturesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->featureCount = record.featureCount;
    {
        auto memberLength = record.featureCount;

        WGPUWGSLLanguageFeatureName* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.features,
                sizeof(WGPUWGSLLanguageFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSupportedWGSLLanguageFeaturesSerialize);

[[maybe_unused]] WireResult WGPUSupportedWGSLLanguageFeaturesDeserialize(
    WGPUSupportedWGSLLanguageFeatures* record,
    const volatile WGPUSupportedWGSLLanguageFeaturesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    if (transfer->featureCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->featureCount = checked_cast<size_t>(transfer->featureCount);
    {
        auto memberLength = record->featureCount;
        const volatile WGPUWGSLLanguageFeatureName* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUWGSLLanguageFeatureName* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->features = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUWGSLLanguageFeatureName*>(memberBuffer),
              sizeof(WGPUWGSLLanguageFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSupportedWGSLLanguageFeaturesDeserialize);

struct WGPUSupportedFeaturesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint64_t featureCount;
};


[[maybe_unused]] size_t WGPUSupportedFeaturesGetExtraRequiredSize([[maybe_unused]] const WGPUSupportedFeatures& record) {
    size_t result = 0;

    {
        auto memberLength = record.featureCount;
        auto size = WireAlignSizeofN<WGPUFeatureName>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSupportedFeaturesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSupportedFeaturesSerialize(
    const WGPUSupportedFeatures& record,
    WGPUSupportedFeaturesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->featureCount = record.featureCount;
    {
        auto memberLength = record.featureCount;

        WGPUFeatureName* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.features,
                sizeof(WGPUFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSupportedFeaturesSerialize);

[[maybe_unused]] WireResult WGPUSupportedFeaturesDeserialize(
    WGPUSupportedFeatures* record,
    const volatile WGPUSupportedFeaturesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    if (transfer->featureCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->featureCount = checked_cast<size_t>(transfer->featureCount);
    {
        auto memberLength = record->featureCount;
        const volatile WGPUFeatureName* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUFeatureName* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->features = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUFeatureName*>(memberBuffer),
              sizeof(WGPUFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSupportedFeaturesDeserialize);

struct WGPUSurfaceCapabilitiesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUTextureUsage usages;
    uint64_t formatCount;
    uint64_t presentModeCount;
    uint64_t alphaModeCount;
};


[[maybe_unused]] size_t WGPUSurfaceCapabilitiesGetExtraRequiredSize([[maybe_unused]] const WGPUSurfaceCapabilities& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    {
        auto memberLength = record.formatCount;
        auto size = WireAlignSizeofN<WGPUTextureFormat>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    {
        auto memberLength = record.presentModeCount;
        auto size = WireAlignSizeofN<WGPUPresentMode>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    {
        auto memberLength = record.alphaModeCount;
        auto size = WireAlignSizeofN<WGPUCompositeAlphaMode>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSurfaceCapabilitiesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSurfaceCapabilitiesSerialize(
    const WGPUSurfaceCapabilities& record,
    WGPUSurfaceCapabilitiesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->usages = record.usages;
    transfer->formatCount = record.formatCount;
    transfer->presentModeCount = record.presentModeCount;
    transfer->alphaModeCount = record.alphaModeCount;
    {
        auto memberLength = record.formatCount;

        WGPUTextureFormat* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.formats,
                sizeof(WGPUTextureFormat) * memberLength);
        }
    }
    {
        auto memberLength = record.presentModeCount;

        WGPUPresentMode* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.presentModes,
                sizeof(WGPUPresentMode) * memberLength);
        }
    }
    {
        auto memberLength = record.alphaModeCount;

        WGPUCompositeAlphaMode* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.alphaModes,
                sizeof(WGPUCompositeAlphaMode) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceCapabilitiesSerialize);

[[maybe_unused]] WireResult WGPUSurfaceCapabilitiesDeserialize(
    WGPUSurfaceCapabilities* record,
    const volatile WGPUSurfaceCapabilitiesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->usages) >= sizeof(transfer->usages), "Deserialize assignment may not narrow.");
    record->usages = transfer->usages;
    if (transfer->formatCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->formatCount = checked_cast<size_t>(transfer->formatCount);
    if (transfer->presentModeCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->presentModeCount = checked_cast<size_t>(transfer->presentModeCount);
    if (transfer->alphaModeCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->alphaModeCount = checked_cast<size_t>(transfer->alphaModeCount);
    {
        auto memberLength = record->formatCount;
        const volatile WGPUTextureFormat* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureFormat* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->formats = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUTextureFormat*>(memberBuffer),
              sizeof(WGPUTextureFormat) * memberLength);
        }
    }
    {
        auto memberLength = record->presentModeCount;
        const volatile WGPUPresentMode* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUPresentMode* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->presentModes = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUPresentMode*>(memberBuffer),
              sizeof(WGPUPresentMode) * memberLength);
        }
    }
    {
        auto memberLength = record->alphaModeCount;
        const volatile WGPUCompositeAlphaMode* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCompositeAlphaMode* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->alphaModes = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUCompositeAlphaMode*>(memberBuffer),
              sizeof(WGPUCompositeAlphaMode) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceCapabilitiesDeserialize);

struct WGPUSurfaceConfigurationTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId device;
    WGPUTextureFormat format;
    WGPUTextureUsage usage;
    uint64_t viewFormatCount;
    WGPUCompositeAlphaMode alphaMode;
    uint32_t width;
    uint32_t height;
    WGPUPresentMode presentMode;
};


[[maybe_unused]] size_t WGPUSurfaceConfigurationGetExtraRequiredSize([[maybe_unused]] const WGPUSurfaceConfiguration& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    {
        auto memberLength = record.viewFormatCount;
        auto size = WireAlignSizeofN<WGPUTextureFormat>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSurfaceConfigurationGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSurfaceConfigurationSerialize(
    const WGPUSurfaceConfiguration& record,
    WGPUSurfaceConfigurationTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetId(record.device, &transfer->device));
    transfer->format = record.format;
    transfer->usage = record.usage;
    transfer->viewFormatCount = record.viewFormatCount;
    transfer->alphaMode = record.alphaMode;
    transfer->width = record.width;
    transfer->height = record.height;
    transfer->presentMode = record.presentMode;
    {
        auto memberLength = record.viewFormatCount;

        WGPUTextureFormat* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.viewFormats,
                sizeof(WGPUTextureFormat) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceConfigurationSerialize);

[[maybe_unused]] WireResult WGPUSurfaceConfigurationDeserialize(
    WGPUSurfaceConfiguration* record,
    const volatile WGPUSurfaceConfigurationTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetFromId(transfer->device, &record->device));
    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->usage) >= sizeof(transfer->usage), "Deserialize assignment may not narrow.");
    record->usage = transfer->usage;
    if (transfer->viewFormatCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->viewFormatCount = checked_cast<size_t>(transfer->viewFormatCount);
    static_assert(sizeof(record->alphaMode) >= sizeof(transfer->alphaMode), "Deserialize assignment may not narrow.");
    record->alphaMode = transfer->alphaMode;
    static_assert(sizeof(record->width) >= sizeof(transfer->width), "Deserialize assignment may not narrow.");
    record->width = transfer->width;
    static_assert(sizeof(record->height) >= sizeof(transfer->height), "Deserialize assignment may not narrow.");
    record->height = transfer->height;
    static_assert(sizeof(record->presentMode) >= sizeof(transfer->presentMode), "Deserialize assignment may not narrow.");
    record->presentMode = transfer->presentMode;
    {
        auto memberLength = record->viewFormatCount;
        const volatile WGPUTextureFormat* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureFormat* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->viewFormats = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUTextureFormat*>(memberBuffer),
              sizeof(WGPUTextureFormat) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceConfigurationDeserialize);

struct WGPUSurfaceTextureTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    ObjectId texture;
    WGPUBool suboptimal;
    WGPUSurfaceGetCurrentTextureStatus status;
};


[[maybe_unused]] size_t WGPUSurfaceTextureGetExtraRequiredSize([[maybe_unused]] const WGPUSurfaceTexture& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSurfaceTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSurfaceTextureSerialize(
    const WGPUSurfaceTexture& record,
    WGPUSurfaceTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {


    WIRE_TRY(provider.GetId(record.texture, &transfer->texture));
    transfer->suboptimal = record.suboptimal;
    transfer->status = record.status;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceTextureSerialize);

[[maybe_unused]] WireResult WGPUSurfaceTextureDeserialize(
    WGPUSurfaceTexture* record,
    const volatile WGPUSurfaceTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {


    WIRE_TRY(resolver.GetFromId(transfer->texture, &record->texture));
    static_assert(sizeof(record->suboptimal) >= sizeof(transfer->suboptimal), "Deserialize assignment may not narrow.");
    record->suboptimal = transfer->suboptimal;
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceTextureDeserialize);

struct WGPUTextureBindingLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUTextureSampleType sampleType;
    WGPUTextureViewDimension viewDimension;
    WGPUBool multisampled;
};


[[maybe_unused]] size_t WGPUTextureBindingLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUTextureBindingLayout& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUTextureBindingLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUTextureBindingLayoutSerialize(
    const WGPUTextureBindingLayout& record,
    WGPUTextureBindingLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->sampleType = record.sampleType;
    transfer->viewDimension = record.viewDimension;
    transfer->multisampled = record.multisampled;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureBindingLayoutSerialize);

[[maybe_unused]] WireResult WGPUTextureBindingLayoutDeserialize(
    WGPUTextureBindingLayout* record,
    const volatile WGPUTextureBindingLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->sampleType) >= sizeof(transfer->sampleType), "Deserialize assignment may not narrow.");
    record->sampleType = transfer->sampleType;
    static_assert(sizeof(record->viewDimension) >= sizeof(transfer->viewDimension), "Deserialize assignment may not narrow.");
    record->viewDimension = transfer->viewDimension;
    static_assert(sizeof(record->multisampled) >= sizeof(transfer->multisampled), "Deserialize assignment may not narrow.");
    record->multisampled = transfer->multisampled;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureBindingLayoutDeserialize);

struct WGPUTextureBindingViewDimensionDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUTextureViewDimension textureBindingViewDimension;
};

static_assert(offsetof(WGPUTextureBindingViewDimensionDescriptorTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUTextureBindingViewDimensionDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUTextureBindingViewDimensionDescriptor& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUTextureBindingViewDimensionDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUTextureBindingViewDimensionDescriptorSerialize(
    const WGPUTextureBindingViewDimensionDescriptor& record,
    WGPUTextureBindingViewDimensionDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_TextureBindingViewDimensionDescriptor);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->textureBindingViewDimension = record.textureBindingViewDimension;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureBindingViewDimensionDescriptorSerialize);

[[maybe_unused]] WireResult WGPUTextureBindingViewDimensionDescriptorDeserialize(
    WGPUTextureBindingViewDimensionDescriptor* record,
    const volatile WGPUTextureBindingViewDimensionDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_TextureBindingViewDimensionDescriptor);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->textureBindingViewDimension) >= sizeof(transfer->textureBindingViewDimension), "Deserialize assignment may not narrow.");
    record->textureBindingViewDimension = transfer->textureBindingViewDimension;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureBindingViewDimensionDescriptorDeserialize);

struct WGPUTextureDataLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    uint64_t offset;
    uint32_t bytesPerRow;
    uint32_t rowsPerImage;
};


[[maybe_unused]] size_t WGPUTextureDataLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUTextureDataLayout& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUTextureDataLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUTextureDataLayoutSerialize(
    const WGPUTextureDataLayout& record,
    WGPUTextureDataLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->offset = record.offset;
    transfer->bytesPerRow = record.bytesPerRow;
    transfer->rowsPerImage = record.rowsPerImage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureDataLayoutSerialize);

[[maybe_unused]] WireResult WGPUTextureDataLayoutDeserialize(
    WGPUTextureDataLayout* record,
    const volatile WGPUTextureDataLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->bytesPerRow) >= sizeof(transfer->bytesPerRow), "Deserialize assignment may not narrow.");
    record->bytesPerRow = transfer->bytesPerRow;
    static_assert(sizeof(record->rowsPerImage) >= sizeof(transfer->rowsPerImage), "Deserialize assignment may not narrow.");
    record->rowsPerImage = transfer->rowsPerImage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureDataLayoutDeserialize);

struct WGPUVertexAttributeTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    WGPUVertexFormat format;
    uint64_t offset;
    uint32_t shaderLocation;
};


[[maybe_unused]] size_t WGPUVertexAttributeGetExtraRequiredSize([[maybe_unused]] const WGPUVertexAttribute& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUVertexAttributeGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUVertexAttributeSerialize(
    const WGPUVertexAttribute& record,
    WGPUVertexAttributeTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->format = record.format;
    transfer->offset = record.offset;
    transfer->shaderLocation = record.shaderLocation;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUVertexAttributeSerialize);

[[maybe_unused]] WireResult WGPUVertexAttributeDeserialize(
    WGPUVertexAttribute* record,
    const volatile WGPUVertexAttributeTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->shaderLocation) >= sizeof(transfer->shaderLocation), "Deserialize assignment may not narrow.");
    record->shaderLocation = transfer->shaderLocation;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUVertexAttributeDeserialize);

struct WGPUYCbCrVkDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint32_t vkFormat;
    uint32_t vkYCbCrModel;
    uint32_t vkYCbCrRange;
    uint32_t vkComponentSwizzleRed;
    uint32_t vkComponentSwizzleGreen;
    uint32_t vkComponentSwizzleBlue;
    uint32_t vkComponentSwizzleAlpha;
    uint32_t vkXChromaOffset;
    uint32_t vkYChromaOffset;
    WGPUFilterMode vkChromaFilter;
    WGPUBool forceExplicitReconstruction;
    uint64_t externalFormat;
};

static_assert(offsetof(WGPUYCbCrVkDescriptorTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUYCbCrVkDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUYCbCrVkDescriptor& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUYCbCrVkDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUYCbCrVkDescriptorSerialize(
    const WGPUYCbCrVkDescriptor& record,
    WGPUYCbCrVkDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_YCbCrVkDescriptor);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->vkFormat = record.vkFormat;
    transfer->vkYCbCrModel = record.vkYCbCrModel;
    transfer->vkYCbCrRange = record.vkYCbCrRange;
    transfer->vkComponentSwizzleRed = record.vkComponentSwizzleRed;
    transfer->vkComponentSwizzleGreen = record.vkComponentSwizzleGreen;
    transfer->vkComponentSwizzleBlue = record.vkComponentSwizzleBlue;
    transfer->vkComponentSwizzleAlpha = record.vkComponentSwizzleAlpha;
    transfer->vkXChromaOffset = record.vkXChromaOffset;
    transfer->vkYChromaOffset = record.vkYChromaOffset;
    transfer->vkChromaFilter = record.vkChromaFilter;
    transfer->forceExplicitReconstruction = record.forceExplicitReconstruction;
    transfer->externalFormat = record.externalFormat;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUYCbCrVkDescriptorSerialize);

[[maybe_unused]] WireResult WGPUYCbCrVkDescriptorDeserialize(
    WGPUYCbCrVkDescriptor* record,
    const volatile WGPUYCbCrVkDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_YCbCrVkDescriptor);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->vkFormat) >= sizeof(transfer->vkFormat), "Deserialize assignment may not narrow.");
    record->vkFormat = transfer->vkFormat;
    static_assert(sizeof(record->vkYCbCrModel) >= sizeof(transfer->vkYCbCrModel), "Deserialize assignment may not narrow.");
    record->vkYCbCrModel = transfer->vkYCbCrModel;
    static_assert(sizeof(record->vkYCbCrRange) >= sizeof(transfer->vkYCbCrRange), "Deserialize assignment may not narrow.");
    record->vkYCbCrRange = transfer->vkYCbCrRange;
    static_assert(sizeof(record->vkComponentSwizzleRed) >= sizeof(transfer->vkComponentSwizzleRed), "Deserialize assignment may not narrow.");
    record->vkComponentSwizzleRed = transfer->vkComponentSwizzleRed;
    static_assert(sizeof(record->vkComponentSwizzleGreen) >= sizeof(transfer->vkComponentSwizzleGreen), "Deserialize assignment may not narrow.");
    record->vkComponentSwizzleGreen = transfer->vkComponentSwizzleGreen;
    static_assert(sizeof(record->vkComponentSwizzleBlue) >= sizeof(transfer->vkComponentSwizzleBlue), "Deserialize assignment may not narrow.");
    record->vkComponentSwizzleBlue = transfer->vkComponentSwizzleBlue;
    static_assert(sizeof(record->vkComponentSwizzleAlpha) >= sizeof(transfer->vkComponentSwizzleAlpha), "Deserialize assignment may not narrow.");
    record->vkComponentSwizzleAlpha = transfer->vkComponentSwizzleAlpha;
    static_assert(sizeof(record->vkXChromaOffset) >= sizeof(transfer->vkXChromaOffset), "Deserialize assignment may not narrow.");
    record->vkXChromaOffset = transfer->vkXChromaOffset;
    static_assert(sizeof(record->vkYChromaOffset) >= sizeof(transfer->vkYChromaOffset), "Deserialize assignment may not narrow.");
    record->vkYChromaOffset = transfer->vkYChromaOffset;
    static_assert(sizeof(record->vkChromaFilter) >= sizeof(transfer->vkChromaFilter), "Deserialize assignment may not narrow.");
    record->vkChromaFilter = transfer->vkChromaFilter;
    static_assert(sizeof(record->forceExplicitReconstruction) >= sizeof(transfer->forceExplicitReconstruction), "Deserialize assignment may not narrow.");
    record->forceExplicitReconstruction = transfer->forceExplicitReconstruction;
    static_assert(sizeof(record->externalFormat) >= sizeof(transfer->externalFormat), "Deserialize assignment may not narrow.");
    record->externalFormat = transfer->externalFormat;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUYCbCrVkDescriptorDeserialize);

struct WGPUAdapterInfoTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer vendor;
    WGPUStringViewTransfer architecture;
    WGPUStringViewTransfer device;
    WGPUStringViewTransfer description;
    WGPUBackendType backendType;
    WGPUAdapterType adapterType;
    uint32_t vendorID;
    uint32_t deviceID;
    WGPUBool compatibilityMode;
};


[[maybe_unused]] size_t WGPUAdapterInfoGetExtraRequiredSize([[maybe_unused]] const WGPUAdapterInfo& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.vendor);
    result += WGPUStringViewGetExtraRequiredSize(record.architecture);
    result += WGPUStringViewGetExtraRequiredSize(record.device);
    result += WGPUStringViewGetExtraRequiredSize(record.description);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUAdapterInfoGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUAdapterInfoSerialize(
    const WGPUAdapterInfo& record,
    WGPUAdapterInfoTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.vendor, &transfer->vendor, buffer));
    WIRE_TRY(WGPUStringViewSerialize(record.architecture, &transfer->architecture, buffer));
    WIRE_TRY(WGPUStringViewSerialize(record.device, &transfer->device, buffer));
    WIRE_TRY(WGPUStringViewSerialize(record.description, &transfer->description, buffer));
    transfer->backendType = record.backendType;
    transfer->adapterType = record.adapterType;
    transfer->vendorID = record.vendorID;
    transfer->deviceID = record.deviceID;
    transfer->compatibilityMode = record.compatibilityMode;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterInfoSerialize);

[[maybe_unused]] WireResult WGPUAdapterInfoDeserialize(
    WGPUAdapterInfo* record,
    const volatile WGPUAdapterInfoTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->vendor, &transfer->vendor, deserializeBuffer, allocator));
    WIRE_TRY(WGPUStringViewDeserialize(&record->architecture, &transfer->architecture, deserializeBuffer, allocator));
    WIRE_TRY(WGPUStringViewDeserialize(&record->device, &transfer->device, deserializeBuffer, allocator));
    WIRE_TRY(WGPUStringViewDeserialize(&record->description, &transfer->description, deserializeBuffer, allocator));
    static_assert(sizeof(record->backendType) >= sizeof(transfer->backendType), "Deserialize assignment may not narrow.");
    record->backendType = transfer->backendType;
    static_assert(sizeof(record->adapterType) >= sizeof(transfer->adapterType), "Deserialize assignment may not narrow.");
    record->adapterType = transfer->adapterType;
    static_assert(sizeof(record->vendorID) >= sizeof(transfer->vendorID), "Deserialize assignment may not narrow.");
    record->vendorID = transfer->vendorID;
    static_assert(sizeof(record->deviceID) >= sizeof(transfer->deviceID), "Deserialize assignment may not narrow.");
    record->deviceID = transfer->deviceID;
    static_assert(sizeof(record->compatibilityMode) >= sizeof(transfer->compatibilityMode), "Deserialize assignment may not narrow.");
    record->compatibilityMode = transfer->compatibilityMode;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterInfoDeserialize);

struct WGPUAdapterPropertiesMemoryHeapsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint64_t heapCount;
};

static_assert(offsetof(WGPUAdapterPropertiesMemoryHeapsTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUAdapterPropertiesMemoryHeapsGetExtraRequiredSize([[maybe_unused]] const WGPUAdapterPropertiesMemoryHeaps& record) {
    size_t result = 0;

    {
        auto memberLength = record.heapCount;
        auto size = WireAlignSizeofN<WGPUMemoryHeapInfoTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUMemoryHeapInfoGetExtraRequiredSize(record.heapInfo[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesMemoryHeapsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesMemoryHeapsSerialize(
    const WGPUAdapterPropertiesMemoryHeaps& record,
    WGPUAdapterPropertiesMemoryHeapsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_AdapterPropertiesMemoryHeaps);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->heapCount = record.heapCount;
    {
        auto memberLength = record.heapCount;

        WGPUMemoryHeapInfoTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.heapInfo,
                sizeof(WGPUMemoryHeapInfoTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesMemoryHeapsSerialize);

[[maybe_unused]] WireResult WGPUAdapterPropertiesMemoryHeapsDeserialize(
    WGPUAdapterPropertiesMemoryHeaps* record,
    const volatile WGPUAdapterPropertiesMemoryHeapsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_AdapterPropertiesMemoryHeaps);
    DAWN_ASSERT(record->chain.next == nullptr);

    if (transfer->heapCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->heapCount = checked_cast<size_t>(transfer->heapCount);
    {
        auto memberLength = record->heapCount;
        const volatile WGPUMemoryHeapInfoTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUMemoryHeapInfo* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->heapInfo = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUMemoryHeapInfoTransfer*>(memberBuffer),
              sizeof(WGPUMemoryHeapInfoTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUAdapterPropertiesMemoryHeapsDeserialize);

struct WGPUBindGroupDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    ObjectId layout;
    uint64_t entryCount;
};


[[maybe_unused]] size_t WGPUBindGroupDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUBindGroupDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    {
        auto memberLength = record.entryCount;
        auto size = WireAlignSizeofN<WGPUBindGroupEntryTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUBindGroupEntryGetExtraRequiredSize(record.entries[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBindGroupDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBindGroupDescriptorSerialize(
    const WGPUBindGroupDescriptor& record,
    WGPUBindGroupDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    WIRE_TRY(provider.GetId(record.layout, &transfer->layout));
    transfer->entryCount = record.entryCount;
    {
        auto memberLength = record.entryCount;

        WGPUBindGroupEntryTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupEntrySerialize(record.entries[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupDescriptorSerialize);

[[maybe_unused]] WireResult WGPUBindGroupDescriptorDeserialize(
    WGPUBindGroupDescriptor* record,
    const volatile WGPUBindGroupDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    WIRE_TRY(resolver.GetFromId(transfer->layout, &record->layout));
    if (transfer->entryCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->entryCount = checked_cast<size_t>(transfer->entryCount);
    {
        auto memberLength = record->entryCount;
        const volatile WGPUBindGroupEntryTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBindGroupEntry* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->entries = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupEntryDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupDescriptorDeserialize);

struct WGPUBindGroupLayoutEntryTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    uint32_t binding;
    WGPUShaderStage visibility;
    WGPUBufferBindingLayoutTransfer buffer;
    WGPUSamplerBindingLayoutTransfer sampler;
    WGPUTextureBindingLayoutTransfer texture;
    WGPUStorageTextureBindingLayoutTransfer storageTexture;
};


[[maybe_unused]] size_t WGPUBindGroupLayoutEntryGetExtraRequiredSize([[maybe_unused]] const WGPUBindGroupLayoutEntry& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUBufferBindingLayoutGetExtraRequiredSize(record.buffer);
    result += WGPUSamplerBindingLayoutGetExtraRequiredSize(record.sampler);
    result += WGPUTextureBindingLayoutGetExtraRequiredSize(record.texture);
    result += WGPUStorageTextureBindingLayoutGetExtraRequiredSize(record.storageTexture);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBindGroupLayoutEntryGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBindGroupLayoutEntrySerialize(
    const WGPUBindGroupLayoutEntry& record,
    WGPUBindGroupLayoutEntryTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->binding = record.binding;
    transfer->visibility = record.visibility;
    WIRE_TRY(WGPUBufferBindingLayoutSerialize(record.buffer, &transfer->buffer, buffer, provider));
    WIRE_TRY(WGPUSamplerBindingLayoutSerialize(record.sampler, &transfer->sampler, buffer, provider));
    WIRE_TRY(WGPUTextureBindingLayoutSerialize(record.texture, &transfer->texture, buffer, provider));
    WIRE_TRY(WGPUStorageTextureBindingLayoutSerialize(record.storageTexture, &transfer->storageTexture, buffer, provider));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupLayoutEntrySerialize);

[[maybe_unused]] WireResult WGPUBindGroupLayoutEntryDeserialize(
    WGPUBindGroupLayoutEntry* record,
    const volatile WGPUBindGroupLayoutEntryTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->binding) >= sizeof(transfer->binding), "Deserialize assignment may not narrow.");
    record->binding = transfer->binding;
    static_assert(sizeof(record->visibility) >= sizeof(transfer->visibility), "Deserialize assignment may not narrow.");
    record->visibility = transfer->visibility;
    WIRE_TRY(WGPUBufferBindingLayoutDeserialize(&record->buffer, &transfer->buffer, deserializeBuffer, allocator, resolver));
    WIRE_TRY(WGPUSamplerBindingLayoutDeserialize(&record->sampler, &transfer->sampler, deserializeBuffer, allocator, resolver));
    WIRE_TRY(WGPUTextureBindingLayoutDeserialize(&record->texture, &transfer->texture, deserializeBuffer, allocator, resolver));
    WIRE_TRY(WGPUStorageTextureBindingLayoutDeserialize(&record->storageTexture, &transfer->storageTexture, deserializeBuffer, allocator, resolver));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupLayoutEntryDeserialize);

struct WGPUBlendStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    WGPUBlendComponentTransfer color;
    WGPUBlendComponentTransfer alpha;
};


[[maybe_unused]] size_t WGPUBlendStateGetExtraRequiredSize([[maybe_unused]] const WGPUBlendState& record) {
    size_t result = 0;

    result += WGPUBlendComponentGetExtraRequiredSize(record.color);
    result += WGPUBlendComponentGetExtraRequiredSize(record.alpha);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBlendStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBlendStateSerialize(
    const WGPUBlendState& record,
    WGPUBlendStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    WIRE_TRY(WGPUBlendComponentSerialize(record.color, &transfer->color, buffer));
    WIRE_TRY(WGPUBlendComponentSerialize(record.alpha, &transfer->alpha, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBlendStateSerialize);

[[maybe_unused]] WireResult WGPUBlendStateDeserialize(
    WGPUBlendState* record,
    const volatile WGPUBlendStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->color) == sizeof(transfer->color), "Deserialize memcpy size must match.");
    memcpy(&record->color, const_cast<const WGPUBlendComponentTransfer*>(&transfer->color), sizeof(WGPUBlendComponentTransfer));
    static_assert(sizeof(record->alpha) == sizeof(transfer->alpha), "Deserialize memcpy size must match.");
    memcpy(&record->alpha, const_cast<const WGPUBlendComponentTransfer*>(&transfer->alpha), sizeof(WGPUBlendComponentTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBlendStateDeserialize);

struct WGPUBufferDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    WGPUBufferUsage usage;
    uint64_t size;
    WGPUBool mappedAtCreation;
};


[[maybe_unused]] size_t WGPUBufferDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUBufferDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBufferDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBufferDescriptorSerialize(
    const WGPUBufferDescriptor& record,
    WGPUBufferDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->usage = record.usage;
    transfer->size = record.size;
    transfer->mappedAtCreation = record.mappedAtCreation;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBufferDescriptorSerialize);

[[maybe_unused]] WireResult WGPUBufferDescriptorDeserialize(
    WGPUBufferDescriptor* record,
    const volatile WGPUBufferDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    static_assert(sizeof(record->usage) >= sizeof(transfer->usage), "Deserialize assignment may not narrow.");
    record->usage = transfer->usage;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;
    static_assert(sizeof(record->mappedAtCreation) >= sizeof(transfer->mappedAtCreation), "Deserialize assignment may not narrow.");
    record->mappedAtCreation = transfer->mappedAtCreation;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBufferDescriptorDeserialize);

struct WGPUCommandBufferDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
};


[[maybe_unused]] size_t WGPUCommandBufferDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUCommandBufferDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUCommandBufferDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUCommandBufferDescriptorSerialize(
    const WGPUCommandBufferDescriptor& record,
    WGPUCommandBufferDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCommandBufferDescriptorSerialize);

[[maybe_unused]] WireResult WGPUCommandBufferDescriptorDeserialize(
    WGPUCommandBufferDescriptor* record,
    const volatile WGPUCommandBufferDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCommandBufferDescriptorDeserialize);

struct WGPUCommandEncoderDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
};


[[maybe_unused]] size_t WGPUCommandEncoderDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUCommandEncoderDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUCommandEncoderDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUCommandEncoderDescriptorSerialize(
    const WGPUCommandEncoderDescriptor& record,
    WGPUCommandEncoderDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCommandEncoderDescriptorSerialize);

[[maybe_unused]] WireResult WGPUCommandEncoderDescriptorDeserialize(
    WGPUCommandEncoderDescriptor* record,
    const volatile WGPUCommandEncoderDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCommandEncoderDescriptorDeserialize);

struct WGPUCompilationMessageTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer message;
    WGPUCompilationMessageType type;
    uint64_t lineNum;
    uint64_t linePos;
    uint64_t offset;
    uint64_t length;
    uint64_t utf16LinePos;
    uint64_t utf16Offset;
    uint64_t utf16Length;
};


[[maybe_unused]] size_t WGPUCompilationMessageGetExtraRequiredSize([[maybe_unused]] const WGPUCompilationMessage& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUCompilationMessageGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUCompilationMessageSerialize(
    const WGPUCompilationMessage& record,
    WGPUCompilationMessageTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));
    transfer->type = record.type;
    transfer->lineNum = record.lineNum;
    transfer->linePos = record.linePos;
    transfer->offset = record.offset;
    transfer->length = record.length;
    transfer->utf16LinePos = record.utf16LinePos;
    transfer->utf16Offset = record.utf16Offset;
    transfer->utf16Length = record.utf16Length;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCompilationMessageSerialize);

[[maybe_unused]] WireResult WGPUCompilationMessageDeserialize(
    WGPUCompilationMessage* record,
    const volatile WGPUCompilationMessageTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));
    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    static_assert(sizeof(record->lineNum) >= sizeof(transfer->lineNum), "Deserialize assignment may not narrow.");
    record->lineNum = transfer->lineNum;
    static_assert(sizeof(record->linePos) >= sizeof(transfer->linePos), "Deserialize assignment may not narrow.");
    record->linePos = transfer->linePos;
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->length) >= sizeof(transfer->length), "Deserialize assignment may not narrow.");
    record->length = transfer->length;
    static_assert(sizeof(record->utf16LinePos) >= sizeof(transfer->utf16LinePos), "Deserialize assignment may not narrow.");
    record->utf16LinePos = transfer->utf16LinePos;
    static_assert(sizeof(record->utf16Offset) >= sizeof(transfer->utf16Offset), "Deserialize assignment may not narrow.");
    record->utf16Offset = transfer->utf16Offset;
    static_assert(sizeof(record->utf16Length) >= sizeof(transfer->utf16Length), "Deserialize assignment may not narrow.");
    record->utf16Length = transfer->utf16Length;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCompilationMessageDeserialize);

struct WGPUComputePassDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    WGPUBool has_timestampWrites;
};


[[maybe_unused]] size_t WGPUComputePassDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUComputePassDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    if (record.timestampWrites != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUPassTimestampWritesTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUPassTimestampWritesGetExtraRequiredSize(record.timestampWrites[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUComputePassDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUComputePassDescriptorSerialize(
    const WGPUComputePassDescriptor& record,
    WGPUComputePassDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    bool has_timestampWrites = record.timestampWrites != nullptr;
    transfer->has_timestampWrites = has_timestampWrites;
    if (has_timestampWrites) {
        auto memberLength = 1u;

        WGPUPassTimestampWritesTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUPassTimestampWritesSerialize(record.timestampWrites[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUComputePassDescriptorSerialize);

[[maybe_unused]] WireResult WGPUComputePassDescriptorDeserialize(
    WGPUComputePassDescriptor* record,
    const volatile WGPUComputePassDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    bool has_timestampWrites = transfer->has_timestampWrites;
    record->timestampWrites = nullptr;
    if (has_timestampWrites) {
        auto memberLength = 1u;
        const volatile WGPUPassTimestampWritesTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUPassTimestampWrites* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->timestampWrites = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUPassTimestampWritesDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUComputePassDescriptorDeserialize);

struct WGPUConstantEntryTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer key;
    double value;
};


[[maybe_unused]] size_t WGPUConstantEntryGetExtraRequiredSize([[maybe_unused]] const WGPUConstantEntry& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.key);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUConstantEntryGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUConstantEntrySerialize(
    const WGPUConstantEntry& record,
    WGPUConstantEntryTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.key, &transfer->key, buffer));
    transfer->value = record.value;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUConstantEntrySerialize);

[[maybe_unused]] WireResult WGPUConstantEntryDeserialize(
    WGPUConstantEntry* record,
    const volatile WGPUConstantEntryTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->key, &transfer->key, deserializeBuffer, allocator));
    static_assert(sizeof(record->value) >= sizeof(transfer->value), "Deserialize assignment may not narrow.");
    record->value = transfer->value;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUConstantEntryDeserialize);

struct WGPUDawnDrmFormatCapabilitiesTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint64_t propertiesCount;
};

static_assert(offsetof(WGPUDawnDrmFormatCapabilitiesTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUDawnDrmFormatCapabilitiesGetExtraRequiredSize([[maybe_unused]] const WGPUDawnDrmFormatCapabilities& record) {
    size_t result = 0;

    {
        auto memberLength = record.propertiesCount;
        auto size = WireAlignSizeofN<WGPUDawnDrmFormatPropertiesTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUDawnDrmFormatPropertiesGetExtraRequiredSize(record.properties[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDawnDrmFormatCapabilitiesGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDawnDrmFormatCapabilitiesSerialize(
    const WGPUDawnDrmFormatCapabilities& record,
    WGPUDawnDrmFormatCapabilitiesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_DawnDrmFormatCapabilities);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->propertiesCount = record.propertiesCount;
    {
        auto memberLength = record.propertiesCount;

        WGPUDawnDrmFormatPropertiesTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.properties,
                sizeof(WGPUDawnDrmFormatPropertiesTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnDrmFormatCapabilitiesSerialize);

[[maybe_unused]] WireResult WGPUDawnDrmFormatCapabilitiesDeserialize(
    WGPUDawnDrmFormatCapabilities* record,
    const volatile WGPUDawnDrmFormatCapabilitiesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_DawnDrmFormatCapabilities);
    DAWN_ASSERT(record->chain.next == nullptr);

    if (transfer->propertiesCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->propertiesCount = checked_cast<size_t>(transfer->propertiesCount);
    {
        auto memberLength = record->propertiesCount;
        const volatile WGPUDawnDrmFormatPropertiesTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUDawnDrmFormatProperties* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->properties = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUDawnDrmFormatPropertiesTransfer*>(memberBuffer),
              sizeof(WGPUDawnDrmFormatPropertiesTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDawnDrmFormatCapabilitiesDeserialize);

struct WGPUDepthStencilStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUTextureFormat format;
    WGPUOptionalBool depthWriteEnabled;
    WGPUCompareFunction depthCompare;
    WGPUStencilFaceStateTransfer stencilFront;
    WGPUStencilFaceStateTransfer stencilBack;
    uint32_t stencilReadMask;
    uint32_t stencilWriteMask;
    int32_t depthBias;
    float depthBiasSlopeScale;
    float depthBiasClamp;
};


[[maybe_unused]] size_t WGPUDepthStencilStateGetExtraRequiredSize([[maybe_unused]] const WGPUDepthStencilState& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStencilFaceStateGetExtraRequiredSize(record.stencilFront);
    result += WGPUStencilFaceStateGetExtraRequiredSize(record.stencilBack);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDepthStencilStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDepthStencilStateSerialize(
    const WGPUDepthStencilState& record,
    WGPUDepthStencilStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->format = record.format;
    transfer->depthWriteEnabled = record.depthWriteEnabled;
    transfer->depthCompare = record.depthCompare;
    WIRE_TRY(WGPUStencilFaceStateSerialize(record.stencilFront, &transfer->stencilFront, buffer));
    WIRE_TRY(WGPUStencilFaceStateSerialize(record.stencilBack, &transfer->stencilBack, buffer));
    transfer->stencilReadMask = record.stencilReadMask;
    transfer->stencilWriteMask = record.stencilWriteMask;
    transfer->depthBias = record.depthBias;
    transfer->depthBiasSlopeScale = record.depthBiasSlopeScale;
    transfer->depthBiasClamp = record.depthBiasClamp;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDepthStencilStateSerialize);

[[maybe_unused]] WireResult WGPUDepthStencilStateDeserialize(
    WGPUDepthStencilState* record,
    const volatile WGPUDepthStencilStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->depthWriteEnabled) >= sizeof(transfer->depthWriteEnabled), "Deserialize assignment may not narrow.");
    record->depthWriteEnabled = transfer->depthWriteEnabled;
    static_assert(sizeof(record->depthCompare) >= sizeof(transfer->depthCompare), "Deserialize assignment may not narrow.");
    record->depthCompare = transfer->depthCompare;
    static_assert(sizeof(record->stencilFront) == sizeof(transfer->stencilFront), "Deserialize memcpy size must match.");
    memcpy(&record->stencilFront, const_cast<const WGPUStencilFaceStateTransfer*>(&transfer->stencilFront), sizeof(WGPUStencilFaceStateTransfer));
    static_assert(sizeof(record->stencilBack) == sizeof(transfer->stencilBack), "Deserialize memcpy size must match.");
    memcpy(&record->stencilBack, const_cast<const WGPUStencilFaceStateTransfer*>(&transfer->stencilBack), sizeof(WGPUStencilFaceStateTransfer));
    static_assert(sizeof(record->stencilReadMask) >= sizeof(transfer->stencilReadMask), "Deserialize assignment may not narrow.");
    record->stencilReadMask = transfer->stencilReadMask;
    static_assert(sizeof(record->stencilWriteMask) >= sizeof(transfer->stencilWriteMask), "Deserialize assignment may not narrow.");
    record->stencilWriteMask = transfer->stencilWriteMask;
    static_assert(sizeof(record->depthBias) >= sizeof(transfer->depthBias), "Deserialize assignment may not narrow.");
    record->depthBias = transfer->depthBias;
    static_assert(sizeof(record->depthBiasSlopeScale) >= sizeof(transfer->depthBiasSlopeScale), "Deserialize assignment may not narrow.");
    record->depthBiasSlopeScale = transfer->depthBiasSlopeScale;
    static_assert(sizeof(record->depthBiasClamp) >= sizeof(transfer->depthBiasClamp), "Deserialize assignment may not narrow.");
    record->depthBiasClamp = transfer->depthBiasClamp;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDepthStencilStateDeserialize);

struct WGPUExternalTextureDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    ObjectId plane0;
    ObjectId plane1;
    WGPUOrigin2DTransfer cropOrigin;
    WGPUExtent2DTransfer cropSize;
    WGPUExtent2DTransfer apparentSize;
    WGPUBool doYuvToRgbConversionOnly;
    WGPUBool has_yuvToRgbConversionMatrix;
    WGPUBool mirrored;
    WGPUExternalTextureRotation rotation;
};


[[maybe_unused]] size_t WGPUExternalTextureDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUExternalTextureDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    result += WGPUOrigin2DGetExtraRequiredSize(record.cropOrigin);
    result += WGPUExtent2DGetExtraRequiredSize(record.cropSize);
    result += WGPUExtent2DGetExtraRequiredSize(record.apparentSize);
    if (record.yuvToRgbConversionMatrix != nullptr)
    {
        auto memberLength = 12u;
        auto size = WireAlignSizeofN<float>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    {
        auto memberLength = 7u;
        auto size = WireAlignSizeofN<float>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    {
        auto memberLength = 7u;
        auto size = WireAlignSizeofN<float>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    {
        auto memberLength = 9u;
        auto size = WireAlignSizeofN<float>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUExternalTextureDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUExternalTextureDescriptorSerialize(
    const WGPUExternalTextureDescriptor& record,
    WGPUExternalTextureDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    WIRE_TRY(provider.GetId(record.plane0, &transfer->plane0));
    WIRE_TRY(provider.GetOptionalId(record.plane1, &transfer->plane1));
    WIRE_TRY(WGPUOrigin2DSerialize(record.cropOrigin, &transfer->cropOrigin, buffer));
    WIRE_TRY(WGPUExtent2DSerialize(record.cropSize, &transfer->cropSize, buffer));
    WIRE_TRY(WGPUExtent2DSerialize(record.apparentSize, &transfer->apparentSize, buffer));
    transfer->doYuvToRgbConversionOnly = record.doYuvToRgbConversionOnly;
    transfer->mirrored = record.mirrored;
    transfer->rotation = record.rotation;
    bool has_yuvToRgbConversionMatrix = record.yuvToRgbConversionMatrix != nullptr;
    transfer->has_yuvToRgbConversionMatrix = has_yuvToRgbConversionMatrix;
    if (has_yuvToRgbConversionMatrix) {
        auto memberLength = 12u;

        float* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.yuvToRgbConversionMatrix,
                sizeof(float) * memberLength);
        }
    }
    {
        auto memberLength = 7u;

        float* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.srcTransferFunctionParameters,
                sizeof(float) * memberLength);
        }
    }
    {
        auto memberLength = 7u;

        float* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.dstTransferFunctionParameters,
                sizeof(float) * memberLength);
        }
    }
    {
        auto memberLength = 9u;

        float* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.gamutConversionMatrix,
                sizeof(float) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExternalTextureDescriptorSerialize);

[[maybe_unused]] WireResult WGPUExternalTextureDescriptorDeserialize(
    WGPUExternalTextureDescriptor* record,
    const volatile WGPUExternalTextureDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    WIRE_TRY(resolver.GetFromId(transfer->plane0, &record->plane0));
    WIRE_TRY(resolver.GetOptionalFromId(transfer->plane1, &record->plane1));
    static_assert(sizeof(record->cropOrigin) == sizeof(transfer->cropOrigin), "Deserialize memcpy size must match.");
    memcpy(&record->cropOrigin, const_cast<const WGPUOrigin2DTransfer*>(&transfer->cropOrigin), sizeof(WGPUOrigin2DTransfer));
    static_assert(sizeof(record->cropSize) == sizeof(transfer->cropSize), "Deserialize memcpy size must match.");
    memcpy(&record->cropSize, const_cast<const WGPUExtent2DTransfer*>(&transfer->cropSize), sizeof(WGPUExtent2DTransfer));
    static_assert(sizeof(record->apparentSize) == sizeof(transfer->apparentSize), "Deserialize memcpy size must match.");
    memcpy(&record->apparentSize, const_cast<const WGPUExtent2DTransfer*>(&transfer->apparentSize), sizeof(WGPUExtent2DTransfer));
    static_assert(sizeof(record->doYuvToRgbConversionOnly) >= sizeof(transfer->doYuvToRgbConversionOnly), "Deserialize assignment may not narrow.");
    record->doYuvToRgbConversionOnly = transfer->doYuvToRgbConversionOnly;
    static_assert(sizeof(record->mirrored) >= sizeof(transfer->mirrored), "Deserialize assignment may not narrow.");
    record->mirrored = transfer->mirrored;
    static_assert(sizeof(record->rotation) >= sizeof(transfer->rotation), "Deserialize assignment may not narrow.");
    record->rotation = transfer->rotation;
    bool has_yuvToRgbConversionMatrix = transfer->has_yuvToRgbConversionMatrix;
    record->yuvToRgbConversionMatrix = nullptr;
    if (has_yuvToRgbConversionMatrix) {
        auto memberLength = 12u;
        const volatile float* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        float* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->yuvToRgbConversionMatrix = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const float*>(memberBuffer),
              sizeof(float) * memberLength);
        }
    }
    {
        auto memberLength = 7u;
        const volatile float* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        float* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->srcTransferFunctionParameters = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const float*>(memberBuffer),
              sizeof(float) * memberLength);
        }
    }
    {
        auto memberLength = 7u;
        const volatile float* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        float* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->dstTransferFunctionParameters = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const float*>(memberBuffer),
              sizeof(float) * memberLength);
        }
    }
    {
        auto memberLength = 9u;
        const volatile float* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        float* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->gamutConversionMatrix = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const float*>(memberBuffer),
              sizeof(float) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUExternalTextureDescriptorDeserialize);

struct WGPUImageCopyBufferTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    WGPUTextureDataLayoutTransfer layout;
    ObjectId buffer;
};


[[maybe_unused]] size_t WGPUImageCopyBufferGetExtraRequiredSize([[maybe_unused]] const WGPUImageCopyBuffer& record) {
    size_t result = 0;

    result += WGPUTextureDataLayoutGetExtraRequiredSize(record.layout);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUImageCopyBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUImageCopyBufferSerialize(
    const WGPUImageCopyBuffer& record,
    WGPUImageCopyBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {


    WIRE_TRY(WGPUTextureDataLayoutSerialize(record.layout, &transfer->layout, buffer, provider));
    WIRE_TRY(provider.GetId(record.buffer, &transfer->buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUImageCopyBufferSerialize);

[[maybe_unused]] WireResult WGPUImageCopyBufferDeserialize(
    WGPUImageCopyBuffer* record,
    const volatile WGPUImageCopyBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {


    WIRE_TRY(WGPUTextureDataLayoutDeserialize(&record->layout, &transfer->layout, deserializeBuffer, allocator, resolver));
    WIRE_TRY(resolver.GetFromId(transfer->buffer, &record->buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUImageCopyBufferDeserialize);

struct WGPUImageCopyExternalTextureTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId externalTexture;
    WGPUOrigin3DTransfer origin;
    WGPUExtent2DTransfer naturalSize;
};


[[maybe_unused]] size_t WGPUImageCopyExternalTextureGetExtraRequiredSize([[maybe_unused]] const WGPUImageCopyExternalTexture& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUOrigin3DGetExtraRequiredSize(record.origin);
    result += WGPUExtent2DGetExtraRequiredSize(record.naturalSize);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUImageCopyExternalTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUImageCopyExternalTextureSerialize(
    const WGPUImageCopyExternalTexture& record,
    WGPUImageCopyExternalTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetId(record.externalTexture, &transfer->externalTexture));
    WIRE_TRY(WGPUOrigin3DSerialize(record.origin, &transfer->origin, buffer));
    WIRE_TRY(WGPUExtent2DSerialize(record.naturalSize, &transfer->naturalSize, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUImageCopyExternalTextureSerialize);

[[maybe_unused]] WireResult WGPUImageCopyExternalTextureDeserialize(
    WGPUImageCopyExternalTexture* record,
    const volatile WGPUImageCopyExternalTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetFromId(transfer->externalTexture, &record->externalTexture));
    static_assert(sizeof(record->origin) == sizeof(transfer->origin), "Deserialize memcpy size must match.");
    memcpy(&record->origin, const_cast<const WGPUOrigin3DTransfer*>(&transfer->origin), sizeof(WGPUOrigin3DTransfer));
    static_assert(sizeof(record->naturalSize) == sizeof(transfer->naturalSize), "Deserialize memcpy size must match.");
    memcpy(&record->naturalSize, const_cast<const WGPUExtent2DTransfer*>(&transfer->naturalSize), sizeof(WGPUExtent2DTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUImageCopyExternalTextureDeserialize);

struct WGPUImageCopyTextureTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    ObjectId texture;
    uint32_t mipLevel;
    WGPUOrigin3DTransfer origin;
    WGPUTextureAspect aspect;
};


[[maybe_unused]] size_t WGPUImageCopyTextureGetExtraRequiredSize([[maybe_unused]] const WGPUImageCopyTexture& record) {
    size_t result = 0;

    result += WGPUOrigin3DGetExtraRequiredSize(record.origin);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUImageCopyTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUImageCopyTextureSerialize(
    const WGPUImageCopyTexture& record,
    WGPUImageCopyTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {


    WIRE_TRY(provider.GetId(record.texture, &transfer->texture));
    transfer->mipLevel = record.mipLevel;
    WIRE_TRY(WGPUOrigin3DSerialize(record.origin, &transfer->origin, buffer));
    transfer->aspect = record.aspect;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUImageCopyTextureSerialize);

[[maybe_unused]] WireResult WGPUImageCopyTextureDeserialize(
    WGPUImageCopyTexture* record,
    const volatile WGPUImageCopyTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {


    WIRE_TRY(resolver.GetFromId(transfer->texture, &record->texture));
    static_assert(sizeof(record->mipLevel) >= sizeof(transfer->mipLevel), "Deserialize assignment may not narrow.");
    record->mipLevel = transfer->mipLevel;
    static_assert(sizeof(record->origin) == sizeof(transfer->origin), "Deserialize memcpy size must match.");
    memcpy(&record->origin, const_cast<const WGPUOrigin3DTransfer*>(&transfer->origin), sizeof(WGPUOrigin3DTransfer));
    static_assert(sizeof(record->aspect) >= sizeof(transfer->aspect), "Deserialize assignment may not narrow.");
    record->aspect = transfer->aspect;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUImageCopyTextureDeserialize);

struct WGPUInstanceDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUInstanceCapabilitiesTransfer capabilities;
    WGPUInstanceCapabilitiesTransfer features;
};


[[maybe_unused]] size_t WGPUInstanceDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUInstanceDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUInstanceCapabilitiesGetExtraRequiredSize(record.capabilities);
    result += WGPUInstanceCapabilitiesGetExtraRequiredSize(record.features);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUInstanceDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUInstanceDescriptorSerialize(
    const WGPUInstanceDescriptor& record,
    WGPUInstanceDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUInstanceCapabilitiesSerialize(record.capabilities, &transfer->capabilities, buffer, provider));
    WIRE_TRY(WGPUInstanceCapabilitiesSerialize(record.features, &transfer->features, buffer, provider));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUInstanceDescriptorSerialize);

[[maybe_unused]] WireResult WGPUInstanceDescriptorDeserialize(
    WGPUInstanceDescriptor* record,
    const volatile WGPUInstanceDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUInstanceCapabilitiesDeserialize(&record->capabilities, &transfer->capabilities, deserializeBuffer, allocator, resolver));
    WIRE_TRY(WGPUInstanceCapabilitiesDeserialize(&record->features, &transfer->features, deserializeBuffer, allocator, resolver));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUInstanceDescriptorDeserialize);

struct WGPUPipelineLayoutDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    uint64_t bindGroupLayoutCount;
    uint32_t immediateDataRangeByteSize;
};


[[maybe_unused]] size_t WGPUPipelineLayoutDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUPipelineLayoutDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    {
        auto memberLength = record.bindGroupLayoutCount;
        auto size = WireAlignSizeofN<ObjectId>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUPipelineLayoutDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUPipelineLayoutDescriptorSerialize(
    const WGPUPipelineLayoutDescriptor& record,
    WGPUPipelineLayoutDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->bindGroupLayoutCount = record.bindGroupLayoutCount;
    transfer->immediateDataRangeByteSize = record.immediateDataRangeByteSize;
    {
        auto memberLength = record.bindGroupLayoutCount;

        ObjectId* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(provider.GetOptionalId(record.bindGroupLayouts[i], &memberBuffer[i]));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPipelineLayoutDescriptorSerialize);

[[maybe_unused]] WireResult WGPUPipelineLayoutDescriptorDeserialize(
    WGPUPipelineLayoutDescriptor* record,
    const volatile WGPUPipelineLayoutDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    if (transfer->bindGroupLayoutCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->bindGroupLayoutCount = checked_cast<size_t>(transfer->bindGroupLayoutCount);
    static_assert(sizeof(record->immediateDataRangeByteSize) >= sizeof(transfer->immediateDataRangeByteSize), "Deserialize assignment may not narrow.");
    record->immediateDataRangeByteSize = transfer->immediateDataRangeByteSize;
    {
        auto memberLength = record->bindGroupLayoutCount;
        const volatile ObjectId* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBindGroupLayout* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->bindGroupLayouts = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(resolver.GetOptionalFromId(memberBuffer[i], &copiedMembers[i]));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPipelineLayoutDescriptorDeserialize);

struct WGPUPipelineLayoutPixelLocalStorageTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint64_t totalPixelLocalStorageSize;
    uint64_t storageAttachmentCount;
};

static_assert(offsetof(WGPUPipelineLayoutPixelLocalStorageTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUPipelineLayoutPixelLocalStorageGetExtraRequiredSize([[maybe_unused]] const WGPUPipelineLayoutPixelLocalStorage& record) {
    size_t result = 0;

    {
        auto memberLength = record.storageAttachmentCount;
        auto size = WireAlignSizeofN<WGPUPipelineLayoutStorageAttachmentTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUPipelineLayoutStorageAttachmentGetExtraRequiredSize(record.storageAttachments[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUPipelineLayoutPixelLocalStorageGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUPipelineLayoutPixelLocalStorageSerialize(
    const WGPUPipelineLayoutPixelLocalStorage& record,
    WGPUPipelineLayoutPixelLocalStorageTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_PipelineLayoutPixelLocalStorage);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->totalPixelLocalStorageSize = record.totalPixelLocalStorageSize;
    transfer->storageAttachmentCount = record.storageAttachmentCount;
    {
        auto memberLength = record.storageAttachmentCount;

        WGPUPipelineLayoutStorageAttachmentTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.storageAttachments,
                sizeof(WGPUPipelineLayoutStorageAttachmentTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPipelineLayoutPixelLocalStorageSerialize);

[[maybe_unused]] WireResult WGPUPipelineLayoutPixelLocalStorageDeserialize(
    WGPUPipelineLayoutPixelLocalStorage* record,
    const volatile WGPUPipelineLayoutPixelLocalStorageTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_PipelineLayoutPixelLocalStorage);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->totalPixelLocalStorageSize) >= sizeof(transfer->totalPixelLocalStorageSize), "Deserialize assignment may not narrow.");
    record->totalPixelLocalStorageSize = transfer->totalPixelLocalStorageSize;
    if (transfer->storageAttachmentCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->storageAttachmentCount = checked_cast<size_t>(transfer->storageAttachmentCount);
    {
        auto memberLength = record->storageAttachmentCount;
        const volatile WGPUPipelineLayoutStorageAttachmentTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUPipelineLayoutStorageAttachment* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->storageAttachments = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUPipelineLayoutStorageAttachmentTransfer*>(memberBuffer),
              sizeof(WGPUPipelineLayoutStorageAttachmentTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUPipelineLayoutPixelLocalStorageDeserialize);

struct WGPUQuerySetDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    WGPUQueryType type;
    uint32_t count;
};


[[maybe_unused]] size_t WGPUQuerySetDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUQuerySetDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUQuerySetDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUQuerySetDescriptorSerialize(
    const WGPUQuerySetDescriptor& record,
    WGPUQuerySetDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->type = record.type;
    transfer->count = record.count;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUQuerySetDescriptorSerialize);

[[maybe_unused]] WireResult WGPUQuerySetDescriptorDeserialize(
    WGPUQuerySetDescriptor* record,
    const volatile WGPUQuerySetDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    static_assert(sizeof(record->count) >= sizeof(transfer->count), "Deserialize assignment may not narrow.");
    record->count = transfer->count;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUQuerySetDescriptorDeserialize);

struct WGPUQueueDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
};


[[maybe_unused]] size_t WGPUQueueDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUQueueDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUQueueDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUQueueDescriptorSerialize(
    const WGPUQueueDescriptor& record,
    WGPUQueueDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUQueueDescriptorSerialize);

[[maybe_unused]] WireResult WGPUQueueDescriptorDeserialize(
    WGPUQueueDescriptor* record,
    const volatile WGPUQueueDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUQueueDescriptorDeserialize);

struct WGPURenderBundleDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
};


[[maybe_unused]] size_t WGPURenderBundleDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPURenderBundleDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderBundleDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderBundleDescriptorSerialize(
    const WGPURenderBundleDescriptor& record,
    WGPURenderBundleDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderBundleDescriptorSerialize);

[[maybe_unused]] WireResult WGPURenderBundleDescriptorDeserialize(
    WGPURenderBundleDescriptor* record,
    const volatile WGPURenderBundleDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderBundleDescriptorDeserialize);

struct WGPURenderBundleEncoderDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    uint64_t colorFormatCount;
    WGPUTextureFormat depthStencilFormat;
    uint32_t sampleCount;
    WGPUBool depthReadOnly;
    WGPUBool stencilReadOnly;
};


[[maybe_unused]] size_t WGPURenderBundleEncoderDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPURenderBundleEncoderDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    {
        auto memberLength = record.colorFormatCount;
        auto size = WireAlignSizeofN<WGPUTextureFormat>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderBundleEncoderDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderBundleEncoderDescriptorSerialize(
    const WGPURenderBundleEncoderDescriptor& record,
    WGPURenderBundleEncoderDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->colorFormatCount = record.colorFormatCount;
    transfer->depthStencilFormat = record.depthStencilFormat;
    transfer->sampleCount = record.sampleCount;
    transfer->depthReadOnly = record.depthReadOnly;
    transfer->stencilReadOnly = record.stencilReadOnly;
    {
        auto memberLength = record.colorFormatCount;

        WGPUTextureFormat* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.colorFormats,
                sizeof(WGPUTextureFormat) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderBundleEncoderDescriptorSerialize);

[[maybe_unused]] WireResult WGPURenderBundleEncoderDescriptorDeserialize(
    WGPURenderBundleEncoderDescriptor* record,
    const volatile WGPURenderBundleEncoderDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    if (transfer->colorFormatCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->colorFormatCount = checked_cast<size_t>(transfer->colorFormatCount);
    static_assert(sizeof(record->depthStencilFormat) >= sizeof(transfer->depthStencilFormat), "Deserialize assignment may not narrow.");
    record->depthStencilFormat = transfer->depthStencilFormat;
    static_assert(sizeof(record->sampleCount) >= sizeof(transfer->sampleCount), "Deserialize assignment may not narrow.");
    record->sampleCount = transfer->sampleCount;
    static_assert(sizeof(record->depthReadOnly) >= sizeof(transfer->depthReadOnly), "Deserialize assignment may not narrow.");
    record->depthReadOnly = transfer->depthReadOnly;
    static_assert(sizeof(record->stencilReadOnly) >= sizeof(transfer->stencilReadOnly), "Deserialize assignment may not narrow.");
    record->stencilReadOnly = transfer->stencilReadOnly;
    {
        auto memberLength = record->colorFormatCount;
        const volatile WGPUTextureFormat* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureFormat* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->colorFormats = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUTextureFormat*>(memberBuffer),
              sizeof(WGPUTextureFormat) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderBundleEncoderDescriptorDeserialize);

struct WGPURenderPassColorAttachmentTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId view;
    uint32_t depthSlice;
    ObjectId resolveTarget;
    WGPULoadOp loadOp;
    WGPUStoreOp storeOp;
    WGPUColorTransfer clearValue;
};


[[maybe_unused]] size_t WGPURenderPassColorAttachmentGetExtraRequiredSize([[maybe_unused]] const WGPURenderPassColorAttachment& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUColorGetExtraRequiredSize(record.clearValue);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPassColorAttachmentGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPassColorAttachmentSerialize(
    const WGPURenderPassColorAttachment& record,
    WGPURenderPassColorAttachmentTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetOptionalId(record.view, &transfer->view));
    transfer->depthSlice = record.depthSlice;
    WIRE_TRY(provider.GetOptionalId(record.resolveTarget, &transfer->resolveTarget));
    transfer->loadOp = record.loadOp;
    transfer->storeOp = record.storeOp;
    WIRE_TRY(WGPUColorSerialize(record.clearValue, &transfer->clearValue, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassColorAttachmentSerialize);

[[maybe_unused]] WireResult WGPURenderPassColorAttachmentDeserialize(
    WGPURenderPassColorAttachment* record,
    const volatile WGPURenderPassColorAttachmentTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetOptionalFromId(transfer->view, &record->view));
    static_assert(sizeof(record->depthSlice) >= sizeof(transfer->depthSlice), "Deserialize assignment may not narrow.");
    record->depthSlice = transfer->depthSlice;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->resolveTarget, &record->resolveTarget));
    static_assert(sizeof(record->loadOp) >= sizeof(transfer->loadOp), "Deserialize assignment may not narrow.");
    record->loadOp = transfer->loadOp;
    static_assert(sizeof(record->storeOp) >= sizeof(transfer->storeOp), "Deserialize assignment may not narrow.");
    record->storeOp = transfer->storeOp;
    static_assert(sizeof(record->clearValue) == sizeof(transfer->clearValue), "Deserialize memcpy size must match.");
    memcpy(&record->clearValue, const_cast<const WGPUColorTransfer*>(&transfer->clearValue), sizeof(WGPUColorTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassColorAttachmentDeserialize);

struct WGPURenderPassStorageAttachmentTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    uint64_t offset;
    ObjectId storage;
    WGPULoadOp loadOp;
    WGPUStoreOp storeOp;
    WGPUColorTransfer clearValue;
};


[[maybe_unused]] size_t WGPURenderPassStorageAttachmentGetExtraRequiredSize([[maybe_unused]] const WGPURenderPassStorageAttachment& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUColorGetExtraRequiredSize(record.clearValue);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPassStorageAttachmentGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPassStorageAttachmentSerialize(
    const WGPURenderPassStorageAttachment& record,
    WGPURenderPassStorageAttachmentTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->offset = record.offset;
    WIRE_TRY(provider.GetId(record.storage, &transfer->storage));
    transfer->loadOp = record.loadOp;
    transfer->storeOp = record.storeOp;
    WIRE_TRY(WGPUColorSerialize(record.clearValue, &transfer->clearValue, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassStorageAttachmentSerialize);

[[maybe_unused]] WireResult WGPURenderPassStorageAttachmentDeserialize(
    WGPURenderPassStorageAttachment* record,
    const volatile WGPURenderPassStorageAttachmentTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    WIRE_TRY(resolver.GetFromId(transfer->storage, &record->storage));
    static_assert(sizeof(record->loadOp) >= sizeof(transfer->loadOp), "Deserialize assignment may not narrow.");
    record->loadOp = transfer->loadOp;
    static_assert(sizeof(record->storeOp) >= sizeof(transfer->storeOp), "Deserialize assignment may not narrow.");
    record->storeOp = transfer->storeOp;
    static_assert(sizeof(record->clearValue) == sizeof(transfer->clearValue), "Deserialize memcpy size must match.");
    memcpy(&record->clearValue, const_cast<const WGPUColorTransfer*>(&transfer->clearValue), sizeof(WGPUColorTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassStorageAttachmentDeserialize);

struct WGPURequiredLimitsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPULimitsTransfer limits;
};


[[maybe_unused]] size_t WGPURequiredLimitsGetExtraRequiredSize([[maybe_unused]] const WGPURequiredLimits& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPULimitsGetExtraRequiredSize(record.limits);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURequiredLimitsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURequiredLimitsSerialize(
    const WGPURequiredLimits& record,
    WGPURequiredLimitsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPULimitsSerialize(record.limits, &transfer->limits, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURequiredLimitsSerialize);

[[maybe_unused]] WireResult WGPURequiredLimitsDeserialize(
    WGPURequiredLimits* record,
    const volatile WGPURequiredLimitsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->limits) == sizeof(transfer->limits), "Deserialize memcpy size must match.");
    memcpy(&record->limits, const_cast<const WGPULimitsTransfer*>(&transfer->limits), sizeof(WGPULimitsTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURequiredLimitsDeserialize);

struct WGPUSamplerDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    WGPUAddressMode addressModeU;
    WGPUAddressMode addressModeV;
    WGPUAddressMode addressModeW;
    WGPUFilterMode magFilter;
    WGPUFilterMode minFilter;
    WGPUMipmapFilterMode mipmapFilter;
    float lodMinClamp;
    float lodMaxClamp;
    WGPUCompareFunction compare;
    uint16_t maxAnisotropy;
};


[[maybe_unused]] size_t WGPUSamplerDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUSamplerDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSamplerDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSamplerDescriptorSerialize(
    const WGPUSamplerDescriptor& record,
    WGPUSamplerDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->addressModeU = record.addressModeU;
    transfer->addressModeV = record.addressModeV;
    transfer->addressModeW = record.addressModeW;
    transfer->magFilter = record.magFilter;
    transfer->minFilter = record.minFilter;
    transfer->mipmapFilter = record.mipmapFilter;
    transfer->lodMinClamp = record.lodMinClamp;
    transfer->lodMaxClamp = record.lodMaxClamp;
    transfer->compare = record.compare;
    transfer->maxAnisotropy = record.maxAnisotropy;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSamplerDescriptorSerialize);

[[maybe_unused]] WireResult WGPUSamplerDescriptorDeserialize(
    WGPUSamplerDescriptor* record,
    const volatile WGPUSamplerDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    static_assert(sizeof(record->addressModeU) >= sizeof(transfer->addressModeU), "Deserialize assignment may not narrow.");
    record->addressModeU = transfer->addressModeU;
    static_assert(sizeof(record->addressModeV) >= sizeof(transfer->addressModeV), "Deserialize assignment may not narrow.");
    record->addressModeV = transfer->addressModeV;
    static_assert(sizeof(record->addressModeW) >= sizeof(transfer->addressModeW), "Deserialize assignment may not narrow.");
    record->addressModeW = transfer->addressModeW;
    static_assert(sizeof(record->magFilter) >= sizeof(transfer->magFilter), "Deserialize assignment may not narrow.");
    record->magFilter = transfer->magFilter;
    static_assert(sizeof(record->minFilter) >= sizeof(transfer->minFilter), "Deserialize assignment may not narrow.");
    record->minFilter = transfer->minFilter;
    static_assert(sizeof(record->mipmapFilter) >= sizeof(transfer->mipmapFilter), "Deserialize assignment may not narrow.");
    record->mipmapFilter = transfer->mipmapFilter;
    static_assert(sizeof(record->lodMinClamp) >= sizeof(transfer->lodMinClamp), "Deserialize assignment may not narrow.");
    record->lodMinClamp = transfer->lodMinClamp;
    static_assert(sizeof(record->lodMaxClamp) >= sizeof(transfer->lodMaxClamp), "Deserialize assignment may not narrow.");
    record->lodMaxClamp = transfer->lodMaxClamp;
    static_assert(sizeof(record->compare) >= sizeof(transfer->compare), "Deserialize assignment may not narrow.");
    record->compare = transfer->compare;
    static_assert(sizeof(record->maxAnisotropy) >= sizeof(transfer->maxAnisotropy), "Deserialize assignment may not narrow.");
    record->maxAnisotropy = transfer->maxAnisotropy;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSamplerDescriptorDeserialize);

struct WGPUShaderModuleDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
};


[[maybe_unused]] size_t WGPUShaderModuleDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUShaderModuleDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUShaderModuleDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUShaderModuleDescriptorSerialize(
    const WGPUShaderModuleDescriptor& record,
    WGPUShaderModuleDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderModuleDescriptorSerialize);

[[maybe_unused]] WireResult WGPUShaderModuleDescriptorDeserialize(
    WGPUShaderModuleDescriptor* record,
    const volatile WGPUShaderModuleDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderModuleDescriptorDeserialize);

struct WGPUShaderSourceWGSLTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    WGPUStringViewTransfer code;
};

static_assert(offsetof(WGPUShaderSourceWGSLTransfer, chain) == 0);

[[maybe_unused]] size_t WGPUShaderSourceWGSLGetExtraRequiredSize([[maybe_unused]] const WGPUShaderSourceWGSL& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.code);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUShaderSourceWGSLGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUShaderSourceWGSLSerialize(
    const WGPUShaderSourceWGSL& record,
    WGPUShaderSourceWGSLTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_ShaderSourceWGSL);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    WIRE_TRY(WGPUStringViewSerialize(record.code, &transfer->code, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderSourceWGSLSerialize);

[[maybe_unused]] WireResult WGPUShaderSourceWGSLDeserialize(
    WGPUShaderSourceWGSL* record,
    const volatile WGPUShaderSourceWGSLTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_ShaderSourceWGSL);
    DAWN_ASSERT(record->chain.next == nullptr);

    WIRE_TRY(WGPUStringViewDeserialize(&record->code, &transfer->code, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUShaderSourceWGSLDeserialize);

struct WGPUSupportedLimitsTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPULimitsTransfer limits;
};


[[maybe_unused]] size_t WGPUSupportedLimitsGetExtraRequiredSize([[maybe_unused]] const WGPUSupportedLimits& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPULimitsGetExtraRequiredSize(record.limits);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSupportedLimitsGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSupportedLimitsSerialize(
    const WGPUSupportedLimits& record,
    WGPUSupportedLimitsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPULimitsSerialize(record.limits, &transfer->limits, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSupportedLimitsSerialize);

[[maybe_unused]] WireResult WGPUSupportedLimitsDeserialize(
    WGPUSupportedLimits* record,
    const volatile WGPUSupportedLimitsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->limits) == sizeof(transfer->limits), "Deserialize memcpy size must match.");
    memcpy(&record->limits, const_cast<const WGPULimitsTransfer*>(&transfer->limits), sizeof(WGPULimitsTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSupportedLimitsDeserialize);

struct WGPUSurfaceDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
};


[[maybe_unused]] size_t WGPUSurfaceDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUSurfaceDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUSurfaceDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUSurfaceDescriptorSerialize(
    const WGPUSurfaceDescriptor& record,
    WGPUSurfaceDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceDescriptorSerialize);

[[maybe_unused]] WireResult WGPUSurfaceDescriptorDeserialize(
    WGPUSurfaceDescriptor* record,
    const volatile WGPUSurfaceDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUSurfaceDescriptorDeserialize);

struct WGPUTextureDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    WGPUTextureUsage usage;
    WGPUTextureDimension dimension;
    WGPUExtent3DTransfer size;
    WGPUTextureFormat format;
    uint32_t mipLevelCount;
    uint32_t sampleCount;
    uint64_t viewFormatCount;
};


[[maybe_unused]] size_t WGPUTextureDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUTextureDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    result += WGPUExtent3DGetExtraRequiredSize(record.size);
    {
        auto memberLength = record.viewFormatCount;
        auto size = WireAlignSizeofN<WGPUTextureFormat>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUTextureDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUTextureDescriptorSerialize(
    const WGPUTextureDescriptor& record,
    WGPUTextureDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->usage = record.usage;
    transfer->dimension = record.dimension;
    WIRE_TRY(WGPUExtent3DSerialize(record.size, &transfer->size, buffer));
    transfer->format = record.format;
    transfer->mipLevelCount = record.mipLevelCount;
    transfer->sampleCount = record.sampleCount;
    transfer->viewFormatCount = record.viewFormatCount;
    {
        auto memberLength = record.viewFormatCount;

        WGPUTextureFormat* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.viewFormats,
                sizeof(WGPUTextureFormat) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureDescriptorSerialize);

[[maybe_unused]] WireResult WGPUTextureDescriptorDeserialize(
    WGPUTextureDescriptor* record,
    const volatile WGPUTextureDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    static_assert(sizeof(record->usage) >= sizeof(transfer->usage), "Deserialize assignment may not narrow.");
    record->usage = transfer->usage;
    static_assert(sizeof(record->dimension) >= sizeof(transfer->dimension), "Deserialize assignment may not narrow.");
    record->dimension = transfer->dimension;
    static_assert(sizeof(record->size) == sizeof(transfer->size), "Deserialize memcpy size must match.");
    memcpy(&record->size, const_cast<const WGPUExtent3DTransfer*>(&transfer->size), sizeof(WGPUExtent3DTransfer));
    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->mipLevelCount) >= sizeof(transfer->mipLevelCount), "Deserialize assignment may not narrow.");
    record->mipLevelCount = transfer->mipLevelCount;
    static_assert(sizeof(record->sampleCount) >= sizeof(transfer->sampleCount), "Deserialize assignment may not narrow.");
    record->sampleCount = transfer->sampleCount;
    if (transfer->viewFormatCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->viewFormatCount = checked_cast<size_t>(transfer->viewFormatCount);
    {
        auto memberLength = record->viewFormatCount;
        const volatile WGPUTextureFormat* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureFormat* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->viewFormats = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUTextureFormat*>(memberBuffer),
              sizeof(WGPUTextureFormat) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureDescriptorDeserialize);

struct WGPUTextureViewDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    WGPUTextureFormat format;
    WGPUTextureViewDimension dimension;
    uint32_t baseMipLevel;
    uint32_t mipLevelCount;
    uint32_t baseArrayLayer;
    uint32_t arrayLayerCount;
    WGPUTextureAspect aspect;
    WGPUTextureUsage usage;
};


[[maybe_unused]] size_t WGPUTextureViewDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUTextureViewDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUTextureViewDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUTextureViewDescriptorSerialize(
    const WGPUTextureViewDescriptor& record,
    WGPUTextureViewDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->format = record.format;
    transfer->dimension = record.dimension;
    transfer->baseMipLevel = record.baseMipLevel;
    transfer->mipLevelCount = record.mipLevelCount;
    transfer->baseArrayLayer = record.baseArrayLayer;
    transfer->arrayLayerCount = record.arrayLayerCount;
    transfer->aspect = record.aspect;
    transfer->usage = record.usage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureViewDescriptorSerialize);

[[maybe_unused]] WireResult WGPUTextureViewDescriptorDeserialize(
    WGPUTextureViewDescriptor* record,
    const volatile WGPUTextureViewDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->dimension) >= sizeof(transfer->dimension), "Deserialize assignment may not narrow.");
    record->dimension = transfer->dimension;
    static_assert(sizeof(record->baseMipLevel) >= sizeof(transfer->baseMipLevel), "Deserialize assignment may not narrow.");
    record->baseMipLevel = transfer->baseMipLevel;
    static_assert(sizeof(record->mipLevelCount) >= sizeof(transfer->mipLevelCount), "Deserialize assignment may not narrow.");
    record->mipLevelCount = transfer->mipLevelCount;
    static_assert(sizeof(record->baseArrayLayer) >= sizeof(transfer->baseArrayLayer), "Deserialize assignment may not narrow.");
    record->baseArrayLayer = transfer->baseArrayLayer;
    static_assert(sizeof(record->arrayLayerCount) >= sizeof(transfer->arrayLayerCount), "Deserialize assignment may not narrow.");
    record->arrayLayerCount = transfer->arrayLayerCount;
    static_assert(sizeof(record->aspect) >= sizeof(transfer->aspect), "Deserialize assignment may not narrow.");
    record->aspect = transfer->aspect;
    static_assert(sizeof(record->usage) >= sizeof(transfer->usage), "Deserialize assignment may not narrow.");
    record->usage = transfer->usage;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUTextureViewDescriptorDeserialize);

struct WGPUVertexBufferLayoutTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");

    uint64_t arrayStride;
    WGPUVertexStepMode stepMode;
    uint64_t attributeCount;
};


[[maybe_unused]] size_t WGPUVertexBufferLayoutGetExtraRequiredSize([[maybe_unused]] const WGPUVertexBufferLayout& record) {
    size_t result = 0;

    {
        auto memberLength = record.attributeCount;
        auto size = WireAlignSizeofN<WGPUVertexAttributeTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUVertexAttributeGetExtraRequiredSize(record.attributes[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUVertexBufferLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUVertexBufferLayoutSerialize(
    const WGPUVertexBufferLayout& record,
    WGPUVertexBufferLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {


    transfer->arrayStride = record.arrayStride;
    transfer->stepMode = record.stepMode;
    transfer->attributeCount = record.attributeCount;
    {
        auto memberLength = record.attributeCount;

        WGPUVertexAttributeTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.attributes,
                sizeof(WGPUVertexAttributeTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUVertexBufferLayoutSerialize);

[[maybe_unused]] WireResult WGPUVertexBufferLayoutDeserialize(
    WGPUVertexBufferLayout* record,
    const volatile WGPUVertexBufferLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {


    static_assert(sizeof(record->arrayStride) >= sizeof(transfer->arrayStride), "Deserialize assignment may not narrow.");
    record->arrayStride = transfer->arrayStride;
    static_assert(sizeof(record->stepMode) >= sizeof(transfer->stepMode), "Deserialize assignment may not narrow.");
    record->stepMode = transfer->stepMode;
    if (transfer->attributeCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->attributeCount = checked_cast<size_t>(transfer->attributeCount);
    {
        auto memberLength = record->attributeCount;
        const volatile WGPUVertexAttributeTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUVertexAttribute* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->attributes = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUVertexAttributeTransfer*>(memberBuffer),
              sizeof(WGPUVertexAttributeTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUVertexBufferLayoutDeserialize);

struct WGPUBindGroupLayoutDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    uint64_t entryCount;
};


[[maybe_unused]] size_t WGPUBindGroupLayoutDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUBindGroupLayoutDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    {
        auto memberLength = record.entryCount;
        auto size = WireAlignSizeofN<WGPUBindGroupLayoutEntryTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUBindGroupLayoutEntryGetExtraRequiredSize(record.entries[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUBindGroupLayoutDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUBindGroupLayoutDescriptorSerialize(
    const WGPUBindGroupLayoutDescriptor& record,
    WGPUBindGroupLayoutDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->entryCount = record.entryCount;
    {
        auto memberLength = record.entryCount;

        WGPUBindGroupLayoutEntryTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupLayoutEntrySerialize(record.entries[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupLayoutDescriptorSerialize);

[[maybe_unused]] WireResult WGPUBindGroupLayoutDescriptorDeserialize(
    WGPUBindGroupLayoutDescriptor* record,
    const volatile WGPUBindGroupLayoutDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    if (transfer->entryCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->entryCount = checked_cast<size_t>(transfer->entryCount);
    {
        auto memberLength = record->entryCount;
        const volatile WGPUBindGroupLayoutEntryTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBindGroupLayoutEntry* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->entries = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupLayoutEntryDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUBindGroupLayoutDescriptorDeserialize);

struct WGPUColorTargetStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUTextureFormat format;
    WGPUBool has_blend;
    WGPUColorWriteMask writeMask;
};


[[maybe_unused]] size_t WGPUColorTargetStateGetExtraRequiredSize([[maybe_unused]] const WGPUColorTargetState& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    if (record.blend != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUBlendStateTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUBlendStateGetExtraRequiredSize(record.blend[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUColorTargetStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUColorTargetStateSerialize(
    const WGPUColorTargetState& record,
    WGPUColorTargetStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->format = record.format;
    transfer->writeMask = record.writeMask;
    bool has_blend = record.blend != nullptr;
    transfer->has_blend = has_blend;
    if (has_blend) {
        auto memberLength = 1u;

        WGPUBlendStateTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.blend,
                sizeof(WGPUBlendStateTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUColorTargetStateSerialize);

[[maybe_unused]] WireResult WGPUColorTargetStateDeserialize(
    WGPUColorTargetState* record,
    const volatile WGPUColorTargetStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->writeMask) >= sizeof(transfer->writeMask), "Deserialize assignment may not narrow.");
    record->writeMask = transfer->writeMask;
    bool has_blend = transfer->has_blend;
    record->blend = nullptr;
    if (has_blend) {
        auto memberLength = 1u;
        const volatile WGPUBlendStateTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBlendState* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->blend = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUBlendStateTransfer*>(memberBuffer),
              sizeof(WGPUBlendStateTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUColorTargetStateDeserialize);

struct WGPUCompilationInfoTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    uint64_t messageCount;
};


[[maybe_unused]] size_t WGPUCompilationInfoGetExtraRequiredSize([[maybe_unused]] const WGPUCompilationInfo& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    {
        auto memberLength = record.messageCount;
        auto size = WireAlignSizeofN<WGPUCompilationMessageTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUCompilationMessageGetExtraRequiredSize(record.messages[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUCompilationInfoGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUCompilationInfoSerialize(
    const WGPUCompilationInfo& record,
    WGPUCompilationInfoTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    transfer->messageCount = record.messageCount;
    {
        auto memberLength = record.messageCount;

        WGPUCompilationMessageTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCompilationMessageSerialize(record.messages[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCompilationInfoSerialize);

[[maybe_unused]] WireResult WGPUCompilationInfoDeserialize(
    WGPUCompilationInfo* record,
    const volatile WGPUCompilationInfoTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    if (transfer->messageCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->messageCount = checked_cast<size_t>(transfer->messageCount);
    {
        auto memberLength = record->messageCount;
        const volatile WGPUCompilationMessageTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCompilationMessage* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->messages = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCompilationMessageDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUCompilationInfoDeserialize);

struct WGPUComputeStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId module;
    WGPUStringViewTransfer entryPoint;
    uint64_t constantCount;
};


[[maybe_unused]] size_t WGPUComputeStateGetExtraRequiredSize([[maybe_unused]] const WGPUComputeState& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.entryPoint);
    {
        auto memberLength = record.constantCount;
        auto size = WireAlignSizeofN<WGPUConstantEntryTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUConstantEntryGetExtraRequiredSize(record.constants[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUComputeStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUComputeStateSerialize(
    const WGPUComputeState& record,
    WGPUComputeStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetId(record.module, &transfer->module));
    WIRE_TRY(WGPUStringViewSerialize(record.entryPoint, &transfer->entryPoint, buffer));
    transfer->constantCount = record.constantCount;
    {
        auto memberLength = record.constantCount;

        WGPUConstantEntryTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUConstantEntrySerialize(record.constants[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUComputeStateSerialize);

[[maybe_unused]] WireResult WGPUComputeStateDeserialize(
    WGPUComputeState* record,
    const volatile WGPUComputeStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetFromId(transfer->module, &record->module));
    WIRE_TRY(WGPUStringViewDeserialize(&record->entryPoint, &transfer->entryPoint, deserializeBuffer, allocator));
    if (transfer->constantCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->constantCount = checked_cast<size_t>(transfer->constantCount);
    {
        auto memberLength = record->constantCount;
        const volatile WGPUConstantEntryTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUConstantEntry* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->constants = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUConstantEntryDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUComputeStateDeserialize);

struct WGPUDeviceDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    uint64_t requiredFeatureCount;
    WGPUBool has_requiredLimits;
    WGPUQueueDescriptorTransfer defaultQueue;
};


[[maybe_unused]] size_t WGPUDeviceDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUDeviceDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    {
        auto memberLength = record.requiredFeatureCount;
        auto size = WireAlignSizeofN<WGPUFeatureName>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    if (record.requiredLimits != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURequiredLimitsTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURequiredLimitsGetExtraRequiredSize(record.requiredLimits[i]);
        }
    }
    result += WGPUQueueDescriptorGetExtraRequiredSize(record.defaultQueue);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUDeviceDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUDeviceDescriptorSerialize(
    const WGPUDeviceDescriptor& record,
    WGPUDeviceDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->requiredFeatureCount = record.requiredFeatureCount;
    WIRE_TRY(WGPUQueueDescriptorSerialize(record.defaultQueue, &transfer->defaultQueue, buffer, provider));
    {
        auto memberLength = record.requiredFeatureCount;

        WGPUFeatureName* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.requiredFeatures,
                sizeof(WGPUFeatureName) * memberLength);
        }
    }
    bool has_requiredLimits = record.requiredLimits != nullptr;
    transfer->has_requiredLimits = has_requiredLimits;
    if (has_requiredLimits) {
        auto memberLength = 1u;

        WGPURequiredLimitsTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURequiredLimitsSerialize(record.requiredLimits[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDeviceDescriptorSerialize);

[[maybe_unused]] WireResult WGPUDeviceDescriptorDeserialize(
    WGPUDeviceDescriptor* record,
    const volatile WGPUDeviceDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    if (transfer->requiredFeatureCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->requiredFeatureCount = checked_cast<size_t>(transfer->requiredFeatureCount);
    WIRE_TRY(WGPUQueueDescriptorDeserialize(&record->defaultQueue, &transfer->defaultQueue, deserializeBuffer, allocator, resolver));
    record->deviceLostCallbackInfo = WGPU_DEVICE_LOST_CALLBACK_INFO_INIT;
    record->uncapturedErrorCallbackInfo = WGPU_UNCAPTURED_ERROR_CALLBACK_INFO_INIT;
    {
        auto memberLength = record->requiredFeatureCount;
        const volatile WGPUFeatureName* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUFeatureName* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->requiredFeatures = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUFeatureName*>(memberBuffer),
              sizeof(WGPUFeatureName) * memberLength);
        }
    }
    bool has_requiredLimits = transfer->has_requiredLimits;
    record->requiredLimits = nullptr;
    if (has_requiredLimits) {
        auto memberLength = 1u;
        const volatile WGPURequiredLimitsTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURequiredLimits* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->requiredLimits = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURequiredLimitsDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUDeviceDescriptorDeserialize);

struct WGPURenderPassDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    uint64_t colorAttachmentCount;
    WGPUBool has_depthStencilAttachment;
    ObjectId occlusionQuerySet;
    WGPUBool has_timestampWrites;
};


[[maybe_unused]] size_t WGPURenderPassDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPURenderPassDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    {
        auto memberLength = record.colorAttachmentCount;
        auto size = WireAlignSizeofN<WGPURenderPassColorAttachmentTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderPassColorAttachmentGetExtraRequiredSize(record.colorAttachments[i]);
        }
    }
    if (record.depthStencilAttachment != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURenderPassDepthStencilAttachmentTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderPassDepthStencilAttachmentGetExtraRequiredSize(record.depthStencilAttachment[i]);
        }
    }
    if (record.timestampWrites != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUPassTimestampWritesTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUPassTimestampWritesGetExtraRequiredSize(record.timestampWrites[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPassDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPassDescriptorSerialize(
    const WGPURenderPassDescriptor& record,
    WGPURenderPassDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    transfer->colorAttachmentCount = record.colorAttachmentCount;
    WIRE_TRY(provider.GetOptionalId(record.occlusionQuerySet, &transfer->occlusionQuerySet));
    {
        auto memberLength = record.colorAttachmentCount;

        WGPURenderPassColorAttachmentTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassColorAttachmentSerialize(record.colorAttachments[i], &memberBuffer[i], buffer, provider));
        }
    }
    bool has_depthStencilAttachment = record.depthStencilAttachment != nullptr;
    transfer->has_depthStencilAttachment = has_depthStencilAttachment;
    if (has_depthStencilAttachment) {
        auto memberLength = 1u;

        WGPURenderPassDepthStencilAttachmentTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassDepthStencilAttachmentSerialize(record.depthStencilAttachment[i], &memberBuffer[i], buffer, provider));
        }
    }
    bool has_timestampWrites = record.timestampWrites != nullptr;
    transfer->has_timestampWrites = has_timestampWrites;
    if (has_timestampWrites) {
        auto memberLength = 1u;

        WGPUPassTimestampWritesTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUPassTimestampWritesSerialize(record.timestampWrites[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassDescriptorSerialize);

[[maybe_unused]] WireResult WGPURenderPassDescriptorDeserialize(
    WGPURenderPassDescriptor* record,
    const volatile WGPURenderPassDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    if (transfer->colorAttachmentCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->colorAttachmentCount = checked_cast<size_t>(transfer->colorAttachmentCount);
    WIRE_TRY(resolver.GetOptionalFromId(transfer->occlusionQuerySet, &record->occlusionQuerySet));
    {
        auto memberLength = record->colorAttachmentCount;
        const volatile WGPURenderPassColorAttachmentTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderPassColorAttachment* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->colorAttachments = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassColorAttachmentDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    bool has_depthStencilAttachment = transfer->has_depthStencilAttachment;
    record->depthStencilAttachment = nullptr;
    if (has_depthStencilAttachment) {
        auto memberLength = 1u;
        const volatile WGPURenderPassDepthStencilAttachmentTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderPassDepthStencilAttachment* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->depthStencilAttachment = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassDepthStencilAttachmentDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    bool has_timestampWrites = transfer->has_timestampWrites;
    record->timestampWrites = nullptr;
    if (has_timestampWrites) {
        auto memberLength = 1u;
        const volatile WGPUPassTimestampWritesTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUPassTimestampWrites* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->timestampWrites = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUPassTimestampWritesDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassDescriptorDeserialize);

struct WGPURenderPassPixelLocalStorageTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUChainedStructTransfer chain;

    uint64_t totalPixelLocalStorageSize;
    uint64_t storageAttachmentCount;
};

static_assert(offsetof(WGPURenderPassPixelLocalStorageTransfer, chain) == 0);

[[maybe_unused]] size_t WGPURenderPassPixelLocalStorageGetExtraRequiredSize([[maybe_unused]] const WGPURenderPassPixelLocalStorage& record) {
    size_t result = 0;

    {
        auto memberLength = record.storageAttachmentCount;
        auto size = WireAlignSizeofN<WGPURenderPassStorageAttachmentTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderPassStorageAttachmentGetExtraRequiredSize(record.storageAttachments[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPassPixelLocalStorageGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPassPixelLocalStorageSerialize(
    const WGPURenderPassPixelLocalStorage& record,
    WGPURenderPassPixelLocalStorageTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    DAWN_ASSERT(transfer->chain.sType == WGPUSType_RenderPassPixelLocalStorage);
    DAWN_ASSERT(transfer->chain.hasNext == (record.chain.next != nullptr));

    transfer->totalPixelLocalStorageSize = record.totalPixelLocalStorageSize;
    transfer->storageAttachmentCount = record.storageAttachmentCount;
    {
        auto memberLength = record.storageAttachmentCount;

        WGPURenderPassStorageAttachmentTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassStorageAttachmentSerialize(record.storageAttachments[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassPixelLocalStorageSerialize);

[[maybe_unused]] WireResult WGPURenderPassPixelLocalStorageDeserialize(
    WGPURenderPassPixelLocalStorage* record,
    const volatile WGPURenderPassPixelLocalStorageTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    DAWN_ASSERT(record->chain.sType == WGPUSType_RenderPassPixelLocalStorage);
    DAWN_ASSERT(record->chain.next == nullptr);

    static_assert(sizeof(record->totalPixelLocalStorageSize) >= sizeof(transfer->totalPixelLocalStorageSize), "Deserialize assignment may not narrow.");
    record->totalPixelLocalStorageSize = transfer->totalPixelLocalStorageSize;
    if (transfer->storageAttachmentCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->storageAttachmentCount = checked_cast<size_t>(transfer->storageAttachmentCount);
    {
        auto memberLength = record->storageAttachmentCount;
        const volatile WGPURenderPassStorageAttachmentTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderPassStorageAttachment* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->storageAttachments = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassStorageAttachmentDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPassPixelLocalStorageDeserialize);

struct WGPUVertexStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId module;
    WGPUStringViewTransfer entryPoint;
    uint64_t constantCount;
    uint64_t bufferCount;
};


[[maybe_unused]] size_t WGPUVertexStateGetExtraRequiredSize([[maybe_unused]] const WGPUVertexState& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.entryPoint);
    {
        auto memberLength = record.constantCount;
        auto size = WireAlignSizeofN<WGPUConstantEntryTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUConstantEntryGetExtraRequiredSize(record.constants[i]);
        }
    }
    {
        auto memberLength = record.bufferCount;
        auto size = WireAlignSizeofN<WGPUVertexBufferLayoutTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUVertexBufferLayoutGetExtraRequiredSize(record.buffers[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUVertexStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUVertexStateSerialize(
    const WGPUVertexState& record,
    WGPUVertexStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetId(record.module, &transfer->module));
    WIRE_TRY(WGPUStringViewSerialize(record.entryPoint, &transfer->entryPoint, buffer));
    transfer->constantCount = record.constantCount;
    transfer->bufferCount = record.bufferCount;
    {
        auto memberLength = record.constantCount;

        WGPUConstantEntryTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUConstantEntrySerialize(record.constants[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = record.bufferCount;

        WGPUVertexBufferLayoutTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUVertexBufferLayoutSerialize(record.buffers[i], &memberBuffer[i], buffer));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUVertexStateSerialize);

[[maybe_unused]] WireResult WGPUVertexStateDeserialize(
    WGPUVertexState* record,
    const volatile WGPUVertexStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetFromId(transfer->module, &record->module));
    WIRE_TRY(WGPUStringViewDeserialize(&record->entryPoint, &transfer->entryPoint, deserializeBuffer, allocator));
    if (transfer->constantCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->constantCount = checked_cast<size_t>(transfer->constantCount);
    if (transfer->bufferCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->bufferCount = checked_cast<size_t>(transfer->bufferCount);
    {
        auto memberLength = record->constantCount;
        const volatile WGPUConstantEntryTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUConstantEntry* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->constants = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUConstantEntryDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = record->bufferCount;
        const volatile WGPUVertexBufferLayoutTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUVertexBufferLayout* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->buffers = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUVertexBufferLayoutDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUVertexStateDeserialize);

struct WGPUComputePipelineDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    ObjectId layout;
    WGPUComputeStateTransfer compute;
};


[[maybe_unused]] size_t WGPUComputePipelineDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPUComputePipelineDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    result += WGPUComputeStateGetExtraRequiredSize(record.compute);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUComputePipelineDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUComputePipelineDescriptorSerialize(
    const WGPUComputePipelineDescriptor& record,
    WGPUComputePipelineDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    WIRE_TRY(provider.GetOptionalId(record.layout, &transfer->layout));
    WIRE_TRY(WGPUComputeStateSerialize(record.compute, &transfer->compute, buffer, provider));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUComputePipelineDescriptorSerialize);

[[maybe_unused]] WireResult WGPUComputePipelineDescriptorDeserialize(
    WGPUComputePipelineDescriptor* record,
    const volatile WGPUComputePipelineDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    WIRE_TRY(resolver.GetOptionalFromId(transfer->layout, &record->layout));
    WIRE_TRY(WGPUComputeStateDeserialize(&record->compute, &transfer->compute, deserializeBuffer, allocator, resolver));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUComputePipelineDescriptorDeserialize);

struct WGPUFragmentStateTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    ObjectId module;
    WGPUStringViewTransfer entryPoint;
    uint64_t constantCount;
    uint64_t targetCount;
};


[[maybe_unused]] size_t WGPUFragmentStateGetExtraRequiredSize([[maybe_unused]] const WGPUFragmentState& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.entryPoint);
    {
        auto memberLength = record.constantCount;
        auto size = WireAlignSizeofN<WGPUConstantEntryTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUConstantEntryGetExtraRequiredSize(record.constants[i]);
        }
    }
    {
        auto memberLength = record.targetCount;
        auto size = WireAlignSizeofN<WGPUColorTargetStateTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUColorTargetStateGetExtraRequiredSize(record.targets[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPUFragmentStateGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPUFragmentStateSerialize(
    const WGPUFragmentState& record,
    WGPUFragmentStateTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(provider.GetId(record.module, &transfer->module));
    WIRE_TRY(WGPUStringViewSerialize(record.entryPoint, &transfer->entryPoint, buffer));
    transfer->constantCount = record.constantCount;
    transfer->targetCount = record.targetCount;
    {
        auto memberLength = record.constantCount;

        WGPUConstantEntryTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUConstantEntrySerialize(record.constants[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = record.targetCount;

        WGPUColorTargetStateTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUColorTargetStateSerialize(record.targets[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUFragmentStateSerialize);

[[maybe_unused]] WireResult WGPUFragmentStateDeserialize(
    WGPUFragmentState* record,
    const volatile WGPUFragmentStateTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(resolver.GetFromId(transfer->module, &record->module));
    WIRE_TRY(WGPUStringViewDeserialize(&record->entryPoint, &transfer->entryPoint, deserializeBuffer, allocator));
    if (transfer->constantCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->constantCount = checked_cast<size_t>(transfer->constantCount);
    if (transfer->targetCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->targetCount = checked_cast<size_t>(transfer->targetCount);
    {
        auto memberLength = record->constantCount;
        const volatile WGPUConstantEntryTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUConstantEntry* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->constants = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUConstantEntryDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = record->targetCount;
        const volatile WGPUColorTargetStateTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUColorTargetState* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->targets = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUColorTargetStateDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPUFragmentStateDeserialize);

struct WGPURenderPipelineDescriptorTransfer {
    static_assert(0 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WGPUBool hasNextInChain;

    WGPUStringViewTransfer label;
    ObjectId layout;
    WGPUVertexStateTransfer vertex;
    WGPUPrimitiveStateTransfer primitive;
    WGPUBool has_depthStencil;
    WGPUMultisampleStateTransfer multisample;
    WGPUBool has_fragment;
};


[[maybe_unused]] size_t WGPURenderPipelineDescriptorGetExtraRequiredSize([[maybe_unused]] const WGPURenderPipelineDescriptor& record) {
    size_t result = 0;

    if (record.nextInChain != nullptr) {
        result += GetChainedStructExtraRequiredSize(record.nextInChain);
    }
    result += WGPUStringViewGetExtraRequiredSize(record.label);
    result += WGPUVertexStateGetExtraRequiredSize(record.vertex);
    result += WGPUPrimitiveStateGetExtraRequiredSize(record.primitive);
    if (record.depthStencil != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUDepthStencilStateTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUDepthStencilStateGetExtraRequiredSize(record.depthStencil[i]);
        }
    }
    result += WGPUMultisampleStateGetExtraRequiredSize(record.multisample);
    if (record.fragment != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUFragmentStateTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUFragmentStateGetExtraRequiredSize(record.fragment[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(WGPURenderPipelineDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult WGPURenderPipelineDescriptorSerialize(
    const WGPURenderPipelineDescriptor& record,
    WGPURenderPipelineDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {

    if (record.nextInChain != nullptr) {
        transfer->hasNextInChain = true;
        WIRE_TRY(SerializeChainedStruct(record.nextInChain, buffer, provider));
    } else {
        transfer->hasNextInChain = false;
    }

    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));
    WIRE_TRY(provider.GetOptionalId(record.layout, &transfer->layout));
    WIRE_TRY(WGPUVertexStateSerialize(record.vertex, &transfer->vertex, buffer, provider));
    WIRE_TRY(WGPUPrimitiveStateSerialize(record.primitive, &transfer->primitive, buffer, provider));
    WIRE_TRY(WGPUMultisampleStateSerialize(record.multisample, &transfer->multisample, buffer, provider));
    bool has_depthStencil = record.depthStencil != nullptr;
    transfer->has_depthStencil = has_depthStencil;
    if (has_depthStencil) {
        auto memberLength = 1u;

        WGPUDepthStencilStateTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUDepthStencilStateSerialize(record.depthStencil[i], &memberBuffer[i], buffer, provider));
        }
    }
    bool has_fragment = record.fragment != nullptr;
    transfer->has_fragment = has_fragment;
    if (has_fragment) {
        auto memberLength = 1u;

        WGPUFragmentStateTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUFragmentStateSerialize(record.fragment[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPipelineDescriptorSerialize);

[[maybe_unused]] WireResult WGPURenderPipelineDescriptorDeserialize(
    WGPURenderPipelineDescriptor* record,
    const volatile WGPURenderPipelineDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {

    record->nextInChain = nullptr;
    if (transfer->hasNextInChain) {
        WIRE_TRY(DeserializeChainedStruct(&record->nextInChain, deserializeBuffer, allocator, resolver));
    }

    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));
    WIRE_TRY(resolver.GetOptionalFromId(transfer->layout, &record->layout));
    WIRE_TRY(WGPUVertexStateDeserialize(&record->vertex, &transfer->vertex, deserializeBuffer, allocator, resolver));
    WIRE_TRY(WGPUPrimitiveStateDeserialize(&record->primitive, &transfer->primitive, deserializeBuffer, allocator, resolver));
    WIRE_TRY(WGPUMultisampleStateDeserialize(&record->multisample, &transfer->multisample, deserializeBuffer, allocator, resolver));
    bool has_depthStencil = transfer->has_depthStencil;
    record->depthStencil = nullptr;
    if (has_depthStencil) {
        auto memberLength = 1u;
        const volatile WGPUDepthStencilStateTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUDepthStencilState* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->depthStencil = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUDepthStencilStateDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    bool has_fragment = transfer->has_fragment;
    record->fragment = nullptr;
    if (has_fragment) {
        auto memberLength = 1u;
        const volatile WGPUFragmentStateTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUFragmentState* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->fragment = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUFragmentStateDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(WGPURenderPipelineDescriptorDeserialize);


size_t GetChainedStructExtraRequiredSize(WGPUChainedStruct* chainedStruct) {
    DAWN_ASSERT(chainedStruct != nullptr);
    size_t result = 0;
    while (chainedStruct != nullptr) {
        uint32_t sType_as_uint;
        std::memcpy(&sType_as_uint, &(chainedStruct->sType), sizeof(uint32_t));
        switch (sType_as_uint) {
            case WGPUSType_ShaderSourceSPIRV: {
                const auto& typedStruct = *reinterpret_cast<WGPUShaderSourceSPIRV const *>(chainedStruct);
                result += WireAlignSizeof<WGPUShaderSourceSPIRVTransfer>();
                result += WGPUShaderSourceSPIRVGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_ShaderSourceWGSL: {
                const auto& typedStruct = *reinterpret_cast<WGPUShaderSourceWGSL const *>(chainedStruct);
                result += WireAlignSizeof<WGPUShaderSourceWGSLTransfer>();
                result += WGPUShaderSourceWGSLGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_RenderPassMaxDrawCount: {
                const auto& typedStruct = *reinterpret_cast<WGPURenderPassMaxDrawCount const *>(chainedStruct);
                result += WireAlignSizeof<WGPURenderPassMaxDrawCountTransfer>();
                result += WGPURenderPassMaxDrawCountGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_AdapterPropertiesSubgroups: {
                const auto& typedStruct = *reinterpret_cast<WGPUAdapterPropertiesSubgroups const *>(chainedStruct);
                result += WireAlignSizeof<WGPUAdapterPropertiesSubgroupsTransfer>();
                result += WGPUAdapterPropertiesSubgroupsGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_TextureBindingViewDimensionDescriptor: {
                const auto& typedStruct = *reinterpret_cast<WGPUTextureBindingViewDimensionDescriptor const *>(chainedStruct);
                result += WireAlignSizeof<WGPUTextureBindingViewDimensionDescriptorTransfer>();
                result += WGPUTextureBindingViewDimensionDescriptorGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_ExternalTextureBindingEntry: {
                const auto& typedStruct = *reinterpret_cast<WGPUExternalTextureBindingEntry const *>(chainedStruct);
                result += WireAlignSizeof<WGPUExternalTextureBindingEntryTransfer>();
                result += WGPUExternalTextureBindingEntryGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_ExternalTextureBindingLayout: {
                const auto& typedStruct = *reinterpret_cast<WGPUExternalTextureBindingLayout const *>(chainedStruct);
                result += WireAlignSizeof<WGPUExternalTextureBindingLayoutTransfer>();
                result += WGPUExternalTextureBindingLayoutGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnTextureInternalUsageDescriptor: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnTextureInternalUsageDescriptor const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnTextureInternalUsageDescriptorTransfer>();
                result += WGPUDawnTextureInternalUsageDescriptorGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnEncoderInternalUsageDescriptor: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnEncoderInternalUsageDescriptor const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnEncoderInternalUsageDescriptorTransfer>();
                result += WGPUDawnEncoderInternalUsageDescriptorGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnBufferDescriptorErrorInfoFromWireClient: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnBufferDescriptorErrorInfoFromWireClient const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnBufferDescriptorErrorInfoFromWireClientTransfer>();
                result += WGPUDawnBufferDescriptorErrorInfoFromWireClientGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnShaderModuleSPIRVOptionsDescriptor: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnShaderModuleSPIRVOptionsDescriptor const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnShaderModuleSPIRVOptionsDescriptorTransfer>();
                result += WGPUDawnShaderModuleSPIRVOptionsDescriptorGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnRenderPassColorAttachmentRenderToSingleSampled: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnRenderPassColorAttachmentRenderToSingleSampled const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnRenderPassColorAttachmentRenderToSingleSampledTransfer>();
                result += WGPUDawnRenderPassColorAttachmentRenderToSingleSampledGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_RenderPassPixelLocalStorage: {
                const auto& typedStruct = *reinterpret_cast<WGPURenderPassPixelLocalStorage const *>(chainedStruct);
                result += WireAlignSizeof<WGPURenderPassPixelLocalStorageTransfer>();
                result += WGPURenderPassPixelLocalStorageGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_PipelineLayoutPixelLocalStorage: {
                const auto& typedStruct = *reinterpret_cast<WGPUPipelineLayoutPixelLocalStorage const *>(chainedStruct);
                result += WireAlignSizeof<WGPUPipelineLayoutPixelLocalStorageTransfer>();
                result += WGPUPipelineLayoutPixelLocalStorageGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_BufferHostMappedPointer: {
                const auto& typedStruct = *reinterpret_cast<WGPUBufferHostMappedPointer const *>(chainedStruct);
                result += WireAlignSizeof<WGPUBufferHostMappedPointerTransfer>();
                result += WGPUBufferHostMappedPointerGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnExperimentalSubgroupLimits: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnExperimentalSubgroupLimits const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnExperimentalSubgroupLimitsTransfer>();
                result += WGPUDawnExperimentalSubgroupLimitsGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_AdapterPropertiesMemoryHeaps: {
                const auto& typedStruct = *reinterpret_cast<WGPUAdapterPropertiesMemoryHeaps const *>(chainedStruct);
                result += WireAlignSizeof<WGPUAdapterPropertiesMemoryHeapsTransfer>();
                result += WGPUAdapterPropertiesMemoryHeapsGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_AdapterPropertiesD3D: {
                const auto& typedStruct = *reinterpret_cast<WGPUAdapterPropertiesD3D const *>(chainedStruct);
                result += WireAlignSizeof<WGPUAdapterPropertiesD3DTransfer>();
                result += WGPUAdapterPropertiesD3DGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_AdapterPropertiesVk: {
                const auto& typedStruct = *reinterpret_cast<WGPUAdapterPropertiesVk const *>(chainedStruct);
                result += WireAlignSizeof<WGPUAdapterPropertiesVkTransfer>();
                result += WGPUAdapterPropertiesVkGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnWireWGSLControl: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnWireWGSLControl const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnWireWGSLControlTransfer>();
                result += WGPUDawnWireWGSLControlGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnDrmFormatCapabilities: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnDrmFormatCapabilities const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnDrmFormatCapabilitiesTransfer>();
                result += WGPUDawnDrmFormatCapabilitiesGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_ShaderModuleCompilationOptions: {
                const auto& typedStruct = *reinterpret_cast<WGPUShaderModuleCompilationOptions const *>(chainedStruct);
                result += WireAlignSizeof<WGPUShaderModuleCompilationOptionsTransfer>();
                result += WGPUShaderModuleCompilationOptionsGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_ColorTargetStateExpandResolveTextureDawn: {
                const auto& typedStruct = *reinterpret_cast<WGPUColorTargetStateExpandResolveTextureDawn const *>(chainedStruct);
                result += WireAlignSizeof<WGPUColorTargetStateExpandResolveTextureDawnTransfer>();
                result += WGPUColorTargetStateExpandResolveTextureDawnGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_RenderPassDescriptorExpandResolveRect: {
                const auto& typedStruct = *reinterpret_cast<WGPURenderPassDescriptorExpandResolveRect const *>(chainedStruct);
                result += WireAlignSizeof<WGPURenderPassDescriptorExpandResolveRectTransfer>();
                result += WGPURenderPassDescriptorExpandResolveRectGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_StaticSamplerBindingLayout: {
                const auto& typedStruct = *reinterpret_cast<WGPUStaticSamplerBindingLayout const *>(chainedStruct);
                result += WireAlignSizeof<WGPUStaticSamplerBindingLayoutTransfer>();
                result += WGPUStaticSamplerBindingLayoutGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_YCbCrVkDescriptor: {
                const auto& typedStruct = *reinterpret_cast<WGPUYCbCrVkDescriptor const *>(chainedStruct);
                result += WireAlignSizeof<WGPUYCbCrVkDescriptorTransfer>();
                result += WGPUYCbCrVkDescriptorGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnExperimentalImmediateDataLimits: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnExperimentalImmediateDataLimits const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnExperimentalImmediateDataLimitsTransfer>();
                result += WGPUDawnExperimentalImmediateDataLimitsGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            case WGPUSType_DawnTexelCopyBufferRowAlignmentLimits: {
                const auto& typedStruct = *reinterpret_cast<WGPUDawnTexelCopyBufferRowAlignmentLimits const *>(chainedStruct);
                result += WireAlignSizeof<WGPUDawnTexelCopyBufferRowAlignmentLimitsTransfer>();
                result += WGPUDawnTexelCopyBufferRowAlignmentLimitsGetExtraRequiredSize(typedStruct);
                chainedStruct = typedStruct.chain.next;
                break;
            }
            default:
                // Invalid enum. Reserve space just for the transfer header (sType and hasNext).
                result += WireAlignSizeof<WGPUChainedStructTransfer>();
                chainedStruct = chainedStruct->next;
                break;
        }
    }
    return result;
}

[[nodiscard]] WireResult SerializeChainedStruct(WGPUChainedStruct* chainedStruct,
                                                SerializeBuffer* buffer,
                                                const ObjectIdProvider& provider) {
    DAWN_ASSERT(chainedStruct != nullptr);
    DAWN_ASSERT(buffer != nullptr);
    do {
        uint32_t sType_as_uint;
        std::memcpy(&sType_as_uint, &(chainedStruct->sType), sizeof(uint32_t));
        switch (sType_as_uint) {
            case WGPUSType_ShaderSourceSPIRV: {
                WGPUShaderSourceSPIRVTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUShaderSourceSPIRVSerialize(*reinterpret_cast<WGPUShaderSourceSPIRV const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_ShaderSourceWGSL: {
                WGPUShaderSourceWGSLTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUShaderSourceWGSLSerialize(*reinterpret_cast<WGPUShaderSourceWGSL const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_RenderPassMaxDrawCount: {
                WGPURenderPassMaxDrawCountTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPURenderPassMaxDrawCountSerialize(*reinterpret_cast<WGPURenderPassMaxDrawCount const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_AdapterPropertiesSubgroups: {
                WGPUAdapterPropertiesSubgroupsTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUAdapterPropertiesSubgroupsSerialize(*reinterpret_cast<WGPUAdapterPropertiesSubgroups const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_TextureBindingViewDimensionDescriptor: {
                WGPUTextureBindingViewDimensionDescriptorTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUTextureBindingViewDimensionDescriptorSerialize(*reinterpret_cast<WGPUTextureBindingViewDimensionDescriptor const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_ExternalTextureBindingEntry: {
                WGPUExternalTextureBindingEntryTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUExternalTextureBindingEntrySerialize(*reinterpret_cast<WGPUExternalTextureBindingEntry const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_ExternalTextureBindingLayout: {
                WGPUExternalTextureBindingLayoutTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUExternalTextureBindingLayoutSerialize(*reinterpret_cast<WGPUExternalTextureBindingLayout const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnTextureInternalUsageDescriptor: {
                WGPUDawnTextureInternalUsageDescriptorTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnTextureInternalUsageDescriptorSerialize(*reinterpret_cast<WGPUDawnTextureInternalUsageDescriptor const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnEncoderInternalUsageDescriptor: {
                WGPUDawnEncoderInternalUsageDescriptorTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnEncoderInternalUsageDescriptorSerialize(*reinterpret_cast<WGPUDawnEncoderInternalUsageDescriptor const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnBufferDescriptorErrorInfoFromWireClient: {
                WGPUDawnBufferDescriptorErrorInfoFromWireClientTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnBufferDescriptorErrorInfoFromWireClientSerialize(*reinterpret_cast<WGPUDawnBufferDescriptorErrorInfoFromWireClient const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnShaderModuleSPIRVOptionsDescriptor: {
                WGPUDawnShaderModuleSPIRVOptionsDescriptorTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnShaderModuleSPIRVOptionsDescriptorSerialize(*reinterpret_cast<WGPUDawnShaderModuleSPIRVOptionsDescriptor const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnRenderPassColorAttachmentRenderToSingleSampled: {
                WGPUDawnRenderPassColorAttachmentRenderToSingleSampledTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnRenderPassColorAttachmentRenderToSingleSampledSerialize(*reinterpret_cast<WGPUDawnRenderPassColorAttachmentRenderToSingleSampled const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_RenderPassPixelLocalStorage: {
                WGPURenderPassPixelLocalStorageTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPURenderPassPixelLocalStorageSerialize(*reinterpret_cast<WGPURenderPassPixelLocalStorage const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_PipelineLayoutPixelLocalStorage: {
                WGPUPipelineLayoutPixelLocalStorageTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUPipelineLayoutPixelLocalStorageSerialize(*reinterpret_cast<WGPUPipelineLayoutPixelLocalStorage const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_BufferHostMappedPointer: {
                WGPUBufferHostMappedPointerTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUBufferHostMappedPointerSerialize(*reinterpret_cast<WGPUBufferHostMappedPointer const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnExperimentalSubgroupLimits: {
                WGPUDawnExperimentalSubgroupLimitsTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnExperimentalSubgroupLimitsSerialize(*reinterpret_cast<WGPUDawnExperimentalSubgroupLimits const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_AdapterPropertiesMemoryHeaps: {
                WGPUAdapterPropertiesMemoryHeapsTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUAdapterPropertiesMemoryHeapsSerialize(*reinterpret_cast<WGPUAdapterPropertiesMemoryHeaps const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_AdapterPropertiesD3D: {
                WGPUAdapterPropertiesD3DTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUAdapterPropertiesD3DSerialize(*reinterpret_cast<WGPUAdapterPropertiesD3D const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_AdapterPropertiesVk: {
                WGPUAdapterPropertiesVkTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUAdapterPropertiesVkSerialize(*reinterpret_cast<WGPUAdapterPropertiesVk const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnWireWGSLControl: {
                WGPUDawnWireWGSLControlTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnWireWGSLControlSerialize(*reinterpret_cast<WGPUDawnWireWGSLControl const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnDrmFormatCapabilities: {
                WGPUDawnDrmFormatCapabilitiesTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnDrmFormatCapabilitiesSerialize(*reinterpret_cast<WGPUDawnDrmFormatCapabilities const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_ShaderModuleCompilationOptions: {
                WGPUShaderModuleCompilationOptionsTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUShaderModuleCompilationOptionsSerialize(*reinterpret_cast<WGPUShaderModuleCompilationOptions const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_ColorTargetStateExpandResolveTextureDawn: {
                WGPUColorTargetStateExpandResolveTextureDawnTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUColorTargetStateExpandResolveTextureDawnSerialize(*reinterpret_cast<WGPUColorTargetStateExpandResolveTextureDawn const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_RenderPassDescriptorExpandResolveRect: {
                WGPURenderPassDescriptorExpandResolveRectTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPURenderPassDescriptorExpandResolveRectSerialize(*reinterpret_cast<WGPURenderPassDescriptorExpandResolveRect const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_StaticSamplerBindingLayout: {
                WGPUStaticSamplerBindingLayoutTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUStaticSamplerBindingLayoutSerialize(*reinterpret_cast<WGPUStaticSamplerBindingLayout const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_YCbCrVkDescriptor: {
                WGPUYCbCrVkDescriptorTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUYCbCrVkDescriptorSerialize(*reinterpret_cast<WGPUYCbCrVkDescriptor const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnExperimentalImmediateDataLimits: {
                WGPUDawnExperimentalImmediateDataLimitsTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnExperimentalImmediateDataLimitsSerialize(*reinterpret_cast<WGPUDawnExperimentalImmediateDataLimits const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            case WGPUSType_DawnTexelCopyBufferRowAlignmentLimits: {
                WGPUDawnTexelCopyBufferRowAlignmentLimitsTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->chain.sType = chainedStruct->sType;
                transfer->chain.hasNext = chainedStruct->next != nullptr;

                WIRE_TRY(WGPUDawnTexelCopyBufferRowAlignmentLimitsSerialize(*reinterpret_cast<WGPUDawnTexelCopyBufferRowAlignmentLimits const*>(chainedStruct), transfer, buffer, provider));

                chainedStruct = chainedStruct->next;
            } break;
            default: {
                // Invalid enum. Serialize just the transfer header with Invalid as the sType.
                // TODO(crbug.com/dawn/369): Unknown sTypes are silently discarded.
                if (sType_as_uint != 0u) {
                    dawn::WarningLog() << "Unknown sType " << sType_as_uint << " discarded.";
                }

                WGPUChainedStructTransfer* transfer;
                WIRE_TRY(buffer->Next(&transfer));
                transfer->sType = WGPUSType(0);
                transfer->hasNext = chainedStruct->next != nullptr;

                // Still move on in case there are valid structs after this.
                chainedStruct = chainedStruct->next;
                break;
            }
        }
    } while (chainedStruct != nullptr);
    return WireResult::Success;
}

WireResult DeserializeChainedStruct(WGPUChainedStruct** outChainNext,
                                    DeserializeBuffer* deserializeBuffer,
                                    DeserializeAllocator* allocator,
                                    const ObjectIdResolver& resolver) {
    bool hasNext;
    do {
        const volatile WGPUChainedStructTransfer* header;
        WIRE_TRY(deserializeBuffer->Peek(&header));
        WGPUSType sType = header->sType;
        switch (sType) {
            case WGPUSType_ShaderSourceSPIRV: {
                const volatile WGPUShaderSourceSPIRVTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUShaderSourceSPIRV* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUShaderSourceSPIRVDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_ShaderSourceWGSL: {
                const volatile WGPUShaderSourceWGSLTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUShaderSourceWGSL* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUShaderSourceWGSLDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_RenderPassMaxDrawCount: {
                const volatile WGPURenderPassMaxDrawCountTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPURenderPassMaxDrawCount* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPURenderPassMaxDrawCountDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_AdapterPropertiesSubgroups: {
                const volatile WGPUAdapterPropertiesSubgroupsTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUAdapterPropertiesSubgroups* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUAdapterPropertiesSubgroupsDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_TextureBindingViewDimensionDescriptor: {
                const volatile WGPUTextureBindingViewDimensionDescriptorTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUTextureBindingViewDimensionDescriptor* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUTextureBindingViewDimensionDescriptorDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_ExternalTextureBindingEntry: {
                const volatile WGPUExternalTextureBindingEntryTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUExternalTextureBindingEntry* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUExternalTextureBindingEntryDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_ExternalTextureBindingLayout: {
                const volatile WGPUExternalTextureBindingLayoutTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUExternalTextureBindingLayout* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUExternalTextureBindingLayoutDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnTextureInternalUsageDescriptor: {
                const volatile WGPUDawnTextureInternalUsageDescriptorTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnTextureInternalUsageDescriptor* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnTextureInternalUsageDescriptorDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnEncoderInternalUsageDescriptor: {
                const volatile WGPUDawnEncoderInternalUsageDescriptorTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnEncoderInternalUsageDescriptor* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnEncoderInternalUsageDescriptorDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnBufferDescriptorErrorInfoFromWireClient: {
                const volatile WGPUDawnBufferDescriptorErrorInfoFromWireClientTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnBufferDescriptorErrorInfoFromWireClient* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnBufferDescriptorErrorInfoFromWireClientDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnShaderModuleSPIRVOptionsDescriptor: {
                const volatile WGPUDawnShaderModuleSPIRVOptionsDescriptorTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnShaderModuleSPIRVOptionsDescriptor* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnShaderModuleSPIRVOptionsDescriptorDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnRenderPassColorAttachmentRenderToSingleSampled: {
                const volatile WGPUDawnRenderPassColorAttachmentRenderToSingleSampledTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnRenderPassColorAttachmentRenderToSingleSampled* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnRenderPassColorAttachmentRenderToSingleSampledDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_RenderPassPixelLocalStorage: {
                const volatile WGPURenderPassPixelLocalStorageTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPURenderPassPixelLocalStorage* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPURenderPassPixelLocalStorageDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_PipelineLayoutPixelLocalStorage: {
                const volatile WGPUPipelineLayoutPixelLocalStorageTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUPipelineLayoutPixelLocalStorage* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUPipelineLayoutPixelLocalStorageDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_BufferHostMappedPointer: {
                const volatile WGPUBufferHostMappedPointerTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUBufferHostMappedPointer* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUBufferHostMappedPointerDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnExperimentalSubgroupLimits: {
                const volatile WGPUDawnExperimentalSubgroupLimitsTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnExperimentalSubgroupLimits* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnExperimentalSubgroupLimitsDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_AdapterPropertiesMemoryHeaps: {
                const volatile WGPUAdapterPropertiesMemoryHeapsTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUAdapterPropertiesMemoryHeaps* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUAdapterPropertiesMemoryHeapsDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_AdapterPropertiesD3D: {
                const volatile WGPUAdapterPropertiesD3DTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUAdapterPropertiesD3D* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUAdapterPropertiesD3DDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_AdapterPropertiesVk: {
                const volatile WGPUAdapterPropertiesVkTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUAdapterPropertiesVk* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUAdapterPropertiesVkDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnWireWGSLControl: {
                const volatile WGPUDawnWireWGSLControlTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnWireWGSLControl* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnWireWGSLControlDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnDrmFormatCapabilities: {
                const volatile WGPUDawnDrmFormatCapabilitiesTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnDrmFormatCapabilities* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnDrmFormatCapabilitiesDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_ShaderModuleCompilationOptions: {
                const volatile WGPUShaderModuleCompilationOptionsTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUShaderModuleCompilationOptions* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUShaderModuleCompilationOptionsDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_ColorTargetStateExpandResolveTextureDawn: {
                const volatile WGPUColorTargetStateExpandResolveTextureDawnTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUColorTargetStateExpandResolveTextureDawn* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUColorTargetStateExpandResolveTextureDawnDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_RenderPassDescriptorExpandResolveRect: {
                const volatile WGPURenderPassDescriptorExpandResolveRectTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPURenderPassDescriptorExpandResolveRect* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPURenderPassDescriptorExpandResolveRectDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_StaticSamplerBindingLayout: {
                const volatile WGPUStaticSamplerBindingLayoutTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUStaticSamplerBindingLayout* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUStaticSamplerBindingLayoutDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_YCbCrVkDescriptor: {
                const volatile WGPUYCbCrVkDescriptorTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUYCbCrVkDescriptor* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUYCbCrVkDescriptorDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnExperimentalImmediateDataLimits: {
                const volatile WGPUDawnExperimentalImmediateDataLimitsTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnExperimentalImmediateDataLimits* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnExperimentalImmediateDataLimitsDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            case WGPUSType_DawnTexelCopyBufferRowAlignmentLimits: {
                const volatile WGPUDawnTexelCopyBufferRowAlignmentLimitsTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUDawnTexelCopyBufferRowAlignmentLimits* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->chain.sType = sType;
                outStruct->chain.next = nullptr;

                *outChainNext = &outStruct->chain;
                outChainNext = &outStruct->chain.next;

                WIRE_TRY(WGPUDawnTexelCopyBufferRowAlignmentLimitsDeserialize(outStruct, transfer, deserializeBuffer, allocator, resolver));

                hasNext = transfer->chain.hasNext;
            } break;
            default: {
                // Invalid enum. Deserialize just the transfer header with Invalid as the sType.
                // TODO(crbug.com/dawn/369): Unknown sTypes are silently discarded.
                if (sType != WGPUSType(0)) {
                    dawn::WarningLog() << "Unknown sType " << sType << " discarded.";
                }

                const volatile WGPUChainedStructTransfer* transfer;
                WIRE_TRY(deserializeBuffer->Read(&transfer));

                WGPUChainedStruct* outStruct;
                WIRE_TRY(GetSpace(allocator, 1u, &outStruct));
                outStruct->sType = WGPUSType(0);
                outStruct->next = nullptr;

                // Still move on in case there are valid structs after this.
                *outChainNext = outStruct;
                outChainNext = &outStruct->next;
                hasNext = transfer->hasNext;
                break;
            }
        }
    } while (hasNext);
    return WireResult::Success;
}


struct AdapterRequestDeviceTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId adapterId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
    ObjectHandle deviceObjectHandle;
    WGPUFutureTransfer deviceLostFuture;
};

static_assert(offsetof(AdapterRequestDeviceTransfer, commandSize) == 0);
static_assert(offsetof(AdapterRequestDeviceTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t AdapterRequestDeviceGetExtraRequiredSize([[maybe_unused]] const AdapterRequestDeviceCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUFutureGetExtraRequiredSize(record.deviceLostFuture);
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUDeviceDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUDeviceDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(AdapterRequestDeviceGetExtraRequiredSize);

[[maybe_unused]] WireResult AdapterRequestDeviceSerialize(
    const AdapterRequestDeviceCmd& record,
    AdapterRequestDeviceTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::AdapterRequestDevice;


    transfer->adapterId = record.adapterId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->deviceObjectHandle = record.deviceObjectHandle;
    WIRE_TRY(WGPUFutureSerialize(record.deviceLostFuture, &transfer->deviceLostFuture, buffer));
    {
        auto memberLength = 1u;

        WGPUDeviceDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUDeviceDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(AdapterRequestDeviceSerialize);

[[maybe_unused]] WireResult AdapterRequestDeviceDeserialize(
    AdapterRequestDeviceCmd* record,
    const volatile AdapterRequestDeviceTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::AdapterRequestDevice);


    static_assert(sizeof(record->adapterId) >= sizeof(transfer->adapterId), "Deserialize assignment may not narrow.");
    record->adapterId = transfer->adapterId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->deviceObjectHandle) >= sizeof(transfer->deviceObjectHandle), "Deserialize assignment may not narrow.");
    record->deviceObjectHandle = transfer->deviceObjectHandle;
    static_assert(sizeof(record->deviceLostFuture) == sizeof(transfer->deviceLostFuture), "Deserialize memcpy size must match.");
    memcpy(&record->deviceLostFuture, const_cast<const WGPUFutureTransfer*>(&transfer->deviceLostFuture), sizeof(WGPUFutureTransfer));
    {
        auto memberLength = 1u;
        const volatile WGPUDeviceDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUDeviceDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUDeviceDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(AdapterRequestDeviceDeserialize);

struct BindGroupLayoutSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(BindGroupLayoutSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(BindGroupLayoutSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t BindGroupLayoutSetLabelGetExtraRequiredSize([[maybe_unused]] const BindGroupLayoutSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(BindGroupLayoutSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult BindGroupLayoutSetLabelSerialize(
    const BindGroupLayoutSetLabelCmd& record,
    BindGroupLayoutSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::BindGroupLayoutSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BindGroupLayoutSetLabelSerialize);

[[maybe_unused]] WireResult BindGroupLayoutSetLabelDeserialize(
    BindGroupLayoutSetLabelCmd* record,
    const volatile BindGroupLayoutSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::BindGroupLayoutSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BindGroupLayoutSetLabelDeserialize);

struct BindGroupSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(BindGroupSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(BindGroupSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t BindGroupSetLabelGetExtraRequiredSize([[maybe_unused]] const BindGroupSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(BindGroupSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult BindGroupSetLabelSerialize(
    const BindGroupSetLabelCmd& record,
    BindGroupSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::BindGroupSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BindGroupSetLabelSerialize);

[[maybe_unused]] WireResult BindGroupSetLabelDeserialize(
    BindGroupSetLabelCmd* record,
    const volatile BindGroupSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::BindGroupSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BindGroupSetLabelDeserialize);

struct BufferDestroyTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(BufferDestroyTransfer, commandSize) == 0);
static_assert(offsetof(BufferDestroyTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t BufferDestroyGetExtraRequiredSize([[maybe_unused]] const BufferDestroyCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(BufferDestroyGetExtraRequiredSize);

[[maybe_unused]] WireResult BufferDestroySerialize(
    const BufferDestroyCmd& record,
    BufferDestroyTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::BufferDestroy;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferDestroySerialize);

[[maybe_unused]] WireResult BufferDestroyDeserialize(
    BufferDestroyCmd* record,
    const volatile BufferDestroyTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::BufferDestroy);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferDestroyDeserialize);

struct BufferMapAsyncTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId bufferId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
    WGPUMapMode mode;
    uint64_t offset;
    uint64_t size;
};

static_assert(offsetof(BufferMapAsyncTransfer, commandSize) == 0);
static_assert(offsetof(BufferMapAsyncTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t BufferMapAsyncGetExtraRequiredSize([[maybe_unused]] const BufferMapAsyncCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(BufferMapAsyncGetExtraRequiredSize);

[[maybe_unused]] WireResult BufferMapAsyncSerialize(
    const BufferMapAsyncCmd& record,
    BufferMapAsyncTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::BufferMapAsync;


    transfer->bufferId = record.bufferId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->mode = record.mode;
    transfer->offset = record.offset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferMapAsyncSerialize);

[[maybe_unused]] WireResult BufferMapAsyncDeserialize(
    BufferMapAsyncCmd* record,
    const volatile BufferMapAsyncTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::BufferMapAsync);


    static_assert(sizeof(record->bufferId) >= sizeof(transfer->bufferId), "Deserialize assignment may not narrow.");
    record->bufferId = transfer->bufferId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->mode) >= sizeof(transfer->mode), "Deserialize assignment may not narrow.");
    record->mode = transfer->mode;
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferMapAsyncDeserialize);

struct BufferSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(BufferSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(BufferSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t BufferSetLabelGetExtraRequiredSize([[maybe_unused]] const BufferSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(BufferSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult BufferSetLabelSerialize(
    const BufferSetLabelCmd& record,
    BufferSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::BufferSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferSetLabelSerialize);

[[maybe_unused]] WireResult BufferSetLabelDeserialize(
    BufferSetLabelCmd* record,
    const volatile BufferSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::BufferSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferSetLabelDeserialize);

struct BufferUnmapTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(BufferUnmapTransfer, commandSize) == 0);
static_assert(offsetof(BufferUnmapTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t BufferUnmapGetExtraRequiredSize([[maybe_unused]] const BufferUnmapCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(BufferUnmapGetExtraRequiredSize);

[[maybe_unused]] WireResult BufferUnmapSerialize(
    const BufferUnmapCmd& record,
    BufferUnmapTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::BufferUnmap;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferUnmapSerialize);

[[maybe_unused]] WireResult BufferUnmapDeserialize(
    BufferUnmapCmd* record,
    const volatile BufferUnmapTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::BufferUnmap);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferUnmapDeserialize);

struct BufferUpdateMappedDataTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId bufferId;
    uint64_t writeDataUpdateInfoLength;
    uint64_t offset;
    uint64_t size;
};

static_assert(offsetof(BufferUpdateMappedDataTransfer, commandSize) == 0);
static_assert(offsetof(BufferUpdateMappedDataTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t BufferUpdateMappedDataGetExtraRequiredSize([[maybe_unused]] const BufferUpdateMappedDataCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(BufferUpdateMappedDataGetExtraRequiredSize);

[[maybe_unused]] WireResult BufferUpdateMappedDataSerialize(
    const BufferUpdateMappedDataCmd& record,
    BufferUpdateMappedDataTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::BufferUpdateMappedData;


    transfer->bufferId = record.bufferId;
    transfer->writeDataUpdateInfoLength = record.writeDataUpdateInfoLength;
    transfer->offset = record.offset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferUpdateMappedDataSerialize);

[[maybe_unused]] WireResult BufferUpdateMappedDataDeserialize(
    BufferUpdateMappedDataCmd* record,
    const volatile BufferUpdateMappedDataTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::BufferUpdateMappedData);


    static_assert(sizeof(record->bufferId) >= sizeof(transfer->bufferId), "Deserialize assignment may not narrow.");
    record->bufferId = transfer->bufferId;
    static_assert(sizeof(record->writeDataUpdateInfoLength) >= sizeof(transfer->writeDataUpdateInfoLength), "Deserialize assignment may not narrow.");
    record->writeDataUpdateInfoLength = transfer->writeDataUpdateInfoLength;
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;
    {
        auto memberLength = record->writeDataUpdateInfoLength;
        const volatile uint8_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint8_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->writeDataUpdateInfo = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint8_t*>(memberBuffer),
              sizeof(uint8_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(BufferUpdateMappedDataDeserialize);

struct CommandBufferSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(CommandBufferSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(CommandBufferSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandBufferSetLabelGetExtraRequiredSize([[maybe_unused]] const CommandBufferSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandBufferSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandBufferSetLabelSerialize(
    const CommandBufferSetLabelCmd& record,
    CommandBufferSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandBufferSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandBufferSetLabelSerialize);

[[maybe_unused]] WireResult CommandBufferSetLabelDeserialize(
    CommandBufferSetLabelCmd* record,
    const volatile CommandBufferSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandBufferSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandBufferSetLabelDeserialize);

struct CommandEncoderBeginComputePassTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUBool has_descriptor;
    ObjectHandle result;
};

static_assert(offsetof(CommandEncoderBeginComputePassTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderBeginComputePassTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderBeginComputePassGetExtraRequiredSize([[maybe_unused]] const CommandEncoderBeginComputePassCmd& record) {
    size_t result = 0;

    if (record.descriptor != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUComputePassDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUComputePassDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderBeginComputePassGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderBeginComputePassSerialize(
    const CommandEncoderBeginComputePassCmd& record,
    CommandEncoderBeginComputePassTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderBeginComputePass;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    bool has_descriptor = record.descriptor != nullptr;
    transfer->has_descriptor = has_descriptor;
    if (has_descriptor) {
        auto memberLength = 1u;

        WGPUComputePassDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUComputePassDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderBeginComputePassSerialize);

[[maybe_unused]] WireResult CommandEncoderBeginComputePassDeserialize(
    CommandEncoderBeginComputePassCmd* record,
    const volatile CommandEncoderBeginComputePassTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderBeginComputePass);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    bool has_descriptor = transfer->has_descriptor;
    record->descriptor = nullptr;
    if (has_descriptor) {
        auto memberLength = 1u;
        const volatile WGPUComputePassDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUComputePassDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUComputePassDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderBeginComputePassDeserialize);

struct CommandEncoderBeginRenderPassTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(CommandEncoderBeginRenderPassTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderBeginRenderPassTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderBeginRenderPassGetExtraRequiredSize([[maybe_unused]] const CommandEncoderBeginRenderPassCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURenderPassDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderPassDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderBeginRenderPassGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderBeginRenderPassSerialize(
    const CommandEncoderBeginRenderPassCmd& record,
    CommandEncoderBeginRenderPassTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderBeginRenderPass;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPURenderPassDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderBeginRenderPassSerialize);

[[maybe_unused]] WireResult CommandEncoderBeginRenderPassDeserialize(
    CommandEncoderBeginRenderPassCmd* record,
    const volatile CommandEncoderBeginRenderPassTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderBeginRenderPass);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPURenderPassDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderPassDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPassDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderBeginRenderPassDeserialize);

struct CommandEncoderClearBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId buffer;
    uint64_t offset;
    uint64_t size;
};

static_assert(offsetof(CommandEncoderClearBufferTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderClearBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderClearBufferGetExtraRequiredSize([[maybe_unused]] const CommandEncoderClearBufferCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderClearBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderClearBufferSerialize(
    const CommandEncoderClearBufferCmd& record,
    CommandEncoderClearBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderClearBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.buffer, &transfer->buffer));
    transfer->offset = record.offset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderClearBufferSerialize);

[[maybe_unused]] WireResult CommandEncoderClearBufferDeserialize(
    CommandEncoderClearBufferCmd* record,
    const volatile CommandEncoderClearBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderClearBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->buffer, &record->buffer));
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderClearBufferDeserialize);

struct CommandEncoderCopyBufferToBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId source;
    uint64_t sourceOffset;
    ObjectId destination;
    uint64_t destinationOffset;
    uint64_t size;
};

static_assert(offsetof(CommandEncoderCopyBufferToBufferTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderCopyBufferToBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderCopyBufferToBufferGetExtraRequiredSize([[maybe_unused]] const CommandEncoderCopyBufferToBufferCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderCopyBufferToBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderCopyBufferToBufferSerialize(
    const CommandEncoderCopyBufferToBufferCmd& record,
    CommandEncoderCopyBufferToBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderCopyBufferToBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.source, &transfer->source));
    transfer->sourceOffset = record.sourceOffset;
    WIRE_TRY(provider.GetId(record.destination, &transfer->destination));
    transfer->destinationOffset = record.destinationOffset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyBufferToBufferSerialize);

[[maybe_unused]] WireResult CommandEncoderCopyBufferToBufferDeserialize(
    CommandEncoderCopyBufferToBufferCmd* record,
    const volatile CommandEncoderCopyBufferToBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderCopyBufferToBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->source, &record->source));
    static_assert(sizeof(record->sourceOffset) >= sizeof(transfer->sourceOffset), "Deserialize assignment may not narrow.");
    record->sourceOffset = transfer->sourceOffset;
    WIRE_TRY(resolver.GetFromId(transfer->destination, &record->destination));
    static_assert(sizeof(record->destinationOffset) >= sizeof(transfer->destinationOffset), "Deserialize assignment may not narrow.");
    record->destinationOffset = transfer->destinationOffset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyBufferToBufferDeserialize);

struct CommandEncoderCopyBufferToTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(CommandEncoderCopyBufferToTextureTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderCopyBufferToTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderCopyBufferToTextureGetExtraRequiredSize([[maybe_unused]] const CommandEncoderCopyBufferToTextureCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyBufferTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyBufferGetExtraRequiredSize(record.source[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.destination[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUExtent3DTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUExtent3DGetExtraRequiredSize(record.copySize[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderCopyBufferToTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderCopyBufferToTextureSerialize(
    const CommandEncoderCopyBufferToTextureCmd& record,
    CommandEncoderCopyBufferToTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderCopyBufferToTexture;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUImageCopyBufferTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyBufferSerialize(record.source[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.destination[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.copySize,
                sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyBufferToTextureSerialize);

[[maybe_unused]] WireResult CommandEncoderCopyBufferToTextureDeserialize(
    CommandEncoderCopyBufferToTextureCmd* record,
    const volatile CommandEncoderCopyBufferToTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderCopyBufferToTexture);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyBufferTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyBuffer* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->source = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyBufferDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->destination = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUExtent3D* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->copySize = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUExtent3DTransfer*>(memberBuffer),
              sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyBufferToTextureDeserialize);

struct CommandEncoderCopyTextureToBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(CommandEncoderCopyTextureToBufferTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderCopyTextureToBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderCopyTextureToBufferGetExtraRequiredSize([[maybe_unused]] const CommandEncoderCopyTextureToBufferCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.source[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyBufferTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyBufferGetExtraRequiredSize(record.destination[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUExtent3DTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUExtent3DGetExtraRequiredSize(record.copySize[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderCopyTextureToBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderCopyTextureToBufferSerialize(
    const CommandEncoderCopyTextureToBufferCmd& record,
    CommandEncoderCopyTextureToBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderCopyTextureToBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.source[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUImageCopyBufferTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyBufferSerialize(record.destination[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.copySize,
                sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyTextureToBufferSerialize);

[[maybe_unused]] WireResult CommandEncoderCopyTextureToBufferDeserialize(
    CommandEncoderCopyTextureToBufferCmd* record,
    const volatile CommandEncoderCopyTextureToBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderCopyTextureToBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->source = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyBufferTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyBuffer* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->destination = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyBufferDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUExtent3D* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->copySize = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUExtent3DTransfer*>(memberBuffer),
              sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyTextureToBufferDeserialize);

struct CommandEncoderCopyTextureToTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(CommandEncoderCopyTextureToTextureTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderCopyTextureToTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderCopyTextureToTextureGetExtraRequiredSize([[maybe_unused]] const CommandEncoderCopyTextureToTextureCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.source[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.destination[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUExtent3DTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUExtent3DGetExtraRequiredSize(record.copySize[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderCopyTextureToTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderCopyTextureToTextureSerialize(
    const CommandEncoderCopyTextureToTextureCmd& record,
    CommandEncoderCopyTextureToTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderCopyTextureToTexture;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.source[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.destination[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.copySize,
                sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyTextureToTextureSerialize);

[[maybe_unused]] WireResult CommandEncoderCopyTextureToTextureDeserialize(
    CommandEncoderCopyTextureToTextureCmd* record,
    const volatile CommandEncoderCopyTextureToTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderCopyTextureToTexture);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->source = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->destination = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUExtent3D* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->copySize = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUExtent3DTransfer*>(memberBuffer),
              sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderCopyTextureToTextureDeserialize);

struct CommandEncoderFinishTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUBool has_descriptor;
    ObjectHandle result;
};

static_assert(offsetof(CommandEncoderFinishTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderFinishTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderFinishGetExtraRequiredSize([[maybe_unused]] const CommandEncoderFinishCmd& record) {
    size_t result = 0;

    if (record.descriptor != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUCommandBufferDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUCommandBufferDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderFinishGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderFinishSerialize(
    const CommandEncoderFinishCmd& record,
    CommandEncoderFinishTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderFinish;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    bool has_descriptor = record.descriptor != nullptr;
    transfer->has_descriptor = has_descriptor;
    if (has_descriptor) {
        auto memberLength = 1u;

        WGPUCommandBufferDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCommandBufferDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderFinishSerialize);

[[maybe_unused]] WireResult CommandEncoderFinishDeserialize(
    CommandEncoderFinishCmd* record,
    const volatile CommandEncoderFinishTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderFinish);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    bool has_descriptor = transfer->has_descriptor;
    record->descriptor = nullptr;
    if (has_descriptor) {
        auto memberLength = 1u;
        const volatile WGPUCommandBufferDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCommandBufferDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCommandBufferDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderFinishDeserialize);

struct CommandEncoderInjectValidationErrorTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(CommandEncoderInjectValidationErrorTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderInjectValidationErrorTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderInjectValidationErrorGetExtraRequiredSize([[maybe_unused]] const CommandEncoderInjectValidationErrorCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderInjectValidationErrorGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderInjectValidationErrorSerialize(
    const CommandEncoderInjectValidationErrorCmd& record,
    CommandEncoderInjectValidationErrorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderInjectValidationError;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderInjectValidationErrorSerialize);

[[maybe_unused]] WireResult CommandEncoderInjectValidationErrorDeserialize(
    CommandEncoderInjectValidationErrorCmd* record,
    const volatile CommandEncoderInjectValidationErrorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderInjectValidationError);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderInjectValidationErrorDeserialize);

struct CommandEncoderInsertDebugMarkerTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer markerLabel;
};

static_assert(offsetof(CommandEncoderInsertDebugMarkerTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderInsertDebugMarkerTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderInsertDebugMarkerGetExtraRequiredSize([[maybe_unused]] const CommandEncoderInsertDebugMarkerCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.markerLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderInsertDebugMarkerGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderInsertDebugMarkerSerialize(
    const CommandEncoderInsertDebugMarkerCmd& record,
    CommandEncoderInsertDebugMarkerTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderInsertDebugMarker;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.markerLabel, &transfer->markerLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderInsertDebugMarkerSerialize);

[[maybe_unused]] WireResult CommandEncoderInsertDebugMarkerDeserialize(
    CommandEncoderInsertDebugMarkerCmd* record,
    const volatile CommandEncoderInsertDebugMarkerTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderInsertDebugMarker);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->markerLabel, &transfer->markerLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderInsertDebugMarkerDeserialize);

struct CommandEncoderPopDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(CommandEncoderPopDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderPopDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderPopDebugGroupGetExtraRequiredSize([[maybe_unused]] const CommandEncoderPopDebugGroupCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderPopDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderPopDebugGroupSerialize(
    const CommandEncoderPopDebugGroupCmd& record,
    CommandEncoderPopDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderPopDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderPopDebugGroupSerialize);

[[maybe_unused]] WireResult CommandEncoderPopDebugGroupDeserialize(
    CommandEncoderPopDebugGroupCmd* record,
    const volatile CommandEncoderPopDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderPopDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderPopDebugGroupDeserialize);

struct CommandEncoderPushDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer groupLabel;
};

static_assert(offsetof(CommandEncoderPushDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderPushDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderPushDebugGroupGetExtraRequiredSize([[maybe_unused]] const CommandEncoderPushDebugGroupCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.groupLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderPushDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderPushDebugGroupSerialize(
    const CommandEncoderPushDebugGroupCmd& record,
    CommandEncoderPushDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderPushDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.groupLabel, &transfer->groupLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderPushDebugGroupSerialize);

[[maybe_unused]] WireResult CommandEncoderPushDebugGroupDeserialize(
    CommandEncoderPushDebugGroupCmd* record,
    const volatile CommandEncoderPushDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderPushDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->groupLabel, &transfer->groupLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderPushDebugGroupDeserialize);

struct CommandEncoderResolveQuerySetTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId querySet;
    uint32_t firstQuery;
    uint32_t queryCount;
    ObjectId destination;
    uint64_t destinationOffset;
};

static_assert(offsetof(CommandEncoderResolveQuerySetTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderResolveQuerySetTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderResolveQuerySetGetExtraRequiredSize([[maybe_unused]] const CommandEncoderResolveQuerySetCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderResolveQuerySetGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderResolveQuerySetSerialize(
    const CommandEncoderResolveQuerySetCmd& record,
    CommandEncoderResolveQuerySetTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderResolveQuerySet;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.querySet, &transfer->querySet));
    transfer->firstQuery = record.firstQuery;
    transfer->queryCount = record.queryCount;
    WIRE_TRY(provider.GetId(record.destination, &transfer->destination));
    transfer->destinationOffset = record.destinationOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderResolveQuerySetSerialize);

[[maybe_unused]] WireResult CommandEncoderResolveQuerySetDeserialize(
    CommandEncoderResolveQuerySetCmd* record,
    const volatile CommandEncoderResolveQuerySetTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderResolveQuerySet);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->querySet, &record->querySet));
    static_assert(sizeof(record->firstQuery) >= sizeof(transfer->firstQuery), "Deserialize assignment may not narrow.");
    record->firstQuery = transfer->firstQuery;
    static_assert(sizeof(record->queryCount) >= sizeof(transfer->queryCount), "Deserialize assignment may not narrow.");
    record->queryCount = transfer->queryCount;
    WIRE_TRY(resolver.GetFromId(transfer->destination, &record->destination));
    static_assert(sizeof(record->destinationOffset) >= sizeof(transfer->destinationOffset), "Deserialize assignment may not narrow.");
    record->destinationOffset = transfer->destinationOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderResolveQuerySetDeserialize);

struct CommandEncoderSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(CommandEncoderSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderSetLabelGetExtraRequiredSize([[maybe_unused]] const CommandEncoderSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderSetLabelSerialize(
    const CommandEncoderSetLabelCmd& record,
    CommandEncoderSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderSetLabelSerialize);

[[maybe_unused]] WireResult CommandEncoderSetLabelDeserialize(
    CommandEncoderSetLabelCmd* record,
    const volatile CommandEncoderSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderSetLabelDeserialize);

struct CommandEncoderWriteBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId buffer;
    uint64_t bufferOffset;
    uint64_t size;
};

static_assert(offsetof(CommandEncoderWriteBufferTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderWriteBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderWriteBufferGetExtraRequiredSize([[maybe_unused]] const CommandEncoderWriteBufferCmd& record) {
    size_t result = 0;

    {
        auto memberLength = record.size;
        auto size = WireAlignSizeofN<uint8_t>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderWriteBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderWriteBufferSerialize(
    const CommandEncoderWriteBufferCmd& record,
    CommandEncoderWriteBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderWriteBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.buffer, &transfer->buffer));
    transfer->bufferOffset = record.bufferOffset;
    transfer->size = record.size;
    {
        auto memberLength = record.size;

        uint8_t* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.data,
                sizeof(uint8_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderWriteBufferSerialize);

[[maybe_unused]] WireResult CommandEncoderWriteBufferDeserialize(
    CommandEncoderWriteBufferCmd* record,
    const volatile CommandEncoderWriteBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderWriteBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->buffer, &record->buffer));
    static_assert(sizeof(record->bufferOffset) >= sizeof(transfer->bufferOffset), "Deserialize assignment may not narrow.");
    record->bufferOffset = transfer->bufferOffset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;
    {
        auto memberLength = record->size;
        const volatile uint8_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint8_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->data = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint8_t*>(memberBuffer),
              sizeof(uint8_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderWriteBufferDeserialize);

struct CommandEncoderWriteTimestampTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId querySet;
    uint32_t queryIndex;
};

static_assert(offsetof(CommandEncoderWriteTimestampTransfer, commandSize) == 0);
static_assert(offsetof(CommandEncoderWriteTimestampTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t CommandEncoderWriteTimestampGetExtraRequiredSize([[maybe_unused]] const CommandEncoderWriteTimestampCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(CommandEncoderWriteTimestampGetExtraRequiredSize);

[[maybe_unused]] WireResult CommandEncoderWriteTimestampSerialize(
    const CommandEncoderWriteTimestampCmd& record,
    CommandEncoderWriteTimestampTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::CommandEncoderWriteTimestamp;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.querySet, &transfer->querySet));
    transfer->queryIndex = record.queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderWriteTimestampSerialize);

[[maybe_unused]] WireResult CommandEncoderWriteTimestampDeserialize(
    CommandEncoderWriteTimestampCmd* record,
    const volatile CommandEncoderWriteTimestampTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::CommandEncoderWriteTimestamp);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->querySet, &record->querySet));
    static_assert(sizeof(record->queryIndex) >= sizeof(transfer->queryIndex), "Deserialize assignment may not narrow.");
    record->queryIndex = transfer->queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(CommandEncoderWriteTimestampDeserialize);

struct ComputePassEncoderDispatchWorkgroupsTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t workgroupCountX;
    uint32_t workgroupCountY;
    uint32_t workgroupCountZ;
};

static_assert(offsetof(ComputePassEncoderDispatchWorkgroupsTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderDispatchWorkgroupsTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderDispatchWorkgroupsGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderDispatchWorkgroupsCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderDispatchWorkgroupsGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderDispatchWorkgroupsSerialize(
    const ComputePassEncoderDispatchWorkgroupsCmd& record,
    ComputePassEncoderDispatchWorkgroupsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderDispatchWorkgroups;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->workgroupCountX = record.workgroupCountX;
    transfer->workgroupCountY = record.workgroupCountY;
    transfer->workgroupCountZ = record.workgroupCountZ;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderDispatchWorkgroupsSerialize);

[[maybe_unused]] WireResult ComputePassEncoderDispatchWorkgroupsDeserialize(
    ComputePassEncoderDispatchWorkgroupsCmd* record,
    const volatile ComputePassEncoderDispatchWorkgroupsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderDispatchWorkgroups);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->workgroupCountX) >= sizeof(transfer->workgroupCountX), "Deserialize assignment may not narrow.");
    record->workgroupCountX = transfer->workgroupCountX;
    static_assert(sizeof(record->workgroupCountY) >= sizeof(transfer->workgroupCountY), "Deserialize assignment may not narrow.");
    record->workgroupCountY = transfer->workgroupCountY;
    static_assert(sizeof(record->workgroupCountZ) >= sizeof(transfer->workgroupCountZ), "Deserialize assignment may not narrow.");
    record->workgroupCountZ = transfer->workgroupCountZ;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderDispatchWorkgroupsDeserialize);

struct ComputePassEncoderDispatchWorkgroupsIndirectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId indirectBuffer;
    uint64_t indirectOffset;
};

static_assert(offsetof(ComputePassEncoderDispatchWorkgroupsIndirectTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderDispatchWorkgroupsIndirectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderDispatchWorkgroupsIndirectGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderDispatchWorkgroupsIndirectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderDispatchWorkgroupsIndirectGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderDispatchWorkgroupsIndirectSerialize(
    const ComputePassEncoderDispatchWorkgroupsIndirectCmd& record,
    ComputePassEncoderDispatchWorkgroupsIndirectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderDispatchWorkgroupsIndirect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.indirectBuffer, &transfer->indirectBuffer));
    transfer->indirectOffset = record.indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderDispatchWorkgroupsIndirectSerialize);

[[maybe_unused]] WireResult ComputePassEncoderDispatchWorkgroupsIndirectDeserialize(
    ComputePassEncoderDispatchWorkgroupsIndirectCmd* record,
    const volatile ComputePassEncoderDispatchWorkgroupsIndirectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderDispatchWorkgroupsIndirect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->indirectBuffer, &record->indirectBuffer));
    static_assert(sizeof(record->indirectOffset) >= sizeof(transfer->indirectOffset), "Deserialize assignment may not narrow.");
    record->indirectOffset = transfer->indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderDispatchWorkgroupsIndirectDeserialize);

struct ComputePassEncoderEndTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(ComputePassEncoderEndTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderEndTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderEndGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderEndCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderEndGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderEndSerialize(
    const ComputePassEncoderEndCmd& record,
    ComputePassEncoderEndTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderEnd;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderEndSerialize);

[[maybe_unused]] WireResult ComputePassEncoderEndDeserialize(
    ComputePassEncoderEndCmd* record,
    const volatile ComputePassEncoderEndTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderEnd);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderEndDeserialize);

struct ComputePassEncoderInsertDebugMarkerTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer markerLabel;
};

static_assert(offsetof(ComputePassEncoderInsertDebugMarkerTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderInsertDebugMarkerTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderInsertDebugMarkerGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderInsertDebugMarkerCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.markerLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderInsertDebugMarkerGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderInsertDebugMarkerSerialize(
    const ComputePassEncoderInsertDebugMarkerCmd& record,
    ComputePassEncoderInsertDebugMarkerTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderInsertDebugMarker;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.markerLabel, &transfer->markerLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderInsertDebugMarkerSerialize);

[[maybe_unused]] WireResult ComputePassEncoderInsertDebugMarkerDeserialize(
    ComputePassEncoderInsertDebugMarkerCmd* record,
    const volatile ComputePassEncoderInsertDebugMarkerTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderInsertDebugMarker);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->markerLabel, &transfer->markerLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderInsertDebugMarkerDeserialize);

struct ComputePassEncoderPopDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(ComputePassEncoderPopDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderPopDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderPopDebugGroupGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderPopDebugGroupCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderPopDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderPopDebugGroupSerialize(
    const ComputePassEncoderPopDebugGroupCmd& record,
    ComputePassEncoderPopDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderPopDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderPopDebugGroupSerialize);

[[maybe_unused]] WireResult ComputePassEncoderPopDebugGroupDeserialize(
    ComputePassEncoderPopDebugGroupCmd* record,
    const volatile ComputePassEncoderPopDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderPopDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderPopDebugGroupDeserialize);

struct ComputePassEncoderPushDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer groupLabel;
};

static_assert(offsetof(ComputePassEncoderPushDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderPushDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderPushDebugGroupGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderPushDebugGroupCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.groupLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderPushDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderPushDebugGroupSerialize(
    const ComputePassEncoderPushDebugGroupCmd& record,
    ComputePassEncoderPushDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderPushDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.groupLabel, &transfer->groupLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderPushDebugGroupSerialize);

[[maybe_unused]] WireResult ComputePassEncoderPushDebugGroupDeserialize(
    ComputePassEncoderPushDebugGroupCmd* record,
    const volatile ComputePassEncoderPushDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderPushDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->groupLabel, &transfer->groupLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderPushDebugGroupDeserialize);

struct ComputePassEncoderSetBindGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t groupIndex;
    ObjectId group;
    uint64_t dynamicOffsetCount;
};

static_assert(offsetof(ComputePassEncoderSetBindGroupTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderSetBindGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderSetBindGroupGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderSetBindGroupCmd& record) {
    size_t result = 0;

    {
        auto memberLength = record.dynamicOffsetCount;
        auto size = WireAlignSizeofN<uint32_t>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderSetBindGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderSetBindGroupSerialize(
    const ComputePassEncoderSetBindGroupCmd& record,
    ComputePassEncoderSetBindGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderSetBindGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->groupIndex = record.groupIndex;
    WIRE_TRY(provider.GetOptionalId(record.group, &transfer->group));
    transfer->dynamicOffsetCount = record.dynamicOffsetCount;
    {
        auto memberLength = record.dynamicOffsetCount;

        uint32_t* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.dynamicOffsets,
                sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderSetBindGroupSerialize);

[[maybe_unused]] WireResult ComputePassEncoderSetBindGroupDeserialize(
    ComputePassEncoderSetBindGroupCmd* record,
    const volatile ComputePassEncoderSetBindGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderSetBindGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->groupIndex) >= sizeof(transfer->groupIndex), "Deserialize assignment may not narrow.");
    record->groupIndex = transfer->groupIndex;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->group, &record->group));
    if (transfer->dynamicOffsetCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->dynamicOffsetCount = checked_cast<size_t>(transfer->dynamicOffsetCount);
    {
        auto memberLength = record->dynamicOffsetCount;
        const volatile uint32_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint32_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->dynamicOffsets = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint32_t*>(memberBuffer),
              sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderSetBindGroupDeserialize);

struct ComputePassEncoderSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(ComputePassEncoderSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderSetLabelGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderSetLabelSerialize(
    const ComputePassEncoderSetLabelCmd& record,
    ComputePassEncoderSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderSetLabelSerialize);

[[maybe_unused]] WireResult ComputePassEncoderSetLabelDeserialize(
    ComputePassEncoderSetLabelCmd* record,
    const volatile ComputePassEncoderSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderSetLabelDeserialize);

struct ComputePassEncoderSetPipelineTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId pipeline;
};

static_assert(offsetof(ComputePassEncoderSetPipelineTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderSetPipelineTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderSetPipelineGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderSetPipelineCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderSetPipelineGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderSetPipelineSerialize(
    const ComputePassEncoderSetPipelineCmd& record,
    ComputePassEncoderSetPipelineTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderSetPipeline;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.pipeline, &transfer->pipeline));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderSetPipelineSerialize);

[[maybe_unused]] WireResult ComputePassEncoderSetPipelineDeserialize(
    ComputePassEncoderSetPipelineCmd* record,
    const volatile ComputePassEncoderSetPipelineTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderSetPipeline);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->pipeline, &record->pipeline));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderSetPipelineDeserialize);

struct ComputePassEncoderWriteTimestampTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId querySet;
    uint32_t queryIndex;
};

static_assert(offsetof(ComputePassEncoderWriteTimestampTransfer, commandSize) == 0);
static_assert(offsetof(ComputePassEncoderWriteTimestampTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePassEncoderWriteTimestampGetExtraRequiredSize([[maybe_unused]] const ComputePassEncoderWriteTimestampCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePassEncoderWriteTimestampGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePassEncoderWriteTimestampSerialize(
    const ComputePassEncoderWriteTimestampCmd& record,
    ComputePassEncoderWriteTimestampTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePassEncoderWriteTimestamp;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.querySet, &transfer->querySet));
    transfer->queryIndex = record.queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderWriteTimestampSerialize);

[[maybe_unused]] WireResult ComputePassEncoderWriteTimestampDeserialize(
    ComputePassEncoderWriteTimestampCmd* record,
    const volatile ComputePassEncoderWriteTimestampTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePassEncoderWriteTimestamp);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->querySet, &record->querySet));
    static_assert(sizeof(record->queryIndex) >= sizeof(transfer->queryIndex), "Deserialize assignment may not narrow.");
    record->queryIndex = transfer->queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePassEncoderWriteTimestampDeserialize);

struct ComputePipelineGetBindGroupLayoutTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t groupIndex;
    ObjectHandle result;
};

static_assert(offsetof(ComputePipelineGetBindGroupLayoutTransfer, commandSize) == 0);
static_assert(offsetof(ComputePipelineGetBindGroupLayoutTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePipelineGetBindGroupLayoutGetExtraRequiredSize([[maybe_unused]] const ComputePipelineGetBindGroupLayoutCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePipelineGetBindGroupLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePipelineGetBindGroupLayoutSerialize(
    const ComputePipelineGetBindGroupLayoutCmd& record,
    ComputePipelineGetBindGroupLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePipelineGetBindGroupLayout;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->groupIndex = record.groupIndex;
    transfer->result = record.result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePipelineGetBindGroupLayoutSerialize);

[[maybe_unused]] WireResult ComputePipelineGetBindGroupLayoutDeserialize(
    ComputePipelineGetBindGroupLayoutCmd* record,
    const volatile ComputePipelineGetBindGroupLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePipelineGetBindGroupLayout);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->groupIndex) >= sizeof(transfer->groupIndex), "Deserialize assignment may not narrow.");
    record->groupIndex = transfer->groupIndex;
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePipelineGetBindGroupLayoutDeserialize);

struct ComputePipelineSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(ComputePipelineSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(ComputePipelineSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ComputePipelineSetLabelGetExtraRequiredSize([[maybe_unused]] const ComputePipelineSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ComputePipelineSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult ComputePipelineSetLabelSerialize(
    const ComputePipelineSetLabelCmd& record,
    ComputePipelineSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ComputePipelineSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePipelineSetLabelSerialize);

[[maybe_unused]] WireResult ComputePipelineSetLabelDeserialize(
    ComputePipelineSetLabelCmd* record,
    const volatile ComputePipelineSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ComputePipelineSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ComputePipelineSetLabelDeserialize);

struct DeviceCreateBindGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateBindGroupTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateBindGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateBindGroupGetExtraRequiredSize([[maybe_unused]] const DeviceCreateBindGroupCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUBindGroupDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUBindGroupDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateBindGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateBindGroupSerialize(
    const DeviceCreateBindGroupCmd& record,
    DeviceCreateBindGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateBindGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUBindGroupDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateBindGroupSerialize);

[[maybe_unused]] WireResult DeviceCreateBindGroupDeserialize(
    DeviceCreateBindGroupCmd* record,
    const volatile DeviceCreateBindGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateBindGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUBindGroupDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBindGroupDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateBindGroupDeserialize);

struct DeviceCreateBindGroupLayoutTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateBindGroupLayoutTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateBindGroupLayoutTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateBindGroupLayoutGetExtraRequiredSize([[maybe_unused]] const DeviceCreateBindGroupLayoutCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUBindGroupLayoutDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUBindGroupLayoutDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateBindGroupLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateBindGroupLayoutSerialize(
    const DeviceCreateBindGroupLayoutCmd& record,
    DeviceCreateBindGroupLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateBindGroupLayout;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUBindGroupLayoutDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupLayoutDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateBindGroupLayoutSerialize);

[[maybe_unused]] WireResult DeviceCreateBindGroupLayoutDeserialize(
    DeviceCreateBindGroupLayoutCmd* record,
    const volatile DeviceCreateBindGroupLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateBindGroupLayout);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUBindGroupLayoutDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBindGroupLayoutDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBindGroupLayoutDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateBindGroupLayoutDeserialize);

struct DeviceCreateBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId deviceId;
    ObjectHandle result;
    uint64_t readHandleCreateInfoLength;
    uint64_t writeHandleCreateInfoLength;
};

static_assert(offsetof(DeviceCreateBufferTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateBufferGetExtraRequiredSize([[maybe_unused]] const DeviceCreateBufferCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUBufferDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUBufferDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateBufferSerialize(
    const DeviceCreateBufferCmd& record,
    DeviceCreateBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateBuffer;


    transfer->deviceId = record.deviceId;
    transfer->result = record.result;
    transfer->readHandleCreateInfoLength = record.readHandleCreateInfoLength;
    transfer->writeHandleCreateInfoLength = record.writeHandleCreateInfoLength;
    {
        auto memberLength = 1u;

        WGPUBufferDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBufferDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateBufferSerialize);

[[maybe_unused]] WireResult DeviceCreateBufferDeserialize(
    DeviceCreateBufferCmd* record,
    const volatile DeviceCreateBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateBuffer);


    static_assert(sizeof(record->deviceId) >= sizeof(transfer->deviceId), "Deserialize assignment may not narrow.");
    record->deviceId = transfer->deviceId;
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    static_assert(sizeof(record->readHandleCreateInfoLength) >= sizeof(transfer->readHandleCreateInfoLength), "Deserialize assignment may not narrow.");
    record->readHandleCreateInfoLength = transfer->readHandleCreateInfoLength;
    static_assert(sizeof(record->writeHandleCreateInfoLength) >= sizeof(transfer->writeHandleCreateInfoLength), "Deserialize assignment may not narrow.");
    record->writeHandleCreateInfoLength = transfer->writeHandleCreateInfoLength;
    {
        auto memberLength = 1u;
        const volatile WGPUBufferDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBufferDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBufferDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = record->readHandleCreateInfoLength;
        const volatile uint8_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint8_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->readHandleCreateInfo = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint8_t*>(memberBuffer),
              sizeof(uint8_t) * memberLength);
        }
    }
    {
        auto memberLength = record->writeHandleCreateInfoLength;
        const volatile uint8_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint8_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->writeHandleCreateInfo = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint8_t*>(memberBuffer),
              sizeof(uint8_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateBufferDeserialize);

struct DeviceCreateCommandEncoderTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUBool has_descriptor;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateCommandEncoderTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateCommandEncoderTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateCommandEncoderGetExtraRequiredSize([[maybe_unused]] const DeviceCreateCommandEncoderCmd& record) {
    size_t result = 0;

    if (record.descriptor != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUCommandEncoderDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUCommandEncoderDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateCommandEncoderGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateCommandEncoderSerialize(
    const DeviceCreateCommandEncoderCmd& record,
    DeviceCreateCommandEncoderTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateCommandEncoder;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    bool has_descriptor = record.descriptor != nullptr;
    transfer->has_descriptor = has_descriptor;
    if (has_descriptor) {
        auto memberLength = 1u;

        WGPUCommandEncoderDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCommandEncoderDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateCommandEncoderSerialize);

[[maybe_unused]] WireResult DeviceCreateCommandEncoderDeserialize(
    DeviceCreateCommandEncoderCmd* record,
    const volatile DeviceCreateCommandEncoderTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateCommandEncoder);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    bool has_descriptor = transfer->has_descriptor;
    record->descriptor = nullptr;
    if (has_descriptor) {
        auto memberLength = 1u;
        const volatile WGPUCommandEncoderDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCommandEncoderDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCommandEncoderDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateCommandEncoderDeserialize);

struct DeviceCreateComputePipelineTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateComputePipelineTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateComputePipelineTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateComputePipelineGetExtraRequiredSize([[maybe_unused]] const DeviceCreateComputePipelineCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUComputePipelineDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUComputePipelineDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateComputePipelineGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateComputePipelineSerialize(
    const DeviceCreateComputePipelineCmd& record,
    DeviceCreateComputePipelineTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateComputePipeline;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUComputePipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUComputePipelineDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateComputePipelineSerialize);

[[maybe_unused]] WireResult DeviceCreateComputePipelineDeserialize(
    DeviceCreateComputePipelineCmd* record,
    const volatile DeviceCreateComputePipelineTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateComputePipeline);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUComputePipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUComputePipelineDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUComputePipelineDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateComputePipelineDeserialize);

struct DeviceCreateComputePipelineAsyncTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId deviceId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
    ObjectHandle pipelineObjectHandle;
};

static_assert(offsetof(DeviceCreateComputePipelineAsyncTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateComputePipelineAsyncTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateComputePipelineAsyncGetExtraRequiredSize([[maybe_unused]] const DeviceCreateComputePipelineAsyncCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUComputePipelineDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUComputePipelineDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateComputePipelineAsyncGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateComputePipelineAsyncSerialize(
    const DeviceCreateComputePipelineAsyncCmd& record,
    DeviceCreateComputePipelineAsyncTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateComputePipelineAsync;


    transfer->deviceId = record.deviceId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->pipelineObjectHandle = record.pipelineObjectHandle;
    {
        auto memberLength = 1u;

        WGPUComputePipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUComputePipelineDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateComputePipelineAsyncSerialize);

[[maybe_unused]] WireResult DeviceCreateComputePipelineAsyncDeserialize(
    DeviceCreateComputePipelineAsyncCmd* record,
    const volatile DeviceCreateComputePipelineAsyncTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateComputePipelineAsync);


    static_assert(sizeof(record->deviceId) >= sizeof(transfer->deviceId), "Deserialize assignment may not narrow.");
    record->deviceId = transfer->deviceId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->pipelineObjectHandle) >= sizeof(transfer->pipelineObjectHandle), "Deserialize assignment may not narrow.");
    record->pipelineObjectHandle = transfer->pipelineObjectHandle;
    {
        auto memberLength = 1u;
        const volatile WGPUComputePipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUComputePipelineDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUComputePipelineDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateComputePipelineAsyncDeserialize);

struct DeviceCreateErrorBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateErrorBufferTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateErrorBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateErrorBufferGetExtraRequiredSize([[maybe_unused]] const DeviceCreateErrorBufferCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUBufferDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUBufferDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateErrorBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateErrorBufferSerialize(
    const DeviceCreateErrorBufferCmd& record,
    DeviceCreateErrorBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateErrorBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUBufferDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBufferDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorBufferSerialize);

[[maybe_unused]] WireResult DeviceCreateErrorBufferDeserialize(
    DeviceCreateErrorBufferCmd* record,
    const volatile DeviceCreateErrorBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateErrorBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUBufferDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUBufferDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUBufferDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorBufferDeserialize);

struct DeviceCreateErrorExternalTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateErrorExternalTextureTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateErrorExternalTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateErrorExternalTextureGetExtraRequiredSize([[maybe_unused]] const DeviceCreateErrorExternalTextureCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateErrorExternalTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateErrorExternalTextureSerialize(
    const DeviceCreateErrorExternalTextureCmd& record,
    DeviceCreateErrorExternalTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateErrorExternalTexture;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorExternalTextureSerialize);

[[maybe_unused]] WireResult DeviceCreateErrorExternalTextureDeserialize(
    DeviceCreateErrorExternalTextureCmd* record,
    const volatile DeviceCreateErrorExternalTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateErrorExternalTexture);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorExternalTextureDeserialize);

struct DeviceCreateErrorShaderModuleTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer errorMessage;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateErrorShaderModuleTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateErrorShaderModuleTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateErrorShaderModuleGetExtraRequiredSize([[maybe_unused]] const DeviceCreateErrorShaderModuleCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUShaderModuleDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUShaderModuleDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    result += WGPUStringViewGetExtraRequiredSize(record.errorMessage);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateErrorShaderModuleGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateErrorShaderModuleSerialize(
    const DeviceCreateErrorShaderModuleCmd& record,
    DeviceCreateErrorShaderModuleTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateErrorShaderModule;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.errorMessage, &transfer->errorMessage, buffer));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUShaderModuleDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUShaderModuleDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorShaderModuleSerialize);

[[maybe_unused]] WireResult DeviceCreateErrorShaderModuleDeserialize(
    DeviceCreateErrorShaderModuleCmd* record,
    const volatile DeviceCreateErrorShaderModuleTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateErrorShaderModule);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->errorMessage, &transfer->errorMessage, deserializeBuffer, allocator));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUShaderModuleDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUShaderModuleDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUShaderModuleDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorShaderModuleDeserialize);

struct DeviceCreateErrorTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateErrorTextureTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateErrorTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateErrorTextureGetExtraRequiredSize([[maybe_unused]] const DeviceCreateErrorTextureCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUTextureDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUTextureDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateErrorTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateErrorTextureSerialize(
    const DeviceCreateErrorTextureCmd& record,
    DeviceCreateErrorTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateErrorTexture;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorTextureSerialize);

[[maybe_unused]] WireResult DeviceCreateErrorTextureDeserialize(
    DeviceCreateErrorTextureCmd* record,
    const volatile DeviceCreateErrorTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateErrorTexture);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateErrorTextureDeserialize);

struct DeviceCreateExternalTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateExternalTextureTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateExternalTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateExternalTextureGetExtraRequiredSize([[maybe_unused]] const DeviceCreateExternalTextureCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUExternalTextureDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUExternalTextureDescriptorGetExtraRequiredSize(record.externalTextureDescriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateExternalTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateExternalTextureSerialize(
    const DeviceCreateExternalTextureCmd& record,
    DeviceCreateExternalTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateExternalTexture;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUExternalTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUExternalTextureDescriptorSerialize(record.externalTextureDescriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateExternalTextureSerialize);

[[maybe_unused]] WireResult DeviceCreateExternalTextureDeserialize(
    DeviceCreateExternalTextureCmd* record,
    const volatile DeviceCreateExternalTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateExternalTexture);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUExternalTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUExternalTextureDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->externalTextureDescriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUExternalTextureDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateExternalTextureDeserialize);

struct DeviceCreatePipelineLayoutTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreatePipelineLayoutTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreatePipelineLayoutTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreatePipelineLayoutGetExtraRequiredSize([[maybe_unused]] const DeviceCreatePipelineLayoutCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUPipelineLayoutDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUPipelineLayoutDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreatePipelineLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreatePipelineLayoutSerialize(
    const DeviceCreatePipelineLayoutCmd& record,
    DeviceCreatePipelineLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreatePipelineLayout;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUPipelineLayoutDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUPipelineLayoutDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreatePipelineLayoutSerialize);

[[maybe_unused]] WireResult DeviceCreatePipelineLayoutDeserialize(
    DeviceCreatePipelineLayoutCmd* record,
    const volatile DeviceCreatePipelineLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreatePipelineLayout);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUPipelineLayoutDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUPipelineLayoutDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUPipelineLayoutDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreatePipelineLayoutDeserialize);

struct DeviceCreateQuerySetTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateQuerySetTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateQuerySetTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateQuerySetGetExtraRequiredSize([[maybe_unused]] const DeviceCreateQuerySetCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUQuerySetDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUQuerySetDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateQuerySetGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateQuerySetSerialize(
    const DeviceCreateQuerySetCmd& record,
    DeviceCreateQuerySetTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateQuerySet;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUQuerySetDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUQuerySetDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateQuerySetSerialize);

[[maybe_unused]] WireResult DeviceCreateQuerySetDeserialize(
    DeviceCreateQuerySetCmd* record,
    const volatile DeviceCreateQuerySetTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateQuerySet);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUQuerySetDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUQuerySetDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUQuerySetDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateQuerySetDeserialize);

struct DeviceCreateRenderBundleEncoderTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateRenderBundleEncoderTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateRenderBundleEncoderTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateRenderBundleEncoderGetExtraRequiredSize([[maybe_unused]] const DeviceCreateRenderBundleEncoderCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURenderBundleEncoderDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderBundleEncoderDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateRenderBundleEncoderGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateRenderBundleEncoderSerialize(
    const DeviceCreateRenderBundleEncoderCmd& record,
    DeviceCreateRenderBundleEncoderTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateRenderBundleEncoder;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPURenderBundleEncoderDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderBundleEncoderDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateRenderBundleEncoderSerialize);

[[maybe_unused]] WireResult DeviceCreateRenderBundleEncoderDeserialize(
    DeviceCreateRenderBundleEncoderCmd* record,
    const volatile DeviceCreateRenderBundleEncoderTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateRenderBundleEncoder);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPURenderBundleEncoderDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderBundleEncoderDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderBundleEncoderDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateRenderBundleEncoderDeserialize);

struct DeviceCreateRenderPipelineTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateRenderPipelineTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateRenderPipelineTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateRenderPipelineGetExtraRequiredSize([[maybe_unused]] const DeviceCreateRenderPipelineCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURenderPipelineDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderPipelineDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateRenderPipelineGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateRenderPipelineSerialize(
    const DeviceCreateRenderPipelineCmd& record,
    DeviceCreateRenderPipelineTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateRenderPipeline;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPURenderPipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPipelineDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateRenderPipelineSerialize);

[[maybe_unused]] WireResult DeviceCreateRenderPipelineDeserialize(
    DeviceCreateRenderPipelineCmd* record,
    const volatile DeviceCreateRenderPipelineTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateRenderPipeline);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPURenderPipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderPipelineDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPipelineDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateRenderPipelineDeserialize);

struct DeviceCreateRenderPipelineAsyncTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId deviceId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
    ObjectHandle pipelineObjectHandle;
};

static_assert(offsetof(DeviceCreateRenderPipelineAsyncTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateRenderPipelineAsyncTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateRenderPipelineAsyncGetExtraRequiredSize([[maybe_unused]] const DeviceCreateRenderPipelineAsyncCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURenderPipelineDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderPipelineDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateRenderPipelineAsyncGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateRenderPipelineAsyncSerialize(
    const DeviceCreateRenderPipelineAsyncCmd& record,
    DeviceCreateRenderPipelineAsyncTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateRenderPipelineAsync;


    transfer->deviceId = record.deviceId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->pipelineObjectHandle = record.pipelineObjectHandle;
    {
        auto memberLength = 1u;

        WGPURenderPipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPipelineDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateRenderPipelineAsyncSerialize);

[[maybe_unused]] WireResult DeviceCreateRenderPipelineAsyncDeserialize(
    DeviceCreateRenderPipelineAsyncCmd* record,
    const volatile DeviceCreateRenderPipelineAsyncTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateRenderPipelineAsync);


    static_assert(sizeof(record->deviceId) >= sizeof(transfer->deviceId), "Deserialize assignment may not narrow.");
    record->deviceId = transfer->deviceId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->pipelineObjectHandle) >= sizeof(transfer->pipelineObjectHandle), "Deserialize assignment may not narrow.");
    record->pipelineObjectHandle = transfer->pipelineObjectHandle;
    {
        auto memberLength = 1u;
        const volatile WGPURenderPipelineDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderPipelineDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderPipelineDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateRenderPipelineAsyncDeserialize);

struct DeviceCreateSamplerTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUBool has_descriptor;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateSamplerTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateSamplerTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateSamplerGetExtraRequiredSize([[maybe_unused]] const DeviceCreateSamplerCmd& record) {
    size_t result = 0;

    if (record.descriptor != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUSamplerDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUSamplerDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateSamplerGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateSamplerSerialize(
    const DeviceCreateSamplerCmd& record,
    DeviceCreateSamplerTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateSampler;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    bool has_descriptor = record.descriptor != nullptr;
    transfer->has_descriptor = has_descriptor;
    if (has_descriptor) {
        auto memberLength = 1u;

        WGPUSamplerDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSamplerDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateSamplerSerialize);

[[maybe_unused]] WireResult DeviceCreateSamplerDeserialize(
    DeviceCreateSamplerCmd* record,
    const volatile DeviceCreateSamplerTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateSampler);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    bool has_descriptor = transfer->has_descriptor;
    record->descriptor = nullptr;
    if (has_descriptor) {
        auto memberLength = 1u;
        const volatile WGPUSamplerDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUSamplerDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSamplerDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateSamplerDeserialize);

struct DeviceCreateShaderModuleTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateShaderModuleTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateShaderModuleTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateShaderModuleGetExtraRequiredSize([[maybe_unused]] const DeviceCreateShaderModuleCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUShaderModuleDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUShaderModuleDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateShaderModuleGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateShaderModuleSerialize(
    const DeviceCreateShaderModuleCmd& record,
    DeviceCreateShaderModuleTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateShaderModule;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUShaderModuleDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUShaderModuleDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateShaderModuleSerialize);

[[maybe_unused]] WireResult DeviceCreateShaderModuleDeserialize(
    DeviceCreateShaderModuleCmd* record,
    const volatile DeviceCreateShaderModuleTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateShaderModule);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUShaderModuleDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUShaderModuleDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUShaderModuleDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateShaderModuleDeserialize);

struct DeviceCreateTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceCreateTextureTransfer, commandSize) == 0);
static_assert(offsetof(DeviceCreateTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceCreateTextureGetExtraRequiredSize([[maybe_unused]] const DeviceCreateTextureCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUTextureDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUTextureDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceCreateTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceCreateTextureSerialize(
    const DeviceCreateTextureCmd& record,
    DeviceCreateTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceCreateTexture;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    {
        auto memberLength = 1u;

        WGPUTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateTextureSerialize);

[[maybe_unused]] WireResult DeviceCreateTextureDeserialize(
    DeviceCreateTextureCmd* record,
    const volatile DeviceCreateTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceCreateTexture);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    {
        auto memberLength = 1u;
        const volatile WGPUTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceCreateTextureDeserialize);

struct DeviceDestroyTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(DeviceDestroyTransfer, commandSize) == 0);
static_assert(offsetof(DeviceDestroyTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceDestroyGetExtraRequiredSize([[maybe_unused]] const DeviceDestroyCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceDestroyGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceDestroySerialize(
    const DeviceDestroyCmd& record,
    DeviceDestroyTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceDestroy;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceDestroySerialize);

[[maybe_unused]] WireResult DeviceDestroyDeserialize(
    DeviceDestroyCmd* record,
    const volatile DeviceDestroyTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceDestroy);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceDestroyDeserialize);

struct DeviceForceLossTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUDeviceLostReason type;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(DeviceForceLossTransfer, commandSize) == 0);
static_assert(offsetof(DeviceForceLossTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceForceLossGetExtraRequiredSize([[maybe_unused]] const DeviceForceLossCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceForceLossGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceForceLossSerialize(
    const DeviceForceLossCmd& record,
    DeviceForceLossTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceForceLoss;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->type = record.type;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceForceLossSerialize);

[[maybe_unused]] WireResult DeviceForceLossDeserialize(
    DeviceForceLossCmd* record,
    const volatile DeviceForceLossTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceForceLoss);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceForceLossDeserialize);

struct DeviceGetQueueTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectHandle result;
};

static_assert(offsetof(DeviceGetQueueTransfer, commandSize) == 0);
static_assert(offsetof(DeviceGetQueueTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceGetQueueGetExtraRequiredSize([[maybe_unused]] const DeviceGetQueueCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceGetQueueGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceGetQueueSerialize(
    const DeviceGetQueueCmd& record,
    DeviceGetQueueTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceGetQueue;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceGetQueueSerialize);

[[maybe_unused]] WireResult DeviceGetQueueDeserialize(
    DeviceGetQueueCmd* record,
    const volatile DeviceGetQueueTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceGetQueue);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceGetQueueDeserialize);

struct DeviceInjectErrorTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUErrorType type;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(DeviceInjectErrorTransfer, commandSize) == 0);
static_assert(offsetof(DeviceInjectErrorTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceInjectErrorGetExtraRequiredSize([[maybe_unused]] const DeviceInjectErrorCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceInjectErrorGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceInjectErrorSerialize(
    const DeviceInjectErrorCmd& record,
    DeviceInjectErrorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceInjectError;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->type = record.type;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceInjectErrorSerialize);

[[maybe_unused]] WireResult DeviceInjectErrorDeserialize(
    DeviceInjectErrorCmd* record,
    const volatile DeviceInjectErrorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceInjectError);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceInjectErrorDeserialize);

struct DevicePopErrorScopeTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId deviceId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
};

static_assert(offsetof(DevicePopErrorScopeTransfer, commandSize) == 0);
static_assert(offsetof(DevicePopErrorScopeTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DevicePopErrorScopeGetExtraRequiredSize([[maybe_unused]] const DevicePopErrorScopeCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DevicePopErrorScopeGetExtraRequiredSize);

[[maybe_unused]] WireResult DevicePopErrorScopeSerialize(
    const DevicePopErrorScopeCmd& record,
    DevicePopErrorScopeTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::DevicePopErrorScope;


    transfer->deviceId = record.deviceId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DevicePopErrorScopeSerialize);

[[maybe_unused]] WireResult DevicePopErrorScopeDeserialize(
    DevicePopErrorScopeCmd* record,
    const volatile DevicePopErrorScopeTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DevicePopErrorScope);


    static_assert(sizeof(record->deviceId) >= sizeof(transfer->deviceId), "Deserialize assignment may not narrow.");
    record->deviceId = transfer->deviceId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DevicePopErrorScopeDeserialize);

struct DevicePushErrorScopeTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUErrorFilter filter;
};

static_assert(offsetof(DevicePushErrorScopeTransfer, commandSize) == 0);
static_assert(offsetof(DevicePushErrorScopeTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DevicePushErrorScopeGetExtraRequiredSize([[maybe_unused]] const DevicePushErrorScopeCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DevicePushErrorScopeGetExtraRequiredSize);

[[maybe_unused]] WireResult DevicePushErrorScopeSerialize(
    const DevicePushErrorScopeCmd& record,
    DevicePushErrorScopeTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DevicePushErrorScope;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->filter = record.filter;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DevicePushErrorScopeSerialize);

[[maybe_unused]] WireResult DevicePushErrorScopeDeserialize(
    DevicePushErrorScopeCmd* record,
    const volatile DevicePushErrorScopeTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DevicePushErrorScope);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->filter) >= sizeof(transfer->filter), "Deserialize assignment may not narrow.");
    record->filter = transfer->filter;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DevicePushErrorScopeDeserialize);

struct DeviceSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(DeviceSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(DeviceSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceSetLabelGetExtraRequiredSize([[maybe_unused]] const DeviceSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceSetLabelSerialize(
    const DeviceSetLabelCmd& record,
    DeviceSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceSetLabelSerialize);

[[maybe_unused]] WireResult DeviceSetLabelDeserialize(
    DeviceSetLabelCmd* record,
    const volatile DeviceSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceSetLabelDeserialize);

struct DeviceTickTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(DeviceTickTransfer, commandSize) == 0);
static_assert(offsetof(DeviceTickTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceTickGetExtraRequiredSize([[maybe_unused]] const DeviceTickCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceTickGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceTickSerialize(
    const DeviceTickCmd& record,
    DeviceTickTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceTick;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceTickSerialize);

[[maybe_unused]] WireResult DeviceTickDeserialize(
    DeviceTickCmd* record,
    const volatile DeviceTickTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceTick);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceTickDeserialize);

struct DeviceValidateTextureDescriptorTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(DeviceValidateTextureDescriptorTransfer, commandSize) == 0);
static_assert(offsetof(DeviceValidateTextureDescriptorTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t DeviceValidateTextureDescriptorGetExtraRequiredSize([[maybe_unused]] const DeviceValidateTextureDescriptorCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUTextureDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUTextureDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(DeviceValidateTextureDescriptorGetExtraRequiredSize);

[[maybe_unused]] WireResult DeviceValidateTextureDescriptorSerialize(
    const DeviceValidateTextureDescriptorCmd& record,
    DeviceValidateTextureDescriptorTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::DeviceValidateTextureDescriptor;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceValidateTextureDescriptorSerialize);

[[maybe_unused]] WireResult DeviceValidateTextureDescriptorDeserialize(
    DeviceValidateTextureDescriptorCmd* record,
    const volatile DeviceValidateTextureDescriptorTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::DeviceValidateTextureDescriptor);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUTextureDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(DeviceValidateTextureDescriptorDeserialize);

struct ExternalTextureDestroyTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(ExternalTextureDestroyTransfer, commandSize) == 0);
static_assert(offsetof(ExternalTextureDestroyTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ExternalTextureDestroyGetExtraRequiredSize([[maybe_unused]] const ExternalTextureDestroyCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ExternalTextureDestroyGetExtraRequiredSize);

[[maybe_unused]] WireResult ExternalTextureDestroySerialize(
    const ExternalTextureDestroyCmd& record,
    ExternalTextureDestroyTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ExternalTextureDestroy;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureDestroySerialize);

[[maybe_unused]] WireResult ExternalTextureDestroyDeserialize(
    ExternalTextureDestroyCmd* record,
    const volatile ExternalTextureDestroyTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ExternalTextureDestroy);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureDestroyDeserialize);

struct ExternalTextureExpireTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(ExternalTextureExpireTransfer, commandSize) == 0);
static_assert(offsetof(ExternalTextureExpireTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ExternalTextureExpireGetExtraRequiredSize([[maybe_unused]] const ExternalTextureExpireCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ExternalTextureExpireGetExtraRequiredSize);

[[maybe_unused]] WireResult ExternalTextureExpireSerialize(
    const ExternalTextureExpireCmd& record,
    ExternalTextureExpireTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ExternalTextureExpire;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureExpireSerialize);

[[maybe_unused]] WireResult ExternalTextureExpireDeserialize(
    ExternalTextureExpireCmd* record,
    const volatile ExternalTextureExpireTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ExternalTextureExpire);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureExpireDeserialize);

struct ExternalTextureRefreshTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(ExternalTextureRefreshTransfer, commandSize) == 0);
static_assert(offsetof(ExternalTextureRefreshTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ExternalTextureRefreshGetExtraRequiredSize([[maybe_unused]] const ExternalTextureRefreshCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ExternalTextureRefreshGetExtraRequiredSize);

[[maybe_unused]] WireResult ExternalTextureRefreshSerialize(
    const ExternalTextureRefreshCmd& record,
    ExternalTextureRefreshTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ExternalTextureRefresh;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureRefreshSerialize);

[[maybe_unused]] WireResult ExternalTextureRefreshDeserialize(
    ExternalTextureRefreshCmd* record,
    const volatile ExternalTextureRefreshTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ExternalTextureRefresh);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureRefreshDeserialize);

struct ExternalTextureSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(ExternalTextureSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(ExternalTextureSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ExternalTextureSetLabelGetExtraRequiredSize([[maybe_unused]] const ExternalTextureSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ExternalTextureSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult ExternalTextureSetLabelSerialize(
    const ExternalTextureSetLabelCmd& record,
    ExternalTextureSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ExternalTextureSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureSetLabelSerialize);

[[maybe_unused]] WireResult ExternalTextureSetLabelDeserialize(
    ExternalTextureSetLabelCmd* record,
    const volatile ExternalTextureSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ExternalTextureSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ExternalTextureSetLabelDeserialize);

struct InstanceProcessEventsTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(InstanceProcessEventsTransfer, commandSize) == 0);
static_assert(offsetof(InstanceProcessEventsTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t InstanceProcessEventsGetExtraRequiredSize([[maybe_unused]] const InstanceProcessEventsCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(InstanceProcessEventsGetExtraRequiredSize);

[[maybe_unused]] WireResult InstanceProcessEventsSerialize(
    const InstanceProcessEventsCmd& record,
    InstanceProcessEventsTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::InstanceProcessEvents;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(InstanceProcessEventsSerialize);

[[maybe_unused]] WireResult InstanceProcessEventsDeserialize(
    InstanceProcessEventsCmd* record,
    const volatile InstanceProcessEventsTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::InstanceProcessEvents);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(InstanceProcessEventsDeserialize);

struct InstanceRequestAdapterTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId instanceId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
    ObjectHandle adapterObjectHandle;
    WGPUBool has_options;
};

static_assert(offsetof(InstanceRequestAdapterTransfer, commandSize) == 0);
static_assert(offsetof(InstanceRequestAdapterTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t InstanceRequestAdapterGetExtraRequiredSize([[maybe_unused]] const InstanceRequestAdapterCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    if (record.options != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURequestAdapterOptionsTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURequestAdapterOptionsGetExtraRequiredSize(record.options[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(InstanceRequestAdapterGetExtraRequiredSize);

[[maybe_unused]] WireResult InstanceRequestAdapterSerialize(
    const InstanceRequestAdapterCmd& record,
    InstanceRequestAdapterTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::InstanceRequestAdapter;


    transfer->instanceId = record.instanceId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->adapterObjectHandle = record.adapterObjectHandle;
    bool has_options = record.options != nullptr;
    transfer->has_options = has_options;
    if (has_options) {
        auto memberLength = 1u;

        WGPURequestAdapterOptionsTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURequestAdapterOptionsSerialize(record.options[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(InstanceRequestAdapterSerialize);

[[maybe_unused]] WireResult InstanceRequestAdapterDeserialize(
    InstanceRequestAdapterCmd* record,
    const volatile InstanceRequestAdapterTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::InstanceRequestAdapter);


    static_assert(sizeof(record->instanceId) >= sizeof(transfer->instanceId), "Deserialize assignment may not narrow.");
    record->instanceId = transfer->instanceId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->adapterObjectHandle) >= sizeof(transfer->adapterObjectHandle), "Deserialize assignment may not narrow.");
    record->adapterObjectHandle = transfer->adapterObjectHandle;
    bool has_options = transfer->has_options;
    record->options = nullptr;
    if (has_options) {
        auto memberLength = 1u;
        const volatile WGPURequestAdapterOptionsTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURequestAdapterOptions* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->options = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURequestAdapterOptionsDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(InstanceRequestAdapterDeserialize);

struct PipelineLayoutSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(PipelineLayoutSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(PipelineLayoutSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t PipelineLayoutSetLabelGetExtraRequiredSize([[maybe_unused]] const PipelineLayoutSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(PipelineLayoutSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult PipelineLayoutSetLabelSerialize(
    const PipelineLayoutSetLabelCmd& record,
    PipelineLayoutSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::PipelineLayoutSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(PipelineLayoutSetLabelSerialize);

[[maybe_unused]] WireResult PipelineLayoutSetLabelDeserialize(
    PipelineLayoutSetLabelCmd* record,
    const volatile PipelineLayoutSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::PipelineLayoutSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(PipelineLayoutSetLabelDeserialize);

struct QuerySetDestroyTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(QuerySetDestroyTransfer, commandSize) == 0);
static_assert(offsetof(QuerySetDestroyTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QuerySetDestroyGetExtraRequiredSize([[maybe_unused]] const QuerySetDestroyCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QuerySetDestroyGetExtraRequiredSize);

[[maybe_unused]] WireResult QuerySetDestroySerialize(
    const QuerySetDestroyCmd& record,
    QuerySetDestroyTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::QuerySetDestroy;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QuerySetDestroySerialize);

[[maybe_unused]] WireResult QuerySetDestroyDeserialize(
    QuerySetDestroyCmd* record,
    const volatile QuerySetDestroyTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QuerySetDestroy);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QuerySetDestroyDeserialize);

struct QuerySetSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(QuerySetSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(QuerySetSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QuerySetSetLabelGetExtraRequiredSize([[maybe_unused]] const QuerySetSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QuerySetSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult QuerySetSetLabelSerialize(
    const QuerySetSetLabelCmd& record,
    QuerySetSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::QuerySetSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QuerySetSetLabelSerialize);

[[maybe_unused]] WireResult QuerySetSetLabelDeserialize(
    QuerySetSetLabelCmd* record,
    const volatile QuerySetSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QuerySetSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QuerySetSetLabelDeserialize);

struct QueueCopyExternalTextureForBrowserTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(QueueCopyExternalTextureForBrowserTransfer, commandSize) == 0);
static_assert(offsetof(QueueCopyExternalTextureForBrowserTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QueueCopyExternalTextureForBrowserGetExtraRequiredSize([[maybe_unused]] const QueueCopyExternalTextureForBrowserCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyExternalTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyExternalTextureGetExtraRequiredSize(record.source[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.destination[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUExtent3DTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUExtent3DGetExtraRequiredSize(record.copySize[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUCopyTextureForBrowserOptionsTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUCopyTextureForBrowserOptionsGetExtraRequiredSize(record.options[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QueueCopyExternalTextureForBrowserGetExtraRequiredSize);

[[maybe_unused]] WireResult QueueCopyExternalTextureForBrowserSerialize(
    const QueueCopyExternalTextureForBrowserCmd& record,
    QueueCopyExternalTextureForBrowserTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::QueueCopyExternalTextureForBrowser;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUImageCopyExternalTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyExternalTextureSerialize(record.source[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.destination[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.copySize,
                sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }
    {
        auto memberLength = 1u;

        WGPUCopyTextureForBrowserOptionsTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCopyTextureForBrowserOptionsSerialize(record.options[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueCopyExternalTextureForBrowserSerialize);

[[maybe_unused]] WireResult QueueCopyExternalTextureForBrowserDeserialize(
    QueueCopyExternalTextureForBrowserCmd* record,
    const volatile QueueCopyExternalTextureForBrowserTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QueueCopyExternalTextureForBrowser);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyExternalTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyExternalTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->source = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyExternalTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->destination = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUExtent3D* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->copySize = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUExtent3DTransfer*>(memberBuffer),
              sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUCopyTextureForBrowserOptionsTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCopyTextureForBrowserOptions* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->options = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCopyTextureForBrowserOptionsDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueCopyExternalTextureForBrowserDeserialize);

struct QueueCopyTextureForBrowserTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(QueueCopyTextureForBrowserTransfer, commandSize) == 0);
static_assert(offsetof(QueueCopyTextureForBrowserTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QueueCopyTextureForBrowserGetExtraRequiredSize([[maybe_unused]] const QueueCopyTextureForBrowserCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.source[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.destination[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUExtent3DTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUExtent3DGetExtraRequiredSize(record.copySize[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUCopyTextureForBrowserOptionsTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUCopyTextureForBrowserOptionsGetExtraRequiredSize(record.options[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QueueCopyTextureForBrowserGetExtraRequiredSize);

[[maybe_unused]] WireResult QueueCopyTextureForBrowserSerialize(
    const QueueCopyTextureForBrowserCmd& record,
    QueueCopyTextureForBrowserTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::QueueCopyTextureForBrowser;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.source[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.destination[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.copySize,
                sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }
    {
        auto memberLength = 1u;

        WGPUCopyTextureForBrowserOptionsTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCopyTextureForBrowserOptionsSerialize(record.options[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueCopyTextureForBrowserSerialize);

[[maybe_unused]] WireResult QueueCopyTextureForBrowserDeserialize(
    QueueCopyTextureForBrowserCmd* record,
    const volatile QueueCopyTextureForBrowserTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QueueCopyTextureForBrowser);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->source = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->destination = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUExtent3D* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->copySize = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUExtent3DTransfer*>(memberBuffer),
              sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUCopyTextureForBrowserOptionsTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCopyTextureForBrowserOptions* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->options = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCopyTextureForBrowserOptionsDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueCopyTextureForBrowserDeserialize);

struct QueueOnSubmittedWorkDoneTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId queueId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
};

static_assert(offsetof(QueueOnSubmittedWorkDoneTransfer, commandSize) == 0);
static_assert(offsetof(QueueOnSubmittedWorkDoneTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QueueOnSubmittedWorkDoneGetExtraRequiredSize([[maybe_unused]] const QueueOnSubmittedWorkDoneCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QueueOnSubmittedWorkDoneGetExtraRequiredSize);

[[maybe_unused]] WireResult QueueOnSubmittedWorkDoneSerialize(
    const QueueOnSubmittedWorkDoneCmd& record,
    QueueOnSubmittedWorkDoneTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::QueueOnSubmittedWorkDone;


    transfer->queueId = record.queueId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueOnSubmittedWorkDoneSerialize);

[[maybe_unused]] WireResult QueueOnSubmittedWorkDoneDeserialize(
    QueueOnSubmittedWorkDoneCmd* record,
    const volatile QueueOnSubmittedWorkDoneTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QueueOnSubmittedWorkDone);


    static_assert(sizeof(record->queueId) >= sizeof(transfer->queueId), "Deserialize assignment may not narrow.");
    record->queueId = transfer->queueId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueOnSubmittedWorkDoneDeserialize);

struct QueueSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(QueueSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(QueueSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QueueSetLabelGetExtraRequiredSize([[maybe_unused]] const QueueSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QueueSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult QueueSetLabelSerialize(
    const QueueSetLabelCmd& record,
    QueueSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::QueueSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueSetLabelSerialize);

[[maybe_unused]] WireResult QueueSetLabelDeserialize(
    QueueSetLabelCmd* record,
    const volatile QueueSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QueueSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueSetLabelDeserialize);

struct QueueSubmitTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint64_t commandCount;
};

static_assert(offsetof(QueueSubmitTransfer, commandSize) == 0);
static_assert(offsetof(QueueSubmitTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QueueSubmitGetExtraRequiredSize([[maybe_unused]] const QueueSubmitCmd& record) {
    size_t result = 0;

    {
        auto memberLength = record.commandCount;
        auto size = WireAlignSizeofN<ObjectId>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QueueSubmitGetExtraRequiredSize);

[[maybe_unused]] WireResult QueueSubmitSerialize(
    const QueueSubmitCmd& record,
    QueueSubmitTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::QueueSubmit;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->commandCount = record.commandCount;
    {
        auto memberLength = record.commandCount;

        ObjectId* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(provider.GetId(record.commands[i], &memberBuffer[i]));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueSubmitSerialize);

[[maybe_unused]] WireResult QueueSubmitDeserialize(
    QueueSubmitCmd* record,
    const volatile QueueSubmitTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QueueSubmit);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    if (transfer->commandCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->commandCount = checked_cast<size_t>(transfer->commandCount);
    {
        auto memberLength = record->commandCount;
        const volatile ObjectId* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCommandBuffer* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->commands = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(resolver.GetFromId(memberBuffer[i], &copiedMembers[i]));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueSubmitDeserialize);

struct QueueWriteBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId queueId;
    ObjectId bufferId;
    uint64_t bufferOffset;
    uint64_t size;
};

static_assert(offsetof(QueueWriteBufferTransfer, commandSize) == 0);
static_assert(offsetof(QueueWriteBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QueueWriteBufferGetExtraRequiredSize([[maybe_unused]] const QueueWriteBufferCmd& record) {
    size_t result = 0;

    {
        auto memberLength = record.size;
        auto size = WireAlignSizeofN<uint8_t>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QueueWriteBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult QueueWriteBufferSerialize(
    const QueueWriteBufferCmd& record,
    QueueWriteBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::QueueWriteBuffer;


    transfer->queueId = record.queueId;
    transfer->bufferId = record.bufferId;
    transfer->bufferOffset = record.bufferOffset;
    transfer->size = record.size;
    {
        auto memberLength = record.size;

        uint8_t* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.data,
                sizeof(uint8_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueWriteBufferSerialize);

[[maybe_unused]] WireResult QueueWriteBufferDeserialize(
    QueueWriteBufferCmd* record,
    const volatile QueueWriteBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QueueWriteBuffer);


    static_assert(sizeof(record->queueId) >= sizeof(transfer->queueId), "Deserialize assignment may not narrow.");
    record->queueId = transfer->queueId;
    static_assert(sizeof(record->bufferId) >= sizeof(transfer->bufferId), "Deserialize assignment may not narrow.");
    record->bufferId = transfer->bufferId;
    static_assert(sizeof(record->bufferOffset) >= sizeof(transfer->bufferOffset), "Deserialize assignment may not narrow.");
    record->bufferOffset = transfer->bufferOffset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;
    {
        auto memberLength = record->size;
        const volatile uint8_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        record->data =
            const_cast<const uint8_t*>(memberBuffer);

    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueWriteBufferDeserialize);

struct QueueWriteTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId queueId;
    uint64_t dataSize;
};

static_assert(offsetof(QueueWriteTextureTransfer, commandSize) == 0);
static_assert(offsetof(QueueWriteTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t QueueWriteTextureGetExtraRequiredSize([[maybe_unused]] const QueueWriteTextureCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUImageCopyTextureTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUImageCopyTextureGetExtraRequiredSize(record.destination[i]);
        }
    }
    {
        auto memberLength = record.dataSize;
        auto size = WireAlignSizeofN<uint8_t>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUTextureDataLayoutTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUTextureDataLayoutGetExtraRequiredSize(record.dataLayout[i]);
        }
    }
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUExtent3DTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUExtent3DGetExtraRequiredSize(record.writeSize[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(QueueWriteTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult QueueWriteTextureSerialize(
    const QueueWriteTextureCmd& record,
    QueueWriteTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::QueueWriteTexture;


    transfer->queueId = record.queueId;
    transfer->dataSize = record.dataSize;
    {
        auto memberLength = 1u;

        WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureSerialize(record.destination[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = record.dataSize;

        uint8_t* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.data,
                sizeof(uint8_t) * memberLength);
        }
    }
    {
        auto memberLength = 1u;

        WGPUTextureDataLayoutTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDataLayoutSerialize(record.dataLayout[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = 1u;

        WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.writeSize,
                sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueWriteTextureSerialize);

[[maybe_unused]] WireResult QueueWriteTextureDeserialize(
    QueueWriteTextureCmd* record,
    const volatile QueueWriteTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::QueueWriteTexture);


    static_assert(sizeof(record->queueId) >= sizeof(transfer->queueId), "Deserialize assignment may not narrow.");
    record->queueId = transfer->queueId;
    static_assert(sizeof(record->dataSize) >= sizeof(transfer->dataSize), "Deserialize assignment may not narrow.");
    record->dataSize = transfer->dataSize;
    {
        auto memberLength = 1u;
        const volatile WGPUImageCopyTextureTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUImageCopyTexture* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->destination = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUImageCopyTextureDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = record->dataSize;
        const volatile uint8_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        record->data =
            const_cast<const uint8_t*>(memberBuffer);

    }
    {
        auto memberLength = 1u;
        const volatile WGPUTextureDataLayoutTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureDataLayout* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->dataLayout = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureDataLayoutDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = 1u;
        const volatile WGPUExtent3DTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUExtent3D* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->writeSize = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUExtent3DTransfer*>(memberBuffer),
              sizeof(WGPUExtent3DTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(QueueWriteTextureDeserialize);

struct RenderBundleEncoderDrawTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t vertexCount;
    uint32_t instanceCount;
    uint32_t firstVertex;
    uint32_t firstInstance;
};

static_assert(offsetof(RenderBundleEncoderDrawTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderDrawTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderDrawGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderDrawCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawSerialize(
    const RenderBundleEncoderDrawCmd& record,
    RenderBundleEncoderDrawTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderDraw;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->vertexCount = record.vertexCount;
    transfer->instanceCount = record.instanceCount;
    transfer->firstVertex = record.firstVertex;
    transfer->firstInstance = record.firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawDeserialize(
    RenderBundleEncoderDrawCmd* record,
    const volatile RenderBundleEncoderDrawTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderDraw);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->vertexCount) >= sizeof(transfer->vertexCount), "Deserialize assignment may not narrow.");
    record->vertexCount = transfer->vertexCount;
    static_assert(sizeof(record->instanceCount) >= sizeof(transfer->instanceCount), "Deserialize assignment may not narrow.");
    record->instanceCount = transfer->instanceCount;
    static_assert(sizeof(record->firstVertex) >= sizeof(transfer->firstVertex), "Deserialize assignment may not narrow.");
    record->firstVertex = transfer->firstVertex;
    static_assert(sizeof(record->firstInstance) >= sizeof(transfer->firstInstance), "Deserialize assignment may not narrow.");
    record->firstInstance = transfer->firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawDeserialize);

struct RenderBundleEncoderDrawIndexedTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t indexCount;
    uint32_t instanceCount;
    uint32_t firstIndex;
    int32_t baseVertex;
    uint32_t firstInstance;
};

static_assert(offsetof(RenderBundleEncoderDrawIndexedTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderDrawIndexedTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderDrawIndexedGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderDrawIndexedCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndexedGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawIndexedSerialize(
    const RenderBundleEncoderDrawIndexedCmd& record,
    RenderBundleEncoderDrawIndexedTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderDrawIndexed;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->indexCount = record.indexCount;
    transfer->instanceCount = record.instanceCount;
    transfer->firstIndex = record.firstIndex;
    transfer->baseVertex = record.baseVertex;
    transfer->firstInstance = record.firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndexedSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawIndexedDeserialize(
    RenderBundleEncoderDrawIndexedCmd* record,
    const volatile RenderBundleEncoderDrawIndexedTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderDrawIndexed);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->indexCount) >= sizeof(transfer->indexCount), "Deserialize assignment may not narrow.");
    record->indexCount = transfer->indexCount;
    static_assert(sizeof(record->instanceCount) >= sizeof(transfer->instanceCount), "Deserialize assignment may not narrow.");
    record->instanceCount = transfer->instanceCount;
    static_assert(sizeof(record->firstIndex) >= sizeof(transfer->firstIndex), "Deserialize assignment may not narrow.");
    record->firstIndex = transfer->firstIndex;
    static_assert(sizeof(record->baseVertex) >= sizeof(transfer->baseVertex), "Deserialize assignment may not narrow.");
    record->baseVertex = transfer->baseVertex;
    static_assert(sizeof(record->firstInstance) >= sizeof(transfer->firstInstance), "Deserialize assignment may not narrow.");
    record->firstInstance = transfer->firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndexedDeserialize);

struct RenderBundleEncoderDrawIndexedIndirectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId indirectBuffer;
    uint64_t indirectOffset;
};

static_assert(offsetof(RenderBundleEncoderDrawIndexedIndirectTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderDrawIndexedIndirectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderDrawIndexedIndirectGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderDrawIndexedIndirectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndexedIndirectGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawIndexedIndirectSerialize(
    const RenderBundleEncoderDrawIndexedIndirectCmd& record,
    RenderBundleEncoderDrawIndexedIndirectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderDrawIndexedIndirect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.indirectBuffer, &transfer->indirectBuffer));
    transfer->indirectOffset = record.indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndexedIndirectSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawIndexedIndirectDeserialize(
    RenderBundleEncoderDrawIndexedIndirectCmd* record,
    const volatile RenderBundleEncoderDrawIndexedIndirectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderDrawIndexedIndirect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->indirectBuffer, &record->indirectBuffer));
    static_assert(sizeof(record->indirectOffset) >= sizeof(transfer->indirectOffset), "Deserialize assignment may not narrow.");
    record->indirectOffset = transfer->indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndexedIndirectDeserialize);

struct RenderBundleEncoderDrawIndirectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId indirectBuffer;
    uint64_t indirectOffset;
};

static_assert(offsetof(RenderBundleEncoderDrawIndirectTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderDrawIndirectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderDrawIndirectGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderDrawIndirectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndirectGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawIndirectSerialize(
    const RenderBundleEncoderDrawIndirectCmd& record,
    RenderBundleEncoderDrawIndirectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderDrawIndirect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.indirectBuffer, &transfer->indirectBuffer));
    transfer->indirectOffset = record.indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndirectSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderDrawIndirectDeserialize(
    RenderBundleEncoderDrawIndirectCmd* record,
    const volatile RenderBundleEncoderDrawIndirectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderDrawIndirect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->indirectBuffer, &record->indirectBuffer));
    static_assert(sizeof(record->indirectOffset) >= sizeof(transfer->indirectOffset), "Deserialize assignment may not narrow.");
    record->indirectOffset = transfer->indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderDrawIndirectDeserialize);

struct RenderBundleEncoderFinishTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUBool has_descriptor;
    ObjectHandle result;
};

static_assert(offsetof(RenderBundleEncoderFinishTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderFinishTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderFinishGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderFinishCmd& record) {
    size_t result = 0;

    if (record.descriptor != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPURenderBundleDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPURenderBundleDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderFinishGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderFinishSerialize(
    const RenderBundleEncoderFinishCmd& record,
    RenderBundleEncoderFinishTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderFinish;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    bool has_descriptor = record.descriptor != nullptr;
    transfer->has_descriptor = has_descriptor;
    if (has_descriptor) {
        auto memberLength = 1u;

        WGPURenderBundleDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderBundleDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderFinishSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderFinishDeserialize(
    RenderBundleEncoderFinishCmd* record,
    const volatile RenderBundleEncoderFinishTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderFinish);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    bool has_descriptor = transfer->has_descriptor;
    record->descriptor = nullptr;
    if (has_descriptor) {
        auto memberLength = 1u;
        const volatile WGPURenderBundleDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderBundleDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPURenderBundleDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderFinishDeserialize);

struct RenderBundleEncoderInsertDebugMarkerTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer markerLabel;
};

static_assert(offsetof(RenderBundleEncoderInsertDebugMarkerTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderInsertDebugMarkerTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderInsertDebugMarkerGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderInsertDebugMarkerCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.markerLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderInsertDebugMarkerGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderInsertDebugMarkerSerialize(
    const RenderBundleEncoderInsertDebugMarkerCmd& record,
    RenderBundleEncoderInsertDebugMarkerTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderInsertDebugMarker;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.markerLabel, &transfer->markerLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderInsertDebugMarkerSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderInsertDebugMarkerDeserialize(
    RenderBundleEncoderInsertDebugMarkerCmd* record,
    const volatile RenderBundleEncoderInsertDebugMarkerTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderInsertDebugMarker);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->markerLabel, &transfer->markerLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderInsertDebugMarkerDeserialize);

struct RenderBundleEncoderPopDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(RenderBundleEncoderPopDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderPopDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderPopDebugGroupGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderPopDebugGroupCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderPopDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderPopDebugGroupSerialize(
    const RenderBundleEncoderPopDebugGroupCmd& record,
    RenderBundleEncoderPopDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderPopDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderPopDebugGroupSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderPopDebugGroupDeserialize(
    RenderBundleEncoderPopDebugGroupCmd* record,
    const volatile RenderBundleEncoderPopDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderPopDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderPopDebugGroupDeserialize);

struct RenderBundleEncoderPushDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer groupLabel;
};

static_assert(offsetof(RenderBundleEncoderPushDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderPushDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderPushDebugGroupGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderPushDebugGroupCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.groupLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderPushDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderPushDebugGroupSerialize(
    const RenderBundleEncoderPushDebugGroupCmd& record,
    RenderBundleEncoderPushDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderPushDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.groupLabel, &transfer->groupLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderPushDebugGroupSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderPushDebugGroupDeserialize(
    RenderBundleEncoderPushDebugGroupCmd* record,
    const volatile RenderBundleEncoderPushDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderPushDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->groupLabel, &transfer->groupLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderPushDebugGroupDeserialize);

struct RenderBundleEncoderSetBindGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t groupIndex;
    ObjectId group;
    uint64_t dynamicOffsetCount;
};

static_assert(offsetof(RenderBundleEncoderSetBindGroupTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderSetBindGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderSetBindGroupGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderSetBindGroupCmd& record) {
    size_t result = 0;

    {
        auto memberLength = record.dynamicOffsetCount;
        auto size = WireAlignSizeofN<uint32_t>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderSetBindGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderSetBindGroupSerialize(
    const RenderBundleEncoderSetBindGroupCmd& record,
    RenderBundleEncoderSetBindGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderSetBindGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->groupIndex = record.groupIndex;
    WIRE_TRY(provider.GetOptionalId(record.group, &transfer->group));
    transfer->dynamicOffsetCount = record.dynamicOffsetCount;
    {
        auto memberLength = record.dynamicOffsetCount;

        uint32_t* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.dynamicOffsets,
                sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetBindGroupSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderSetBindGroupDeserialize(
    RenderBundleEncoderSetBindGroupCmd* record,
    const volatile RenderBundleEncoderSetBindGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderSetBindGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->groupIndex) >= sizeof(transfer->groupIndex), "Deserialize assignment may not narrow.");
    record->groupIndex = transfer->groupIndex;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->group, &record->group));
    if (transfer->dynamicOffsetCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->dynamicOffsetCount = checked_cast<size_t>(transfer->dynamicOffsetCount);
    {
        auto memberLength = record->dynamicOffsetCount;
        const volatile uint32_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint32_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->dynamicOffsets = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint32_t*>(memberBuffer),
              sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetBindGroupDeserialize);

struct RenderBundleEncoderSetIndexBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId buffer;
    WGPUIndexFormat format;
    uint64_t offset;
    uint64_t size;
};

static_assert(offsetof(RenderBundleEncoderSetIndexBufferTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderSetIndexBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderSetIndexBufferGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderSetIndexBufferCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderSetIndexBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderSetIndexBufferSerialize(
    const RenderBundleEncoderSetIndexBufferCmd& record,
    RenderBundleEncoderSetIndexBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderSetIndexBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.buffer, &transfer->buffer));
    transfer->format = record.format;
    transfer->offset = record.offset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetIndexBufferSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderSetIndexBufferDeserialize(
    RenderBundleEncoderSetIndexBufferCmd* record,
    const volatile RenderBundleEncoderSetIndexBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderSetIndexBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->buffer, &record->buffer));
    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetIndexBufferDeserialize);

struct RenderBundleEncoderSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(RenderBundleEncoderSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderSetLabelGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderSetLabelSerialize(
    const RenderBundleEncoderSetLabelCmd& record,
    RenderBundleEncoderSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetLabelSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderSetLabelDeserialize(
    RenderBundleEncoderSetLabelCmd* record,
    const volatile RenderBundleEncoderSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetLabelDeserialize);

struct RenderBundleEncoderSetPipelineTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId pipeline;
};

static_assert(offsetof(RenderBundleEncoderSetPipelineTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderSetPipelineTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderSetPipelineGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderSetPipelineCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderSetPipelineGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderSetPipelineSerialize(
    const RenderBundleEncoderSetPipelineCmd& record,
    RenderBundleEncoderSetPipelineTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderSetPipeline;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.pipeline, &transfer->pipeline));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetPipelineSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderSetPipelineDeserialize(
    RenderBundleEncoderSetPipelineCmd* record,
    const volatile RenderBundleEncoderSetPipelineTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderSetPipeline);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->pipeline, &record->pipeline));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetPipelineDeserialize);

struct RenderBundleEncoderSetVertexBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t slot;
    ObjectId buffer;
    uint64_t offset;
    uint64_t size;
};

static_assert(offsetof(RenderBundleEncoderSetVertexBufferTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleEncoderSetVertexBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleEncoderSetVertexBufferGetExtraRequiredSize([[maybe_unused]] const RenderBundleEncoderSetVertexBufferCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleEncoderSetVertexBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleEncoderSetVertexBufferSerialize(
    const RenderBundleEncoderSetVertexBufferCmd& record,
    RenderBundleEncoderSetVertexBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleEncoderSetVertexBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->slot = record.slot;
    WIRE_TRY(provider.GetOptionalId(record.buffer, &transfer->buffer));
    transfer->offset = record.offset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetVertexBufferSerialize);

[[maybe_unused]] WireResult RenderBundleEncoderSetVertexBufferDeserialize(
    RenderBundleEncoderSetVertexBufferCmd* record,
    const volatile RenderBundleEncoderSetVertexBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleEncoderSetVertexBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->slot) >= sizeof(transfer->slot), "Deserialize assignment may not narrow.");
    record->slot = transfer->slot;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->buffer, &record->buffer));
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleEncoderSetVertexBufferDeserialize);

struct RenderBundleSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(RenderBundleSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(RenderBundleSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderBundleSetLabelGetExtraRequiredSize([[maybe_unused]] const RenderBundleSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderBundleSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderBundleSetLabelSerialize(
    const RenderBundleSetLabelCmd& record,
    RenderBundleSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderBundleSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleSetLabelSerialize);

[[maybe_unused]] WireResult RenderBundleSetLabelDeserialize(
    RenderBundleSetLabelCmd* record,
    const volatile RenderBundleSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderBundleSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderBundleSetLabelDeserialize);

struct RenderPassEncoderBeginOcclusionQueryTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t queryIndex;
};

static_assert(offsetof(RenderPassEncoderBeginOcclusionQueryTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderBeginOcclusionQueryTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderBeginOcclusionQueryGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderBeginOcclusionQueryCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderBeginOcclusionQueryGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderBeginOcclusionQuerySerialize(
    const RenderPassEncoderBeginOcclusionQueryCmd& record,
    RenderPassEncoderBeginOcclusionQueryTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderBeginOcclusionQuery;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->queryIndex = record.queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderBeginOcclusionQuerySerialize);

[[maybe_unused]] WireResult RenderPassEncoderBeginOcclusionQueryDeserialize(
    RenderPassEncoderBeginOcclusionQueryCmd* record,
    const volatile RenderPassEncoderBeginOcclusionQueryTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderBeginOcclusionQuery);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->queryIndex) >= sizeof(transfer->queryIndex), "Deserialize assignment may not narrow.");
    record->queryIndex = transfer->queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderBeginOcclusionQueryDeserialize);

struct RenderPassEncoderDrawTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t vertexCount;
    uint32_t instanceCount;
    uint32_t firstVertex;
    uint32_t firstInstance;
};

static_assert(offsetof(RenderPassEncoderDrawTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderDrawTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderDrawGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderDrawCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderDrawGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderDrawSerialize(
    const RenderPassEncoderDrawCmd& record,
    RenderPassEncoderDrawTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderDraw;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->vertexCount = record.vertexCount;
    transfer->instanceCount = record.instanceCount;
    transfer->firstVertex = record.firstVertex;
    transfer->firstInstance = record.firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawSerialize);

[[maybe_unused]] WireResult RenderPassEncoderDrawDeserialize(
    RenderPassEncoderDrawCmd* record,
    const volatile RenderPassEncoderDrawTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderDraw);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->vertexCount) >= sizeof(transfer->vertexCount), "Deserialize assignment may not narrow.");
    record->vertexCount = transfer->vertexCount;
    static_assert(sizeof(record->instanceCount) >= sizeof(transfer->instanceCount), "Deserialize assignment may not narrow.");
    record->instanceCount = transfer->instanceCount;
    static_assert(sizeof(record->firstVertex) >= sizeof(transfer->firstVertex), "Deserialize assignment may not narrow.");
    record->firstVertex = transfer->firstVertex;
    static_assert(sizeof(record->firstInstance) >= sizeof(transfer->firstInstance), "Deserialize assignment may not narrow.");
    record->firstInstance = transfer->firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawDeserialize);

struct RenderPassEncoderDrawIndexedTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t indexCount;
    uint32_t instanceCount;
    uint32_t firstIndex;
    int32_t baseVertex;
    uint32_t firstInstance;
};

static_assert(offsetof(RenderPassEncoderDrawIndexedTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderDrawIndexedTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderDrawIndexedGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderDrawIndexedCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndexedGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderDrawIndexedSerialize(
    const RenderPassEncoderDrawIndexedCmd& record,
    RenderPassEncoderDrawIndexedTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderDrawIndexed;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->indexCount = record.indexCount;
    transfer->instanceCount = record.instanceCount;
    transfer->firstIndex = record.firstIndex;
    transfer->baseVertex = record.baseVertex;
    transfer->firstInstance = record.firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndexedSerialize);

[[maybe_unused]] WireResult RenderPassEncoderDrawIndexedDeserialize(
    RenderPassEncoderDrawIndexedCmd* record,
    const volatile RenderPassEncoderDrawIndexedTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderDrawIndexed);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->indexCount) >= sizeof(transfer->indexCount), "Deserialize assignment may not narrow.");
    record->indexCount = transfer->indexCount;
    static_assert(sizeof(record->instanceCount) >= sizeof(transfer->instanceCount), "Deserialize assignment may not narrow.");
    record->instanceCount = transfer->instanceCount;
    static_assert(sizeof(record->firstIndex) >= sizeof(transfer->firstIndex), "Deserialize assignment may not narrow.");
    record->firstIndex = transfer->firstIndex;
    static_assert(sizeof(record->baseVertex) >= sizeof(transfer->baseVertex), "Deserialize assignment may not narrow.");
    record->baseVertex = transfer->baseVertex;
    static_assert(sizeof(record->firstInstance) >= sizeof(transfer->firstInstance), "Deserialize assignment may not narrow.");
    record->firstInstance = transfer->firstInstance;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndexedDeserialize);

struct RenderPassEncoderDrawIndexedIndirectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId indirectBuffer;
    uint64_t indirectOffset;
};

static_assert(offsetof(RenderPassEncoderDrawIndexedIndirectTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderDrawIndexedIndirectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderDrawIndexedIndirectGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderDrawIndexedIndirectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndexedIndirectGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderDrawIndexedIndirectSerialize(
    const RenderPassEncoderDrawIndexedIndirectCmd& record,
    RenderPassEncoderDrawIndexedIndirectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderDrawIndexedIndirect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.indirectBuffer, &transfer->indirectBuffer));
    transfer->indirectOffset = record.indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndexedIndirectSerialize);

[[maybe_unused]] WireResult RenderPassEncoderDrawIndexedIndirectDeserialize(
    RenderPassEncoderDrawIndexedIndirectCmd* record,
    const volatile RenderPassEncoderDrawIndexedIndirectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderDrawIndexedIndirect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->indirectBuffer, &record->indirectBuffer));
    static_assert(sizeof(record->indirectOffset) >= sizeof(transfer->indirectOffset), "Deserialize assignment may not narrow.");
    record->indirectOffset = transfer->indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndexedIndirectDeserialize);

struct RenderPassEncoderDrawIndirectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId indirectBuffer;
    uint64_t indirectOffset;
};

static_assert(offsetof(RenderPassEncoderDrawIndirectTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderDrawIndirectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderDrawIndirectGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderDrawIndirectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndirectGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderDrawIndirectSerialize(
    const RenderPassEncoderDrawIndirectCmd& record,
    RenderPassEncoderDrawIndirectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderDrawIndirect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.indirectBuffer, &transfer->indirectBuffer));
    transfer->indirectOffset = record.indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndirectSerialize);

[[maybe_unused]] WireResult RenderPassEncoderDrawIndirectDeserialize(
    RenderPassEncoderDrawIndirectCmd* record,
    const volatile RenderPassEncoderDrawIndirectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderDrawIndirect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->indirectBuffer, &record->indirectBuffer));
    static_assert(sizeof(record->indirectOffset) >= sizeof(transfer->indirectOffset), "Deserialize assignment may not narrow.");
    record->indirectOffset = transfer->indirectOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderDrawIndirectDeserialize);

struct RenderPassEncoderEndTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(RenderPassEncoderEndTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderEndTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderEndGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderEndCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderEndGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderEndSerialize(
    const RenderPassEncoderEndCmd& record,
    RenderPassEncoderEndTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderEnd;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderEndSerialize);

[[maybe_unused]] WireResult RenderPassEncoderEndDeserialize(
    RenderPassEncoderEndCmd* record,
    const volatile RenderPassEncoderEndTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderEnd);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderEndDeserialize);

struct RenderPassEncoderEndOcclusionQueryTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(RenderPassEncoderEndOcclusionQueryTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderEndOcclusionQueryTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderEndOcclusionQueryGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderEndOcclusionQueryCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderEndOcclusionQueryGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderEndOcclusionQuerySerialize(
    const RenderPassEncoderEndOcclusionQueryCmd& record,
    RenderPassEncoderEndOcclusionQueryTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderEndOcclusionQuery;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderEndOcclusionQuerySerialize);

[[maybe_unused]] WireResult RenderPassEncoderEndOcclusionQueryDeserialize(
    RenderPassEncoderEndOcclusionQueryCmd* record,
    const volatile RenderPassEncoderEndOcclusionQueryTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderEndOcclusionQuery);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderEndOcclusionQueryDeserialize);

struct RenderPassEncoderExecuteBundlesTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint64_t bundleCount;
};

static_assert(offsetof(RenderPassEncoderExecuteBundlesTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderExecuteBundlesTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderExecuteBundlesGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderExecuteBundlesCmd& record) {
    size_t result = 0;

    {
        auto memberLength = record.bundleCount;
        auto size = WireAlignSizeofN<ObjectId>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderExecuteBundlesGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderExecuteBundlesSerialize(
    const RenderPassEncoderExecuteBundlesCmd& record,
    RenderPassEncoderExecuteBundlesTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderExecuteBundles;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->bundleCount = record.bundleCount;
    {
        auto memberLength = record.bundleCount;

        ObjectId* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(provider.GetId(record.bundles[i], &memberBuffer[i]));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderExecuteBundlesSerialize);

[[maybe_unused]] WireResult RenderPassEncoderExecuteBundlesDeserialize(
    RenderPassEncoderExecuteBundlesCmd* record,
    const volatile RenderPassEncoderExecuteBundlesTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderExecuteBundles);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    if (transfer->bundleCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->bundleCount = checked_cast<size_t>(transfer->bundleCount);
    {
        auto memberLength = record->bundleCount;
        const volatile ObjectId* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPURenderBundle* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->bundles = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(resolver.GetFromId(memberBuffer[i], &copiedMembers[i]));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderExecuteBundlesDeserialize);

struct RenderPassEncoderInsertDebugMarkerTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer markerLabel;
};

static_assert(offsetof(RenderPassEncoderInsertDebugMarkerTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderInsertDebugMarkerTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderInsertDebugMarkerGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderInsertDebugMarkerCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.markerLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderInsertDebugMarkerGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderInsertDebugMarkerSerialize(
    const RenderPassEncoderInsertDebugMarkerCmd& record,
    RenderPassEncoderInsertDebugMarkerTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderInsertDebugMarker;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.markerLabel, &transfer->markerLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderInsertDebugMarkerSerialize);

[[maybe_unused]] WireResult RenderPassEncoderInsertDebugMarkerDeserialize(
    RenderPassEncoderInsertDebugMarkerCmd* record,
    const volatile RenderPassEncoderInsertDebugMarkerTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderInsertDebugMarker);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->markerLabel, &transfer->markerLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderInsertDebugMarkerDeserialize);

struct RenderPassEncoderMultiDrawIndexedIndirectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId indirectBuffer;
    uint64_t indirectOffset;
    uint32_t maxDrawCount;
    ObjectId drawCountBuffer;
    uint64_t drawCountBufferOffset;
};

static_assert(offsetof(RenderPassEncoderMultiDrawIndexedIndirectTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderMultiDrawIndexedIndirectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderMultiDrawIndexedIndirectGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderMultiDrawIndexedIndirectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderMultiDrawIndexedIndirectGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderMultiDrawIndexedIndirectSerialize(
    const RenderPassEncoderMultiDrawIndexedIndirectCmd& record,
    RenderPassEncoderMultiDrawIndexedIndirectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderMultiDrawIndexedIndirect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.indirectBuffer, &transfer->indirectBuffer));
    transfer->indirectOffset = record.indirectOffset;
    transfer->maxDrawCount = record.maxDrawCount;
    WIRE_TRY(provider.GetOptionalId(record.drawCountBuffer, &transfer->drawCountBuffer));
    transfer->drawCountBufferOffset = record.drawCountBufferOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderMultiDrawIndexedIndirectSerialize);

[[maybe_unused]] WireResult RenderPassEncoderMultiDrawIndexedIndirectDeserialize(
    RenderPassEncoderMultiDrawIndexedIndirectCmd* record,
    const volatile RenderPassEncoderMultiDrawIndexedIndirectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderMultiDrawIndexedIndirect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->indirectBuffer, &record->indirectBuffer));
    static_assert(sizeof(record->indirectOffset) >= sizeof(transfer->indirectOffset), "Deserialize assignment may not narrow.");
    record->indirectOffset = transfer->indirectOffset;
    static_assert(sizeof(record->maxDrawCount) >= sizeof(transfer->maxDrawCount), "Deserialize assignment may not narrow.");
    record->maxDrawCount = transfer->maxDrawCount;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->drawCountBuffer, &record->drawCountBuffer));
    static_assert(sizeof(record->drawCountBufferOffset) >= sizeof(transfer->drawCountBufferOffset), "Deserialize assignment may not narrow.");
    record->drawCountBufferOffset = transfer->drawCountBufferOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderMultiDrawIndexedIndirectDeserialize);

struct RenderPassEncoderMultiDrawIndirectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId indirectBuffer;
    uint64_t indirectOffset;
    uint32_t maxDrawCount;
    ObjectId drawCountBuffer;
    uint64_t drawCountBufferOffset;
};

static_assert(offsetof(RenderPassEncoderMultiDrawIndirectTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderMultiDrawIndirectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderMultiDrawIndirectGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderMultiDrawIndirectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderMultiDrawIndirectGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderMultiDrawIndirectSerialize(
    const RenderPassEncoderMultiDrawIndirectCmd& record,
    RenderPassEncoderMultiDrawIndirectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderMultiDrawIndirect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.indirectBuffer, &transfer->indirectBuffer));
    transfer->indirectOffset = record.indirectOffset;
    transfer->maxDrawCount = record.maxDrawCount;
    WIRE_TRY(provider.GetOptionalId(record.drawCountBuffer, &transfer->drawCountBuffer));
    transfer->drawCountBufferOffset = record.drawCountBufferOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderMultiDrawIndirectSerialize);

[[maybe_unused]] WireResult RenderPassEncoderMultiDrawIndirectDeserialize(
    RenderPassEncoderMultiDrawIndirectCmd* record,
    const volatile RenderPassEncoderMultiDrawIndirectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderMultiDrawIndirect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->indirectBuffer, &record->indirectBuffer));
    static_assert(sizeof(record->indirectOffset) >= sizeof(transfer->indirectOffset), "Deserialize assignment may not narrow.");
    record->indirectOffset = transfer->indirectOffset;
    static_assert(sizeof(record->maxDrawCount) >= sizeof(transfer->maxDrawCount), "Deserialize assignment may not narrow.");
    record->maxDrawCount = transfer->maxDrawCount;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->drawCountBuffer, &record->drawCountBuffer));
    static_assert(sizeof(record->drawCountBufferOffset) >= sizeof(transfer->drawCountBufferOffset), "Deserialize assignment may not narrow.");
    record->drawCountBufferOffset = transfer->drawCountBufferOffset;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderMultiDrawIndirectDeserialize);

struct RenderPassEncoderPixelLocalStorageBarrierTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(RenderPassEncoderPixelLocalStorageBarrierTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderPixelLocalStorageBarrierTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderPixelLocalStorageBarrierGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderPixelLocalStorageBarrierCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderPixelLocalStorageBarrierGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderPixelLocalStorageBarrierSerialize(
    const RenderPassEncoderPixelLocalStorageBarrierCmd& record,
    RenderPassEncoderPixelLocalStorageBarrierTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderPixelLocalStorageBarrier;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderPixelLocalStorageBarrierSerialize);

[[maybe_unused]] WireResult RenderPassEncoderPixelLocalStorageBarrierDeserialize(
    RenderPassEncoderPixelLocalStorageBarrierCmd* record,
    const volatile RenderPassEncoderPixelLocalStorageBarrierTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderPixelLocalStorageBarrier);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderPixelLocalStorageBarrierDeserialize);

struct RenderPassEncoderPopDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(RenderPassEncoderPopDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderPopDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderPopDebugGroupGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderPopDebugGroupCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderPopDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderPopDebugGroupSerialize(
    const RenderPassEncoderPopDebugGroupCmd& record,
    RenderPassEncoderPopDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderPopDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderPopDebugGroupSerialize);

[[maybe_unused]] WireResult RenderPassEncoderPopDebugGroupDeserialize(
    RenderPassEncoderPopDebugGroupCmd* record,
    const volatile RenderPassEncoderPopDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderPopDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderPopDebugGroupDeserialize);

struct RenderPassEncoderPushDebugGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer groupLabel;
};

static_assert(offsetof(RenderPassEncoderPushDebugGroupTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderPushDebugGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderPushDebugGroupGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderPushDebugGroupCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.groupLabel);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderPushDebugGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderPushDebugGroupSerialize(
    const RenderPassEncoderPushDebugGroupCmd& record,
    RenderPassEncoderPushDebugGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderPushDebugGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.groupLabel, &transfer->groupLabel, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderPushDebugGroupSerialize);

[[maybe_unused]] WireResult RenderPassEncoderPushDebugGroupDeserialize(
    RenderPassEncoderPushDebugGroupCmd* record,
    const volatile RenderPassEncoderPushDebugGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderPushDebugGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->groupLabel, &transfer->groupLabel, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderPushDebugGroupDeserialize);

struct RenderPassEncoderSetBindGroupTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t groupIndex;
    ObjectId group;
    uint64_t dynamicOffsetCount;
};

static_assert(offsetof(RenderPassEncoderSetBindGroupTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetBindGroupTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetBindGroupGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetBindGroupCmd& record) {
    size_t result = 0;

    {
        auto memberLength = record.dynamicOffsetCount;
        auto size = WireAlignSizeofN<uint32_t>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetBindGroupGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetBindGroupSerialize(
    const RenderPassEncoderSetBindGroupCmd& record,
    RenderPassEncoderSetBindGroupTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetBindGroup;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->groupIndex = record.groupIndex;
    WIRE_TRY(provider.GetOptionalId(record.group, &transfer->group));
    transfer->dynamicOffsetCount = record.dynamicOffsetCount;
    {
        auto memberLength = record.dynamicOffsetCount;

        uint32_t* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.dynamicOffsets,
                sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetBindGroupSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetBindGroupDeserialize(
    RenderPassEncoderSetBindGroupCmd* record,
    const volatile RenderPassEncoderSetBindGroupTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetBindGroup);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->groupIndex) >= sizeof(transfer->groupIndex), "Deserialize assignment may not narrow.");
    record->groupIndex = transfer->groupIndex;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->group, &record->group));
    if (transfer->dynamicOffsetCount > std::numeric_limits<size_t>::max()) return WireResult::FatalError;
    record->dynamicOffsetCount = checked_cast<size_t>(transfer->dynamicOffsetCount);
    {
        auto memberLength = record->dynamicOffsetCount;
        const volatile uint32_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint32_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->dynamicOffsets = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint32_t*>(memberBuffer),
              sizeof(uint32_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetBindGroupDeserialize);

struct RenderPassEncoderSetBlendConstantTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(RenderPassEncoderSetBlendConstantTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetBlendConstantTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetBlendConstantGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetBlendConstantCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUColorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUColorGetExtraRequiredSize(record.color[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetBlendConstantGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetBlendConstantSerialize(
    const RenderPassEncoderSetBlendConstantCmd& record,
    RenderPassEncoderSetBlendConstantTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetBlendConstant;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUColorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.color,
                sizeof(WGPUColorTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetBlendConstantSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetBlendConstantDeserialize(
    RenderPassEncoderSetBlendConstantCmd* record,
    const volatile RenderPassEncoderSetBlendConstantTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetBlendConstant);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUColorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUColor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->color = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUColorTransfer*>(memberBuffer),
              sizeof(WGPUColorTransfer) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetBlendConstantDeserialize);

struct RenderPassEncoderSetIndexBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId buffer;
    WGPUIndexFormat format;
    uint64_t offset;
    uint64_t size;
};

static_assert(offsetof(RenderPassEncoderSetIndexBufferTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetIndexBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetIndexBufferGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetIndexBufferCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetIndexBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetIndexBufferSerialize(
    const RenderPassEncoderSetIndexBufferCmd& record,
    RenderPassEncoderSetIndexBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetIndexBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.buffer, &transfer->buffer));
    transfer->format = record.format;
    transfer->offset = record.offset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetIndexBufferSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetIndexBufferDeserialize(
    RenderPassEncoderSetIndexBufferCmd* record,
    const volatile RenderPassEncoderSetIndexBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetIndexBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->buffer, &record->buffer));
    static_assert(sizeof(record->format) >= sizeof(transfer->format), "Deserialize assignment may not narrow.");
    record->format = transfer->format;
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetIndexBufferDeserialize);

struct RenderPassEncoderSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(RenderPassEncoderSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetLabelGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetLabelSerialize(
    const RenderPassEncoderSetLabelCmd& record,
    RenderPassEncoderSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetLabelSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetLabelDeserialize(
    RenderPassEncoderSetLabelCmd* record,
    const volatile RenderPassEncoderSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetLabelDeserialize);

struct RenderPassEncoderSetPipelineTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId pipeline;
};

static_assert(offsetof(RenderPassEncoderSetPipelineTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetPipelineTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetPipelineGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetPipelineCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetPipelineGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetPipelineSerialize(
    const RenderPassEncoderSetPipelineCmd& record,
    RenderPassEncoderSetPipelineTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetPipeline;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.pipeline, &transfer->pipeline));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetPipelineSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetPipelineDeserialize(
    RenderPassEncoderSetPipelineCmd* record,
    const volatile RenderPassEncoderSetPipelineTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetPipeline);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->pipeline, &record->pipeline));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetPipelineDeserialize);

struct RenderPassEncoderSetScissorRectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t x;
    uint32_t y;
    uint32_t width;
    uint32_t height;
};

static_assert(offsetof(RenderPassEncoderSetScissorRectTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetScissorRectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetScissorRectGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetScissorRectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetScissorRectGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetScissorRectSerialize(
    const RenderPassEncoderSetScissorRectCmd& record,
    RenderPassEncoderSetScissorRectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetScissorRect;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->x = record.x;
    transfer->y = record.y;
    transfer->width = record.width;
    transfer->height = record.height;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetScissorRectSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetScissorRectDeserialize(
    RenderPassEncoderSetScissorRectCmd* record,
    const volatile RenderPassEncoderSetScissorRectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetScissorRect);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->x) >= sizeof(transfer->x), "Deserialize assignment may not narrow.");
    record->x = transfer->x;
    static_assert(sizeof(record->y) >= sizeof(transfer->y), "Deserialize assignment may not narrow.");
    record->y = transfer->y;
    static_assert(sizeof(record->width) >= sizeof(transfer->width), "Deserialize assignment may not narrow.");
    record->width = transfer->width;
    static_assert(sizeof(record->height) >= sizeof(transfer->height), "Deserialize assignment may not narrow.");
    record->height = transfer->height;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetScissorRectDeserialize);

struct RenderPassEncoderSetStencilReferenceTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t reference;
};

static_assert(offsetof(RenderPassEncoderSetStencilReferenceTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetStencilReferenceTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetStencilReferenceGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetStencilReferenceCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetStencilReferenceGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetStencilReferenceSerialize(
    const RenderPassEncoderSetStencilReferenceCmd& record,
    RenderPassEncoderSetStencilReferenceTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetStencilReference;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->reference = record.reference;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetStencilReferenceSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetStencilReferenceDeserialize(
    RenderPassEncoderSetStencilReferenceCmd* record,
    const volatile RenderPassEncoderSetStencilReferenceTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetStencilReference);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->reference) >= sizeof(transfer->reference), "Deserialize assignment may not narrow.");
    record->reference = transfer->reference;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetStencilReferenceDeserialize);

struct RenderPassEncoderSetVertexBufferTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t slot;
    ObjectId buffer;
    uint64_t offset;
    uint64_t size;
};

static_assert(offsetof(RenderPassEncoderSetVertexBufferTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetVertexBufferTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetVertexBufferGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetVertexBufferCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetVertexBufferGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetVertexBufferSerialize(
    const RenderPassEncoderSetVertexBufferCmd& record,
    RenderPassEncoderSetVertexBufferTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetVertexBuffer;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->slot = record.slot;
    WIRE_TRY(provider.GetOptionalId(record.buffer, &transfer->buffer));
    transfer->offset = record.offset;
    transfer->size = record.size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetVertexBufferSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetVertexBufferDeserialize(
    RenderPassEncoderSetVertexBufferCmd* record,
    const volatile RenderPassEncoderSetVertexBufferTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetVertexBuffer);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->slot) >= sizeof(transfer->slot), "Deserialize assignment may not narrow.");
    record->slot = transfer->slot;
    WIRE_TRY(resolver.GetOptionalFromId(transfer->buffer, &record->buffer));
    static_assert(sizeof(record->offset) >= sizeof(transfer->offset), "Deserialize assignment may not narrow.");
    record->offset = transfer->offset;
    static_assert(sizeof(record->size) >= sizeof(transfer->size), "Deserialize assignment may not narrow.");
    record->size = transfer->size;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetVertexBufferDeserialize);

struct RenderPassEncoderSetViewportTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    float x;
    float y;
    float width;
    float height;
    float minDepth;
    float maxDepth;
};

static_assert(offsetof(RenderPassEncoderSetViewportTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderSetViewportTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderSetViewportGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderSetViewportCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderSetViewportGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderSetViewportSerialize(
    const RenderPassEncoderSetViewportCmd& record,
    RenderPassEncoderSetViewportTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderSetViewport;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->x = record.x;
    transfer->y = record.y;
    transfer->width = record.width;
    transfer->height = record.height;
    transfer->minDepth = record.minDepth;
    transfer->maxDepth = record.maxDepth;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetViewportSerialize);

[[maybe_unused]] WireResult RenderPassEncoderSetViewportDeserialize(
    RenderPassEncoderSetViewportCmd* record,
    const volatile RenderPassEncoderSetViewportTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderSetViewport);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->x) >= sizeof(transfer->x), "Deserialize assignment may not narrow.");
    record->x = transfer->x;
    static_assert(sizeof(record->y) >= sizeof(transfer->y), "Deserialize assignment may not narrow.");
    record->y = transfer->y;
    static_assert(sizeof(record->width) >= sizeof(transfer->width), "Deserialize assignment may not narrow.");
    record->width = transfer->width;
    static_assert(sizeof(record->height) >= sizeof(transfer->height), "Deserialize assignment may not narrow.");
    record->height = transfer->height;
    static_assert(sizeof(record->minDepth) >= sizeof(transfer->minDepth), "Deserialize assignment may not narrow.");
    record->minDepth = transfer->minDepth;
    static_assert(sizeof(record->maxDepth) >= sizeof(transfer->maxDepth), "Deserialize assignment may not narrow.");
    record->maxDepth = transfer->maxDepth;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderSetViewportDeserialize);

struct RenderPassEncoderWriteTimestampTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    ObjectId querySet;
    uint32_t queryIndex;
};

static_assert(offsetof(RenderPassEncoderWriteTimestampTransfer, commandSize) == 0);
static_assert(offsetof(RenderPassEncoderWriteTimestampTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPassEncoderWriteTimestampGetExtraRequiredSize([[maybe_unused]] const RenderPassEncoderWriteTimestampCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPassEncoderWriteTimestampGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPassEncoderWriteTimestampSerialize(
    const RenderPassEncoderWriteTimestampCmd& record,
    RenderPassEncoderWriteTimestampTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPassEncoderWriteTimestamp;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(provider.GetId(record.querySet, &transfer->querySet));
    transfer->queryIndex = record.queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderWriteTimestampSerialize);

[[maybe_unused]] WireResult RenderPassEncoderWriteTimestampDeserialize(
    RenderPassEncoderWriteTimestampCmd* record,
    const volatile RenderPassEncoderWriteTimestampTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPassEncoderWriteTimestamp);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(resolver.GetFromId(transfer->querySet, &record->querySet));
    static_assert(sizeof(record->queryIndex) >= sizeof(transfer->queryIndex), "Deserialize assignment may not narrow.");
    record->queryIndex = transfer->queryIndex;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPassEncoderWriteTimestampDeserialize);

struct RenderPipelineGetBindGroupLayoutTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    uint32_t groupIndex;
    ObjectHandle result;
};

static_assert(offsetof(RenderPipelineGetBindGroupLayoutTransfer, commandSize) == 0);
static_assert(offsetof(RenderPipelineGetBindGroupLayoutTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPipelineGetBindGroupLayoutGetExtraRequiredSize([[maybe_unused]] const RenderPipelineGetBindGroupLayoutCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPipelineGetBindGroupLayoutGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPipelineGetBindGroupLayoutSerialize(
    const RenderPipelineGetBindGroupLayoutCmd& record,
    RenderPipelineGetBindGroupLayoutTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPipelineGetBindGroupLayout;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->groupIndex = record.groupIndex;
    transfer->result = record.result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPipelineGetBindGroupLayoutSerialize);

[[maybe_unused]] WireResult RenderPipelineGetBindGroupLayoutDeserialize(
    RenderPipelineGetBindGroupLayoutCmd* record,
    const volatile RenderPipelineGetBindGroupLayoutTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPipelineGetBindGroupLayout);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->groupIndex) >= sizeof(transfer->groupIndex), "Deserialize assignment may not narrow.");
    record->groupIndex = transfer->groupIndex;
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPipelineGetBindGroupLayoutDeserialize);

struct RenderPipelineSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(RenderPipelineSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(RenderPipelineSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t RenderPipelineSetLabelGetExtraRequiredSize([[maybe_unused]] const RenderPipelineSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(RenderPipelineSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult RenderPipelineSetLabelSerialize(
    const RenderPipelineSetLabelCmd& record,
    RenderPipelineSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::RenderPipelineSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPipelineSetLabelSerialize);

[[maybe_unused]] WireResult RenderPipelineSetLabelDeserialize(
    RenderPipelineSetLabelCmd* record,
    const volatile RenderPipelineSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::RenderPipelineSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(RenderPipelineSetLabelDeserialize);

struct SamplerSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(SamplerSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(SamplerSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t SamplerSetLabelGetExtraRequiredSize([[maybe_unused]] const SamplerSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(SamplerSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult SamplerSetLabelSerialize(
    const SamplerSetLabelCmd& record,
    SamplerSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::SamplerSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SamplerSetLabelSerialize);

[[maybe_unused]] WireResult SamplerSetLabelDeserialize(
    SamplerSetLabelCmd* record,
    const volatile SamplerSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::SamplerSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SamplerSetLabelDeserialize);

struct ShaderModuleGetCompilationInfoTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId shaderModuleId;
    ObjectHandle eventManagerHandle;
    WGPUFutureTransfer future;
};

static_assert(offsetof(ShaderModuleGetCompilationInfoTransfer, commandSize) == 0);
static_assert(offsetof(ShaderModuleGetCompilationInfoTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ShaderModuleGetCompilationInfoGetExtraRequiredSize([[maybe_unused]] const ShaderModuleGetCompilationInfoCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ShaderModuleGetCompilationInfoGetExtraRequiredSize);

[[maybe_unused]] WireResult ShaderModuleGetCompilationInfoSerialize(
    const ShaderModuleGetCompilationInfoCmd& record,
    ShaderModuleGetCompilationInfoTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::ShaderModuleGetCompilationInfo;


    transfer->shaderModuleId = record.shaderModuleId;
    transfer->eventManagerHandle = record.eventManagerHandle;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ShaderModuleGetCompilationInfoSerialize);

[[maybe_unused]] WireResult ShaderModuleGetCompilationInfoDeserialize(
    ShaderModuleGetCompilationInfoCmd* record,
    const volatile ShaderModuleGetCompilationInfoTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ShaderModuleGetCompilationInfo);


    static_assert(sizeof(record->shaderModuleId) >= sizeof(transfer->shaderModuleId), "Deserialize assignment may not narrow.");
    record->shaderModuleId = transfer->shaderModuleId;
    static_assert(sizeof(record->eventManagerHandle) >= sizeof(transfer->eventManagerHandle), "Deserialize assignment may not narrow.");
    record->eventManagerHandle = transfer->eventManagerHandle;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ShaderModuleGetCompilationInfoDeserialize);

struct ShaderModuleSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(ShaderModuleSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(ShaderModuleSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ShaderModuleSetLabelGetExtraRequiredSize([[maybe_unused]] const ShaderModuleSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ShaderModuleSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult ShaderModuleSetLabelSerialize(
    const ShaderModuleSetLabelCmd& record,
    ShaderModuleSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::ShaderModuleSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ShaderModuleSetLabelSerialize);

[[maybe_unused]] WireResult ShaderModuleSetLabelDeserialize(
    ShaderModuleSetLabelCmd* record,
    const volatile ShaderModuleSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::ShaderModuleSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ShaderModuleSetLabelDeserialize);

struct SurfaceConfigureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(SurfaceConfigureTransfer, commandSize) == 0);
static_assert(offsetof(SurfaceConfigureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t SurfaceConfigureGetExtraRequiredSize([[maybe_unused]] const SurfaceConfigureCmd& record) {
    size_t result = 0;

    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUSurfaceConfigurationTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUSurfaceConfigurationGetExtraRequiredSize(record.config[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(SurfaceConfigureGetExtraRequiredSize);

[[maybe_unused]] WireResult SurfaceConfigureSerialize(
    const SurfaceConfigureCmd& record,
    SurfaceConfigureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::SurfaceConfigure;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    {
        auto memberLength = 1u;

        WGPUSurfaceConfigurationTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSurfaceConfigurationSerialize(record.config[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceConfigureSerialize);

[[maybe_unused]] WireResult SurfaceConfigureDeserialize(
    SurfaceConfigureCmd* record,
    const volatile SurfaceConfigureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::SurfaceConfigure);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    {
        auto memberLength = 1u;
        const volatile WGPUSurfaceConfigurationTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUSurfaceConfiguration* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->config = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSurfaceConfigurationDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceConfigureDeserialize);

struct SurfaceGetCurrentTextureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId surfaceId;
    ObjectId configuredDeviceId;
    ObjectHandle textureHandle;
};

static_assert(offsetof(SurfaceGetCurrentTextureTransfer, commandSize) == 0);
static_assert(offsetof(SurfaceGetCurrentTextureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t SurfaceGetCurrentTextureGetExtraRequiredSize([[maybe_unused]] const SurfaceGetCurrentTextureCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(SurfaceGetCurrentTextureGetExtraRequiredSize);

[[maybe_unused]] WireResult SurfaceGetCurrentTextureSerialize(
    const SurfaceGetCurrentTextureCmd& record,
    SurfaceGetCurrentTextureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::SurfaceGetCurrentTexture;


    transfer->surfaceId = record.surfaceId;
    transfer->configuredDeviceId = record.configuredDeviceId;
    transfer->textureHandle = record.textureHandle;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceGetCurrentTextureSerialize);

[[maybe_unused]] WireResult SurfaceGetCurrentTextureDeserialize(
    SurfaceGetCurrentTextureCmd* record,
    const volatile SurfaceGetCurrentTextureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::SurfaceGetCurrentTexture);


    static_assert(sizeof(record->surfaceId) >= sizeof(transfer->surfaceId), "Deserialize assignment may not narrow.");
    record->surfaceId = transfer->surfaceId;
    static_assert(sizeof(record->configuredDeviceId) >= sizeof(transfer->configuredDeviceId), "Deserialize assignment may not narrow.");
    record->configuredDeviceId = transfer->configuredDeviceId;
    static_assert(sizeof(record->textureHandle) >= sizeof(transfer->textureHandle), "Deserialize assignment may not narrow.");
    record->textureHandle = transfer->textureHandle;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceGetCurrentTextureDeserialize);

struct SurfacePresentTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(SurfacePresentTransfer, commandSize) == 0);
static_assert(offsetof(SurfacePresentTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t SurfacePresentGetExtraRequiredSize([[maybe_unused]] const SurfacePresentCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(SurfacePresentGetExtraRequiredSize);

[[maybe_unused]] WireResult SurfacePresentSerialize(
    const SurfacePresentCmd& record,
    SurfacePresentTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::SurfacePresent;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfacePresentSerialize);

[[maybe_unused]] WireResult SurfacePresentDeserialize(
    SurfacePresentCmd* record,
    const volatile SurfacePresentTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::SurfacePresent);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfacePresentDeserialize);

struct SurfaceSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(SurfaceSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(SurfaceSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t SurfaceSetLabelGetExtraRequiredSize([[maybe_unused]] const SurfaceSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(SurfaceSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult SurfaceSetLabelSerialize(
    const SurfaceSetLabelCmd& record,
    SurfaceSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::SurfaceSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceSetLabelSerialize);

[[maybe_unused]] WireResult SurfaceSetLabelDeserialize(
    SurfaceSetLabelCmd* record,
    const volatile SurfaceSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::SurfaceSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceSetLabelDeserialize);

struct SurfaceUnconfigureTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(SurfaceUnconfigureTransfer, commandSize) == 0);
static_assert(offsetof(SurfaceUnconfigureTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t SurfaceUnconfigureGetExtraRequiredSize([[maybe_unused]] const SurfaceUnconfigureCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(SurfaceUnconfigureGetExtraRequiredSize);

[[maybe_unused]] WireResult SurfaceUnconfigureSerialize(
    const SurfaceUnconfigureCmd& record,
    SurfaceUnconfigureTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::SurfaceUnconfigure;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceUnconfigureSerialize);

[[maybe_unused]] WireResult SurfaceUnconfigureDeserialize(
    SurfaceUnconfigureCmd* record,
    const volatile SurfaceUnconfigureTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::SurfaceUnconfigure);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(SurfaceUnconfigureDeserialize);

struct TextureCreateErrorViewTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUBool has_descriptor;
    ObjectHandle result;
};

static_assert(offsetof(TextureCreateErrorViewTransfer, commandSize) == 0);
static_assert(offsetof(TextureCreateErrorViewTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t TextureCreateErrorViewGetExtraRequiredSize([[maybe_unused]] const TextureCreateErrorViewCmd& record) {
    size_t result = 0;

    if (record.descriptor != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUTextureViewDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUTextureViewDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(TextureCreateErrorViewGetExtraRequiredSize);

[[maybe_unused]] WireResult TextureCreateErrorViewSerialize(
    const TextureCreateErrorViewCmd& record,
    TextureCreateErrorViewTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::TextureCreateErrorView;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    bool has_descriptor = record.descriptor != nullptr;
    transfer->has_descriptor = has_descriptor;
    if (has_descriptor) {
        auto memberLength = 1u;

        WGPUTextureViewDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureViewDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureCreateErrorViewSerialize);

[[maybe_unused]] WireResult TextureCreateErrorViewDeserialize(
    TextureCreateErrorViewCmd* record,
    const volatile TextureCreateErrorViewTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::TextureCreateErrorView);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    bool has_descriptor = transfer->has_descriptor;
    record->descriptor = nullptr;
    if (has_descriptor) {
        auto memberLength = 1u;
        const volatile WGPUTextureViewDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureViewDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureViewDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureCreateErrorViewDeserialize);

struct TextureCreateViewTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUBool has_descriptor;
    ObjectHandle result;
};

static_assert(offsetof(TextureCreateViewTransfer, commandSize) == 0);
static_assert(offsetof(TextureCreateViewTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t TextureCreateViewGetExtraRequiredSize([[maybe_unused]] const TextureCreateViewCmd& record) {
    size_t result = 0;

    if (record.descriptor != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUTextureViewDescriptorTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUTextureViewDescriptorGetExtraRequiredSize(record.descriptor[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(TextureCreateViewGetExtraRequiredSize);

[[maybe_unused]] WireResult TextureCreateViewSerialize(
    const TextureCreateViewCmd& record,
    TextureCreateViewTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::TextureCreateView;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    transfer->result = record.result;
    bool has_descriptor = record.descriptor != nullptr;
    transfer->has_descriptor = has_descriptor;
    if (has_descriptor) {
        auto memberLength = 1u;

        WGPUTextureViewDescriptorTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureViewDescriptorSerialize(record.descriptor[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureCreateViewSerialize);

[[maybe_unused]] WireResult TextureCreateViewDeserialize(
    TextureCreateViewCmd* record,
    const volatile TextureCreateViewTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::TextureCreateView);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    static_assert(sizeof(record->result) >= sizeof(transfer->result), "Deserialize assignment may not narrow.");
    record->result = transfer->result;
    bool has_descriptor = transfer->has_descriptor;
    record->descriptor = nullptr;
    if (has_descriptor) {
        auto memberLength = 1u;
        const volatile WGPUTextureViewDescriptorTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUTextureViewDescriptor* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->descriptor = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUTextureViewDescriptorDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureCreateViewDeserialize);

struct TextureDestroyTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
};

static_assert(offsetof(TextureDestroyTransfer, commandSize) == 0);
static_assert(offsetof(TextureDestroyTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t TextureDestroyGetExtraRequiredSize([[maybe_unused]] const TextureDestroyCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(TextureDestroyGetExtraRequiredSize);

[[maybe_unused]] WireResult TextureDestroySerialize(
    const TextureDestroyCmd& record,
    TextureDestroyTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::TextureDestroy;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureDestroySerialize);

[[maybe_unused]] WireResult TextureDestroyDeserialize(
    TextureDestroyCmd* record,
    const volatile TextureDestroyTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::TextureDestroy);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureDestroyDeserialize);

struct TextureSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(TextureSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(TextureSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t TextureSetLabelGetExtraRequiredSize([[maybe_unused]] const TextureSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(TextureSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult TextureSetLabelSerialize(
    const TextureSetLabelCmd& record,
    TextureSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::TextureSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureSetLabelSerialize);

[[maybe_unused]] WireResult TextureSetLabelDeserialize(
    TextureSetLabelCmd* record,
    const volatile TextureSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::TextureSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureSetLabelDeserialize);

struct TextureViewSetLabelTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectId self;
    WGPUStringViewTransfer label;
};

static_assert(offsetof(TextureViewSetLabelTransfer, commandSize) == 0);
static_assert(offsetof(TextureViewSetLabelTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t TextureViewSetLabelGetExtraRequiredSize([[maybe_unused]] const TextureViewSetLabelCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.label);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(TextureViewSetLabelGetExtraRequiredSize);

[[maybe_unused]] WireResult TextureViewSetLabelSerialize(
    const TextureViewSetLabelCmd& record,
    TextureViewSetLabelTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = WireCmd::TextureViewSetLabel;


    WIRE_TRY(provider.GetId(record.self, &transfer->self));
    WIRE_TRY(WGPUStringViewSerialize(record.label, &transfer->label, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureViewSetLabelSerialize);

[[maybe_unused]] WireResult TextureViewSetLabelDeserialize(
    TextureViewSetLabelCmd* record,
    const volatile TextureViewSetLabelTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == WireCmd::TextureViewSetLabel);
    record->selfId = transfer->self;


    WIRE_TRY(resolver.GetFromId(transfer->self, &record->self));
    WIRE_TRY(WGPUStringViewDeserialize(&record->label, &transfer->label, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(TextureViewSetLabelDeserialize);

struct UnregisterObjectTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    WireCmd commandId;

    ObjectType objectType;
    ObjectId objectId;
};

static_assert(offsetof(UnregisterObjectTransfer, commandSize) == 0);
static_assert(offsetof(UnregisterObjectTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t UnregisterObjectGetExtraRequiredSize([[maybe_unused]] const UnregisterObjectCmd& record) {
    size_t result = 0;

    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(UnregisterObjectGetExtraRequiredSize);

[[maybe_unused]] WireResult UnregisterObjectSerialize(
    const UnregisterObjectCmd& record,
    UnregisterObjectTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = WireCmd::UnregisterObject;


    transfer->objectType = record.objectType;
    transfer->objectId = record.objectId;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(UnregisterObjectSerialize);

[[maybe_unused]] WireResult UnregisterObjectDeserialize(
    UnregisterObjectCmd* record,
    const volatile UnregisterObjectTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == WireCmd::UnregisterObject);


    static_assert(sizeof(record->objectType) >= sizeof(transfer->objectType), "Deserialize assignment may not narrow.");
    record->objectType = transfer->objectType;
    static_assert(sizeof(record->objectId) >= sizeof(transfer->objectId), "Deserialize assignment may not narrow.");
    record->objectId = transfer->objectId;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(UnregisterObjectDeserialize);


struct ReturnAdapterRequestDeviceCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPURequestDeviceStatus status;
    WGPUStringViewTransfer message;
    WGPUBool has_limits;
    uint32_t featuresCount;
};

static_assert(offsetof(ReturnAdapterRequestDeviceCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnAdapterRequestDeviceCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnAdapterRequestDeviceCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnAdapterRequestDeviceCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    if (record.limits != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUSupportedLimitsTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUSupportedLimitsGetExtraRequiredSize(record.limits[i]);
        }
    }
    {
        auto memberLength = record.featuresCount;
        auto size = WireAlignSizeofN<WGPUFeatureName>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnAdapterRequestDeviceCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnAdapterRequestDeviceCallbackSerialize(
    const ReturnAdapterRequestDeviceCallbackCmd& record,
    ReturnAdapterRequestDeviceCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = ReturnWireCmd::AdapterRequestDeviceCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));
    transfer->featuresCount = record.featuresCount;
    bool has_limits = record.limits != nullptr;
    transfer->has_limits = has_limits;
    if (has_limits) {
        auto memberLength = 1u;

        WGPUSupportedLimitsTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSupportedLimitsSerialize(record.limits[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = record.featuresCount;

        WGPUFeatureName* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.features,
                sizeof(WGPUFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnAdapterRequestDeviceCallbackSerialize);

[[maybe_unused]] WireResult ReturnAdapterRequestDeviceCallbackDeserialize(
    ReturnAdapterRequestDeviceCallbackCmd* record,
    const volatile ReturnAdapterRequestDeviceCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::AdapterRequestDeviceCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));
    static_assert(sizeof(record->featuresCount) >= sizeof(transfer->featuresCount), "Deserialize assignment may not narrow.");
    record->featuresCount = transfer->featuresCount;
    bool has_limits = transfer->has_limits;
    record->limits = nullptr;
    if (has_limits) {
        auto memberLength = 1u;
        const volatile WGPUSupportedLimitsTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUSupportedLimits* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->limits = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSupportedLimitsDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = record->featuresCount;
        const volatile WGPUFeatureName* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUFeatureName* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->features = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUFeatureName*>(memberBuffer),
              sizeof(WGPUFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnAdapterRequestDeviceCallbackDeserialize);

struct ReturnBufferMapAsyncCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPUMapAsyncStatus status;
    WGPUStringViewTransfer message;
    uint64_t readDataUpdateInfoLength;
};

static_assert(offsetof(ReturnBufferMapAsyncCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnBufferMapAsyncCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnBufferMapAsyncCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnBufferMapAsyncCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnBufferMapAsyncCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnBufferMapAsyncCallbackSerialize(
    const ReturnBufferMapAsyncCallbackCmd& record,
    ReturnBufferMapAsyncCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::BufferMapAsyncCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));
    transfer->readDataUpdateInfoLength = record.readDataUpdateInfoLength;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnBufferMapAsyncCallbackSerialize);

[[maybe_unused]] WireResult ReturnBufferMapAsyncCallbackDeserialize(
    ReturnBufferMapAsyncCallbackCmd* record,
    const volatile ReturnBufferMapAsyncCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::BufferMapAsyncCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));
    static_assert(sizeof(record->readDataUpdateInfoLength) >= sizeof(transfer->readDataUpdateInfoLength), "Deserialize assignment may not narrow.");
    record->readDataUpdateInfoLength = transfer->readDataUpdateInfoLength;
    {
        auto memberLength = record->readDataUpdateInfoLength;
        const volatile uint8_t* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        uint8_t* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->readDataUpdateInfo = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const uint8_t*>(memberBuffer),
              sizeof(uint8_t) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnBufferMapAsyncCallbackDeserialize);

struct ReturnDeviceCreateComputePipelineAsyncCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPUCreatePipelineAsyncStatus status;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(ReturnDeviceCreateComputePipelineAsyncCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnDeviceCreateComputePipelineAsyncCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnDeviceCreateComputePipelineAsyncCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnDeviceCreateComputePipelineAsyncCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnDeviceCreateComputePipelineAsyncCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnDeviceCreateComputePipelineAsyncCallbackSerialize(
    const ReturnDeviceCreateComputePipelineAsyncCallbackCmd& record,
    ReturnDeviceCreateComputePipelineAsyncCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::DeviceCreateComputePipelineAsyncCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceCreateComputePipelineAsyncCallbackSerialize);

[[maybe_unused]] WireResult ReturnDeviceCreateComputePipelineAsyncCallbackDeserialize(
    ReturnDeviceCreateComputePipelineAsyncCallbackCmd* record,
    const volatile ReturnDeviceCreateComputePipelineAsyncCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::DeviceCreateComputePipelineAsyncCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceCreateComputePipelineAsyncCallbackDeserialize);

struct ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPUCreatePipelineAsyncStatus status;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnDeviceCreateRenderPipelineAsyncCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnDeviceCreateRenderPipelineAsyncCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnDeviceCreateRenderPipelineAsyncCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnDeviceCreateRenderPipelineAsyncCallbackSerialize(
    const ReturnDeviceCreateRenderPipelineAsyncCallbackCmd& record,
    ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::DeviceCreateRenderPipelineAsyncCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceCreateRenderPipelineAsyncCallbackSerialize);

[[maybe_unused]] WireResult ReturnDeviceCreateRenderPipelineAsyncCallbackDeserialize(
    ReturnDeviceCreateRenderPipelineAsyncCallbackCmd* record,
    const volatile ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::DeviceCreateRenderPipelineAsyncCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceCreateRenderPipelineAsyncCallbackDeserialize);

struct ReturnDeviceLoggingCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle device;
    WGPULoggingType type;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(ReturnDeviceLoggingCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnDeviceLoggingCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnDeviceLoggingCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnDeviceLoggingCallbackCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnDeviceLoggingCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnDeviceLoggingCallbackSerialize(
    const ReturnDeviceLoggingCallbackCmd& record,
    ReturnDeviceLoggingCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::DeviceLoggingCallback;


    transfer->device = record.device;
    transfer->type = record.type;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceLoggingCallbackSerialize);

[[maybe_unused]] WireResult ReturnDeviceLoggingCallbackDeserialize(
    ReturnDeviceLoggingCallbackCmd* record,
    const volatile ReturnDeviceLoggingCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::DeviceLoggingCallback);


    static_assert(sizeof(record->device) >= sizeof(transfer->device), "Deserialize assignment may not narrow.");
    record->device = transfer->device;
    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceLoggingCallbackDeserialize);

struct ReturnDeviceLostCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPUDeviceLostReason reason;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(ReturnDeviceLostCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnDeviceLostCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnDeviceLostCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnDeviceLostCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnDeviceLostCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnDeviceLostCallbackSerialize(
    const ReturnDeviceLostCallbackCmd& record,
    ReturnDeviceLostCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::DeviceLostCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->reason = record.reason;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceLostCallbackSerialize);

[[maybe_unused]] WireResult ReturnDeviceLostCallbackDeserialize(
    ReturnDeviceLostCallbackCmd* record,
    const volatile ReturnDeviceLostCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::DeviceLostCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->reason) >= sizeof(transfer->reason), "Deserialize assignment may not narrow.");
    record->reason = transfer->reason;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceLostCallbackDeserialize);

struct ReturnDevicePopErrorScopeCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPUPopErrorScopeStatus status;
    WGPUErrorType type;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(ReturnDevicePopErrorScopeCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnDevicePopErrorScopeCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnDevicePopErrorScopeCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnDevicePopErrorScopeCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnDevicePopErrorScopeCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnDevicePopErrorScopeCallbackSerialize(
    const ReturnDevicePopErrorScopeCallbackCmd& record,
    ReturnDevicePopErrorScopeCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::DevicePopErrorScopeCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;
    transfer->type = record.type;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDevicePopErrorScopeCallbackSerialize);

[[maybe_unused]] WireResult ReturnDevicePopErrorScopeCallbackDeserialize(
    ReturnDevicePopErrorScopeCallbackCmd* record,
    const volatile ReturnDevicePopErrorScopeCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::DevicePopErrorScopeCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;
    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDevicePopErrorScopeCallbackDeserialize);

struct ReturnDeviceUncapturedErrorCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle device;
    WGPUErrorType type;
    WGPUStringViewTransfer message;
};

static_assert(offsetof(ReturnDeviceUncapturedErrorCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnDeviceUncapturedErrorCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnDeviceUncapturedErrorCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnDeviceUncapturedErrorCallbackCmd& record) {
    size_t result = 0;

    result += WGPUStringViewGetExtraRequiredSize(record.message);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnDeviceUncapturedErrorCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnDeviceUncapturedErrorCallbackSerialize(
    const ReturnDeviceUncapturedErrorCallbackCmd& record,
    ReturnDeviceUncapturedErrorCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::DeviceUncapturedErrorCallback;


    transfer->device = record.device;
    transfer->type = record.type;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceUncapturedErrorCallbackSerialize);

[[maybe_unused]] WireResult ReturnDeviceUncapturedErrorCallbackDeserialize(
    ReturnDeviceUncapturedErrorCallbackCmd* record,
    const volatile ReturnDeviceUncapturedErrorCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::DeviceUncapturedErrorCallback);


    static_assert(sizeof(record->device) >= sizeof(transfer->device), "Deserialize assignment may not narrow.");
    record->device = transfer->device;
    static_assert(sizeof(record->type) >= sizeof(transfer->type), "Deserialize assignment may not narrow.");
    record->type = transfer->type;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnDeviceUncapturedErrorCallbackDeserialize);

struct ReturnInstanceRequestAdapterCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPURequestAdapterStatus status;
    WGPUStringViewTransfer message;
    WGPUBool has_info;
    WGPUBool has_limits;
    uint32_t featuresCount;
};

static_assert(offsetof(ReturnInstanceRequestAdapterCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnInstanceRequestAdapterCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnInstanceRequestAdapterCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnInstanceRequestAdapterCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    result += WGPUStringViewGetExtraRequiredSize(record.message);
    if (record.info != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUAdapterInfoTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUAdapterInfoGetExtraRequiredSize(record.info[i]);
        }
    }
    if (record.limits != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUSupportedLimitsTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUSupportedLimitsGetExtraRequiredSize(record.limits[i]);
        }
    }
    {
        auto memberLength = record.featuresCount;
        auto size = WireAlignSizeofN<WGPUFeatureName>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnInstanceRequestAdapterCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnInstanceRequestAdapterCallbackSerialize(
    const ReturnInstanceRequestAdapterCallbackCmd& record,
    ReturnInstanceRequestAdapterCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = ReturnWireCmd::InstanceRequestAdapterCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;
    WIRE_TRY(WGPUStringViewSerialize(record.message, &transfer->message, buffer));
    transfer->featuresCount = record.featuresCount;
    bool has_info = record.info != nullptr;
    transfer->has_info = has_info;
    if (has_info) {
        auto memberLength = 1u;

        WGPUAdapterInfoTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUAdapterInfoSerialize(record.info[i], &memberBuffer[i], buffer, provider));
        }
    }
    bool has_limits = record.limits != nullptr;
    transfer->has_limits = has_limits;
    if (has_limits) {
        auto memberLength = 1u;

        WGPUSupportedLimitsTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSupportedLimitsSerialize(record.limits[i], &memberBuffer[i], buffer, provider));
        }
    }
    {
        auto memberLength = record.featuresCount;

        WGPUFeatureName* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        if (memberLength != 0) {
            memcpy(
                memberBuffer, record.features,
                sizeof(WGPUFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnInstanceRequestAdapterCallbackSerialize);

[[maybe_unused]] WireResult ReturnInstanceRequestAdapterCallbackDeserialize(
    ReturnInstanceRequestAdapterCallbackCmd* record,
    const volatile ReturnInstanceRequestAdapterCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::InstanceRequestAdapterCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;
    WIRE_TRY(WGPUStringViewDeserialize(&record->message, &transfer->message, deserializeBuffer, allocator));
    static_assert(sizeof(record->featuresCount) >= sizeof(transfer->featuresCount), "Deserialize assignment may not narrow.");
    record->featuresCount = transfer->featuresCount;
    bool has_info = transfer->has_info;
    record->info = nullptr;
    if (has_info) {
        auto memberLength = 1u;
        const volatile WGPUAdapterInfoTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUAdapterInfo* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->info = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUAdapterInfoDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    bool has_limits = transfer->has_limits;
    record->limits = nullptr;
    if (has_limits) {
        auto memberLength = 1u;
        const volatile WGPUSupportedLimitsTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUSupportedLimits* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->limits = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUSupportedLimitsDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }
    {
        auto memberLength = record->featuresCount;
        const volatile WGPUFeatureName* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUFeatureName* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->features = copiedMembers;

        if (memberLength != 0) {
            memcpy(
                copiedMembers,
                const_cast<const WGPUFeatureName*>(memberBuffer),
              sizeof(WGPUFeatureName) * memberLength);
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnInstanceRequestAdapterCallbackDeserialize);

struct ReturnQueueWorkDoneCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPUQueueWorkDoneStatus status;
};

static_assert(offsetof(ReturnQueueWorkDoneCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnQueueWorkDoneCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnQueueWorkDoneCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnQueueWorkDoneCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnQueueWorkDoneCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnQueueWorkDoneCallbackSerialize(
    const ReturnQueueWorkDoneCallbackCmd& record,
    ReturnQueueWorkDoneCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer) {
    transfer->commandId = ReturnWireCmd::QueueWorkDoneCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnQueueWorkDoneCallbackSerialize);

[[maybe_unused]] WireResult ReturnQueueWorkDoneCallbackDeserialize(
    ReturnQueueWorkDoneCallbackCmd* record,
    const volatile ReturnQueueWorkDoneCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::QueueWorkDoneCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnQueueWorkDoneCallbackDeserialize);

struct ReturnShaderModuleGetCompilationInfoCallbackTransfer : CmdHeader {
    static_assert(1 <= 1,
                  "Record must be at most one of is_cmd, extensible, and chained.");
    ReturnWireCmd commandId;

    ObjectHandle eventManager;
    WGPUFutureTransfer future;
    WGPUCompilationInfoRequestStatus status;
    WGPUBool has_info;
};

static_assert(offsetof(ReturnShaderModuleGetCompilationInfoCallbackTransfer, commandSize) == 0);
static_assert(offsetof(ReturnShaderModuleGetCompilationInfoCallbackTransfer, commandId) == sizeof(CmdHeader));

[[maybe_unused]] size_t ReturnShaderModuleGetCompilationInfoCallbackGetExtraRequiredSize([[maybe_unused]] const ReturnShaderModuleGetCompilationInfoCallbackCmd& record) {
    size_t result = 0;

    result += WGPUFutureGetExtraRequiredSize(record.future);
    if (record.info != nullptr)
    {
        auto memberLength = 1u;
        auto size = WireAlignSizeofN<WGPUCompilationInfoTransfer>(memberLength);
        DAWN_ASSERT(size);
        result += *size;
        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            result += WGPUCompilationInfoGetExtraRequiredSize(record.info[i]);
        }
    }
    return result;
}
// GetExtraRequiredSize isn't used for structures that are value members of other structures
// because we assume they cannot contain pointers themselves.
DAWN_UNUSED_FUNC(ReturnShaderModuleGetCompilationInfoCallbackGetExtraRequiredSize);

[[maybe_unused]] WireResult ReturnShaderModuleGetCompilationInfoCallbackSerialize(
    const ReturnShaderModuleGetCompilationInfoCallbackCmd& record,
    ReturnShaderModuleGetCompilationInfoCallbackTransfer* transfer,
    [[maybe_unused]] SerializeBuffer* buffer, const ObjectIdProvider& provider) {
    transfer->commandId = ReturnWireCmd::ShaderModuleGetCompilationInfoCallback;


    transfer->eventManager = record.eventManager;
    WIRE_TRY(WGPUFutureSerialize(record.future, &transfer->future, buffer));
    transfer->status = record.status;
    bool has_info = record.info != nullptr;
    transfer->has_info = has_info;
    if (has_info) {
        auto memberLength = 1u;

        WGPUCompilationInfoTransfer* memberBuffer;
        WIRE_TRY(buffer->NextN(memberLength, &memberBuffer));

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCompilationInfoSerialize(record.info[i], &memberBuffer[i], buffer, provider));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnShaderModuleGetCompilationInfoCallbackSerialize);

[[maybe_unused]] WireResult ReturnShaderModuleGetCompilationInfoCallbackDeserialize(
    ReturnShaderModuleGetCompilationInfoCallbackCmd* record,
    const volatile ReturnShaderModuleGetCompilationInfoCallbackTransfer* transfer,
    DeserializeBuffer* deserializeBuffer,
    [[maybe_unused]] DeserializeAllocator* allocator, const ObjectIdResolver& resolver) {
    DAWN_ASSERT(transfer->commandId == ReturnWireCmd::ShaderModuleGetCompilationInfoCallback);


    static_assert(sizeof(record->eventManager) >= sizeof(transfer->eventManager), "Deserialize assignment may not narrow.");
    record->eventManager = transfer->eventManager;
    static_assert(sizeof(record->future) == sizeof(transfer->future), "Deserialize memcpy size must match.");
    memcpy(&record->future, const_cast<const WGPUFutureTransfer*>(&transfer->future), sizeof(WGPUFutureTransfer));
    static_assert(sizeof(record->status) >= sizeof(transfer->status), "Deserialize assignment may not narrow.");
    record->status = transfer->status;
    bool has_info = transfer->has_info;
    record->info = nullptr;
    if (has_info) {
        auto memberLength = 1u;
        const volatile WGPUCompilationInfoTransfer* memberBuffer;
        WIRE_TRY(deserializeBuffer->ReadN(memberLength, &memberBuffer));

        WGPUCompilationInfo* copiedMembers;
        WIRE_TRY(GetSpace(allocator, memberLength, &copiedMembers));
        record->info = copiedMembers;

        for (decltype(memberLength) i = 0; i < memberLength; ++i) {
            WIRE_TRY(WGPUCompilationInfoDeserialize(&copiedMembers[i], &memberBuffer[i], deserializeBuffer, allocator, resolver));
        }
    }

    return WireResult::Success;
}
DAWN_UNUSED_FUNC(ReturnShaderModuleGetCompilationInfoCallbackDeserialize);

// Implementation of ObjectIdResolver that always errors.
// Used when the generator adds a provider argument because of a chained
// struct, but in practice, a chained struct in that location is invalid.
class ErrorObjectIdResolver final : public ObjectIdResolver {
    public:
  WireResult GetFromId(ObjectId id, WGPUAdapter* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUAdapter* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUBindGroup* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUBindGroup* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUBindGroupLayout* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUBindGroupLayout* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUBuffer* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUBuffer* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUCommandBuffer* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUCommandBuffer* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUCommandEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUCommandEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUComputePassEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUComputePassEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUComputePipeline* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUComputePipeline* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUDevice* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUDevice* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUExternalTexture* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUExternalTexture* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUInstance* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUInstance* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUPipelineLayout* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUPipelineLayout* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUQuerySet* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUQuerySet* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUQueue* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUQueue* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPURenderBundle* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPURenderBundle* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPURenderBundleEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPURenderBundleEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPURenderPassEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPURenderPassEncoder* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPURenderPipeline* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPURenderPipeline* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUSampler* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUSampler* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUShaderModule* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUShaderModule* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUSurface* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUSurface* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUTexture* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUTexture* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetFromId(ObjectId id, WGPUTextureView* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalFromId(ObjectId id, WGPUTextureView* out) const override {
      return WireResult::FatalError;
  }
};

// Implementation of ObjectIdProvider that always errors.
// Used when the generator adds a provider argument because of a chained
// struct, but in practice, a chained struct in that location is invalid.
class ErrorObjectIdProvider final : public ObjectIdProvider {
    public:
  WireResult GetId(WGPUAdapter object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUAdapter object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUBindGroup object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUBindGroup object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUBindGroupLayout object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUBindGroupLayout object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUBuffer object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUBuffer object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUCommandBuffer object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUCommandBuffer object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUCommandEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUCommandEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUComputePassEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUComputePassEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUComputePipeline object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUComputePipeline object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUDevice object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUDevice object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUExternalTexture object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUExternalTexture object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUInstance object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUInstance object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUPipelineLayout object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUPipelineLayout object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUQuerySet object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUQuerySet object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUQueue object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUQueue object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPURenderBundle object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPURenderBundle object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPURenderBundleEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPURenderBundleEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPURenderPassEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPURenderPassEncoder object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPURenderPipeline object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPURenderPipeline object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUSampler object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUSampler object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUShaderModule object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUShaderModule object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUSurface object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUSurface object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUTexture object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUTexture object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetId(WGPUTextureView object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
  WireResult GetOptionalId(WGPUTextureView object, ObjectId* out) const override {
      return WireResult::FatalError;
  }
};

}  // anonymous namespace

size_t AdapterRequestDeviceCmd::GetRequiredSize() const {
    return WireAlignSizeof<AdapterRequestDeviceTransfer>() + AdapterRequestDeviceGetExtraRequiredSize(*this);
}

WireResult AdapterRequestDeviceCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    AdapterRequestDeviceTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (AdapterRequestDeviceSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult AdapterRequestDeviceCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult AdapterRequestDeviceCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile AdapterRequestDeviceTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return AdapterRequestDeviceDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult AdapterRequestDeviceCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t BindGroupLayoutSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<BindGroupLayoutSetLabelTransfer>() + BindGroupLayoutSetLabelGetExtraRequiredSize(*this);
}

WireResult BindGroupLayoutSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    BindGroupLayoutSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (BindGroupLayoutSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult BindGroupLayoutSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult BindGroupLayoutSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile BindGroupLayoutSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return BindGroupLayoutSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult BindGroupLayoutSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t BindGroupSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<BindGroupSetLabelTransfer>() + BindGroupSetLabelGetExtraRequiredSize(*this);
}

WireResult BindGroupSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    BindGroupSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (BindGroupSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult BindGroupSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult BindGroupSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile BindGroupSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return BindGroupSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult BindGroupSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t BufferDestroyCmd::GetRequiredSize() const {
    return WireAlignSizeof<BufferDestroyTransfer>() + BufferDestroyGetExtraRequiredSize(*this);
}

WireResult BufferDestroyCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    BufferDestroyTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (BufferDestroySerialize(*this, transfer, serializeBuffer, provider));
}
WireResult BufferDestroyCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult BufferDestroyCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile BufferDestroyTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return BufferDestroyDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult BufferDestroyCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t BufferMapAsyncCmd::GetRequiredSize() const {
    return WireAlignSizeof<BufferMapAsyncTransfer>() + BufferMapAsyncGetExtraRequiredSize(*this);
}

WireResult BufferMapAsyncCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    BufferMapAsyncTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (BufferMapAsyncSerialize(*this, transfer, serializeBuffer));
}
WireResult BufferMapAsyncCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult BufferMapAsyncCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile BufferMapAsyncTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return BufferMapAsyncDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult BufferMapAsyncCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t BufferSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<BufferSetLabelTransfer>() + BufferSetLabelGetExtraRequiredSize(*this);
}

WireResult BufferSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    BufferSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (BufferSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult BufferSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult BufferSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile BufferSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return BufferSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult BufferSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t BufferUnmapCmd::GetRequiredSize() const {
    return WireAlignSizeof<BufferUnmapTransfer>() + BufferUnmapGetExtraRequiredSize(*this);
}

WireResult BufferUnmapCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    BufferUnmapTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (BufferUnmapSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult BufferUnmapCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult BufferUnmapCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile BufferUnmapTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return BufferUnmapDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult BufferUnmapCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t BufferUpdateMappedDataCmd::GetRequiredSize() const {
    return WireAlignSizeof<BufferUpdateMappedDataTransfer>() + BufferUpdateMappedDataGetExtraRequiredSize(*this);
}

WireResult BufferUpdateMappedDataCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    BufferUpdateMappedDataTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (BufferUpdateMappedDataSerialize(*this, transfer, serializeBuffer));
}
WireResult BufferUpdateMappedDataCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult BufferUpdateMappedDataCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile BufferUpdateMappedDataTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return BufferUpdateMappedDataDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult BufferUpdateMappedDataCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t CommandBufferSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandBufferSetLabelTransfer>() + CommandBufferSetLabelGetExtraRequiredSize(*this);
}

WireResult CommandBufferSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandBufferSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandBufferSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandBufferSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandBufferSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandBufferSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandBufferSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandBufferSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderBeginComputePassCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderBeginComputePassTransfer>() + CommandEncoderBeginComputePassGetExtraRequiredSize(*this);
}

WireResult CommandEncoderBeginComputePassCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderBeginComputePassTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderBeginComputePassSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderBeginComputePassCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderBeginComputePassCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderBeginComputePassTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderBeginComputePassDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderBeginComputePassCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderBeginRenderPassCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderBeginRenderPassTransfer>() + CommandEncoderBeginRenderPassGetExtraRequiredSize(*this);
}

WireResult CommandEncoderBeginRenderPassCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderBeginRenderPassTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderBeginRenderPassSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderBeginRenderPassCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderBeginRenderPassCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderBeginRenderPassTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderBeginRenderPassDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderBeginRenderPassCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderClearBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderClearBufferTransfer>() + CommandEncoderClearBufferGetExtraRequiredSize(*this);
}

WireResult CommandEncoderClearBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderClearBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderClearBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderClearBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderClearBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderClearBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderClearBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderClearBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderCopyBufferToBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderCopyBufferToBufferTransfer>() + CommandEncoderCopyBufferToBufferGetExtraRequiredSize(*this);
}

WireResult CommandEncoderCopyBufferToBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderCopyBufferToBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderCopyBufferToBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderCopyBufferToBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderCopyBufferToBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderCopyBufferToBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderCopyBufferToBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderCopyBufferToBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderCopyBufferToTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderCopyBufferToTextureTransfer>() + CommandEncoderCopyBufferToTextureGetExtraRequiredSize(*this);
}

WireResult CommandEncoderCopyBufferToTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderCopyBufferToTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderCopyBufferToTextureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderCopyBufferToTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderCopyBufferToTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderCopyBufferToTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderCopyBufferToTextureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderCopyBufferToTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderCopyTextureToBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderCopyTextureToBufferTransfer>() + CommandEncoderCopyTextureToBufferGetExtraRequiredSize(*this);
}

WireResult CommandEncoderCopyTextureToBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderCopyTextureToBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderCopyTextureToBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderCopyTextureToBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderCopyTextureToBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderCopyTextureToBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderCopyTextureToBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderCopyTextureToBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderCopyTextureToTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderCopyTextureToTextureTransfer>() + CommandEncoderCopyTextureToTextureGetExtraRequiredSize(*this);
}

WireResult CommandEncoderCopyTextureToTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderCopyTextureToTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderCopyTextureToTextureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderCopyTextureToTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderCopyTextureToTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderCopyTextureToTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderCopyTextureToTextureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderCopyTextureToTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderFinishCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderFinishTransfer>() + CommandEncoderFinishGetExtraRequiredSize(*this);
}

WireResult CommandEncoderFinishCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderFinishTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderFinishSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderFinishCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderFinishCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderFinishTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderFinishDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderFinishCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderInjectValidationErrorCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderInjectValidationErrorTransfer>() + CommandEncoderInjectValidationErrorGetExtraRequiredSize(*this);
}

WireResult CommandEncoderInjectValidationErrorCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderInjectValidationErrorTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderInjectValidationErrorSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderInjectValidationErrorCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderInjectValidationErrorCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderInjectValidationErrorTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderInjectValidationErrorDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderInjectValidationErrorCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderInsertDebugMarkerCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderInsertDebugMarkerTransfer>() + CommandEncoderInsertDebugMarkerGetExtraRequiredSize(*this);
}

WireResult CommandEncoderInsertDebugMarkerCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderInsertDebugMarkerSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderInsertDebugMarkerCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderInsertDebugMarkerCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderInsertDebugMarkerDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderInsertDebugMarkerCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderPopDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderPopDebugGroupTransfer>() + CommandEncoderPopDebugGroupGetExtraRequiredSize(*this);
}

WireResult CommandEncoderPopDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderPopDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderPopDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderPopDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderPopDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderPopDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderPushDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderPushDebugGroupTransfer>() + CommandEncoderPushDebugGroupGetExtraRequiredSize(*this);
}

WireResult CommandEncoderPushDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderPushDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderPushDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderPushDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderPushDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderPushDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderResolveQuerySetCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderResolveQuerySetTransfer>() + CommandEncoderResolveQuerySetGetExtraRequiredSize(*this);
}

WireResult CommandEncoderResolveQuerySetCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderResolveQuerySetTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderResolveQuerySetSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderResolveQuerySetCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderResolveQuerySetCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderResolveQuerySetTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderResolveQuerySetDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderResolveQuerySetCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderSetLabelTransfer>() + CommandEncoderSetLabelGetExtraRequiredSize(*this);
}

WireResult CommandEncoderSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderWriteBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderWriteBufferTransfer>() + CommandEncoderWriteBufferGetExtraRequiredSize(*this);
}

WireResult CommandEncoderWriteBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderWriteBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderWriteBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderWriteBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderWriteBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderWriteBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderWriteBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderWriteBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t CommandEncoderWriteTimestampCmd::GetRequiredSize() const {
    return WireAlignSizeof<CommandEncoderWriteTimestampTransfer>() + CommandEncoderWriteTimestampGetExtraRequiredSize(*this);
}

WireResult CommandEncoderWriteTimestampCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    CommandEncoderWriteTimestampTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (CommandEncoderWriteTimestampSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult CommandEncoderWriteTimestampCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult CommandEncoderWriteTimestampCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile CommandEncoderWriteTimestampTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return CommandEncoderWriteTimestampDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult CommandEncoderWriteTimestampCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderDispatchWorkgroupsCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderDispatchWorkgroupsTransfer>() + ComputePassEncoderDispatchWorkgroupsGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderDispatchWorkgroupsCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderDispatchWorkgroupsTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderDispatchWorkgroupsSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderDispatchWorkgroupsCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderDispatchWorkgroupsCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderDispatchWorkgroupsTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderDispatchWorkgroupsDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderDispatchWorkgroupsCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderDispatchWorkgroupsIndirectCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderDispatchWorkgroupsIndirectTransfer>() + ComputePassEncoderDispatchWorkgroupsIndirectGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderDispatchWorkgroupsIndirectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderDispatchWorkgroupsIndirectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderDispatchWorkgroupsIndirectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderDispatchWorkgroupsIndirectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderDispatchWorkgroupsIndirectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderDispatchWorkgroupsIndirectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderDispatchWorkgroupsIndirectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderDispatchWorkgroupsIndirectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderEndCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderEndTransfer>() + ComputePassEncoderEndGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderEndCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderEndTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderEndSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderEndCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderEndCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderEndTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderEndDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderEndCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderInsertDebugMarkerCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderInsertDebugMarkerTransfer>() + ComputePassEncoderInsertDebugMarkerGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderInsertDebugMarkerCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderInsertDebugMarkerSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderInsertDebugMarkerCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderInsertDebugMarkerCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderInsertDebugMarkerDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderInsertDebugMarkerCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderPopDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderPopDebugGroupTransfer>() + ComputePassEncoderPopDebugGroupGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderPopDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderPopDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderPopDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderPopDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderPopDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderPopDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderPushDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderPushDebugGroupTransfer>() + ComputePassEncoderPushDebugGroupGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderPushDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderPushDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderPushDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderPushDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderPushDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderPushDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderSetBindGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderSetBindGroupTransfer>() + ComputePassEncoderSetBindGroupGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderSetBindGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderSetBindGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderSetBindGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderSetBindGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderSetBindGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderSetBindGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderSetBindGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderSetBindGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderSetLabelTransfer>() + ComputePassEncoderSetLabelGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderSetPipelineCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderSetPipelineTransfer>() + ComputePassEncoderSetPipelineGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderSetPipelineCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderSetPipelineTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderSetPipelineSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderSetPipelineCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderSetPipelineCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderSetPipelineTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderSetPipelineDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderSetPipelineCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePassEncoderWriteTimestampCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePassEncoderWriteTimestampTransfer>() + ComputePassEncoderWriteTimestampGetExtraRequiredSize(*this);
}

WireResult ComputePassEncoderWriteTimestampCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePassEncoderWriteTimestampTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePassEncoderWriteTimestampSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePassEncoderWriteTimestampCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePassEncoderWriteTimestampCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePassEncoderWriteTimestampTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePassEncoderWriteTimestampDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePassEncoderWriteTimestampCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePipelineGetBindGroupLayoutCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePipelineGetBindGroupLayoutTransfer>() + ComputePipelineGetBindGroupLayoutGetExtraRequiredSize(*this);
}

WireResult ComputePipelineGetBindGroupLayoutCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePipelineGetBindGroupLayoutTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePipelineGetBindGroupLayoutSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePipelineGetBindGroupLayoutCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePipelineGetBindGroupLayoutCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePipelineGetBindGroupLayoutTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePipelineGetBindGroupLayoutDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePipelineGetBindGroupLayoutCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ComputePipelineSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<ComputePipelineSetLabelTransfer>() + ComputePipelineSetLabelGetExtraRequiredSize(*this);
}

WireResult ComputePipelineSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ComputePipelineSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ComputePipelineSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ComputePipelineSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ComputePipelineSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ComputePipelineSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ComputePipelineSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ComputePipelineSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateBindGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateBindGroupTransfer>() + DeviceCreateBindGroupGetExtraRequiredSize(*this);
}

WireResult DeviceCreateBindGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateBindGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateBindGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateBindGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateBindGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateBindGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateBindGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateBindGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateBindGroupLayoutCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateBindGroupLayoutTransfer>() + DeviceCreateBindGroupLayoutGetExtraRequiredSize(*this);
}

WireResult DeviceCreateBindGroupLayoutCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateBindGroupLayoutTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateBindGroupLayoutSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateBindGroupLayoutCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateBindGroupLayoutCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateBindGroupLayoutTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateBindGroupLayoutDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateBindGroupLayoutCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateBufferTransfer>() + DeviceCreateBufferGetExtraRequiredSize(*this);
}

WireResult DeviceCreateBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateCommandEncoderCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateCommandEncoderTransfer>() + DeviceCreateCommandEncoderGetExtraRequiredSize(*this);
}

WireResult DeviceCreateCommandEncoderCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateCommandEncoderTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateCommandEncoderSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateCommandEncoderCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateCommandEncoderCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateCommandEncoderTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateCommandEncoderDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateCommandEncoderCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateComputePipelineCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateComputePipelineTransfer>() + DeviceCreateComputePipelineGetExtraRequiredSize(*this);
}

WireResult DeviceCreateComputePipelineCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateComputePipelineTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateComputePipelineSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateComputePipelineCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateComputePipelineCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateComputePipelineTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateComputePipelineDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateComputePipelineCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateComputePipelineAsyncCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateComputePipelineAsyncTransfer>() + DeviceCreateComputePipelineAsyncGetExtraRequiredSize(*this);
}

WireResult DeviceCreateComputePipelineAsyncCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateComputePipelineAsyncTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateComputePipelineAsyncSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateComputePipelineAsyncCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateComputePipelineAsyncCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateComputePipelineAsyncTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateComputePipelineAsyncDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateComputePipelineAsyncCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateErrorBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateErrorBufferTransfer>() + DeviceCreateErrorBufferGetExtraRequiredSize(*this);
}

WireResult DeviceCreateErrorBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateErrorBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateErrorBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateErrorBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateErrorBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateErrorBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateErrorBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateErrorBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateErrorExternalTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateErrorExternalTextureTransfer>() + DeviceCreateErrorExternalTextureGetExtraRequiredSize(*this);
}

WireResult DeviceCreateErrorExternalTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateErrorExternalTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateErrorExternalTextureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateErrorExternalTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateErrorExternalTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateErrorExternalTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateErrorExternalTextureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateErrorExternalTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateErrorShaderModuleCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateErrorShaderModuleTransfer>() + DeviceCreateErrorShaderModuleGetExtraRequiredSize(*this);
}

WireResult DeviceCreateErrorShaderModuleCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateErrorShaderModuleTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateErrorShaderModuleSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateErrorShaderModuleCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateErrorShaderModuleCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateErrorShaderModuleTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateErrorShaderModuleDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateErrorShaderModuleCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateErrorTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateErrorTextureTransfer>() + DeviceCreateErrorTextureGetExtraRequiredSize(*this);
}

WireResult DeviceCreateErrorTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateErrorTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateErrorTextureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateErrorTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateErrorTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateErrorTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateErrorTextureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateErrorTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateExternalTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateExternalTextureTransfer>() + DeviceCreateExternalTextureGetExtraRequiredSize(*this);
}

WireResult DeviceCreateExternalTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateExternalTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateExternalTextureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateExternalTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateExternalTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateExternalTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateExternalTextureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateExternalTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreatePipelineLayoutCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreatePipelineLayoutTransfer>() + DeviceCreatePipelineLayoutGetExtraRequiredSize(*this);
}

WireResult DeviceCreatePipelineLayoutCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreatePipelineLayoutTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreatePipelineLayoutSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreatePipelineLayoutCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreatePipelineLayoutCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreatePipelineLayoutTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreatePipelineLayoutDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreatePipelineLayoutCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateQuerySetCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateQuerySetTransfer>() + DeviceCreateQuerySetGetExtraRequiredSize(*this);
}

WireResult DeviceCreateQuerySetCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateQuerySetTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateQuerySetSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateQuerySetCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateQuerySetCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateQuerySetTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateQuerySetDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateQuerySetCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateRenderBundleEncoderCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateRenderBundleEncoderTransfer>() + DeviceCreateRenderBundleEncoderGetExtraRequiredSize(*this);
}

WireResult DeviceCreateRenderBundleEncoderCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateRenderBundleEncoderTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateRenderBundleEncoderSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateRenderBundleEncoderCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateRenderBundleEncoderCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateRenderBundleEncoderTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateRenderBundleEncoderDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateRenderBundleEncoderCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateRenderPipelineCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateRenderPipelineTransfer>() + DeviceCreateRenderPipelineGetExtraRequiredSize(*this);
}

WireResult DeviceCreateRenderPipelineCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateRenderPipelineTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateRenderPipelineSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateRenderPipelineCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateRenderPipelineCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateRenderPipelineTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateRenderPipelineDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateRenderPipelineCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateRenderPipelineAsyncCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateRenderPipelineAsyncTransfer>() + DeviceCreateRenderPipelineAsyncGetExtraRequiredSize(*this);
}

WireResult DeviceCreateRenderPipelineAsyncCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateRenderPipelineAsyncTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateRenderPipelineAsyncSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateRenderPipelineAsyncCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateRenderPipelineAsyncCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateRenderPipelineAsyncTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateRenderPipelineAsyncDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateRenderPipelineAsyncCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateSamplerCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateSamplerTransfer>() + DeviceCreateSamplerGetExtraRequiredSize(*this);
}

WireResult DeviceCreateSamplerCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateSamplerTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateSamplerSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateSamplerCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateSamplerCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateSamplerTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateSamplerDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateSamplerCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateShaderModuleCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateShaderModuleTransfer>() + DeviceCreateShaderModuleGetExtraRequiredSize(*this);
}

WireResult DeviceCreateShaderModuleCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateShaderModuleTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateShaderModuleSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateShaderModuleCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateShaderModuleCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateShaderModuleTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateShaderModuleDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateShaderModuleCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceCreateTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceCreateTextureTransfer>() + DeviceCreateTextureGetExtraRequiredSize(*this);
}

WireResult DeviceCreateTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceCreateTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceCreateTextureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceCreateTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceCreateTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceCreateTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceCreateTextureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceCreateTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceDestroyCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceDestroyTransfer>() + DeviceDestroyGetExtraRequiredSize(*this);
}

WireResult DeviceDestroyCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceDestroyTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceDestroySerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceDestroyCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceDestroyCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceDestroyTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceDestroyDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceDestroyCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceForceLossCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceForceLossTransfer>() + DeviceForceLossGetExtraRequiredSize(*this);
}

WireResult DeviceForceLossCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceForceLossTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceForceLossSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceForceLossCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceForceLossCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceForceLossTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceForceLossDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceForceLossCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceGetQueueCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceGetQueueTransfer>() + DeviceGetQueueGetExtraRequiredSize(*this);
}

WireResult DeviceGetQueueCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceGetQueueTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceGetQueueSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceGetQueueCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceGetQueueCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceGetQueueTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceGetQueueDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceGetQueueCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceInjectErrorCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceInjectErrorTransfer>() + DeviceInjectErrorGetExtraRequiredSize(*this);
}

WireResult DeviceInjectErrorCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceInjectErrorTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceInjectErrorSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceInjectErrorCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceInjectErrorCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceInjectErrorTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceInjectErrorDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceInjectErrorCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DevicePopErrorScopeCmd::GetRequiredSize() const {
    return WireAlignSizeof<DevicePopErrorScopeTransfer>() + DevicePopErrorScopeGetExtraRequiredSize(*this);
}

WireResult DevicePopErrorScopeCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    DevicePopErrorScopeTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DevicePopErrorScopeSerialize(*this, transfer, serializeBuffer));
}
WireResult DevicePopErrorScopeCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult DevicePopErrorScopeCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile DevicePopErrorScopeTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DevicePopErrorScopeDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult DevicePopErrorScopeCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t DevicePushErrorScopeCmd::GetRequiredSize() const {
    return WireAlignSizeof<DevicePushErrorScopeTransfer>() + DevicePushErrorScopeGetExtraRequiredSize(*this);
}

WireResult DevicePushErrorScopeCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DevicePushErrorScopeTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DevicePushErrorScopeSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DevicePushErrorScopeCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DevicePushErrorScopeCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DevicePushErrorScopeTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DevicePushErrorScopeDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DevicePushErrorScopeCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceSetLabelTransfer>() + DeviceSetLabelGetExtraRequiredSize(*this);
}

WireResult DeviceSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceTickCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceTickTransfer>() + DeviceTickGetExtraRequiredSize(*this);
}

WireResult DeviceTickCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceTickTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceTickSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceTickCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceTickCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceTickTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceTickDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceTickCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t DeviceValidateTextureDescriptorCmd::GetRequiredSize() const {
    return WireAlignSizeof<DeviceValidateTextureDescriptorTransfer>() + DeviceValidateTextureDescriptorGetExtraRequiredSize(*this);
}

WireResult DeviceValidateTextureDescriptorCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    DeviceValidateTextureDescriptorTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (DeviceValidateTextureDescriptorSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult DeviceValidateTextureDescriptorCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult DeviceValidateTextureDescriptorCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile DeviceValidateTextureDescriptorTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return DeviceValidateTextureDescriptorDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult DeviceValidateTextureDescriptorCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ExternalTextureDestroyCmd::GetRequiredSize() const {
    return WireAlignSizeof<ExternalTextureDestroyTransfer>() + ExternalTextureDestroyGetExtraRequiredSize(*this);
}

WireResult ExternalTextureDestroyCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ExternalTextureDestroyTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ExternalTextureDestroySerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ExternalTextureDestroyCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ExternalTextureDestroyCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ExternalTextureDestroyTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ExternalTextureDestroyDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ExternalTextureDestroyCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ExternalTextureExpireCmd::GetRequiredSize() const {
    return WireAlignSizeof<ExternalTextureExpireTransfer>() + ExternalTextureExpireGetExtraRequiredSize(*this);
}

WireResult ExternalTextureExpireCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ExternalTextureExpireTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ExternalTextureExpireSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ExternalTextureExpireCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ExternalTextureExpireCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ExternalTextureExpireTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ExternalTextureExpireDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ExternalTextureExpireCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ExternalTextureRefreshCmd::GetRequiredSize() const {
    return WireAlignSizeof<ExternalTextureRefreshTransfer>() + ExternalTextureRefreshGetExtraRequiredSize(*this);
}

WireResult ExternalTextureRefreshCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ExternalTextureRefreshTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ExternalTextureRefreshSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ExternalTextureRefreshCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ExternalTextureRefreshCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ExternalTextureRefreshTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ExternalTextureRefreshDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ExternalTextureRefreshCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ExternalTextureSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<ExternalTextureSetLabelTransfer>() + ExternalTextureSetLabelGetExtraRequiredSize(*this);
}

WireResult ExternalTextureSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ExternalTextureSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ExternalTextureSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ExternalTextureSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ExternalTextureSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ExternalTextureSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ExternalTextureSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ExternalTextureSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t InstanceProcessEventsCmd::GetRequiredSize() const {
    return WireAlignSizeof<InstanceProcessEventsTransfer>() + InstanceProcessEventsGetExtraRequiredSize(*this);
}

WireResult InstanceProcessEventsCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    InstanceProcessEventsTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (InstanceProcessEventsSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult InstanceProcessEventsCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult InstanceProcessEventsCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile InstanceProcessEventsTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return InstanceProcessEventsDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult InstanceProcessEventsCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t InstanceRequestAdapterCmd::GetRequiredSize() const {
    return WireAlignSizeof<InstanceRequestAdapterTransfer>() + InstanceRequestAdapterGetExtraRequiredSize(*this);
}

WireResult InstanceRequestAdapterCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    InstanceRequestAdapterTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (InstanceRequestAdapterSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult InstanceRequestAdapterCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult InstanceRequestAdapterCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile InstanceRequestAdapterTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return InstanceRequestAdapterDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult InstanceRequestAdapterCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t PipelineLayoutSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<PipelineLayoutSetLabelTransfer>() + PipelineLayoutSetLabelGetExtraRequiredSize(*this);
}

WireResult PipelineLayoutSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    PipelineLayoutSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (PipelineLayoutSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult PipelineLayoutSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult PipelineLayoutSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile PipelineLayoutSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return PipelineLayoutSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult PipelineLayoutSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t QuerySetDestroyCmd::GetRequiredSize() const {
    return WireAlignSizeof<QuerySetDestroyTransfer>() + QuerySetDestroyGetExtraRequiredSize(*this);
}

WireResult QuerySetDestroyCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    QuerySetDestroyTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QuerySetDestroySerialize(*this, transfer, serializeBuffer, provider));
}
WireResult QuerySetDestroyCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult QuerySetDestroyCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile QuerySetDestroyTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QuerySetDestroyDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult QuerySetDestroyCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t QuerySetSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<QuerySetSetLabelTransfer>() + QuerySetSetLabelGetExtraRequiredSize(*this);
}

WireResult QuerySetSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    QuerySetSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QuerySetSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult QuerySetSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult QuerySetSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile QuerySetSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QuerySetSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult QuerySetSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t QueueCopyExternalTextureForBrowserCmd::GetRequiredSize() const {
    return WireAlignSizeof<QueueCopyExternalTextureForBrowserTransfer>() + QueueCopyExternalTextureForBrowserGetExtraRequiredSize(*this);
}

WireResult QueueCopyExternalTextureForBrowserCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    QueueCopyExternalTextureForBrowserTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QueueCopyExternalTextureForBrowserSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult QueueCopyExternalTextureForBrowserCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult QueueCopyExternalTextureForBrowserCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile QueueCopyExternalTextureForBrowserTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QueueCopyExternalTextureForBrowserDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult QueueCopyExternalTextureForBrowserCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t QueueCopyTextureForBrowserCmd::GetRequiredSize() const {
    return WireAlignSizeof<QueueCopyTextureForBrowserTransfer>() + QueueCopyTextureForBrowserGetExtraRequiredSize(*this);
}

WireResult QueueCopyTextureForBrowserCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    QueueCopyTextureForBrowserTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QueueCopyTextureForBrowserSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult QueueCopyTextureForBrowserCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult QueueCopyTextureForBrowserCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile QueueCopyTextureForBrowserTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QueueCopyTextureForBrowserDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult QueueCopyTextureForBrowserCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t QueueOnSubmittedWorkDoneCmd::GetRequiredSize() const {
    return WireAlignSizeof<QueueOnSubmittedWorkDoneTransfer>() + QueueOnSubmittedWorkDoneGetExtraRequiredSize(*this);
}

WireResult QueueOnSubmittedWorkDoneCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    QueueOnSubmittedWorkDoneTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QueueOnSubmittedWorkDoneSerialize(*this, transfer, serializeBuffer));
}
WireResult QueueOnSubmittedWorkDoneCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult QueueOnSubmittedWorkDoneCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile QueueOnSubmittedWorkDoneTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QueueOnSubmittedWorkDoneDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult QueueOnSubmittedWorkDoneCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t QueueSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<QueueSetLabelTransfer>() + QueueSetLabelGetExtraRequiredSize(*this);
}

WireResult QueueSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    QueueSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QueueSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult QueueSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult QueueSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile QueueSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QueueSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult QueueSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t QueueSubmitCmd::GetRequiredSize() const {
    return WireAlignSizeof<QueueSubmitTransfer>() + QueueSubmitGetExtraRequiredSize(*this);
}

WireResult QueueSubmitCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    QueueSubmitTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QueueSubmitSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult QueueSubmitCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult QueueSubmitCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile QueueSubmitTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QueueSubmitDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult QueueSubmitCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t QueueWriteBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<QueueWriteBufferTransfer>() + QueueWriteBufferGetExtraRequiredSize(*this);
}

WireResult QueueWriteBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    QueueWriteBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QueueWriteBufferSerialize(*this, transfer, serializeBuffer));
}
WireResult QueueWriteBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult QueueWriteBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile QueueWriteBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QueueWriteBufferDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult QueueWriteBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t QueueWriteTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<QueueWriteTextureTransfer>() + QueueWriteTextureGetExtraRequiredSize(*this);
}

WireResult QueueWriteTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    QueueWriteTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (QueueWriteTextureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult QueueWriteTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult QueueWriteTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile QueueWriteTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return QueueWriteTextureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult QueueWriteTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderDrawCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderDrawTransfer>() + RenderBundleEncoderDrawGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderDrawCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderDrawTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderDrawSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderDrawCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderDrawCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderDrawTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderDrawDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderDrawCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderDrawIndexedCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderDrawIndexedTransfer>() + RenderBundleEncoderDrawIndexedGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderDrawIndexedCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderDrawIndexedTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderDrawIndexedSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderDrawIndexedCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderDrawIndexedCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderDrawIndexedTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderDrawIndexedDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderDrawIndexedCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderDrawIndexedIndirectCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderDrawIndexedIndirectTransfer>() + RenderBundleEncoderDrawIndexedIndirectGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderDrawIndexedIndirectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderDrawIndexedIndirectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderDrawIndexedIndirectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderDrawIndexedIndirectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderDrawIndexedIndirectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderDrawIndexedIndirectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderDrawIndexedIndirectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderDrawIndexedIndirectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderDrawIndirectCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderDrawIndirectTransfer>() + RenderBundleEncoderDrawIndirectGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderDrawIndirectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderDrawIndirectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderDrawIndirectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderDrawIndirectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderDrawIndirectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderDrawIndirectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderDrawIndirectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderDrawIndirectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderFinishCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderFinishTransfer>() + RenderBundleEncoderFinishGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderFinishCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderFinishTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderFinishSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderFinishCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderFinishCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderFinishTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderFinishDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderFinishCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderInsertDebugMarkerCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderInsertDebugMarkerTransfer>() + RenderBundleEncoderInsertDebugMarkerGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderInsertDebugMarkerCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderInsertDebugMarkerSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderInsertDebugMarkerCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderInsertDebugMarkerCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderInsertDebugMarkerDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderInsertDebugMarkerCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderPopDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderPopDebugGroupTransfer>() + RenderBundleEncoderPopDebugGroupGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderPopDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderPopDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderPopDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderPopDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderPopDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderPopDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderPushDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderPushDebugGroupTransfer>() + RenderBundleEncoderPushDebugGroupGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderPushDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderPushDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderPushDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderPushDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderPushDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderPushDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderSetBindGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderSetBindGroupTransfer>() + RenderBundleEncoderSetBindGroupGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderSetBindGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderSetBindGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderSetBindGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderSetBindGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderSetBindGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderSetBindGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderSetBindGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderSetBindGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderSetIndexBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderSetIndexBufferTransfer>() + RenderBundleEncoderSetIndexBufferGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderSetIndexBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderSetIndexBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderSetIndexBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderSetIndexBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderSetIndexBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderSetIndexBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderSetIndexBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderSetIndexBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderSetLabelTransfer>() + RenderBundleEncoderSetLabelGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderSetPipelineCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderSetPipelineTransfer>() + RenderBundleEncoderSetPipelineGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderSetPipelineCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderSetPipelineTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderSetPipelineSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderSetPipelineCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderSetPipelineCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderSetPipelineTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderSetPipelineDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderSetPipelineCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleEncoderSetVertexBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleEncoderSetVertexBufferTransfer>() + RenderBundleEncoderSetVertexBufferGetExtraRequiredSize(*this);
}

WireResult RenderBundleEncoderSetVertexBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleEncoderSetVertexBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleEncoderSetVertexBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleEncoderSetVertexBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleEncoderSetVertexBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleEncoderSetVertexBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleEncoderSetVertexBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleEncoderSetVertexBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderBundleSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderBundleSetLabelTransfer>() + RenderBundleSetLabelGetExtraRequiredSize(*this);
}

WireResult RenderBundleSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderBundleSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderBundleSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderBundleSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderBundleSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderBundleSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderBundleSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderBundleSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderBeginOcclusionQueryCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderBeginOcclusionQueryTransfer>() + RenderPassEncoderBeginOcclusionQueryGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderBeginOcclusionQueryCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderBeginOcclusionQueryTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderBeginOcclusionQuerySerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderBeginOcclusionQueryCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderBeginOcclusionQueryCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderBeginOcclusionQueryTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderBeginOcclusionQueryDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderBeginOcclusionQueryCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderDrawCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderDrawTransfer>() + RenderPassEncoderDrawGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderDrawCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderDrawTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderDrawSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderDrawCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderDrawCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderDrawTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderDrawDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderDrawCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderDrawIndexedCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderDrawIndexedTransfer>() + RenderPassEncoderDrawIndexedGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderDrawIndexedCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderDrawIndexedTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderDrawIndexedSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderDrawIndexedCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderDrawIndexedCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderDrawIndexedTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderDrawIndexedDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderDrawIndexedCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderDrawIndexedIndirectCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderDrawIndexedIndirectTransfer>() + RenderPassEncoderDrawIndexedIndirectGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderDrawIndexedIndirectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderDrawIndexedIndirectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderDrawIndexedIndirectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderDrawIndexedIndirectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderDrawIndexedIndirectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderDrawIndexedIndirectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderDrawIndexedIndirectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderDrawIndexedIndirectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderDrawIndirectCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderDrawIndirectTransfer>() + RenderPassEncoderDrawIndirectGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderDrawIndirectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderDrawIndirectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderDrawIndirectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderDrawIndirectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderDrawIndirectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderDrawIndirectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderDrawIndirectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderDrawIndirectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderEndCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderEndTransfer>() + RenderPassEncoderEndGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderEndCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderEndTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderEndSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderEndCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderEndCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderEndTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderEndDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderEndCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderEndOcclusionQueryCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderEndOcclusionQueryTransfer>() + RenderPassEncoderEndOcclusionQueryGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderEndOcclusionQueryCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderEndOcclusionQueryTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderEndOcclusionQuerySerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderEndOcclusionQueryCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderEndOcclusionQueryCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderEndOcclusionQueryTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderEndOcclusionQueryDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderEndOcclusionQueryCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderExecuteBundlesCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderExecuteBundlesTransfer>() + RenderPassEncoderExecuteBundlesGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderExecuteBundlesCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderExecuteBundlesTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderExecuteBundlesSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderExecuteBundlesCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderExecuteBundlesCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderExecuteBundlesTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderExecuteBundlesDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderExecuteBundlesCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderInsertDebugMarkerCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderInsertDebugMarkerTransfer>() + RenderPassEncoderInsertDebugMarkerGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderInsertDebugMarkerCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderInsertDebugMarkerSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderInsertDebugMarkerCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderInsertDebugMarkerCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderInsertDebugMarkerTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderInsertDebugMarkerDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderInsertDebugMarkerCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderMultiDrawIndexedIndirectCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderMultiDrawIndexedIndirectTransfer>() + RenderPassEncoderMultiDrawIndexedIndirectGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderMultiDrawIndexedIndirectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderMultiDrawIndexedIndirectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderMultiDrawIndexedIndirectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderMultiDrawIndexedIndirectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderMultiDrawIndexedIndirectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderMultiDrawIndexedIndirectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderMultiDrawIndexedIndirectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderMultiDrawIndexedIndirectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderMultiDrawIndirectCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderMultiDrawIndirectTransfer>() + RenderPassEncoderMultiDrawIndirectGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderMultiDrawIndirectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderMultiDrawIndirectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderMultiDrawIndirectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderMultiDrawIndirectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderMultiDrawIndirectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderMultiDrawIndirectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderMultiDrawIndirectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderMultiDrawIndirectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderPixelLocalStorageBarrierCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderPixelLocalStorageBarrierTransfer>() + RenderPassEncoderPixelLocalStorageBarrierGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderPixelLocalStorageBarrierCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderPixelLocalStorageBarrierTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderPixelLocalStorageBarrierSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderPixelLocalStorageBarrierCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderPixelLocalStorageBarrierCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderPixelLocalStorageBarrierTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderPixelLocalStorageBarrierDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderPixelLocalStorageBarrierCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderPopDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderPopDebugGroupTransfer>() + RenderPassEncoderPopDebugGroupGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderPopDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderPopDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderPopDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderPopDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderPopDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderPopDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderPopDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderPushDebugGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderPushDebugGroupTransfer>() + RenderPassEncoderPushDebugGroupGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderPushDebugGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderPushDebugGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderPushDebugGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderPushDebugGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderPushDebugGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderPushDebugGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderPushDebugGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetBindGroupCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetBindGroupTransfer>() + RenderPassEncoderSetBindGroupGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetBindGroupCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetBindGroupTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetBindGroupSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetBindGroupCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetBindGroupCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetBindGroupTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetBindGroupDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetBindGroupCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetBlendConstantCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetBlendConstantTransfer>() + RenderPassEncoderSetBlendConstantGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetBlendConstantCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetBlendConstantTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetBlendConstantSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetBlendConstantCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetBlendConstantCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetBlendConstantTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetBlendConstantDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetBlendConstantCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetIndexBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetIndexBufferTransfer>() + RenderPassEncoderSetIndexBufferGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetIndexBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetIndexBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetIndexBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetIndexBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetIndexBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetIndexBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetIndexBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetIndexBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetLabelTransfer>() + RenderPassEncoderSetLabelGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetPipelineCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetPipelineTransfer>() + RenderPassEncoderSetPipelineGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetPipelineCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetPipelineTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetPipelineSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetPipelineCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetPipelineCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetPipelineTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetPipelineDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetPipelineCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetScissorRectCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetScissorRectTransfer>() + RenderPassEncoderSetScissorRectGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetScissorRectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetScissorRectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetScissorRectSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetScissorRectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetScissorRectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetScissorRectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetScissorRectDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetScissorRectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetStencilReferenceCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetStencilReferenceTransfer>() + RenderPassEncoderSetStencilReferenceGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetStencilReferenceCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetStencilReferenceTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetStencilReferenceSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetStencilReferenceCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetStencilReferenceCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetStencilReferenceTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetStencilReferenceDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetStencilReferenceCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetVertexBufferCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetVertexBufferTransfer>() + RenderPassEncoderSetVertexBufferGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetVertexBufferCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetVertexBufferTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetVertexBufferSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetVertexBufferCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetVertexBufferCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetVertexBufferTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetVertexBufferDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetVertexBufferCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderSetViewportCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderSetViewportTransfer>() + RenderPassEncoderSetViewportGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderSetViewportCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderSetViewportTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderSetViewportSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderSetViewportCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderSetViewportCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderSetViewportTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderSetViewportDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderSetViewportCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPassEncoderWriteTimestampCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPassEncoderWriteTimestampTransfer>() + RenderPassEncoderWriteTimestampGetExtraRequiredSize(*this);
}

WireResult RenderPassEncoderWriteTimestampCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPassEncoderWriteTimestampTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPassEncoderWriteTimestampSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPassEncoderWriteTimestampCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPassEncoderWriteTimestampCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPassEncoderWriteTimestampTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPassEncoderWriteTimestampDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPassEncoderWriteTimestampCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPipelineGetBindGroupLayoutCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPipelineGetBindGroupLayoutTransfer>() + RenderPipelineGetBindGroupLayoutGetExtraRequiredSize(*this);
}

WireResult RenderPipelineGetBindGroupLayoutCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPipelineGetBindGroupLayoutTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPipelineGetBindGroupLayoutSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPipelineGetBindGroupLayoutCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPipelineGetBindGroupLayoutCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPipelineGetBindGroupLayoutTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPipelineGetBindGroupLayoutDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPipelineGetBindGroupLayoutCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t RenderPipelineSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<RenderPipelineSetLabelTransfer>() + RenderPipelineSetLabelGetExtraRequiredSize(*this);
}

WireResult RenderPipelineSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    RenderPipelineSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (RenderPipelineSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult RenderPipelineSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult RenderPipelineSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile RenderPipelineSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return RenderPipelineSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult RenderPipelineSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t SamplerSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<SamplerSetLabelTransfer>() + SamplerSetLabelGetExtraRequiredSize(*this);
}

WireResult SamplerSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    SamplerSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (SamplerSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult SamplerSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult SamplerSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile SamplerSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return SamplerSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult SamplerSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ShaderModuleGetCompilationInfoCmd::GetRequiredSize() const {
    return WireAlignSizeof<ShaderModuleGetCompilationInfoTransfer>() + ShaderModuleGetCompilationInfoGetExtraRequiredSize(*this);
}

WireResult ShaderModuleGetCompilationInfoCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ShaderModuleGetCompilationInfoTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ShaderModuleGetCompilationInfoSerialize(*this, transfer, serializeBuffer));
}
WireResult ShaderModuleGetCompilationInfoCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ShaderModuleGetCompilationInfoCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ShaderModuleGetCompilationInfoTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ShaderModuleGetCompilationInfoDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ShaderModuleGetCompilationInfoCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ShaderModuleSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<ShaderModuleSetLabelTransfer>() + ShaderModuleSetLabelGetExtraRequiredSize(*this);
}

WireResult ShaderModuleSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ShaderModuleSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ShaderModuleSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ShaderModuleSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ShaderModuleSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ShaderModuleSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ShaderModuleSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ShaderModuleSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t SurfaceConfigureCmd::GetRequiredSize() const {
    return WireAlignSizeof<SurfaceConfigureTransfer>() + SurfaceConfigureGetExtraRequiredSize(*this);
}

WireResult SurfaceConfigureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    SurfaceConfigureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (SurfaceConfigureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult SurfaceConfigureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult SurfaceConfigureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile SurfaceConfigureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return SurfaceConfigureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult SurfaceConfigureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t SurfaceGetCurrentTextureCmd::GetRequiredSize() const {
    return WireAlignSizeof<SurfaceGetCurrentTextureTransfer>() + SurfaceGetCurrentTextureGetExtraRequiredSize(*this);
}

WireResult SurfaceGetCurrentTextureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    SurfaceGetCurrentTextureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (SurfaceGetCurrentTextureSerialize(*this, transfer, serializeBuffer));
}
WireResult SurfaceGetCurrentTextureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult SurfaceGetCurrentTextureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile SurfaceGetCurrentTextureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return SurfaceGetCurrentTextureDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult SurfaceGetCurrentTextureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t SurfacePresentCmd::GetRequiredSize() const {
    return WireAlignSizeof<SurfacePresentTransfer>() + SurfacePresentGetExtraRequiredSize(*this);
}

WireResult SurfacePresentCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    SurfacePresentTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (SurfacePresentSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult SurfacePresentCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult SurfacePresentCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile SurfacePresentTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return SurfacePresentDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult SurfacePresentCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t SurfaceSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<SurfaceSetLabelTransfer>() + SurfaceSetLabelGetExtraRequiredSize(*this);
}

WireResult SurfaceSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    SurfaceSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (SurfaceSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult SurfaceSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult SurfaceSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile SurfaceSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return SurfaceSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult SurfaceSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t SurfaceUnconfigureCmd::GetRequiredSize() const {
    return WireAlignSizeof<SurfaceUnconfigureTransfer>() + SurfaceUnconfigureGetExtraRequiredSize(*this);
}

WireResult SurfaceUnconfigureCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    SurfaceUnconfigureTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (SurfaceUnconfigureSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult SurfaceUnconfigureCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult SurfaceUnconfigureCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile SurfaceUnconfigureTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return SurfaceUnconfigureDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult SurfaceUnconfigureCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t TextureCreateErrorViewCmd::GetRequiredSize() const {
    return WireAlignSizeof<TextureCreateErrorViewTransfer>() + TextureCreateErrorViewGetExtraRequiredSize(*this);
}

WireResult TextureCreateErrorViewCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    TextureCreateErrorViewTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (TextureCreateErrorViewSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult TextureCreateErrorViewCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult TextureCreateErrorViewCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile TextureCreateErrorViewTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return TextureCreateErrorViewDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult TextureCreateErrorViewCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t TextureCreateViewCmd::GetRequiredSize() const {
    return WireAlignSizeof<TextureCreateViewTransfer>() + TextureCreateViewGetExtraRequiredSize(*this);
}

WireResult TextureCreateViewCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    TextureCreateViewTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (TextureCreateViewSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult TextureCreateViewCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult TextureCreateViewCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile TextureCreateViewTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return TextureCreateViewDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult TextureCreateViewCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t TextureDestroyCmd::GetRequiredSize() const {
    return WireAlignSizeof<TextureDestroyTransfer>() + TextureDestroyGetExtraRequiredSize(*this);
}

WireResult TextureDestroyCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    TextureDestroyTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (TextureDestroySerialize(*this, transfer, serializeBuffer, provider));
}
WireResult TextureDestroyCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult TextureDestroyCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile TextureDestroyTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return TextureDestroyDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult TextureDestroyCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t TextureSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<TextureSetLabelTransfer>() + TextureSetLabelGetExtraRequiredSize(*this);
}

WireResult TextureSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    TextureSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (TextureSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult TextureSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult TextureSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile TextureSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return TextureSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult TextureSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t TextureViewSetLabelCmd::GetRequiredSize() const {
    return WireAlignSizeof<TextureViewSetLabelTransfer>() + TextureViewSetLabelGetExtraRequiredSize(*this);
}

WireResult TextureViewSetLabelCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    TextureViewSetLabelTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (TextureViewSetLabelSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult TextureViewSetLabelCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult TextureViewSetLabelCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile TextureViewSetLabelTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return TextureViewSetLabelDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult TextureViewSetLabelCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t UnregisterObjectCmd::GetRequiredSize() const {
    return WireAlignSizeof<UnregisterObjectTransfer>() + UnregisterObjectGetExtraRequiredSize(*this);
}

WireResult UnregisterObjectCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    UnregisterObjectTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (UnregisterObjectSerialize(*this, transfer, serializeBuffer));
}
WireResult UnregisterObjectCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult UnregisterObjectCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile UnregisterObjectTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return UnregisterObjectDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult UnregisterObjectCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}


size_t ReturnAdapterRequestDeviceCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnAdapterRequestDeviceCallbackTransfer>() + ReturnAdapterRequestDeviceCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnAdapterRequestDeviceCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ReturnAdapterRequestDeviceCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnAdapterRequestDeviceCallbackSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ReturnAdapterRequestDeviceCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ReturnAdapterRequestDeviceCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ReturnAdapterRequestDeviceCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnAdapterRequestDeviceCallbackDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ReturnAdapterRequestDeviceCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ReturnBufferMapAsyncCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnBufferMapAsyncCallbackTransfer>() + ReturnBufferMapAsyncCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnBufferMapAsyncCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnBufferMapAsyncCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnBufferMapAsyncCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnBufferMapAsyncCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnBufferMapAsyncCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnBufferMapAsyncCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnBufferMapAsyncCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnBufferMapAsyncCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnDeviceCreateComputePipelineAsyncCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnDeviceCreateComputePipelineAsyncCallbackTransfer>() + ReturnDeviceCreateComputePipelineAsyncCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnDeviceCreateComputePipelineAsyncCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnDeviceCreateComputePipelineAsyncCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnDeviceCreateComputePipelineAsyncCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnDeviceCreateComputePipelineAsyncCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnDeviceCreateComputePipelineAsyncCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnDeviceCreateComputePipelineAsyncCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnDeviceCreateComputePipelineAsyncCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnDeviceCreateComputePipelineAsyncCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnDeviceCreateRenderPipelineAsyncCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer>() + ReturnDeviceCreateRenderPipelineAsyncCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnDeviceCreateRenderPipelineAsyncCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnDeviceCreateRenderPipelineAsyncCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnDeviceCreateRenderPipelineAsyncCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnDeviceCreateRenderPipelineAsyncCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnDeviceCreateRenderPipelineAsyncCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnDeviceCreateRenderPipelineAsyncCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnDeviceCreateRenderPipelineAsyncCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnDeviceLoggingCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnDeviceLoggingCallbackTransfer>() + ReturnDeviceLoggingCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnDeviceLoggingCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnDeviceLoggingCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnDeviceLoggingCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnDeviceLoggingCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnDeviceLoggingCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnDeviceLoggingCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnDeviceLoggingCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnDeviceLoggingCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnDeviceLostCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnDeviceLostCallbackTransfer>() + ReturnDeviceLostCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnDeviceLostCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnDeviceLostCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnDeviceLostCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnDeviceLostCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnDeviceLostCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnDeviceLostCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnDeviceLostCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnDeviceLostCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnDevicePopErrorScopeCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnDevicePopErrorScopeCallbackTransfer>() + ReturnDevicePopErrorScopeCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnDevicePopErrorScopeCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnDevicePopErrorScopeCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnDevicePopErrorScopeCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnDevicePopErrorScopeCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnDevicePopErrorScopeCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnDevicePopErrorScopeCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnDevicePopErrorScopeCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnDevicePopErrorScopeCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnDeviceUncapturedErrorCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnDeviceUncapturedErrorCallbackTransfer>() + ReturnDeviceUncapturedErrorCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnDeviceUncapturedErrorCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnDeviceUncapturedErrorCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnDeviceUncapturedErrorCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnDeviceUncapturedErrorCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnDeviceUncapturedErrorCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnDeviceUncapturedErrorCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnDeviceUncapturedErrorCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnDeviceUncapturedErrorCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnInstanceRequestAdapterCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnInstanceRequestAdapterCallbackTransfer>() + ReturnInstanceRequestAdapterCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnInstanceRequestAdapterCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ReturnInstanceRequestAdapterCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnInstanceRequestAdapterCallbackSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ReturnInstanceRequestAdapterCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ReturnInstanceRequestAdapterCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver& resolver) {
    const volatile ReturnInstanceRequestAdapterCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnInstanceRequestAdapterCallbackDeserialize(this, transfer, deserializeBuffer, allocator, resolver);
}
WireResult ReturnInstanceRequestAdapterCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    ErrorObjectIdResolver resolver;
    return Deserialize(deserializeBuffer, allocator, resolver);
}

size_t ReturnQueueWorkDoneCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnQueueWorkDoneCallbackTransfer>() + ReturnQueueWorkDoneCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnQueueWorkDoneCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ReturnQueueWorkDoneCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnQueueWorkDoneCallbackSerialize(*this, transfer, serializeBuffer));
}
WireResult ReturnQueueWorkDoneCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider&) const {
    return Serialize(commandSize, serializeBuffer);
}

WireResult ReturnQueueWorkDoneCallbackCmd::Deserialize(DeserializeBuffer* deserializeBuffer, DeserializeAllocator* allocator) {
    const volatile ReturnQueueWorkDoneCallbackTransfer* transfer;
    WIRE_TRY(deserializeBuffer->Read(&transfer));
    return ReturnQueueWorkDoneCallbackDeserialize(this, transfer, deserializeBuffer, allocator);
}
WireResult ReturnQueueWorkDoneCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocator,
    const ObjectIdResolver&) {
    return Deserialize(deserializeBuffer, allocator);
}

size_t ReturnShaderModuleGetCompilationInfoCallbackCmd::GetRequiredSize() const {
    return WireAlignSizeof<ReturnShaderModuleGetCompilationInfoCallbackTransfer>() + ReturnShaderModuleGetCompilationInfoCallbackGetExtraRequiredSize(*this);
}

WireResult ReturnShaderModuleGetCompilationInfoCallbackCmd::Serialize(
    size_t commandSize,
    SerializeBuffer* serializeBuffer,
    const ObjectIdProvider& provider) const {
    ReturnShaderModuleGetCompilationInfoCallbackTransfer* transfer;
    WIRE_TRY(serializeBuffer->Next(&transfer));
    transfer->commandSize = commandSize;
    return (ReturnShaderModuleGetCompilationInfoCallbackSerialize(*this, transfer, serializeBuffer, provider));
}
WireResult ReturnShaderModuleGetCompilationInfoCallbackCmd::Serialize(size_t commandSize, SerializeBuffer* serializeBuffer) const {
    ErrorObjectIdProvider provider;
    return Serialize(commandSize, serializeBuffer, provider);
}

WireResult ReturnShaderModuleGetCompilationInfoCallbackCmd::Deserialize(
    DeserializeBuffer* deserializeBuffer,
    DeserializeAllocator* allocato