// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_TRACING_BACKGROUND_TRACING_RULE_H_
#define CONTENT_BROWSER_TRACING_BACKGROUND_TRACING_RULE_H_

#include <memory>
#include <optional>

#include "base/observer_list_types.h"
#include "base/timer/timer.h"
#include "content/common/content_export.h"
#include "third_party/perfetto/protos/perfetto/config/chrome/scenario_config.gen.h"
#include "third_party/perfetto/protos/perfetto/trace/chrome/chrome_metadata.pbzero.h"

namespace content {

class CONTENT_EXPORT BackgroundTracingRule : public base::CheckedObserver {
 public:
  using MetadataProto =
      perfetto::protos::pbzero::BackgroundTracingMetadata::TriggerRule;
  // Returns true if the trigger caused a scenario to either begin recording or
  // finalize the trace depending on the config.
  using RuleTriggeredCallback =
      base::RepeatingCallback<bool(const BackgroundTracingRule*)>;

  BackgroundTracingRule();

  BackgroundTracingRule(const BackgroundTracingRule&) = delete;
  BackgroundTracingRule& operator=(const BackgroundTracingRule&) = delete;

  ~BackgroundTracingRule() override;

  virtual void Install(RuleTriggeredCallback);
  virtual void Uninstall();
  virtual perfetto::protos::gen::TriggerRule ToProtoForTesting() const;
  virtual void GenerateMetadataProto(MetadataProto* out) const;

  // Probability that we should allow a tigger to  happen.
  double trigger_chance() const { return trigger_chance_; }
  std::optional<base::TimeDelta> delay() const { return delay_; }

  static std::unique_ptr<BackgroundTracingRule> Create(
      const perfetto::protos::gen::TriggerRule& config);
  static bool Append(
      const std::vector<perfetto::protos::gen::TriggerRule>& configs,
      std::vector<std::unique_ptr<BackgroundTracingRule>>& rules);

  const std::string& rule_name() const { return rule_name_; }
  std::optional<int32_t> triggered_value() const { return triggered_value_; }
  uint64_t flow_id() const { return flow_id_; }

  bool is_crash() const { return is_crash_; }

  bool OnRuleTriggered(std::optional<int32_t> value, uint64_t flow_id);

 protected:
  virtual std::string GetDefaultRuleName() const;

  virtual void DoInstall() = 0;
  virtual void DoUninstall() = 0;

  bool installed() const { return installed_; }

 private:
  void Setup(const perfetto::protos::gen::TriggerRule& config);

  RuleTriggeredCallback trigger_callback_;
  bool installed_ = false;
  double trigger_chance_ = 1.0;
  std::optional<base::TimeDelta> delay_;
  std::optional<base::TimeDelta> activation_delay_;
  base::OneShotTimer trigger_timer_;
  base::OneShotTimer activation_timer_;
  std::string rule_name_;
  std::optional<int32_t> triggered_value_;
  uint64_t flow_id_;
  bool is_crash_ = false;
};

}  // namespace content

#endif  // CONTENT_BROWSER_TRACING_BACKGROUND_TRACING_RULE_H_
