{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A type in the [type/@gLib@/.Variant] type system.
-- 
-- [type/@gLib@/.Variant] types are represented as strings, but have a strict
-- syntax described below. All [type/@gLib@/.VariantType]s passed to GLib must be
-- valid, and they are typically expected to be static (i.e. not provided by
-- user input) as they determine how binary [type/@gLib@/.Variant] data is
-- interpreted.
-- 
-- To convert a static string to a [type/@gLib@/.VariantType] in C, use the
-- @/GLib.VARIANT_TYPE/@ casting macro. When GLib is compiled with checks
-- enabled, it will validate the type. To check if an arbitrary string is a
-- valid [type/@gLib@/.VariantType], use 'GI.GLib.Structs.VariantType.variantTypeStringIsValid'.
-- 
-- == GVariant Type System
-- 
-- This section introduces the [type/@gLib@/.Variant] type system. It is based, in
-- large part, on the D-Bus type system, with two major changes and
-- some minor lifting of restrictions. The
-- <http://dbus.freedesktop.org/doc/dbus-specification.html D-Bus specification>,
-- therefore, provides a significant amount of
-- information that is useful when working with [type/@gLib@/.Variant].
-- 
-- The first major change with respect to the D-Bus type system is the
-- introduction of maybe (or ‘nullable’) types.  Any type in [type/@gLib@/.Variant]
-- can be converted to a maybe type, in which case, @nothing@ (or @null@)
-- becomes a valid value.  Maybe types have been added by introducing the
-- character @m@ to type strings.
-- 
-- The second major change is that the [type/@gLib@/.Variant] type system supports
-- the concept of ‘indefinite types’ — types that are less specific than
-- the normal types found in D-Bus.  For example, it is possible to speak
-- of ‘an array of any type’ in [type/@gLib@/.Variant], where the D-Bus type system
-- would require you to speak of ‘an array of integers’ or ‘an array of
-- strings’.  Indefinite types have been added by introducing the
-- characters @*@, @?@ and @r@ to type strings.
-- 
-- Finally, all arbitrary restrictions relating to the complexity of
-- types are lifted along with the restriction that dictionary entries
-- may only appear nested inside of arrays.
-- 
-- Just as in D-Bus, [type/@gLib@/.Variant] types are described with strings (‘type
-- strings’).  Subject to the differences mentioned above, these strings
-- are of the same form as those found in D-Bus.  Note, however: D-Bus
-- always works in terms of messages and therefore individual type
-- strings appear nowhere in its interface.  Instead, ‘signatures’
-- are a concatenation of the strings of the type of each argument in a
-- message.  [type/@gLib@/.Variant] deals with single values directly so
-- [type/@gLib@/.Variant] type strings always describe the type of exactly one
-- value.  This means that a D-Bus signature string is generally not a valid
-- [type/@gLib@/.Variant] type string — except in the case that it is the signature
-- of a message containing exactly one argument.
-- 
-- An indefinite type is similar in spirit to what may be called an
-- abstract type in other type systems.  No value can exist that has an
-- indefinite type as its type, but values can exist that have types
-- that are subtypes of indefinite types.  That is to say,
-- 'GI.GLib.Structs.Variant.variantGetType' will never return an indefinite type, but
-- calling 'GI.GLib.Structs.Variant.variantIsOfType' with an indefinite type may return
-- true.  For example, you cannot have a value that represents ‘an
-- array of no particular type’, but you can have an ‘array of integers’
-- which certainly matches the type of ‘an array of no particular type’,
-- since ‘array of integers’ is a subtype of ‘array of no particular
-- type’.
-- 
-- This is similar to how instances of abstract classes may not
-- directly exist in other type systems, but instances of their
-- non-abstract subtypes may.  For example, in GTK, no object that has
-- the type of <https://docs.gtk.org/gtk4/class.Widget.html `GtkWidget`> can
-- exist (since @GtkWidget@ is an abstract class), but a <https://docs.gtk.org/gtk4/class.Window.html `GtkWindow`>
-- can certainly be instantiated, and you would say that a @GtkWindow@ is a
-- @GtkWidget@ (since @GtkWindow@ is a subclass of @GtkWidget@).
-- 
-- Two types may not be compared by value; use 'GI.GLib.Structs.VariantType.variantTypeEqual'
-- or 'GI.GLib.Structs.VariantType.variantTypeIsSubtypeOf'  May be copied using
-- 'GI.GLib.Structs.VariantType.variantTypeCopy' and freed using 'GI.GLib.Structs.VariantType.variantTypeFree'.
-- 
-- == GVariant Type Strings
-- 
-- A [type/@gLib@/.Variant] type string can be any of the following:
-- 
-- * any basic type string (listed below)
-- * @v@, @r@ or @*@
-- * one of the characters @a@ or @m@, followed by another type string
-- * the character @(@, followed by a concatenation of zero or more other
--   type strings, followed by the character @)@
-- * the character @{@, followed by a basic type string (see below),
--   followed by another type string, followed by the character @}@
-- 
-- 
-- A basic type string describes a basic type (as per
-- 'GI.GLib.Structs.VariantType.variantTypeIsBasic') and is always a single character in
-- length. The valid basic type strings are @b@, @y@, @n@, @q@, @i@, @u@, @x@,
-- @t@, @h@, @d@, @s@, @o@, @g@ and @?@.
-- 
-- The above definition is recursive to arbitrary depth. @aaaaai@ and
-- @(ui(nq((y)))s)@ are both valid type strings, as is
-- @a(aa(ui)(qna{ya(yd)}))@. In order to not hit memory limits,
-- [type/@gLib@/.Variant] imposes a limit on recursion depth of 65 nested
-- containers. This is the limit in the D-Bus specification (64) plus one to
-- allow a <http://developer.gnome.org/glib/stable/../gio/class.DBusMessage.html `GDBusMessage`> to be nested in
-- a top-level tuple.
-- 
-- The meaning of each of the characters is as follows:
-- 
-- * @b@: the type string of @G_VARIANT_TYPE_BOOLEAN@; a boolean value.
-- * @y@: the type string of @G_VARIANT_TYPE_BYTE@; a byte.
-- * @n@: the type string of @G_VARIANT_TYPE_INT16@; a signed 16 bit integer.
-- * @q@: the type string of @G_VARIANT_TYPE_UINT16@; an unsigned 16 bit integer.
-- * @i@: the type string of @G_VARIANT_TYPE_INT32@; a signed 32 bit integer.
-- * @u@: the type string of @G_VARIANT_TYPE_UINT32@; an unsigned 32 bit integer.
-- * @x@: the type string of @G_VARIANT_TYPE_INT64@; a signed 64 bit integer.
-- * @t@: the type string of @G_VARIANT_TYPE_UINT64@; an unsigned 64 bit integer.
-- * @h@: the type string of @G_VARIANT_TYPE_HANDLE@; a signed 32 bit value
--   that, by convention, is used as an index into an array of file
--   descriptors that are sent alongside a D-Bus message.
-- * @d@: the type string of @G_VARIANT_TYPE_DOUBLE@; a double precision
--   floating point value.
-- * @s@: the type string of @G_VARIANT_TYPE_STRING@; a string.
-- * @o@: the type string of @G_VARIANT_TYPE_OBJECT_PATH@; a string in the form
--   of a D-Bus object path.
-- * @g@: the type string of @G_VARIANT_TYPE_SIGNATURE@; a string in the form of
--   a D-Bus type signature.
-- * @?@: the type string of @G_VARIANT_TYPE_BASIC@; an indefinite type that
--   is a supertype of any of the basic types.
-- * @v@: the type string of @G_VARIANT_TYPE_VARIANT@; a container type that
--   contain any other type of value.
-- * @a@: used as a prefix on another type string to mean an array of that
--   type; the type string @ai@, for example, is the type of an array of
--   signed 32-bit integers.
-- * @m@: used as a prefix on another type string to mean a ‘maybe’, or
--   ‘nullable’, version of that type; the type string @ms@, for example,
--   is the type of a value that maybe contains a string, or maybe contains
--   nothing.
-- * @()@: used to enclose zero or more other concatenated type strings to
--   create a tuple type; the type string @(is)@, for example, is the type of
--   a pair of an integer and a string.
-- * @r@: the type string of @G_VARIANT_TYPE_TUPLE@; an indefinite type that is
--   a supertype of any tuple type, regardless of the number of items.
-- * @{}@: used to enclose a basic type string concatenated with another type
--   string to create a dictionary entry type, which usually appears inside of
--   an array to form a dictionary; the type string @a{sd}@, for example, is
--   the type of a dictionary that maps strings to double precision floating
--   point values.
--   
--   The first type (the basic type) is the key type and the second type is
--   the value type. The reason that the first type is restricted to being a
--   basic type is so that it can easily be hashed.
-- * @*@: the type string of @G_VARIANT_TYPE_ANY@; the indefinite type that is
--   a supertype of all types.  Note that, as with all type strings, this
--   character represents exactly one type. It cannot be used inside of tuples
--   to mean ‘any number of items’.
-- 
-- 
-- Any type string of a container that contains an indefinite type is,
-- itself, an indefinite type. For example, the type string @a*@
-- (corresponding to @G_VARIANT_TYPE_ARRAY@) is an indefinite type
-- that is a supertype of every array type. @(*s)@ is a supertype
-- of all tuples that contain exactly two items where the second
-- item is a string.
-- 
-- @a{?*}@ is an indefinite type that is a supertype of all arrays
-- containing dictionary entries where the key is any basic type and
-- the value is any type at all.  This is, by definition, a dictionary,
-- so this type string corresponds to @G_VARIANT_TYPE_DICTIONARY@. Note
-- that, due to the restriction that the key of a dictionary entry must
-- be a basic type, @{**}@ is not a valid type string.
-- 
-- /Since: 2.24/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.VariantType
    ( 

-- * Exported types
    VariantType(..)                         ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [copy]("GI.GLib.Structs.VariantType#g:method:copy"), [dupString]("GI.GLib.Structs.VariantType#g:method:dupString"), [element]("GI.GLib.Structs.VariantType#g:method:element"), [equal]("GI.GLib.Structs.VariantType#g:method:equal"), [first]("GI.GLib.Structs.VariantType#g:method:first"), [free]("GI.GLib.Structs.VariantType#g:method:free"), [hash]("GI.GLib.Structs.VariantType#g:method:hash"), [isArray]("GI.GLib.Structs.VariantType#g:method:isArray"), [isBasic]("GI.GLib.Structs.VariantType#g:method:isBasic"), [isContainer]("GI.GLib.Structs.VariantType#g:method:isContainer"), [isDefinite]("GI.GLib.Structs.VariantType#g:method:isDefinite"), [isDictEntry]("GI.GLib.Structs.VariantType#g:method:isDictEntry"), [isMaybe]("GI.GLib.Structs.VariantType#g:method:isMaybe"), [isSubtypeOf]("GI.GLib.Structs.VariantType#g:method:isSubtypeOf"), [isTuple]("GI.GLib.Structs.VariantType#g:method:isTuple"), [isVariant]("GI.GLib.Structs.VariantType#g:method:isVariant"), [key]("GI.GLib.Structs.VariantType#g:method:key"), [nItems]("GI.GLib.Structs.VariantType#g:method:nItems"), [next]("GI.GLib.Structs.VariantType#g:method:next"), [value]("GI.GLib.Structs.VariantType#g:method:value").
-- 
-- ==== Getters
-- [getStringLength]("GI.GLib.Structs.VariantType#g:method:getStringLength").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveVariantTypeMethod                ,
#endif

-- ** checked_ #method:checked_#

    variantTypeChecked_                     ,


-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    VariantTypeCopyMethodInfo               ,
#endif
    variantTypeCopy                         ,


-- ** dupString #method:dupString#

#if defined(ENABLE_OVERLOADING)
    VariantTypeDupStringMethodInfo          ,
#endif
    variantTypeDupString                    ,


-- ** element #method:element#

#if defined(ENABLE_OVERLOADING)
    VariantTypeElementMethodInfo            ,
#endif
    variantTypeElement                      ,


-- ** equal #method:equal#

#if defined(ENABLE_OVERLOADING)
    VariantTypeEqualMethodInfo              ,
#endif
    variantTypeEqual                        ,


-- ** first #method:first#

#if defined(ENABLE_OVERLOADING)
    VariantTypeFirstMethodInfo              ,
#endif
    variantTypeFirst                        ,


-- ** free #method:free#

#if defined(ENABLE_OVERLOADING)
    VariantTypeFreeMethodInfo               ,
#endif
    variantTypeFree                         ,


-- ** getStringLength #method:getStringLength#

#if defined(ENABLE_OVERLOADING)
    VariantTypeGetStringLengthMethodInfo    ,
#endif
    variantTypeGetStringLength              ,


-- ** hash #method:hash#

#if defined(ENABLE_OVERLOADING)
    VariantTypeHashMethodInfo               ,
#endif
    variantTypeHash                         ,


-- ** isArray #method:isArray#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsArrayMethodInfo            ,
#endif
    variantTypeIsArray                      ,


-- ** isBasic #method:isBasic#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsBasicMethodInfo            ,
#endif
    variantTypeIsBasic                      ,


-- ** isContainer #method:isContainer#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsContainerMethodInfo        ,
#endif
    variantTypeIsContainer                  ,


-- ** isDefinite #method:isDefinite#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsDefiniteMethodInfo         ,
#endif
    variantTypeIsDefinite                   ,


-- ** isDictEntry #method:isDictEntry#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsDictEntryMethodInfo        ,
#endif
    variantTypeIsDictEntry                  ,


-- ** isMaybe #method:isMaybe#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsMaybeMethodInfo            ,
#endif
    variantTypeIsMaybe                      ,


-- ** isSubtypeOf #method:isSubtypeOf#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsSubtypeOfMethodInfo        ,
#endif
    variantTypeIsSubtypeOf                  ,


-- ** isTuple #method:isTuple#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsTupleMethodInfo            ,
#endif
    variantTypeIsTuple                      ,


-- ** isVariant #method:isVariant#

#if defined(ENABLE_OVERLOADING)
    VariantTypeIsVariantMethodInfo          ,
#endif
    variantTypeIsVariant                    ,


-- ** key #method:key#

#if defined(ENABLE_OVERLOADING)
    VariantTypeKeyMethodInfo                ,
#endif
    variantTypeKey                          ,


-- ** nItems #method:nItems#

#if defined(ENABLE_OVERLOADING)
    VariantTypeNItemsMethodInfo             ,
#endif
    variantTypeNItems                       ,


-- ** new #method:new#

    variantTypeNew                          ,


-- ** newArray #method:newArray#

    variantTypeNewArray                     ,


-- ** newDictEntry #method:newDictEntry#

    variantTypeNewDictEntry                 ,


-- ** newMaybe #method:newMaybe#

    variantTypeNewMaybe                     ,


-- ** newTuple #method:newTuple#

    variantTypeNewTuple                     ,


-- ** next #method:next#

#if defined(ENABLE_OVERLOADING)
    VariantTypeNextMethodInfo               ,
#endif
    variantTypeNext                         ,


-- ** stringGetDepth_ #method:stringGetDepth_#

    variantTypeStringGetDepth_              ,


-- ** stringIsValid #method:stringIsValid#

    variantTypeStringIsValid                ,


-- ** stringScan #method:stringScan#

    variantTypeStringScan                   ,


-- ** value #method:value#

#if defined(ENABLE_OVERLOADING)
    VariantTypeValueMethodInfo              ,
#endif
    variantTypeValue                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Memory-managed wrapper type.
newtype VariantType = VariantType (SP.ManagedPtr VariantType)
    deriving (Eq)

instance SP.ManagedPtrNewtype VariantType where
    toManagedPtr (VariantType p) = p

foreign import ccall "g_variant_type_get_gtype" c_g_variant_type_get_gtype :: 
    IO GType

type instance O.ParentTypes VariantType = '[]
instance O.HasParentTypes VariantType

instance B.Types.TypedObject VariantType where
    glibType = c_g_variant_type_get_gtype

instance B.Types.GBoxed VariantType

-- | Convert t'VariantType' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe VariantType) where
    gvalueGType_ = c_g_variant_type_get_gtype
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr VariantType)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr VariantType)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed VariantType ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList VariantType
type instance O.AttributeList VariantType = VariantTypeAttributeList
type VariantTypeAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method VariantType::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "type_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a valid [GVariant type string](./struct.VariantType.html#gvariant-type-strings)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_new" g_variant_type_new :: 
    CString ->                              -- type_string : TBasicType TUTF8
    IO (Ptr VariantType)

-- | Creates a new [type/@gLib@/.VariantType] corresponding to the type string given
-- by /@typeString@/.
-- 
-- It is appropriate to call 'GI.GLib.Structs.VariantType.variantTypeFree' on the return value.
-- 
-- It is a programmer error to call this function with an invalid type
-- string.  Use 'GI.GLib.Structs.VariantType.variantTypeStringIsValid' if you are unsure.
-- 
-- /Since: 2.24/
variantTypeNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@typeString@/: a valid <http://developer.gnome.org/glib/stable/./struct.VariantType.html#gvariant-type-strings GVariant type string>
    -> m VariantType
    -- ^ __Returns:__ a new [type/@gLib@/.VariantType]
variantTypeNew typeString = liftIO $ do
    typeString' <- textToCString typeString
    result <- g_variant_type_new typeString'
    checkUnexpectedReturnNULL "variantTypeNew" result
    result' <- (wrapBoxed VariantType) result
    freeMem typeString'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::new_array
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "element"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an element type" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_new_array" g_variant_type_new_array :: 
    Ptr VariantType ->                      -- element : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Constructs the type corresponding to an array of elements of the
-- type /@type@/.
-- 
-- It is appropriate to call 'GI.GLib.Structs.VariantType.variantTypeFirst' on the return value.
variantTypeNewArray ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@element@/: an element type
    -> m VariantType
    -- ^ __Returns:__ a new array type
    -- Since 2.24
variantTypeNewArray element = liftIO $ do
    element' <- unsafeManagedPtrGetPtr element
    result <- g_variant_type_new_array element'
    checkUnexpectedReturnNULL "variantTypeNewArray" result
    result' <- (wrapBoxed VariantType) result
    touchManagedPtr element
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::new_dict_entry
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "key"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a basic type to use for the key"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a type to use for the value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_new_dict_entry" g_variant_type_new_dict_entry :: 
    Ptr VariantType ->                      -- key : TInterface (Name {namespace = "GLib", name = "VariantType"})
    Ptr VariantType ->                      -- value : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Constructs the type corresponding to a dictionary entry with a key
-- of type /@key@/ and a value of type /@value@/.
-- 
-- It is appropriate to call 'GI.GLib.Structs.VariantType.variantTypeFree' on the return value.
variantTypeNewDictEntry ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@key@/: a basic type to use for the key
    -> VariantType
    -- ^ /@value@/: a type to use for the value
    -> m VariantType
    -- ^ __Returns:__ a new dictionary entry type
    -- Since 2.24
variantTypeNewDictEntry key value = liftIO $ do
    key' <- unsafeManagedPtrGetPtr key
    value' <- unsafeManagedPtrGetPtr value
    result <- g_variant_type_new_dict_entry key' value'
    checkUnexpectedReturnNULL "variantTypeNewDictEntry" result
    result' <- (wrapBoxed VariantType) result
    touchManagedPtr key
    touchManagedPtr value
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::new_maybe
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "element"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an element type" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_new_maybe" g_variant_type_new_maybe :: 
    Ptr VariantType ->                      -- element : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Constructs the type corresponding to a ‘maybe’ instance containing
-- type /@type@/ or @Nothing@.
-- 
-- It is appropriate to call 'GI.GLib.Structs.VariantType.variantTypeFree' on the return value.
variantTypeNewMaybe ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@element@/: an element type
    -> m VariantType
    -- ^ __Returns:__ a new ‘maybe’ type
    -- Since 2.24
variantTypeNewMaybe element = liftIO $ do
    element' <- unsafeManagedPtrGetPtr element
    result <- g_variant_type_new_maybe element'
    checkUnexpectedReturnNULL "variantTypeNewMaybe" result
    result' <- (wrapBoxed VariantType) result
    touchManagedPtr element
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::new_tuple
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "items"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 1
--                 (TInterface Name { namespace = "GLib" , name = "VariantType" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of types, one for each item"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @items, or `-1`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of @items, or `-1`"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_new_tuple" g_variant_type_new_tuple :: 
    Ptr (Ptr VariantType) ->                -- items : TCArray False (-1) 1 (TInterface (Name {namespace = "GLib", name = "VariantType"}))
    Int32 ->                                -- length : TBasicType TInt
    IO (Ptr VariantType)

-- | Constructs a new tuple type, from /@items@/.
-- 
-- /@length@/ is the number of items in /@items@/, or @-1@ to indicate that
-- /@items@/ is @NULL@-terminated.
-- 
-- It is appropriate to call 'GI.GLib.Structs.VariantType.variantTypeFree' on the return value.
variantTypeNewTuple ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [VariantType]
    -- ^ /@items@/: an array of types, one for each item
    -> m VariantType
    -- ^ __Returns:__ a new tuple type
    -- Since 2.24
variantTypeNewTuple items = liftIO $ do
    let length_ = fromIntegral $ P.length items
    items' <- mapM unsafeManagedPtrGetPtr items
    items'' <- packPtrArray items'
    result <- g_variant_type_new_tuple items'' length_
    checkUnexpectedReturnNULL "variantTypeNewTuple" result
    result' <- (wrapBoxed VariantType) result
    mapM_ touchManagedPtr items
    freeMem items''
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to copy" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_copy" g_variant_type_copy :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Makes a copy of a [type/@gLib@/.VariantType].
-- 
-- It is appropriate to call 'GI.GLib.Structs.VariantType.variantTypeFree' on the return value.
-- /@type@/ may not be @NULL@.
variantTypeCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to copy
    -> m VariantType
    -- ^ __Returns:__ a new [type/@gLib@/.VariantType]
    -- Since 2.24
variantTypeCopy type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_copy type_'
    checkUnexpectedReturnNULL "variantTypeCopy" result
    result' <- (wrapBoxed VariantType) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeCopyMethodInfo
instance (signature ~ (m VariantType), MonadIO m) => O.OverloadedMethod VariantTypeCopyMethodInfo VariantType signature where
    overloadedMethod = variantTypeCopy

instance O.OverloadedMethodInfo VariantTypeCopyMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeCopy"
        })


#endif

-- method VariantType::dup_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to copy" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_dup_string" g_variant_type_dup_string :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CString

-- | Returns a newly-allocated copy of the type string corresponding to /@type@/.
-- 
-- The returned string is nul-terminated.  It is appropriate to call
-- 'GI.GLib.Functions.free' on the return value.
variantTypeDupString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to copy
    -> m T.Text
    -- ^ __Returns:__ the corresponding type string
    -- Since 2.24
variantTypeDupString type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_dup_string type_'
    checkUnexpectedReturnNULL "variantTypeDupString" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeDupStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod VariantTypeDupStringMethodInfo VariantType signature where
    overloadedMethod = variantTypeDupString

instance O.OverloadedMethodInfo VariantTypeDupStringMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeDupString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeDupString"
        })


#endif

-- method VariantType::element
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array or \8216maybe\8217 type"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_element" g_variant_type_element :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Determines the element type of an array or ‘maybe’ type.
-- 
-- This function may only be used with array or ‘maybe’ types.
variantTypeElement ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: an array or ‘maybe’ type
    -> m VariantType
    -- ^ __Returns:__ the element type of /@type@/
    -- Since 2.24
variantTypeElement type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_element type_'
    checkUnexpectedReturnNULL "variantTypeElement" result
    result' <- (newBoxed VariantType) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeElementMethodInfo
instance (signature ~ (m VariantType), MonadIO m) => O.OverloadedMethod VariantTypeElementMethodInfo VariantType signature where
    overloadedMethod = variantTypeElement

instance O.OverloadedMethodInfo VariantTypeElementMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeElement",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeElement"
        })


#endif

-- method VariantType::equal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type1"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to compare" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type2"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another type to compare"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_equal" g_variant_type_equal :: 
    Ptr VariantType ->                      -- type1 : TInterface (Name {namespace = "GLib", name = "VariantType"})
    Ptr VariantType ->                      -- type2 : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Compares /@type1@/ and /@type2@/ for equality.
-- 
-- Only returns true if the types are exactly equal.  Even if one type
-- is an indefinite type and the other is a subtype of it, false will
-- be returned if they are not exactly equal.  If you want to check for
-- subtypes, use 'GI.GLib.Structs.VariantType.variantTypeIsSubtypeOf'.
-- 
-- The argument types of /@type1@/ and /@type2@/ are only @gconstpointer@ to
-- allow use with [type/@gLib@/.HashTable] without function pointer casting.  For
-- both arguments, a valid [type/@gLib@/.VariantType] must be provided.
variantTypeEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type1@/: type to compare
    -> VariantType
    -- ^ /@type2@/: another type to compare
    -> m Bool
    -- ^ __Returns:__ true if /@type1@/ and /@type2@/ are exactly equal
    -- Since 2.24
variantTypeEqual type1 type2 = liftIO $ do
    type1' <- unsafeManagedPtrGetPtr type1
    type2' <- unsafeManagedPtrGetPtr type2
    result <- g_variant_type_equal type1' type2'
    let result' = (/= 0) result
    touchManagedPtr type1
    touchManagedPtr type2
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeEqualMethodInfo
instance (signature ~ (VariantType -> m Bool), MonadIO m) => O.OverloadedMethod VariantTypeEqualMethodInfo VariantType signature where
    overloadedMethod = variantTypeEqual

instance O.OverloadedMethodInfo VariantTypeEqualMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeEqual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeEqual"
        })


#endif

-- method VariantType::first
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a tuple or dictionary entry type"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_first" g_variant_type_first :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Determines the first item type of a tuple or dictionary entry
-- type.
-- 
-- This function may only be used with tuple or dictionary entry types,
-- but must not be used with the generic tuple type
-- @G_VARIANT_TYPE_TUPLE@.
-- 
-- In the case of a dictionary entry type, this returns the type of
-- the key.
-- 
-- @NULL@ is returned in case of /@type@/ being @G_VARIANT_TYPE_UNIT@.
-- 
-- This call, together with 'GI.GLib.Structs.VariantType.variantTypeNext' provides an iterator
-- interface over tuple and dictionary entry types.
variantTypeFirst ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: a tuple or dictionary entry type
    -> m (Maybe VariantType)
    -- ^ __Returns:__ the first item type of /@type@/, or @NULL@
    --   if the type has no item types
    -- Since 2.24
variantTypeFirst type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_first type_'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed VariantType) result'
        return result''
    touchManagedPtr type_
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VariantTypeFirstMethodInfo
instance (signature ~ (m (Maybe VariantType)), MonadIO m) => O.OverloadedMethod VariantTypeFirstMethodInfo VariantType signature where
    overloadedMethod = variantTypeFirst

instance O.OverloadedMethodInfo VariantTypeFirstMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeFirst",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeFirst"
        })


#endif

-- method VariantType::free
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to free" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_free" g_variant_type_free :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO ()

-- | Frees a [type/@gLib@/.VariantType] that was allocated with
-- 'GI.GLib.Structs.VariantType.variantTypeCopy', 'GI.GLib.Structs.VariantType.variantTypeNew' or one of the
-- container type constructor functions.
-- 
-- In the case that /@type@/ is @NULL@, this function does nothing.
-- 
-- Since 2.24
variantTypeFree ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (VariantType)
    -- ^ /@type@/: type to free
    -> m ()
variantTypeFree type_ = liftIO $ do
    maybeType_ <- case type_ of
        Nothing -> return FP.nullPtr
        Just jType_ -> do
            jType_' <- unsafeManagedPtrGetPtr jType_
            return jType_'
    g_variant_type_free maybeType_
    whenJust type_ touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data VariantTypeFreeMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod VariantTypeFreeMethodInfo VariantType signature where
    overloadedMethod i = variantTypeFree (Just i)

instance O.OverloadedMethodInfo VariantTypeFreeMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeFree",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeFree"
        })


#endif

-- method VariantType::get_string_length
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to measure" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSize)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_get_string_length" g_variant_type_get_string_length :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO FCT.CSize

-- | Returns the length of the type string corresponding to the given /@type@/.
-- 
-- This function must be used to determine the valid extent of
-- the memory region returned by t'GI.GLib.Structs.VariantType.VariantType'.@/peek_string/@().
variantTypeGetStringLength ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to measure
    -> m FCT.CSize
    -- ^ __Returns:__ the length of the corresponding type string
    -- Since 2.24
variantTypeGetStringLength type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_get_string_length type_'
    touchManagedPtr type_
    return result

#if defined(ENABLE_OVERLOADING)
data VariantTypeGetStringLengthMethodInfo
instance (signature ~ (m FCT.CSize), MonadIO m) => O.OverloadedMethod VariantTypeGetStringLengthMethodInfo VariantType signature where
    overloadedMethod = variantTypeGetStringLength

instance O.OverloadedMethodInfo VariantTypeGetStringLengthMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeGetStringLength",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeGetStringLength"
        })


#endif

-- method VariantType::hash
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to hash" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_hash" g_variant_type_hash :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO Word32

-- | Hashes /@type@/.
-- 
-- The argument type of /@type@/ is only @gconstpointer@ to allow use with
-- [type/@gLib@/.HashTable] without function pointer casting.  A valid
-- [type/@gLib@/.VariantType] must be provided.
variantTypeHash ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to hash
    -> m Word32
    -- ^ __Returns:__ the hash value
    -- Since 2.24
variantTypeHash type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_hash type_'
    touchManagedPtr type_
    return result

#if defined(ENABLE_OVERLOADING)
data VariantTypeHashMethodInfo
instance (signature ~ (m Word32), MonadIO m) => O.OverloadedMethod VariantTypeHashMethodInfo VariantType signature where
    overloadedMethod = variantTypeHash

instance O.OverloadedMethodInfo VariantTypeHashMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeHash",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeHash"
        })


#endif

-- method VariantType::is_array
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_array" g_variant_type_is_array :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is an array type.
-- 
-- This is true if the type string for /@type@/ starts with an @a@.
-- 
-- This function returns true for any indefinite type for which every
-- definite subtype is an array type — @G_VARIANT_TYPE_ARRAY@, for
-- example.
variantTypeIsArray ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is an array type
    -- Since 2.24
variantTypeIsArray type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_array type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsArrayMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsArrayMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsArray

instance O.OverloadedMethodInfo VariantTypeIsArrayMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsArray",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsArray"
        })


#endif

-- method VariantType::is_basic
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_basic" g_variant_type_is_basic :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is a basic type.
-- 
-- Basic types are booleans, bytes, integers, doubles, strings, object
-- paths and signatures.
-- 
-- Only a basic type may be used as the key of a dictionary entry.
-- 
-- This function returns @FALSE@ for all indefinite types except
-- @G_VARIANT_TYPE_BASIC@.
variantTypeIsBasic ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is a basic type
    -- Since 2.24
variantTypeIsBasic type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_basic type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsBasicMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsBasicMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsBasic

instance O.OverloadedMethodInfo VariantTypeIsBasicMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsBasic",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsBasic"
        })


#endif

-- method VariantType::is_container
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_container" g_variant_type_is_container :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is a container type.
-- 
-- Container types are any array, maybe, tuple, or dictionary
-- entry types plus the variant type.
-- 
-- This function returns true for any indefinite type for which every
-- definite subtype is a container — @G_VARIANT_TYPE_ARRAY@, for
-- example.
variantTypeIsContainer ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is a container type
    -- Since 2.24
variantTypeIsContainer type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_container type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsContainerMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsContainerMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsContainer

instance O.OverloadedMethodInfo VariantTypeIsContainerMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsContainer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsContainer"
        })


#endif

-- method VariantType::is_definite
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_definite" g_variant_type_is_definite :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is definite (ie: not indefinite).
-- 
-- A type is definite if its type string does not contain any indefinite
-- type characters (@*@, @?@, or @r@).
-- 
-- A [type/@gLib@/.Variant] instance may not have an indefinite type, so calling
-- this function on the result of 'GI.GLib.Structs.Variant.variantGetType' will always
-- result in true being returned.  Calling this function on an
-- indefinite type like @G_VARIANT_TYPE_ARRAY@, however, will result in
-- @FALSE@ being returned.
variantTypeIsDefinite ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is definite
    -- Since 2.24
variantTypeIsDefinite type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_definite type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsDefiniteMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsDefiniteMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsDefinite

instance O.OverloadedMethodInfo VariantTypeIsDefiniteMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsDefinite",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsDefinite"
        })


#endif

-- method VariantType::is_dict_entry
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_dict_entry" g_variant_type_is_dict_entry :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is a dictionary entry type.
-- 
-- This is true if the type string for /@type@/ starts with a @{@.
-- 
-- This function returns true for any indefinite type for which every
-- definite subtype is a dictionary entry type —
-- @G_VARIANT_TYPE_DICT_ENTRY@, for example.
variantTypeIsDictEntry ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is a dictionary entry type
    -- Since 2.24
variantTypeIsDictEntry type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_dict_entry type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsDictEntryMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsDictEntryMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsDictEntry

instance O.OverloadedMethodInfo VariantTypeIsDictEntryMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsDictEntry",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsDictEntry"
        })


#endif

-- method VariantType::is_maybe
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_maybe" g_variant_type_is_maybe :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is a ‘maybe’ type.
-- 
-- This is true if the type string for /@type@/ starts with an @m@.
-- 
-- This function returns true for any indefinite type for which every
-- definite subtype is a ‘maybe’ type — @G_VARIANT_TYPE_MAYBE@, for
-- example.
variantTypeIsMaybe ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is a ‘maybe’ type
    -- Since 2.24
variantTypeIsMaybe type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_maybe type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsMaybeMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsMaybeMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsMaybe

instance O.OverloadedMethodInfo VariantTypeIsMaybeMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsMaybe",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsMaybe"
        })


#endif

-- method VariantType::is_subtype_of
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "supertype"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type of potential supertype"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_subtype_of" g_variant_type_is_subtype_of :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    Ptr VariantType ->                      -- supertype : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Checks if /@type@/ is a subtype of /@supertype@/.
-- 
-- This function returns true if /@type@/ is a subtype of /@supertype@/.  All
-- types are considered to be subtypes of themselves.  Aside from that,
-- only indefinite types can have subtypes.
variantTypeIsSubtypeOf ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> VariantType
    -- ^ /@supertype@/: type of potential supertype
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is a subtype of /@supertype@/
    -- Since 2.24
variantTypeIsSubtypeOf type_ supertype = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    supertype' <- unsafeManagedPtrGetPtr supertype
    result <- g_variant_type_is_subtype_of type_' supertype'
    let result' = (/= 0) result
    touchManagedPtr type_
    touchManagedPtr supertype
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsSubtypeOfMethodInfo
instance (signature ~ (VariantType -> m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsSubtypeOfMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsSubtypeOf

instance O.OverloadedMethodInfo VariantTypeIsSubtypeOfMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsSubtypeOf",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsSubtypeOf"
        })


#endif

-- method VariantType::is_tuple
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_tuple" g_variant_type_is_tuple :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is a tuple type.
-- 
-- This is true if the type string for /@type@/ starts with a @(@ or if /@type@/ is
-- @G_VARIANT_TYPE_TUPLE@.
-- 
-- This function returns true for any indefinite type for which every
-- definite subtype is a tuple type — @G_VARIANT_TYPE_TUPLE@, for
-- example.
variantTypeIsTuple ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is a tuple type
    -- Since 2.24
variantTypeIsTuple type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_tuple type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsTupleMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsTupleMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsTuple

instance O.OverloadedMethodInfo VariantTypeIsTupleMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsTuple",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsTuple"
        })


#endif

-- method VariantType::is_variant
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type to check" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_is_variant" g_variant_type_is_variant :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO CInt

-- | Determines if the given /@type@/ is the variant type.
variantTypeIsVariant ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: type to check
    -> m Bool
    -- ^ __Returns:__ true if /@type@/ is the variant type
    -- Since 2.24
variantTypeIsVariant type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_is_variant type_'
    let result' = (/= 0) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeIsVariantMethodInfo
instance (signature ~ (m Bool), MonadIO m) => O.OverloadedMethod VariantTypeIsVariantMethodInfo VariantType signature where
    overloadedMethod = variantTypeIsVariant

instance O.OverloadedMethodInfo VariantTypeIsVariantMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeIsVariant",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeIsVariant"
        })


#endif

-- method VariantType::key
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a dictionary entry type"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_key" g_variant_type_key :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Determines the key type of a dictionary entry type.
-- 
-- This function may only be used with a dictionary entry type.  Other
-- than the additional restriction, this call is equivalent to
-- 'GI.GLib.Structs.VariantType.variantTypeFirst'.
variantTypeKey ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: a dictionary entry type
    -> m VariantType
    -- ^ __Returns:__ the key type of the dictionary entry
    -- Since 2.24
variantTypeKey type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_key type_'
    checkUnexpectedReturnNULL "variantTypeKey" result
    result' <- (newBoxed VariantType) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeKeyMethodInfo
instance (signature ~ (m VariantType), MonadIO m) => O.OverloadedMethod VariantTypeKeyMethodInfo VariantType signature where
    overloadedMethod = variantTypeKey

instance O.OverloadedMethodInfo VariantTypeKeyMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeKey",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeKey"
        })


#endif

-- method VariantType::n_items
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a tuple or dictionary entry type"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSize)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_n_items" g_variant_type_n_items :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO FCT.CSize

-- | Determines the number of items contained in a tuple or
-- dictionary entry type.
-- 
-- This function may only be used with tuple or dictionary entry types,
-- but must not be used with the generic tuple type
-- @G_VARIANT_TYPE_TUPLE@.
-- 
-- In the case of a dictionary entry type, this function will always
-- return @2@.
variantTypeNItems ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: a tuple or dictionary entry type
    -> m FCT.CSize
    -- ^ __Returns:__ the number of items in /@type@/
    -- Since 2.24
variantTypeNItems type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_n_items type_'
    touchManagedPtr type_
    return result

#if defined(ENABLE_OVERLOADING)
data VariantTypeNItemsMethodInfo
instance (signature ~ (m FCT.CSize), MonadIO m) => O.OverloadedMethod VariantTypeNItemsMethodInfo VariantType signature where
    overloadedMethod = variantTypeNItems

instance O.OverloadedMethodInfo VariantTypeNItemsMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeNItems",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeNItems"
        })


#endif

-- method VariantType::next
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a type from a previous call"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_next" g_variant_type_next :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Determines the next item type of a tuple or dictionary entry
-- type.
-- 
-- /@type@/ must be the result of a previous call to
-- 'GI.GLib.Structs.VariantType.variantTypeFirst' or 'GI.GLib.Structs.VariantType.variantTypeNext'.
-- 
-- If called on the key type of a dictionary entry then this call
-- returns the value type.  If called on the value type of a dictionary
-- entry then this call returns @NULL@.
-- 
-- For tuples, @NULL@ is returned when /@type@/ is the last item in the tuple.
variantTypeNext ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: a type from a previous call
    -> m (Maybe VariantType)
    -- ^ __Returns:__ the next type after /@type@/, or @NULL@ if
    --   there are no further types
    -- Since 2.24
variantTypeNext type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_next type_'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed VariantType) result'
        return result''
    touchManagedPtr type_
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data VariantTypeNextMethodInfo
instance (signature ~ (m (Maybe VariantType)), MonadIO m) => O.OverloadedMethod VariantTypeNextMethodInfo VariantType signature where
    overloadedMethod = variantTypeNext

instance O.OverloadedMethodInfo VariantTypeNextMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeNext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeNext"
        })


#endif

-- method VariantType::value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "VariantType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a dictionary entry type"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_value" g_variant_type_value :: 
    Ptr VariantType ->                      -- type : TInterface (Name {namespace = "GLib", name = "VariantType"})
    IO (Ptr VariantType)

-- | Determines the value type of a dictionary entry type.
-- 
-- This function may only be used with a dictionary entry type.
variantTypeValue ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    VariantType
    -- ^ /@type@/: a dictionary entry type
    -> m VariantType
    -- ^ __Returns:__ the value type of the dictionary entry
    -- Since 2.24
variantTypeValue type_ = liftIO $ do
    type_' <- unsafeManagedPtrGetPtr type_
    result <- g_variant_type_value type_'
    checkUnexpectedReturnNULL "variantTypeValue" result
    result' <- (newBoxed VariantType) result
    touchManagedPtr type_
    return result'

#if defined(ENABLE_OVERLOADING)
data VariantTypeValueMethodInfo
instance (signature ~ (m VariantType), MonadIO m) => O.OverloadedMethod VariantTypeValueMethodInfo VariantType signature where
    overloadedMethod = variantTypeValue

instance O.OverloadedMethodInfo VariantTypeValueMethodInfo VariantType where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.VariantType.variantTypeValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-VariantType.html#v:variantTypeValue"
        })


#endif

-- method VariantType::checked_
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "type_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Nothing , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "VariantType" })
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_checked_" g_variant_type_checked_ :: 
    CString ->                              -- type_string : TBasicType TUTF8
    IO (Ptr VariantType)

-- | /No description available in the introspection data./
variantTypeChecked_ ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -> m VariantType
variantTypeChecked_ typeString = liftIO $ do
    typeString' <- textToCString typeString
    result <- g_variant_type_checked_ typeString'
    checkUnexpectedReturnNULL "variantTypeChecked_" result
    result' <- (newBoxed VariantType) result
    freeMem typeString'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::string_get_depth_
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "type_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Nothing , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSize)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_string_get_depth_" g_variant_type_string_get_depth_ :: 
    CString ->                              -- type_string : TBasicType TUTF8
    IO FCT.CSize

-- | /No description available in the introspection data./
variantTypeStringGetDepth_ ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -> m FCT.CSize
variantTypeStringGetDepth_ typeString = liftIO $ do
    typeString' <- textToCString typeString
    result <- g_variant_type_string_get_depth_ typeString'
    freeMem typeString'
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::string_is_valid
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "type_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to any string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_string_is_valid" g_variant_type_string_is_valid :: 
    CString ->                              -- type_string : TBasicType TUTF8
    IO CInt

-- | Checks if /@typeString@/ is a valid
-- <http://developer.gnome.org/glib/stable/./struct.VariantType.html#gvariant-type-strings GVariant type string>.
-- 
-- This call is equivalent to calling 'GI.GLib.Structs.VariantType.variantTypeStringScan' and
-- confirming that the following character is a nul terminator.
variantTypeStringIsValid ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@typeString@/: a pointer to any string
    -> m Bool
    -- ^ __Returns:__ true if /@typeString@/ is exactly one valid type string
    -- Since 2.24
variantTypeStringIsValid typeString = liftIO $ do
    typeString' <- textToCString typeString
    result <- g_variant_type_string_is_valid typeString'
    let result' = (/= 0) result
    freeMem typeString'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method VariantType::string_scan
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to any string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "limit"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the end of @string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "endptr"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the end pointer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_variant_type_string_scan" g_variant_type_string_scan :: 
    CString ->                              -- string : TBasicType TUTF8
    CString ->                              -- limit : TBasicType TUTF8
    Ptr CString ->                          -- endptr : TBasicType TUTF8
    IO CInt

-- | Scan for a single complete and valid GVariant type string in /@string@/.
-- 
-- The memory pointed to by /@limit@/ (or bytes beyond it) is never
-- accessed.
-- 
-- If a valid type string is found, /@endptr@/ is updated to point to the
-- first character past the end of the string that was found and 'P.True'
-- is returned.
-- 
-- If there is no valid type string starting at /@string@/, or if the type
-- string does not end before /@limit@/ then 'P.False' is returned.
-- 
-- For the simple case of checking if a string is a valid type string,
-- see 'GI.GLib.Structs.VariantType.variantTypeStringIsValid'.
-- 
-- /Since: 2.24/
variantTypeStringScan ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@string@/: a pointer to any string
    -> Maybe (T.Text)
    -- ^ /@limit@/: the end of /@string@/
    -> m ((Bool, T.Text))
    -- ^ __Returns:__ true if a valid type string was found
variantTypeStringScan string limit = liftIO $ do
    string' <- textToCString string
    maybeLimit <- case limit of
        Nothing -> return FP.nullPtr
        Just jLimit -> do
            jLimit' <- textToCString jLimit
            return jLimit'
    endptr <- callocMem :: IO (Ptr CString)
    result <- g_variant_type_string_scan string' maybeLimit endptr
    let result' = (/= 0) result
    endptr' <- peek endptr
    endptr'' <- cstringToText endptr'
    freeMem endptr'
    freeMem string'
    freeMem maybeLimit
    freeMem endptr
    return (result', endptr'')

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveVariantTypeMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveVariantTypeMethod "copy" o = VariantTypeCopyMethodInfo
    ResolveVariantTypeMethod "dupString" o = VariantTypeDupStringMethodInfo
    ResolveVariantTypeMethod "element" o = VariantTypeElementMethodInfo
    ResolveVariantTypeMethod "equal" o = VariantTypeEqualMethodInfo
    ResolveVariantTypeMethod "first" o = VariantTypeFirstMethodInfo
    ResolveVariantTypeMethod "free" o = VariantTypeFreeMethodInfo
    ResolveVariantTypeMethod "hash" o = VariantTypeHashMethodInfo
    ResolveVariantTypeMethod "isArray" o = VariantTypeIsArrayMethodInfo
    ResolveVariantTypeMethod "isBasic" o = VariantTypeIsBasicMethodInfo
    ResolveVariantTypeMethod "isContainer" o = VariantTypeIsContainerMethodInfo
    ResolveVariantTypeMethod "isDefinite" o = VariantTypeIsDefiniteMethodInfo
    ResolveVariantTypeMethod "isDictEntry" o = VariantTypeIsDictEntryMethodInfo
    ResolveVariantTypeMethod "isMaybe" o = VariantTypeIsMaybeMethodInfo
    ResolveVariantTypeMethod "isSubtypeOf" o = VariantTypeIsSubtypeOfMethodInfo
    ResolveVariantTypeMethod "isTuple" o = VariantTypeIsTupleMethodInfo
    ResolveVariantTypeMethod "isVariant" o = VariantTypeIsVariantMethodInfo
    ResolveVariantTypeMethod "key" o = VariantTypeKeyMethodInfo
    ResolveVariantTypeMethod "nItems" o = VariantTypeNItemsMethodInfo
    ResolveVariantTypeMethod "next" o = VariantTypeNextMethodInfo
    ResolveVariantTypeMethod "value" o = VariantTypeValueMethodInfo
    ResolveVariantTypeMethod "getStringLength" o = VariantTypeGetStringLengthMethodInfo
    ResolveVariantTypeMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveVariantTypeMethod t VariantType, O.OverloadedMethod info VariantType p) => OL.IsLabel t (VariantType -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveVariantTypeMethod t VariantType, O.OverloadedMethod info VariantType p, R.HasField t VariantType p) => R.HasField t VariantType p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveVariantTypeMethod t VariantType, O.OverloadedMethodInfo info VariantType) => OL.IsLabel t (O.MethodProxy info VariantType) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


