{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Specifies a transfer function for a color component to be applied
-- while rendering.
-- 
-- The available functions include linear, piecewise-linear,
-- gamma and step functions.
-- 
-- Note that the transfer function is applied to un-premultiplied
-- values, and all results are clamped to the [0, 1] range.
-- 
-- /Since: 4.20/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Structs.ComponentTransfer
    ( 

-- * Exported types
    ComponentTransfer(..)                   ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [copy]("GI.Gsk.Structs.ComponentTransfer#g:method:copy"), [free]("GI.Gsk.Structs.ComponentTransfer#g:method:free").
-- 
-- ==== Getters
-- /None/.
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveComponentTransferMethod          ,
#endif

-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    ComponentTransferCopyMethodInfo         ,
#endif
    componentTransferCopy                   ,


-- ** equal #method:equal#

    componentTransferEqual                  ,


-- ** free #method:free#

#if defined(ENABLE_OVERLOADING)
    ComponentTransferFreeMethodInfo         ,
#endif
    componentTransferFree                   ,


-- ** newDiscrete #method:newDiscrete#

    componentTransferNewDiscrete            ,


-- ** newGamma #method:newGamma#

    componentTransferNewGamma               ,


-- ** newIdentity #method:newIdentity#

    componentTransferNewIdentity            ,


-- ** newLevels #method:newLevels#

    componentTransferNewLevels              ,


-- ** newLinear #method:newLinear#

    componentTransferNewLinear              ,


-- ** newTable #method:newTable#

    componentTransferNewTable               ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Memory-managed wrapper type.
newtype ComponentTransfer = ComponentTransfer (SP.ManagedPtr ComponentTransfer)
    deriving (Eq)

instance SP.ManagedPtrNewtype ComponentTransfer where
    toManagedPtr (ComponentTransfer p) = p

foreign import ccall "gsk_component_transfer_get_type" c_gsk_component_transfer_get_type :: 
    IO GType

type instance O.ParentTypes ComponentTransfer = '[]
instance O.HasParentTypes ComponentTransfer

instance B.Types.TypedObject ComponentTransfer where
    glibType = c_gsk_component_transfer_get_type

instance B.Types.GBoxed ComponentTransfer

-- | Convert t'ComponentTransfer' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe ComponentTransfer) where
    gvalueGType_ = c_gsk_component_transfer_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr ComponentTransfer)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr ComponentTransfer)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed ComponentTransfer ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList ComponentTransfer
type instance O.AttributeList ComponentTransfer = ComponentTransferAttributeList
type ComponentTransferAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method ComponentTransfer::new_discrete
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "n"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Number of values" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "values"
--           , argType = TCArray False (-1) 0 (TBasicType TFloat)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Values" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n"
--              , argType = TBasicType TUInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "Number of values" , sinceVersion = Nothing }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "ComponentTransfer" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_new_discrete" gsk_component_transfer_new_discrete :: 
    Word32 ->                               -- n : TBasicType TUInt
    Ptr CFloat ->                           -- values : TCArray False (-1) 0 (TBasicType TFloat)
    IO (Ptr ComponentTransfer)

-- | Creates a new component transfer that applies
-- a step function.
-- 
-- The new value is computed as
-- 
--     C\' = values[k]
-- 
-- where k is the smallest value such that
-- 
--     k \/ n \<= C \< (k + 1) \/ n
-- 
-- \<figure>
--   \<picture>
--     \<source srcset=\"discrete-dark.png\" media=\"(prefers-color-scheme: dark)\">
--       \<img alt=\"Component transfer: discrete\" src=\"discrete-light.png\">
--   \<\/picture>
-- \<\/figure>
-- 
-- /Since: 4.20/
componentTransferNewDiscrete ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Float]
    -- ^ /@values@/: Values
    -> m ComponentTransfer
    -- ^ __Returns:__ a new @GskComponentTransfer@
componentTransferNewDiscrete values = liftIO $ do
    let n = fromIntegral $ P.length values
    values' <- (packMapStorableArray realToFrac) values
    result <- gsk_component_transfer_new_discrete n values'
    checkUnexpectedReturnNULL "componentTransferNewDiscrete" result
    result' <- (wrapBoxed ComponentTransfer) result
    freeMem values'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ComponentTransfer::new_gamma
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "amp"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Amplitude" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "exp"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Exponent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "ofs"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Offset" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "ComponentTransfer" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_new_gamma" gsk_component_transfer_new_gamma :: 
    CFloat ->                               -- amp : TBasicType TFloat
    CFloat ->                               -- exp : TBasicType TFloat
    CFloat ->                               -- ofs : TBasicType TFloat
    IO (Ptr ComponentTransfer)

-- | Creates a new component transfer that applies
-- a gamma transform.
-- 
-- The new value is computed as
-- 
--     C\' = amp * pow (C, exp) + ofs
-- 
-- \<figure>
--   \<picture>
--     \<source srcset=\"gamma-dark.png\" media=\"(prefers-color-scheme: dark)\">
--       \<img alt=\"Component transfer: gamma\" src=\"gamma-light.png\">
--   \<\/picture>
-- \<\/figure>
-- 
-- /Since: 4.20/
componentTransferNewGamma ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Float
    -- ^ /@amp@/: Amplitude
    -> Float
    -- ^ /@exp@/: Exponent
    -> Float
    -- ^ /@ofs@/: Offset
    -> m ComponentTransfer
    -- ^ __Returns:__ a new @GskComponentTransfer@
componentTransferNewGamma amp exp ofs = liftIO $ do
    let amp' = realToFrac amp
    let exp' = realToFrac exp
    let ofs' = realToFrac ofs
    result <- gsk_component_transfer_new_gamma amp' exp' ofs'
    checkUnexpectedReturnNULL "componentTransferNewGamma" result
    result' <- (wrapBoxed ComponentTransfer) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ComponentTransfer::new_identity
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "ComponentTransfer" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_new_identity" gsk_component_transfer_new_identity :: 
    IO (Ptr ComponentTransfer)

-- | Creates a new component transfer that doesn\'t
-- change the component value.
-- 
-- \<figure>
--   \<picture>
--     \<source srcset=\"identity-dark.png\" media=\"(prefers-color-scheme: dark)\">
--       \<img alt=\"Component transfer: identity\" src=\"identity-light.png\">
--   \<\/picture>
-- \<\/figure>
-- 
-- /Since: 4.20/
componentTransferNewIdentity ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m ComponentTransfer
    -- ^ __Returns:__ a new @GskComponentTransfer@
componentTransferNewIdentity  = liftIO $ do
    result <- gsk_component_transfer_new_identity
    checkUnexpectedReturnNULL "componentTransferNewIdentity" result
    result' <- (wrapBoxed ComponentTransfer) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ComponentTransfer::new_levels
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "n"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Number of levels" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "ComponentTransfer" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_new_levels" gsk_component_transfer_new_levels :: 
    CFloat ->                               -- n : TBasicType TFloat
    IO (Ptr ComponentTransfer)

-- | Creates a new component transfer that limits
-- the values of the component to @n@ levels.
-- 
-- The new value is computed as
-- 
--     C\' = (floor (C * n) + 0.5) \/ n
-- 
-- \<figure>
--   \<picture>
--     \<source srcset=\"levels-dark.png\" media=\"(prefers-color-scheme: dark)\">
--       \<img alt=\"Component transfer: levels\" src=\"levels-light.png\">
--   \<\/picture>
-- \<\/figure>
-- 
-- /Since: 4.20/
componentTransferNewLevels ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Float
    -- ^ /@n@/: Number of levels
    -> m ComponentTransfer
    -- ^ __Returns:__ a new @GskComponentTransfer@
componentTransferNewLevels n = liftIO $ do
    let n' = realToFrac n
    result <- gsk_component_transfer_new_levels n'
    checkUnexpectedReturnNULL "componentTransferNewLevels" result
    result' <- (wrapBoxed ComponentTransfer) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ComponentTransfer::new_linear
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "m"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Slope" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "b"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Offset" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "ComponentTransfer" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_new_linear" gsk_component_transfer_new_linear :: 
    CFloat ->                               -- m : TBasicType TFloat
    CFloat ->                               -- b : TBasicType TFloat
    IO (Ptr ComponentTransfer)

-- | Creates a new component transfer that applies
-- a linear transform.
-- 
-- The new value is computed as
-- 
--     C\' = C * m + b
-- 
-- \<figure>
--   \<picture>
--     \<source srcset=\"linear-dark.png\" media=\"(prefers-color-scheme: dark)\">
--       \<img alt=\"Component transfer: linear\" src=\"linear-light.png\">
--   \<\/picture>
-- \<\/figure>
-- 
-- /Since: 4.20/
componentTransferNewLinear ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Float
    -- ^ /@m@/: Slope
    -> Float
    -- ^ /@b@/: Offset
    -> m ComponentTransfer
    -- ^ __Returns:__ a new @GskComponentTransfer@
componentTransferNewLinear m b = liftIO $ do
    let m' = realToFrac m
    let b' = realToFrac b
    result <- gsk_component_transfer_new_linear m' b'
    checkUnexpectedReturnNULL "componentTransferNewLinear" result
    result' <- (wrapBoxed ComponentTransfer) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ComponentTransfer::new_table
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "n"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Number of values" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "values"
--           , argType = TCArray False (-1) 0 (TBasicType TFloat)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Values" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n"
--              , argType = TBasicType TUInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "Number of values" , sinceVersion = Nothing }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "ComponentTransfer" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_new_table" gsk_component_transfer_new_table :: 
    Word32 ->                               -- n : TBasicType TUInt
    Ptr CFloat ->                           -- values : TCArray False (-1) 0 (TBasicType TFloat)
    IO (Ptr ComponentTransfer)

-- | Creates a new component transfer that applies
-- a piecewise linear function.
-- 
-- The new value is computed as
-- 
--     C\' = values[k] + (C - k \/ (n - 1)) * n * (values[k + 1] - values[k])
-- 
-- where k is the smallest value such that
-- 
--     k \/ (n - 1) \<= C \< (k + 1) \/ (n - 1)
-- 
-- \<figure>
--   \<picture>
--     \<source srcset=\"table-dark.png\" media=\"(prefers-color-scheme: dark)\">
--       \<img alt=\"Component transfer: table\" src=\"table-light.png\">
--   \<\/picture>
-- \<\/figure>
-- 
-- /Since: 4.20/
componentTransferNewTable ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Float]
    -- ^ /@values@/: Values
    -> m ComponentTransfer
    -- ^ __Returns:__ a new @GskComponentTransfer@
componentTransferNewTable values = liftIO $ do
    let n = fromIntegral $ P.length values
    values' <- (packMapStorableArray realToFrac) values
    result <- gsk_component_transfer_new_table n values'
    checkUnexpectedReturnNULL "componentTransferNewTable" result
    result' <- (wrapBoxed ComponentTransfer) result
    freeMem values'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method ComponentTransfer::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "other"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "ComponentTransfer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a component transfer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "ComponentTransfer" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_copy" gsk_component_transfer_copy :: 
    Ptr ComponentTransfer ->                -- other : TInterface (Name {namespace = "Gsk", name = "ComponentTransfer"})
    IO (Ptr ComponentTransfer)

-- | Creates a copy of /@other@/.
-- 
-- /Since: 4.20/
componentTransferCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ComponentTransfer
    -- ^ /@other@/: a component transfer
    -> m ComponentTransfer
    -- ^ __Returns:__ a newly allocated copy of /@other@/
componentTransferCopy other = liftIO $ do
    other' <- unsafeManagedPtrGetPtr other
    result <- gsk_component_transfer_copy other'
    checkUnexpectedReturnNULL "componentTransferCopy" result
    result' <- (wrapBoxed ComponentTransfer) result
    touchManagedPtr other
    return result'

#if defined(ENABLE_OVERLOADING)
data ComponentTransferCopyMethodInfo
instance (signature ~ (m ComponentTransfer), MonadIO m) => O.OverloadedMethod ComponentTransferCopyMethodInfo ComponentTransfer signature where
    overloadedMethod = componentTransferCopy

instance O.OverloadedMethodInfo ComponentTransferCopyMethodInfo ComponentTransfer where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.ComponentTransfer.componentTransferCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-ComponentTransfer.html#v:componentTransferCopy"
        })


#endif

-- method ComponentTransfer::free
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "ComponentTransfer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a component transfer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_free" gsk_component_transfer_free :: 
    Ptr ComponentTransfer ->                -- self : TInterface (Name {namespace = "Gsk", name = "ComponentTransfer"})
    IO ()

-- | Frees a component transfer.
-- 
-- /Since: 4.20/
componentTransferFree ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ComponentTransfer
    -- ^ /@self@/: a component transfer
    -> m ()
componentTransferFree self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    gsk_component_transfer_free self'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data ComponentTransferFreeMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod ComponentTransferFreeMethodInfo ComponentTransfer signature where
    overloadedMethod = componentTransferFree

instance O.OverloadedMethodInfo ComponentTransferFreeMethodInfo ComponentTransfer where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.ComponentTransfer.componentTransferFree",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-ComponentTransfer.html#v:componentTransferFree"
        })


#endif

-- method ComponentTransfer::equal
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a component transfer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "other"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another component transfer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_component_transfer_equal" gsk_component_transfer_equal :: 
    Ptr () ->                               -- self : TBasicType TPtr
    Ptr () ->                               -- other : TBasicType TPtr
    IO CInt

-- | Compares two component transfers for equality.
-- 
-- /Since: 4.20/
componentTransferEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Ptr ()
    -- ^ /@self@/: a component transfer
    -> Ptr ()
    -- ^ /@other@/: another component transfer
    -> m Bool
    -- ^ __Returns:__ true if /@self@/ and /@other@/ are equal
componentTransferEqual self other = liftIO $ do
    result <- gsk_component_transfer_equal self other
    let result' = (/= 0) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveComponentTransferMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveComponentTransferMethod "copy" o = ComponentTransferCopyMethodInfo
    ResolveComponentTransferMethod "free" o = ComponentTransferFreeMethodInfo
    ResolveComponentTransferMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveComponentTransferMethod t ComponentTransfer, O.OverloadedMethod info ComponentTransfer p) => OL.IsLabel t (ComponentTransfer -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveComponentTransferMethod t ComponentTransfer, O.OverloadedMethod info ComponentTransfer p, R.HasField t ComponentTransfer p) => R.HasField t ComponentTransfer p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveComponentTransferMethod t ComponentTransfer, O.OverloadedMethodInfo info ComponentTransfer) => OL.IsLabel t (O.MethodProxy info ComponentTransfer) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


