{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The base class for gesture recognition.
-- 
-- Although @GtkGesture@ is quite generalized to serve as a base for
-- multi-touch gestures, it is suitable to implement single-touch and
-- pointer-based gestures (using the special 'P.Nothing' @GdkEventSequence@
-- value for these).
-- 
-- The number of touches that a @GtkGesture@ need to be recognized is
-- controlled by the [Gesture:nPoints]("GI.Gtk.Objects.Gesture#g:attr:nPoints") property, if a
-- gesture is keeping track of less or more than that number of sequences,
-- it won\'t check whether the gesture is recognized.
-- 
-- As soon as the gesture has the expected number of touches, it will check
-- regularly if it is recognized, the criteria to consider a gesture as
-- \"recognized\" is left to @GtkGesture@ subclasses.
-- 
-- A recognized gesture will then emit the following signals:
-- 
-- * [Gesture::begin]("GI.Gtk.Objects.Gesture#g:signal:begin") when the gesture is recognized.
-- * [Gesture::update]("GI.Gtk.Objects.Gesture#g:signal:update"), whenever an input event is processed.
-- * [Gesture::end]("GI.Gtk.Objects.Gesture#g:signal:end") when the gesture is no longer recognized.
-- 
-- 
-- == Event propagation
-- 
-- In order to receive events, a gesture needs to set a propagation phase
-- through 'GI.Gtk.Objects.EventController.eventControllerSetPropagationPhase'.
-- 
-- In the capture phase, events are propagated from the toplevel down
-- to the target widget, and gestures that are attached to containers
-- above the widget get a chance to interact with the event before it
-- reaches the target.
-- 
-- In the bubble phase, events are propagated up from the target widget
-- to the toplevel, and gestures that are attached to containers above
-- the widget get a chance to interact with events that have not been
-- handled yet.
-- 
-- == States of a sequence
-- 
-- Whenever input interaction happens, a single event may trigger a cascade
-- of @GtkGesture@s, both across the parents of the widget receiving the
-- event and in parallel within an individual widget. It is a responsibility
-- of the widgets using those gestures to set the state of touch sequences
-- accordingly in order to enable cooperation of gestures around the
-- @GdkEventSequence@s triggering those.
-- 
-- Within a widget, gestures can be grouped through 'GI.Gtk.Objects.Gesture.gestureGroup'.
-- Grouped gestures synchronize the state of sequences, so calling
-- 'GI.Gtk.Objects.Gesture.gestureSetState' on one will effectively propagate
-- the state throughout the group.
-- 
-- By default, all sequences start out in the 'GI.Gtk.Enums.EventSequenceStateNone' state,
-- sequences in this state trigger the gesture event handler, but event
-- propagation will continue unstopped by gestures.
-- 
-- If a sequence enters into the 'GI.Gtk.Enums.EventSequenceStateDenied' state, the gesture
-- group will effectively ignore the sequence, letting events go unstopped
-- through the gesture, but the \"slot\" will still remain occupied while
-- the touch is active.
-- 
-- If a sequence enters in the 'GI.Gtk.Enums.EventSequenceStateClaimed' state, the gesture
-- group will grab all interaction on the sequence, by:
-- 
-- * Setting the same sequence to 'GI.Gtk.Enums.EventSequenceStateDenied' on every other
--   gesture group within the widget, and every gesture on parent widgets
--   in the propagation chain.
-- * Emitting [Gesture::cancel]("GI.Gtk.Objects.Gesture#g:signal:cancel") on every gesture in widgets
--   underneath in the propagation chain.
-- * Stopping event propagation after the gesture group handles the event.
-- 
-- 
-- Note: if a sequence is set early to 'GI.Gtk.Enums.EventSequenceStateClaimed' on
-- 'GI.Gdk.Enums.EventTypeTouchBegin'\/'GI.Gdk.Enums.EventTypeButtonPress' (so those events are captured before
-- reaching the event widget, this implies 'GI.Gtk.Enums.PropagationPhaseCapture'), one similar
-- event will be emulated if the sequence changes to 'GI.Gtk.Enums.EventSequenceStateDenied'.
-- This way event coherence is preserved before event propagation is unstopped
-- again.
-- 
-- Sequence states can\'t be changed freely.
-- See 'GI.Gtk.Objects.Gesture.gestureSetState' to know about the possible
-- lifetimes of a @GdkEventSequence@.
-- 
-- == Touchpad gestures
-- 
-- On the platforms that support it, @GtkGesture@ will handle transparently
-- touchpad gesture events. The only precautions users of @GtkGesture@ should
-- do to enable this support are:
-- 
-- * If the gesture has 'GI.Gtk.Enums.PropagationPhaseNone', ensuring events of type
--   'GI.Gdk.Enums.EventTypeTouchpadSwipe' and 'GI.Gdk.Enums.EventTypeTouchpadPinch' are handled by the @GtkGesture@
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.Gesture
    ( 

-- * Exported types
    Gesture(..)                             ,
    IsGesture                               ,
    toGesture                               ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [group]("GI.Gtk.Objects.Gesture#g:method:group"), [handlesSequence]("GI.Gtk.Objects.Gesture#g:method:handlesSequence"), [isActive]("GI.Gtk.Objects.Gesture#g:method:isActive"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isGroupedWith]("GI.Gtk.Objects.Gesture#g:method:isGroupedWith"), [isRecognized]("GI.Gtk.Objects.Gesture#g:method:isRecognized"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [reset]("GI.Gtk.Objects.EventController#g:method:reset"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [ungroup]("GI.Gtk.Objects.Gesture#g:method:ungroup"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getBoundingBox]("GI.Gtk.Objects.Gesture#g:method:getBoundingBox"), [getBoundingBoxCenter]("GI.Gtk.Objects.Gesture#g:method:getBoundingBoxCenter"), [getCurrentEvent]("GI.Gtk.Objects.EventController#g:method:getCurrentEvent"), [getCurrentEventDevice]("GI.Gtk.Objects.EventController#g:method:getCurrentEventDevice"), [getCurrentEventState]("GI.Gtk.Objects.EventController#g:method:getCurrentEventState"), [getCurrentEventTime]("GI.Gtk.Objects.EventController#g:method:getCurrentEventTime"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDevice]("GI.Gtk.Objects.Gesture#g:method:getDevice"), [getGroup]("GI.Gtk.Objects.Gesture#g:method:getGroup"), [getLastEvent]("GI.Gtk.Objects.Gesture#g:method:getLastEvent"), [getLastUpdatedSequence]("GI.Gtk.Objects.Gesture#g:method:getLastUpdatedSequence"), [getName]("GI.Gtk.Objects.EventController#g:method:getName"), [getPoint]("GI.Gtk.Objects.Gesture#g:method:getPoint"), [getPropagationLimit]("GI.Gtk.Objects.EventController#g:method:getPropagationLimit"), [getPropagationPhase]("GI.Gtk.Objects.EventController#g:method:getPropagationPhase"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getSequenceState]("GI.Gtk.Objects.Gesture#g:method:getSequenceState"), [getSequences]("GI.Gtk.Objects.Gesture#g:method:getSequences"), [getWidget]("GI.Gtk.Objects.EventController#g:method:getWidget").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setName]("GI.Gtk.Objects.EventController#g:method:setName"), [setPropagationLimit]("GI.Gtk.Objects.EventController#g:method:setPropagationLimit"), [setPropagationPhase]("GI.Gtk.Objects.EventController#g:method:setPropagationPhase"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setSequenceState]("GI.Gtk.Objects.Gesture#g:method:setSequenceState"), [setState]("GI.Gtk.Objects.Gesture#g:method:setState"), [setStaticName]("GI.Gtk.Objects.EventController#g:method:setStaticName").

#if defined(ENABLE_OVERLOADING)
    ResolveGestureMethod                    ,
#endif

-- ** getBoundingBox #method:getBoundingBox#

#if defined(ENABLE_OVERLOADING)
    GestureGetBoundingBoxMethodInfo         ,
#endif
    gestureGetBoundingBox                   ,


-- ** getBoundingBoxCenter #method:getBoundingBoxCenter#

#if defined(ENABLE_OVERLOADING)
    GestureGetBoundingBoxCenterMethodInfo   ,
#endif
    gestureGetBoundingBoxCenter             ,


-- ** getDevice #method:getDevice#

#if defined(ENABLE_OVERLOADING)
    GestureGetDeviceMethodInfo              ,
#endif
    gestureGetDevice                        ,


-- ** getGroup #method:getGroup#

#if defined(ENABLE_OVERLOADING)
    GestureGetGroupMethodInfo               ,
#endif
    gestureGetGroup                         ,


-- ** getLastEvent #method:getLastEvent#

#if defined(ENABLE_OVERLOADING)
    GestureGetLastEventMethodInfo           ,
#endif
    gestureGetLastEvent                     ,


-- ** getLastUpdatedSequence #method:getLastUpdatedSequence#

#if defined(ENABLE_OVERLOADING)
    GestureGetLastUpdatedSequenceMethodInfo ,
#endif
    gestureGetLastUpdatedSequence           ,


-- ** getPoint #method:getPoint#

#if defined(ENABLE_OVERLOADING)
    GestureGetPointMethodInfo               ,
#endif
    gestureGetPoint                         ,


-- ** getSequenceState #method:getSequenceState#

#if defined(ENABLE_OVERLOADING)
    GestureGetSequenceStateMethodInfo       ,
#endif
    gestureGetSequenceState                 ,


-- ** getSequences #method:getSequences#

#if defined(ENABLE_OVERLOADING)
    GestureGetSequencesMethodInfo           ,
#endif
    gestureGetSequences                     ,


-- ** group #method:group#

#if defined(ENABLE_OVERLOADING)
    GestureGroupMethodInfo                  ,
#endif
    gestureGroup                            ,


-- ** handlesSequence #method:handlesSequence#

#if defined(ENABLE_OVERLOADING)
    GestureHandlesSequenceMethodInfo        ,
#endif
    gestureHandlesSequence                  ,


-- ** isActive #method:isActive#

#if defined(ENABLE_OVERLOADING)
    GestureIsActiveMethodInfo               ,
#endif
    gestureIsActive                         ,


-- ** isGroupedWith #method:isGroupedWith#

#if defined(ENABLE_OVERLOADING)
    GestureIsGroupedWithMethodInfo          ,
#endif
    gestureIsGroupedWith                    ,


-- ** isRecognized #method:isRecognized#

#if defined(ENABLE_OVERLOADING)
    GestureIsRecognizedMethodInfo           ,
#endif
    gestureIsRecognized                     ,


-- ** setSequenceState #method:setSequenceState#

#if defined(ENABLE_OVERLOADING)
    GestureSetSequenceStateMethodInfo       ,
#endif
    gestureSetSequenceState                 ,


-- ** setState #method:setState#

#if defined(ENABLE_OVERLOADING)
    GestureSetStateMethodInfo               ,
#endif
    gestureSetState                         ,


-- ** ungroup #method:ungroup#

#if defined(ENABLE_OVERLOADING)
    GestureUngroupMethodInfo                ,
#endif
    gestureUngroup                          ,




 -- * Properties


-- ** nPoints #attr:nPoints#
-- | The number of touch points that trigger
-- recognition on this gesture.

#if defined(ENABLE_OVERLOADING)
    GestureNPointsPropertyInfo              ,
#endif
    constructGestureNPoints                 ,
#if defined(ENABLE_OVERLOADING)
    gestureNPoints                          ,
#endif
    getGestureNPoints                       ,




 -- * Signals


-- ** begin #signal:begin#

    GestureBeginCallback                    ,
#if defined(ENABLE_OVERLOADING)
    GestureBeginSignalInfo                  ,
#endif
    afterGestureBegin                       ,
    onGestureBegin                          ,


-- ** cancel #signal:cancel#

    GestureCancelCallback                   ,
#if defined(ENABLE_OVERLOADING)
    GestureCancelSignalInfo                 ,
#endif
    afterGestureCancel                      ,
    onGestureCancel                         ,


-- ** end #signal:end#

    GestureEndCallback                      ,
#if defined(ENABLE_OVERLOADING)
    GestureEndSignalInfo                    ,
#endif
    afterGestureEnd                         ,
    onGestureEnd                            ,


-- ** sequenceStateChanged #signal:sequenceStateChanged#

    GestureSequenceStateChangedCallback     ,
#if defined(ENABLE_OVERLOADING)
    GestureSequenceStateChangedSignalInfo   ,
#endif
    afterGestureSequenceStateChanged        ,
    onGestureSequenceStateChanged           ,


-- ** update #signal:update#

    GestureUpdateCallback                   ,
#if defined(ENABLE_OVERLOADING)
    GestureUpdateSignalInfo                 ,
#endif
    afterGestureUpdate                      ,
    onGestureUpdate                         ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.EventSequence as Gdk.EventSequence
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.ComponentTransfer as Gsk.ComponentTransfer
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Structs.EventSequence as Gdk.EventSequence
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController

#endif

-- | Memory-managed wrapper type.
newtype Gesture = Gesture (SP.ManagedPtr Gesture)
    deriving (Eq)

instance SP.ManagedPtrNewtype Gesture where
    toManagedPtr (Gesture p) = p

foreign import ccall "gtk_gesture_get_type"
    c_gtk_gesture_get_type :: IO B.Types.GType

instance B.Types.TypedObject Gesture where
    glibType = c_gtk_gesture_get_type

instance B.Types.GObject Gesture

-- | Type class for types which can be safely cast to t'Gesture', for instance with `toGesture`.
class (SP.GObject o, O.IsDescendantOf Gesture o) => IsGesture o
instance (SP.GObject o, O.IsDescendantOf Gesture o) => IsGesture o

instance O.HasParentTypes Gesture
type instance O.ParentTypes Gesture = '[Gtk.EventController.EventController, GObject.Object.Object]

-- | Cast to t'Gesture', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toGesture :: (MIO.MonadIO m, IsGesture o) => o -> m Gesture
toGesture = MIO.liftIO . B.ManagedPtr.unsafeCastTo Gesture

-- | Convert t'Gesture' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Gesture) where
    gvalueGType_ = c_gtk_gesture_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Gesture)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Gesture)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Gesture ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveGestureMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveGestureMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveGestureMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveGestureMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveGestureMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveGestureMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveGestureMethod "group" o = GestureGroupMethodInfo
    ResolveGestureMethod "handlesSequence" o = GestureHandlesSequenceMethodInfo
    ResolveGestureMethod "isActive" o = GestureIsActiveMethodInfo
    ResolveGestureMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveGestureMethod "isGroupedWith" o = GestureIsGroupedWithMethodInfo
    ResolveGestureMethod "isRecognized" o = GestureIsRecognizedMethodInfo
    ResolveGestureMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveGestureMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveGestureMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveGestureMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveGestureMethod "reset" o = Gtk.EventController.EventControllerResetMethodInfo
    ResolveGestureMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveGestureMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveGestureMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveGestureMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveGestureMethod "ungroup" o = GestureUngroupMethodInfo
    ResolveGestureMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveGestureMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveGestureMethod "getBoundingBox" o = GestureGetBoundingBoxMethodInfo
    ResolveGestureMethod "getBoundingBoxCenter" o = GestureGetBoundingBoxCenterMethodInfo
    ResolveGestureMethod "getCurrentEvent" o = Gtk.EventController.EventControllerGetCurrentEventMethodInfo
    ResolveGestureMethod "getCurrentEventDevice" o = Gtk.EventController.EventControllerGetCurrentEventDeviceMethodInfo
    ResolveGestureMethod "getCurrentEventState" o = Gtk.EventController.EventControllerGetCurrentEventStateMethodInfo
    ResolveGestureMethod "getCurrentEventTime" o = Gtk.EventController.EventControllerGetCurrentEventTimeMethodInfo
    ResolveGestureMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveGestureMethod "getDevice" o = GestureGetDeviceMethodInfo
    ResolveGestureMethod "getGroup" o = GestureGetGroupMethodInfo
    ResolveGestureMethod "getLastEvent" o = GestureGetLastEventMethodInfo
    ResolveGestureMethod "getLastUpdatedSequence" o = GestureGetLastUpdatedSequenceMethodInfo
    ResolveGestureMethod "getName" o = Gtk.EventController.EventControllerGetNameMethodInfo
    ResolveGestureMethod "getPoint" o = GestureGetPointMethodInfo
    ResolveGestureMethod "getPropagationLimit" o = Gtk.EventController.EventControllerGetPropagationLimitMethodInfo
    ResolveGestureMethod "getPropagationPhase" o = Gtk.EventController.EventControllerGetPropagationPhaseMethodInfo
    ResolveGestureMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveGestureMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveGestureMethod "getSequenceState" o = GestureGetSequenceStateMethodInfo
    ResolveGestureMethod "getSequences" o = GestureGetSequencesMethodInfo
    ResolveGestureMethod "getWidget" o = Gtk.EventController.EventControllerGetWidgetMethodInfo
    ResolveGestureMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveGestureMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveGestureMethod "setName" o = Gtk.EventController.EventControllerSetNameMethodInfo
    ResolveGestureMethod "setPropagationLimit" o = Gtk.EventController.EventControllerSetPropagationLimitMethodInfo
    ResolveGestureMethod "setPropagationPhase" o = Gtk.EventController.EventControllerSetPropagationPhaseMethodInfo
    ResolveGestureMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveGestureMethod "setSequenceState" o = GestureSetSequenceStateMethodInfo
    ResolveGestureMethod "setState" o = GestureSetStateMethodInfo
    ResolveGestureMethod "setStaticName" o = Gtk.EventController.EventControllerSetStaticNameMethodInfo
    ResolveGestureMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveGestureMethod t Gesture, O.OverloadedMethod info Gesture p) => OL.IsLabel t (Gesture -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveGestureMethod t Gesture, O.OverloadedMethod info Gesture p, R.HasField t Gesture p) => R.HasField t Gesture p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveGestureMethod t Gesture, O.OverloadedMethodInfo info Gesture) => OL.IsLabel t (O.MethodProxy info Gesture) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Gesture::begin
-- | Emitted when the gesture is recognized.
-- 
-- This means the number of touch sequences matches
-- [Gesture:nPoints]("GI.Gtk.Objects.Gesture#g:attr:nPoints").
-- 
-- Note: These conditions may also happen when an extra touch
-- (eg. a third touch on a 2-touches gesture) is lifted, in that
-- situation /@sequence@/ won\'t pertain to the current set of active
-- touches, so don\'t rely on this being true.
type GestureBeginCallback =
    Maybe Gdk.EventSequence.EventSequence
    -- ^ /@sequence@/: the @GdkEventSequence@ that made the gesture
    --   to be recognized
    -> IO ()

type C_GestureBeginCallback =
    Ptr Gesture ->                          -- object
    Ptr Gdk.EventSequence.EventSequence ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_GestureBeginCallback`.
foreign import ccall "wrapper"
    mk_GestureBeginCallback :: C_GestureBeginCallback -> IO (FunPtr C_GestureBeginCallback)

wrap_GestureBeginCallback :: 
    GObject a => (a -> GestureBeginCallback) ->
    C_GestureBeginCallback
wrap_GestureBeginCallback gi'cb gi'selfPtr sequence _ = do
    maybeSequence <-
        if sequence == FP.nullPtr
        then return Nothing
        else do
            B.ManagedPtr.withTransient  sequence $ \sequence' -> do
                return $ Just sequence'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeSequence


-- | Connect a signal handler for the [begin](#signal:begin) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gesture #begin callback
-- @
-- 
-- 
onGestureBegin :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureBeginCallback) -> m SignalHandlerId
onGestureBegin obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureBeginCallback wrapped
    wrapped'' <- mk_GestureBeginCallback wrapped'
    connectSignalFunPtr obj "begin" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [begin](#signal:begin) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gesture #begin callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGestureBegin :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureBeginCallback) -> m SignalHandlerId
afterGestureBegin obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureBeginCallback wrapped
    wrapped'' <- mk_GestureBeginCallback wrapped'
    connectSignalFunPtr obj "begin" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GestureBeginSignalInfo
instance SignalInfo GestureBeginSignalInfo where
    type HaskellCallbackType GestureBeginSignalInfo = GestureBeginCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GestureBeginCallback cb
        cb'' <- mk_GestureBeginCallback cb'
        connectSignalFunPtr obj "begin" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture::begin"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#g:signal:begin"})

#endif

-- signal Gesture::cancel
-- | Emitted whenever a sequence is cancelled.
-- 
-- This usually happens on active touches when
-- 'GI.Gtk.Objects.EventController.eventControllerReset' is called on /@gesture@/
-- (manually, due to grabs...), or the individual /@sequence@/
-- was claimed by parent widgets\' controllers (see
-- 'GI.Gtk.Objects.Gesture.gestureSetSequenceState').
-- 
-- /@gesture@/ must forget everything about /@sequence@/ as in
-- response to this signal.
type GestureCancelCallback =
    Maybe Gdk.EventSequence.EventSequence
    -- ^ /@sequence@/: the @GdkEventSequence@ that was cancelled
    -> IO ()

type C_GestureCancelCallback =
    Ptr Gesture ->                          -- object
    Ptr Gdk.EventSequence.EventSequence ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_GestureCancelCallback`.
foreign import ccall "wrapper"
    mk_GestureCancelCallback :: C_GestureCancelCallback -> IO (FunPtr C_GestureCancelCallback)

wrap_GestureCancelCallback :: 
    GObject a => (a -> GestureCancelCallback) ->
    C_GestureCancelCallback
wrap_GestureCancelCallback gi'cb gi'selfPtr sequence _ = do
    maybeSequence <-
        if sequence == FP.nullPtr
        then return Nothing
        else do
            B.ManagedPtr.withTransient  sequence $ \sequence' -> do
                return $ Just sequence'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeSequence


-- | Connect a signal handler for the [cancel](#signal:cancel) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gesture #cancel callback
-- @
-- 
-- 
onGestureCancel :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureCancelCallback) -> m SignalHandlerId
onGestureCancel obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureCancelCallback wrapped
    wrapped'' <- mk_GestureCancelCallback wrapped'
    connectSignalFunPtr obj "cancel" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [cancel](#signal:cancel) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gesture #cancel callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGestureCancel :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureCancelCallback) -> m SignalHandlerId
afterGestureCancel obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureCancelCallback wrapped
    wrapped'' <- mk_GestureCancelCallback wrapped'
    connectSignalFunPtr obj "cancel" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GestureCancelSignalInfo
instance SignalInfo GestureCancelSignalInfo where
    type HaskellCallbackType GestureCancelSignalInfo = GestureCancelCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GestureCancelCallback cb
        cb'' <- mk_GestureCancelCallback cb'
        connectSignalFunPtr obj "cancel" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture::cancel"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#g:signal:cancel"})

#endif

-- signal Gesture::end
-- | Emitted when /@gesture@/ either stopped recognizing the event
-- sequences as something to be handled, or the number of touch
-- sequences became higher or lower than [Gesture:nPoints]("GI.Gtk.Objects.Gesture#g:attr:nPoints").
-- 
-- Note: /@sequence@/ might not pertain to the group of sequences that
-- were previously triggering recognition on /@gesture@/ (ie. a just
-- pressed touch sequence that exceeds [Gesture:nPoints]("GI.Gtk.Objects.Gesture#g:attr:nPoints")).
-- This situation may be detected by checking through
-- 'GI.Gtk.Objects.Gesture.gestureHandlesSequence'.
type GestureEndCallback =
    Maybe Gdk.EventSequence.EventSequence
    -- ^ /@sequence@/: the @GdkEventSequence@ that made gesture
    --   recognition to finish
    -> IO ()

type C_GestureEndCallback =
    Ptr Gesture ->                          -- object
    Ptr Gdk.EventSequence.EventSequence ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_GestureEndCallback`.
foreign import ccall "wrapper"
    mk_GestureEndCallback :: C_GestureEndCallback -> IO (FunPtr C_GestureEndCallback)

wrap_GestureEndCallback :: 
    GObject a => (a -> GestureEndCallback) ->
    C_GestureEndCallback
wrap_GestureEndCallback gi'cb gi'selfPtr sequence _ = do
    maybeSequence <-
        if sequence == FP.nullPtr
        then return Nothing
        else do
            B.ManagedPtr.withTransient  sequence $ \sequence' -> do
                return $ Just sequence'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeSequence


-- | Connect a signal handler for the [end](#signal:end) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gesture #end callback
-- @
-- 
-- 
onGestureEnd :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureEndCallback) -> m SignalHandlerId
onGestureEnd obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureEndCallback wrapped
    wrapped'' <- mk_GestureEndCallback wrapped'
    connectSignalFunPtr obj "end" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [end](#signal:end) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gesture #end callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGestureEnd :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureEndCallback) -> m SignalHandlerId
afterGestureEnd obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureEndCallback wrapped
    wrapped'' <- mk_GestureEndCallback wrapped'
    connectSignalFunPtr obj "end" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GestureEndSignalInfo
instance SignalInfo GestureEndSignalInfo where
    type HaskellCallbackType GestureEndSignalInfo = GestureEndCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GestureEndCallback cb
        cb'' <- mk_GestureEndCallback cb'
        connectSignalFunPtr obj "end" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture::end"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#g:signal:end"})

#endif

-- signal Gesture::sequence-state-changed
-- | Emitted whenever a sequence state changes.
-- 
-- See 'GI.Gtk.Objects.Gesture.gestureSetSequenceState' to know
-- more about the expectable sequence lifetimes.
type GestureSequenceStateChangedCallback =
    Maybe Gdk.EventSequence.EventSequence
    -- ^ /@sequence@/: the @GdkEventSequence@ that was cancelled
    -> Gtk.Enums.EventSequenceState
    -- ^ /@state@/: the new sequence state
    -> IO ()

type C_GestureSequenceStateChangedCallback =
    Ptr Gesture ->                          -- object
    Ptr Gdk.EventSequence.EventSequence ->
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_GestureSequenceStateChangedCallback`.
foreign import ccall "wrapper"
    mk_GestureSequenceStateChangedCallback :: C_GestureSequenceStateChangedCallback -> IO (FunPtr C_GestureSequenceStateChangedCallback)

wrap_GestureSequenceStateChangedCallback :: 
    GObject a => (a -> GestureSequenceStateChangedCallback) ->
    C_GestureSequenceStateChangedCallback
wrap_GestureSequenceStateChangedCallback gi'cb gi'selfPtr sequence state _ = do
    maybeSequence <-
        if sequence == FP.nullPtr
        then return Nothing
        else do
            B.ManagedPtr.withTransient  sequence $ \sequence' -> do
                return $ Just sequence'
    let state' = (toEnum . fromIntegral) state
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeSequence state'


-- | Connect a signal handler for the [sequenceStateChanged](#signal:sequenceStateChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gesture #sequenceStateChanged callback
-- @
-- 
-- 
onGestureSequenceStateChanged :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureSequenceStateChangedCallback) -> m SignalHandlerId
onGestureSequenceStateChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureSequenceStateChangedCallback wrapped
    wrapped'' <- mk_GestureSequenceStateChangedCallback wrapped'
    connectSignalFunPtr obj "sequence-state-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [sequenceStateChanged](#signal:sequenceStateChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gesture #sequenceStateChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGestureSequenceStateChanged :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureSequenceStateChangedCallback) -> m SignalHandlerId
afterGestureSequenceStateChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureSequenceStateChangedCallback wrapped
    wrapped'' <- mk_GestureSequenceStateChangedCallback wrapped'
    connectSignalFunPtr obj "sequence-state-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GestureSequenceStateChangedSignalInfo
instance SignalInfo GestureSequenceStateChangedSignalInfo where
    type HaskellCallbackType GestureSequenceStateChangedSignalInfo = GestureSequenceStateChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GestureSequenceStateChangedCallback cb
        cb'' <- mk_GestureSequenceStateChangedCallback cb'
        connectSignalFunPtr obj "sequence-state-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture::sequence-state-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#g:signal:sequenceStateChanged"})

#endif

-- signal Gesture::update
-- | Emitted whenever an event is handled while the gesture is recognized.
-- 
-- /@sequence@/ is guaranteed to pertain to the set of active touches.
type GestureUpdateCallback =
    Maybe Gdk.EventSequence.EventSequence
    -- ^ /@sequence@/: the @GdkEventSequence@ that was updated
    -> IO ()

type C_GestureUpdateCallback =
    Ptr Gesture ->                          -- object
    Ptr Gdk.EventSequence.EventSequence ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_GestureUpdateCallback`.
foreign import ccall "wrapper"
    mk_GestureUpdateCallback :: C_GestureUpdateCallback -> IO (FunPtr C_GestureUpdateCallback)

wrap_GestureUpdateCallback :: 
    GObject a => (a -> GestureUpdateCallback) ->
    C_GestureUpdateCallback
wrap_GestureUpdateCallback gi'cb gi'selfPtr sequence _ = do
    maybeSequence <-
        if sequence == FP.nullPtr
        then return Nothing
        else do
            B.ManagedPtr.withTransient  sequence $ \sequence' -> do
                return $ Just sequence'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeSequence


-- | Connect a signal handler for the [update](#signal:update) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' gesture #update callback
-- @
-- 
-- 
onGestureUpdate :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureUpdateCallback) -> m SignalHandlerId
onGestureUpdate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureUpdateCallback wrapped
    wrapped'' <- mk_GestureUpdateCallback wrapped'
    connectSignalFunPtr obj "update" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [update](#signal:update) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' gesture #update callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterGestureUpdate :: (IsGesture a, MonadIO m) => a -> ((?self :: a) => GestureUpdateCallback) -> m SignalHandlerId
afterGestureUpdate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_GestureUpdateCallback wrapped
    wrapped'' <- mk_GestureUpdateCallback wrapped'
    connectSignalFunPtr obj "update" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data GestureUpdateSignalInfo
instance SignalInfo GestureUpdateSignalInfo where
    type HaskellCallbackType GestureUpdateSignalInfo = GestureUpdateCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_GestureUpdateCallback cb
        cb'' <- mk_GestureUpdateCallback cb'
        connectSignalFunPtr obj "update" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture::update"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#g:signal:update"})

#endif

-- VVV Prop "n-points"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@n-points@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gesture #nPoints
-- @
getGestureNPoints :: (MonadIO m, IsGesture o) => o -> m Word32
getGestureNPoints obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "n-points"

-- | Construct a t'GValueConstruct' with valid value for the “@n-points@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGestureNPoints :: (IsGesture o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructGestureNPoints val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "n-points" val

#if defined(ENABLE_OVERLOADING)
data GestureNPointsPropertyInfo
instance AttrInfo GestureNPointsPropertyInfo where
    type AttrAllowedOps GestureNPointsPropertyInfo = '[ 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint GestureNPointsPropertyInfo = IsGesture
    type AttrSetTypeConstraint GestureNPointsPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint GestureNPointsPropertyInfo = (~) Word32
    type AttrTransferType GestureNPointsPropertyInfo = Word32
    type AttrGetType GestureNPointsPropertyInfo = Word32
    type AttrLabel GestureNPointsPropertyInfo = "n-points"
    type AttrOrigin GestureNPointsPropertyInfo = Gesture
    attrGet = getGestureNPoints
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructGestureNPoints
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.nPoints"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#g:attr:nPoints"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Gesture
type instance O.AttributeList Gesture = GestureAttributeList
type GestureAttributeList = ('[ '("nPoints", GestureNPointsPropertyInfo), '("name", Gtk.EventController.EventControllerNamePropertyInfo), '("propagationLimit", Gtk.EventController.EventControllerPropagationLimitPropertyInfo), '("propagationPhase", Gtk.EventController.EventControllerPropagationPhasePropertyInfo), '("widget", Gtk.EventController.EventControllerWidgetPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
gestureNPoints :: AttrLabelProxy "nPoints"
gestureNPoints = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Gesture = GestureSignalList
type GestureSignalList = ('[ '("begin", GestureBeginSignalInfo), '("cancel", GestureCancelSignalInfo), '("end", GestureEndSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("sequenceStateChanged", GestureSequenceStateChangedSignalInfo), '("update", GestureUpdateSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Gesture::get_bounding_box
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rect"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "bounding box containing all active touches."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_bounding_box" gtk_gesture_get_bounding_box :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gdk.Rectangle.Rectangle ->          -- rect : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    IO CInt

-- | If there are touch sequences being currently handled by /@gesture@/,
-- returns 'P.True' and fills in /@rect@/ with the bounding box containing
-- all active touches.
-- 
-- Otherwise, 'P.False' will be returned.
-- 
-- Note: This function will yield unexpected results on touchpad
-- gestures. Since there is no correlation between physical and
-- pixel distances, these will look as if constrained in an
-- infinitely small area, /@rect@/ width and height will thus be 0
-- regardless of the number of touchpoints.
gestureGetBoundingBox ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m ((Bool, Gdk.Rectangle.Rectangle))
    -- ^ __Returns:__ 'P.True' if there are active touches, 'P.False' otherwise
gestureGetBoundingBox gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    rect <- SP.callocBoxedBytes 16 :: IO (Ptr Gdk.Rectangle.Rectangle)
    result <- gtk_gesture_get_bounding_box gesture' rect
    let result' = (/= 0) result
    rect' <- (wrapBoxed Gdk.Rectangle.Rectangle) rect
    touchManagedPtr gesture
    return (result', rect')

#if defined(ENABLE_OVERLOADING)
data GestureGetBoundingBoxMethodInfo
instance (signature ~ (m ((Bool, Gdk.Rectangle.Rectangle))), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetBoundingBoxMethodInfo a signature where
    overloadedMethod = gestureGetBoundingBox

instance O.OverloadedMethodInfo GestureGetBoundingBoxMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetBoundingBox",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetBoundingBox"
        })


#endif

-- method Gesture::get_bounding_box_center
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "X coordinate for the bounding box center"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Y coordinate for the bounding box center"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_bounding_box_center" gtk_gesture_get_bounding_box_center :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr CDouble ->                          -- x : TBasicType TDouble
    Ptr CDouble ->                          -- y : TBasicType TDouble
    IO CInt

-- | If there are touch sequences being currently handled by /@gesture@/,
-- returns 'P.True' and fills in /@x@/ and /@y@/ with the center of the bounding
-- box containing all active touches.
-- 
-- Otherwise, 'P.False' will be returned.
gestureGetBoundingBoxCenter ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m ((Bool, Double, Double))
    -- ^ __Returns:__ 'P.False' if no active touches are present, 'P.True' otherwise
gestureGetBoundingBoxCenter gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    x <- allocMem :: IO (Ptr CDouble)
    y <- allocMem :: IO (Ptr CDouble)
    result <- gtk_gesture_get_bounding_box_center gesture' x y
    let result' = (/= 0) result
    x' <- peek x
    let x'' = realToFrac x'
    y' <- peek y
    let y'' = realToFrac y'
    touchManagedPtr gesture
    freeMem x
    freeMem y
    return (result', x'', y'')

#if defined(ENABLE_OVERLOADING)
data GestureGetBoundingBoxCenterMethodInfo
instance (signature ~ (m ((Bool, Double, Double))), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetBoundingBoxCenterMethodInfo a signature where
    overloadedMethod = gestureGetBoundingBoxCenter

instance O.OverloadedMethodInfo GestureGetBoundingBoxCenterMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetBoundingBoxCenter",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetBoundingBoxCenter"
        })


#endif

-- method Gesture::get_device
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Device" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_device" gtk_gesture_get_device :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO (Ptr Gdk.Device.Device)

-- | Returns the logical @GdkDevice@ that is currently operating
-- on /@gesture@/.
-- 
-- This returns 'P.Nothing' if the gesture is not being interacted.
gestureGetDevice ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m (Maybe Gdk.Device.Device)
    -- ^ __Returns:__ a @GdkDevice@
gestureGetDevice gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    result <- gtk_gesture_get_device gesture'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.Device.Device) result'
        return result''
    touchManagedPtr gesture
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data GestureGetDeviceMethodInfo
instance (signature ~ (m (Maybe Gdk.Device.Device)), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetDeviceMethodInfo a signature where
    overloadedMethod = gestureGetDevice

instance O.OverloadedMethodInfo GestureGetDeviceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetDevice",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetDevice"
        })


#endif

-- method Gesture::get_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList (TInterface Name { namespace = "Gtk" , name = "Gesture" }))
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_group" gtk_gesture_get_group :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO (Ptr (GList (Ptr Gesture)))

-- | Returns all gestures in the group of /@gesture@/
gestureGetGroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m [Gesture]
    -- ^ __Returns:__ The list
    --   of @GtkGesture@s, free with @/g_list_free()/@
gestureGetGroup gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    result <- gtk_gesture_get_group gesture'
    result' <- unpackGList result
    result'' <- mapM (newObject Gesture) result'
    g_list_free result
    touchManagedPtr gesture
    return result''

#if defined(ENABLE_OVERLOADING)
data GestureGetGroupMethodInfo
instance (signature ~ (m [Gesture]), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetGroupMethodInfo a signature where
    overloadedMethod = gestureGetGroup

instance O.OverloadedMethodInfo GestureGetGroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetGroup"
        })


#endif

-- method Gesture::get_last_event
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "sequence"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "EventSequence" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkEventSequence`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Event" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_last_event" gtk_gesture_get_last_event :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gdk.EventSequence.EventSequence ->  -- sequence : TInterface (Name {namespace = "Gdk", name = "EventSequence"})
    IO (Ptr Gdk.Event.Event)

-- | Returns the last event that was processed for /@sequence@/.
-- 
-- Note that the returned pointer is only valid as long as the
-- /@sequence@/ is still interpreted by the /@gesture@/. If in doubt,
-- you should make a copy of the event.
gestureGetLastEvent ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> Maybe (Gdk.EventSequence.EventSequence)
    -- ^ /@sequence@/: a @GdkEventSequence@
    -> m (Maybe Gdk.Event.Event)
    -- ^ __Returns:__ The last event from /@sequence@/
gestureGetLastEvent gesture sequence = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    maybeSequence <- case sequence of
        Nothing -> return FP.nullPtr
        Just jSequence -> do
            jSequence' <- unsafeManagedPtrGetPtr jSequence
            return jSequence'
    result <- gtk_gesture_get_last_event gesture' maybeSequence
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newPtr Gdk.Event.Event) result'
        return result''
    touchManagedPtr gesture
    whenJust sequence touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data GestureGetLastEventMethodInfo
instance (signature ~ (Maybe (Gdk.EventSequence.EventSequence) -> m (Maybe Gdk.Event.Event)), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetLastEventMethodInfo a signature where
    overloadedMethod = gestureGetLastEvent

instance O.OverloadedMethodInfo GestureGetLastEventMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetLastEvent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetLastEvent"
        })


#endif

-- method Gesture::get_last_updated_sequence
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gdk" , name = "EventSequence" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_last_updated_sequence" gtk_gesture_get_last_updated_sequence :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO (Ptr Gdk.EventSequence.EventSequence)

-- | Returns the @GdkEventSequence@ that was last updated on /@gesture@/.
gestureGetLastUpdatedSequence ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m (Maybe Gdk.EventSequence.EventSequence)
    -- ^ __Returns:__ The last updated sequence
gestureGetLastUpdatedSequence gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    result <- gtk_gesture_get_last_updated_sequence gesture'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Gdk.EventSequence.EventSequence) result'
        return result''
    touchManagedPtr gesture
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data GestureGetLastUpdatedSequenceMethodInfo
instance (signature ~ (m (Maybe Gdk.EventSequence.EventSequence)), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetLastUpdatedSequenceMethodInfo a signature where
    overloadedMethod = gestureGetLastUpdatedSequence

instance O.OverloadedMethodInfo GestureGetLastUpdatedSequenceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetLastUpdatedSequence",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetLastUpdatedSequence"
        })


#endif

-- method Gesture::get_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "sequence"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "EventSequence" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a `GdkEventSequence`, or %NULL for pointer events"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for X axis of the sequence coordinates"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TDouble
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for Y axis of the sequence coordinates"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_point" gtk_gesture_get_point :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gdk.EventSequence.EventSequence ->  -- sequence : TInterface (Name {namespace = "Gdk", name = "EventSequence"})
    Ptr CDouble ->                          -- x : TBasicType TDouble
    Ptr CDouble ->                          -- y : TBasicType TDouble
    IO CInt

-- | If /@sequence@/ is currently being interpreted by /@gesture@/,
-- returns 'P.True' and fills in /@x@/ and /@y@/ with the last coordinates
-- stored for that event sequence.
-- 
-- The coordinates are always relative to the widget allocation.
gestureGetPoint ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> Maybe (Gdk.EventSequence.EventSequence)
    -- ^ /@sequence@/: a @GdkEventSequence@, or 'P.Nothing' for pointer events
    -> m ((Bool, Double, Double))
    -- ^ __Returns:__ 'P.True' if /@sequence@/ is currently interpreted
gestureGetPoint gesture sequence = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    maybeSequence <- case sequence of
        Nothing -> return FP.nullPtr
        Just jSequence -> do
            jSequence' <- unsafeManagedPtrGetPtr jSequence
            return jSequence'
    x <- allocMem :: IO (Ptr CDouble)
    y <- allocMem :: IO (Ptr CDouble)
    result <- gtk_gesture_get_point gesture' maybeSequence x y
    let result' = (/= 0) result
    x' <- peek x
    let x'' = realToFrac x'
    y' <- peek y
    let y'' = realToFrac y'
    touchManagedPtr gesture
    whenJust sequence touchManagedPtr
    freeMem x
    freeMem y
    return (result', x'', y'')

#if defined(ENABLE_OVERLOADING)
data GestureGetPointMethodInfo
instance (signature ~ (Maybe (Gdk.EventSequence.EventSequence) -> m ((Bool, Double, Double))), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetPointMethodInfo a signature where
    overloadedMethod = gestureGetPoint

instance O.OverloadedMethodInfo GestureGetPointMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetPoint"
        })


#endif

-- method Gesture::get_sequence_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "sequence"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "EventSequence" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkEventSequence`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gtk" , name = "EventSequenceState" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_sequence_state" gtk_gesture_get_sequence_state :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gdk.EventSequence.EventSequence ->  -- sequence : TInterface (Name {namespace = "Gdk", name = "EventSequence"})
    IO CUInt

-- | Returns the /@sequence@/ state, as seen by /@gesture@/.
gestureGetSequenceState ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> Gdk.EventSequence.EventSequence
    -- ^ /@sequence@/: a @GdkEventSequence@
    -> m Gtk.Enums.EventSequenceState
    -- ^ __Returns:__ The sequence state in /@gesture@/
gestureGetSequenceState gesture sequence = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    sequence' <- unsafeManagedPtrGetPtr sequence
    result <- gtk_gesture_get_sequence_state gesture' sequence'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr gesture
    touchManagedPtr sequence
    return result'

#if defined(ENABLE_OVERLOADING)
data GestureGetSequenceStateMethodInfo
instance (signature ~ (Gdk.EventSequence.EventSequence -> m Gtk.Enums.EventSequenceState), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetSequenceStateMethodInfo a signature where
    overloadedMethod = gestureGetSequenceState

instance O.OverloadedMethodInfo GestureGetSequenceStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetSequenceState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetSequenceState"
        })


#endif

-- method Gesture::get_sequences
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList
--                  (TInterface Name { namespace = "Gdk" , name = "EventSequence" }))
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_get_sequences" gtk_gesture_get_sequences :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO (Ptr (GList (Ptr Gdk.EventSequence.EventSequence)))

-- | Returns the list of @GdkEventSequences@ currently being interpreted
-- by /@gesture@/.
gestureGetSequences ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m [Gdk.EventSequence.EventSequence]
    -- ^ __Returns:__ A list
    --   of @GdkEventSequence@, the list elements are owned by GTK and must
    --   not be freed or modified, the list itself must be deleted
    --   through @/g_list_free()/@
gestureGetSequences gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    result <- gtk_gesture_get_sequences gesture'
    result' <- unpackGList result
    result'' <- mapM (newBoxed Gdk.EventSequence.EventSequence) result'
    g_list_free result
    touchManagedPtr gesture
    return result''

#if defined(ENABLE_OVERLOADING)
data GestureGetSequencesMethodInfo
instance (signature ~ (m [Gdk.EventSequence.EventSequence]), MonadIO m, IsGesture a) => O.OverloadedMethod GestureGetSequencesMethodInfo a signature where
    overloadedMethod = gestureGetSequences

instance O.OverloadedMethodInfo GestureGetSequencesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGetSequences",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGetSequences"
        })


#endif

-- method Gesture::group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "group_gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "`GtkGesture` to group @gesture with"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_group" gtk_gesture_group :: 
    Ptr Gesture ->                          -- group_gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO ()

-- | Adds /@gesture@/ to the same group than /@groupGesture@/.
-- 
-- Gestures are by default isolated in their own groups.
-- 
-- Both gestures must have been added to the same widget before
-- they can be grouped.
-- 
-- When gestures are grouped, the state of @GdkEventSequences@
-- is kept in sync for all of those, so calling
-- 'GI.Gtk.Objects.Gesture.gestureSetSequenceState', on one will transfer
-- the same value to the others.
-- 
-- Groups also perform an \"implicit grabbing\" of sequences, if a
-- @GdkEventSequence@ state is set to 'GI.Gtk.Enums.EventSequenceStateClaimed'
-- on one group, every other gesture group attached to the same
-- @GtkWidget@ will switch the state for that sequence to
-- 'GI.Gtk.Enums.EventSequenceStateDenied'.
gestureGroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a, IsGesture b) =>
    a
    -- ^ /@groupGesture@/: @GtkGesture@ to group /@gesture@/ with
    -> b
    -- ^ /@gesture@/: a @GtkGesture@
    -> m ()
gestureGroup groupGesture gesture = liftIO $ do
    groupGesture' <- unsafeManagedPtrCastPtr groupGesture
    gesture' <- unsafeManagedPtrCastPtr gesture
    gtk_gesture_group groupGesture' gesture'
    touchManagedPtr groupGesture
    touchManagedPtr gesture
    return ()

#if defined(ENABLE_OVERLOADING)
data GestureGroupMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsGesture a, IsGesture b) => O.OverloadedMethod GestureGroupMethodInfo a signature where
    overloadedMethod = gestureGroup

instance O.OverloadedMethodInfo GestureGroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureGroup"
        })


#endif

-- method Gesture::handles_sequence
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "sequence"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "EventSequence" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkEventSequence`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_handles_sequence" gtk_gesture_handles_sequence :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gdk.EventSequence.EventSequence ->  -- sequence : TInterface (Name {namespace = "Gdk", name = "EventSequence"})
    IO CInt

-- | Returns 'P.True' if /@gesture@/ is currently handling events
-- corresponding to /@sequence@/.
gestureHandlesSequence ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> Maybe (Gdk.EventSequence.EventSequence)
    -- ^ /@sequence@/: a @GdkEventSequence@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@gesture@/ is handling /@sequence@/, 'P.False' otherwise
gestureHandlesSequence gesture sequence = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    maybeSequence <- case sequence of
        Nothing -> return FP.nullPtr
        Just jSequence -> do
            jSequence' <- unsafeManagedPtrGetPtr jSequence
            return jSequence'
    result <- gtk_gesture_handles_sequence gesture' maybeSequence
    let result' = (/= 0) result
    touchManagedPtr gesture
    whenJust sequence touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data GestureHandlesSequenceMethodInfo
instance (signature ~ (Maybe (Gdk.EventSequence.EventSequence) -> m Bool), MonadIO m, IsGesture a) => O.OverloadedMethod GestureHandlesSequenceMethodInfo a signature where
    overloadedMethod = gestureHandlesSequence

instance O.OverloadedMethodInfo GestureHandlesSequenceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureHandlesSequence",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureHandlesSequence"
        })


#endif

-- method Gesture::is_active
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_is_active" gtk_gesture_is_active :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO CInt

-- | Returns 'P.True' if the gesture is currently active.
-- 
-- A gesture is active while there are touch sequences
-- interacting with it.
gestureIsActive ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if gesture is active
gestureIsActive gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    result <- gtk_gesture_is_active gesture'
    let result' = (/= 0) result
    touchManagedPtr gesture
    return result'

#if defined(ENABLE_OVERLOADING)
data GestureIsActiveMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsGesture a) => O.OverloadedMethod GestureIsActiveMethodInfo a signature where
    overloadedMethod = gestureIsActive

instance O.OverloadedMethodInfo GestureIsActiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureIsActive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureIsActive"
        })


#endif

-- method Gesture::is_grouped_with
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "other"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another `GtkGesture`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_is_grouped_with" gtk_gesture_is_grouped_with :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gesture ->                          -- other : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO CInt

-- | Returns 'P.True' if both gestures pertain to the same group.
gestureIsGroupedWith ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a, IsGesture b) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> b
    -- ^ /@other@/: another @GtkGesture@
    -> m Bool
    -- ^ __Returns:__ whether the gestures are grouped
gestureIsGroupedWith gesture other = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    other' <- unsafeManagedPtrCastPtr other
    result <- gtk_gesture_is_grouped_with gesture' other'
    let result' = (/= 0) result
    touchManagedPtr gesture
    touchManagedPtr other
    return result'

#if defined(ENABLE_OVERLOADING)
data GestureIsGroupedWithMethodInfo
instance (signature ~ (b -> m Bool), MonadIO m, IsGesture a, IsGesture b) => O.OverloadedMethod GestureIsGroupedWithMethodInfo a signature where
    overloadedMethod = gestureIsGroupedWith

instance O.OverloadedMethodInfo GestureIsGroupedWithMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureIsGroupedWith",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureIsGroupedWith"
        })


#endif

-- method Gesture::is_recognized
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_is_recognized" gtk_gesture_is_recognized :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO CInt

-- | Returns 'P.True' if the gesture is currently recognized.
-- 
-- A gesture is recognized if there are as many interacting
-- touch sequences as required by /@gesture@/.
gestureIsRecognized ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if gesture is recognized
gestureIsRecognized gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    result <- gtk_gesture_is_recognized gesture'
    let result' = (/= 0) result
    touchManagedPtr gesture
    return result'

#if defined(ENABLE_OVERLOADING)
data GestureIsRecognizedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsGesture a) => O.OverloadedMethod GestureIsRecognizedMethodInfo a signature where
    overloadedMethod = gestureIsRecognized

instance O.OverloadedMethodInfo GestureIsRecognizedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureIsRecognized",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureIsRecognized"
        })


#endif

-- method Gesture::set_sequence_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "sequence"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "EventSequence" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GdkEventSequence`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "EventSequenceState" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the sequence state" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_set_sequence_state" gtk_gesture_set_sequence_state :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    Ptr Gdk.EventSequence.EventSequence ->  -- sequence : TInterface (Name {namespace = "Gdk", name = "EventSequence"})
    CUInt ->                                -- state : TInterface (Name {namespace = "Gtk", name = "EventSequenceState"})
    IO CInt

{-# DEPRECATED gestureSetSequenceState ["(Since version 4.10.)","Use 'GI.Gtk.Objects.Gesture.gestureSetState'"] #-}
-- | Sets the state of /@sequence@/ in /@gesture@/.
-- 
-- Sequences start in state 'GI.Gtk.Enums.EventSequenceStateNone', and whenever
-- they change state, they can never go back to that state. Likewise,
-- sequences in state 'GI.Gtk.Enums.EventSequenceStateDenied' cannot turn back to
-- a not denied state. With these rules, the lifetime of an event
-- sequence is constrained to the next four:
-- 
-- * None
-- * None → Denied
-- * None → Claimed
-- * None → Claimed → Denied
-- 
-- Note: Due to event handling ordering, it may be unsafe to set the
-- state on another gesture within a [Gesture::begin]("GI.Gtk.Objects.Gesture#g:signal:begin") signal
-- handler, as the callback might be executed before the other gesture
-- knows about the sequence. A safe way to perform this could be:
-- 
-- 
-- === /c code/
-- >static void
-- >first_gesture_begin_cb (GtkGesture       *first_gesture,
-- >                        GdkEventSequence *sequence,
-- >                        gpointer          user_data)
-- >{
-- >  gtk_gesture_set_sequence_state (first_gesture, sequence, GTK_EVENT_SEQUENCE_CLAIMED);
-- >  gtk_gesture_set_sequence_state (second_gesture, sequence, GTK_EVENT_SEQUENCE_DENIED);
-- >}
-- >
-- >static void
-- >second_gesture_begin_cb (GtkGesture       *second_gesture,
-- >                         GdkEventSequence *sequence,
-- >                         gpointer          user_data)
-- >{
-- >  if (gtk_gesture_get_sequence_state (first_gesture, sequence) == GTK_EVENT_SEQUENCE_CLAIMED)
-- >    gtk_gesture_set_sequence_state (second_gesture, sequence, GTK_EVENT_SEQUENCE_DENIED);
-- >}
-- 
-- 
-- If both gestures are in the same group, just set the state on
-- the gesture emitting the event, the sequence will be already
-- be initialized to the group\'s global state when the second
-- gesture processes the event.
gestureSetSequenceState ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> Gdk.EventSequence.EventSequence
    -- ^ /@sequence@/: a @GdkEventSequence@
    -> Gtk.Enums.EventSequenceState
    -- ^ /@state@/: the sequence state
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@sequence@/ is handled by /@gesture@/,
    --   and the state is changed successfully
gestureSetSequenceState gesture sequence state = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    sequence' <- unsafeManagedPtrGetPtr sequence
    let state' = (fromIntegral . fromEnum) state
    result <- gtk_gesture_set_sequence_state gesture' sequence' state'
    let result' = (/= 0) result
    touchManagedPtr gesture
    touchManagedPtr sequence
    return result'

#if defined(ENABLE_OVERLOADING)
data GestureSetSequenceStateMethodInfo
instance (signature ~ (Gdk.EventSequence.EventSequence -> Gtk.Enums.EventSequenceState -> m Bool), MonadIO m, IsGesture a) => O.OverloadedMethod GestureSetSequenceStateMethodInfo a signature where
    overloadedMethod = gestureSetSequenceState

instance O.OverloadedMethodInfo GestureSetSequenceStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureSetSequenceState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureSetSequenceState"
        })


#endif

-- method Gesture::set_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "EventSequenceState" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the sequence state" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_set_state" gtk_gesture_set_state :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    CUInt ->                                -- state : TInterface (Name {namespace = "Gtk", name = "EventSequenceState"})
    IO CInt

-- | Sets the state of all sequences that /@gesture@/ is currently
-- interacting with.
-- 
-- Sequences start in state 'GI.Gtk.Enums.EventSequenceStateNone', and whenever
-- they change state, they can never go back to that state. Likewise,
-- sequences in state 'GI.Gtk.Enums.EventSequenceStateDenied' cannot turn back to
-- a not denied state. With these rules, the lifetime of an event
-- sequence is constrained to the next four:
-- 
-- * None
-- * None → Denied
-- * None → Claimed
-- * None → Claimed → Denied
-- 
-- Note: Due to event handling ordering, it may be unsafe to set the
-- state on another gesture within a [Gesture::begin]("GI.Gtk.Objects.Gesture#g:signal:begin") signal
-- handler, as the callback might be executed before the other gesture
-- knows about the sequence. A safe way to perform this could be:
-- 
-- 
-- === /c code/
-- >static void
-- >first_gesture_begin_cb (GtkGesture       *first_gesture,
-- >                        GdkEventSequence *sequence,
-- >                        gpointer          user_data)
-- >{
-- >  gtk_gesture_set_state (first_gesture, GTK_EVENT_SEQUENCE_CLAIMED);
-- >  gtk_gesture_set_state (second_gesture, GTK_EVENT_SEQUENCE_DENIED);
-- >}
-- >
-- >static void
-- >second_gesture_begin_cb (GtkGesture       *second_gesture,
-- >                         GdkEventSequence *sequence,
-- >                         gpointer          user_data)
-- >{
-- >  if (gtk_gesture_get_sequence_state (first_gesture, sequence) == GTK_EVENT_SEQUENCE_CLAIMED)
-- >    gtk_gesture_set_state (second_gesture, GTK_EVENT_SEQUENCE_DENIED);
-- >}
-- 
-- 
-- If both gestures are in the same group, just set the state on
-- the gesture emitting the event, the sequence will be already
-- be initialized to the group\'s global state when the second
-- gesture processes the event.
gestureSetState ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> Gtk.Enums.EventSequenceState
    -- ^ /@state@/: the sequence state
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the state of at least one sequence
    --   was changed successfully
gestureSetState gesture state = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    let state' = (fromIntegral . fromEnum) state
    result <- gtk_gesture_set_state gesture' state'
    let result' = (/= 0) result
    touchManagedPtr gesture
    return result'

#if defined(ENABLE_OVERLOADING)
data GestureSetStateMethodInfo
instance (signature ~ (Gtk.Enums.EventSequenceState -> m Bool), MonadIO m, IsGesture a) => O.OverloadedMethod GestureSetStateMethodInfo a signature where
    overloadedMethod = gestureSetState

instance O.OverloadedMethodInfo GestureSetStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureSetState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureSetState"
        })


#endif

-- method Gesture::ungroup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "gesture"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Gesture" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkGesture`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_gesture_ungroup" gtk_gesture_ungroup :: 
    Ptr Gesture ->                          -- gesture : TInterface (Name {namespace = "Gtk", name = "Gesture"})
    IO ()

-- | Separates /@gesture@/ into an isolated group.
gestureUngroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsGesture a) =>
    a
    -- ^ /@gesture@/: a @GtkGesture@
    -> m ()
gestureUngroup gesture = liftIO $ do
    gesture' <- unsafeManagedPtrCastPtr gesture
    gtk_gesture_ungroup gesture'
    touchManagedPtr gesture
    return ()

#if defined(ENABLE_OVERLOADING)
data GestureUngroupMethodInfo
instance (signature ~ (m ()), MonadIO m, IsGesture a) => O.OverloadedMethod GestureUngroupMethodInfo a signature where
    overloadedMethod = gestureUngroup

instance O.OverloadedMethodInfo GestureUngroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Gesture.gestureUngroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Gesture.html#v:gestureUngroup"
        })


#endif


