/* Mednafen - Multi-system Emulator
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <boolean.h>
#include <streams/file_stream.h>

#include "../mednafen.h"

#include <sys/types.h>

#include <string.h>
#include <time.h>

#include "../general.h"
#include "../FileStream.h"
#include "../MemoryStream.h"

#include "CDAccess.h"
#include "CDAccess_PBP.h"
#include "CDUtility.h"

#include "audioreader.h"

#include <libretro.h>

#include "zlib.h"

extern "C" {
   #include "deps/libkirk/kirk_engine.h"
   #include "deps/libkirk/amctrl.h"
   #include "deps/libkirk/des.h"
}

extern retro_log_printf_t log_cb;

// very hacky but currently the only way to update the disc start offset class variable from libretro.cpp
extern int CD_SelectedDisc;
int PBP_DiscCount;

// Disk-image(rip) track/sector formats
enum
{
   DI_FORMAT_AUDIO       = 0x00,
   DI_FORMAT_MODE1       = 0x01,
   DI_FORMAT_MODE1_RAW   = 0x02,
   DI_FORMAT_MODE2       = 0x03,
   DI_FORMAT_MODE2_FORM1 = 0x04,
   DI_FORMAT_MODE2_FORM2 = 0x05,
   DI_FORMAT_MODE2_RAW   = 0x06,
   _DI_FORMAT_COUNT
};

static const char *DI_CUE_Strings[7] = 
{
   "AUDIO",
   "MODE1/2048",
   "MODE1/2352",

   // FIXME: These are just guesses:
   "MODE2/2336",
   "MODE2/2048",
   "MODE2/2324",
   "MODE2/2352"
};


static unsigned char dnas_key1A90[] = {0xED, 0xE2, 0x5D, 0x2D, 0xBB, 0xF8, 0x12, 0xE5, 0x3C, 0x5C, 0x59, 0x32, 0xFA, 0xE3, 0xE2, 0x43};

// Structure to describe the header of a PGD file.
typedef struct {
   unsigned char vkey[16];

   int open_flag;
   int key_index;
   int drm_type;
   int mac_type;
   int cipher_type;

   int data_size;
   int align_size;
   int block_size;
   int block_nr;
   int data_offset;
   int table_offset;

   unsigned char *buf;
} PGD_HEADER;

bool CDAccess_PBP::ImageOpen(const char *path, bool image_memcache)
{
   uint8 magic[4];
   char psar_sig[12];
   std::string base_dir, file_base, file_ext;
   char sbi_ext[4] = { 's', 'b', 'i', 0 };

   MDFN_GetFilePathComponents(path, &base_dir, &file_base, &file_ext);

   if(image_memcache)
      fp = new MemoryStream(new FileStream(path, MODE_READ));
   else
      fp = new FileStream(path, MODE_READ);

   // check for valid pbp
   if(fp->read(magic, 4) != 4 || magic[0] != 0 || magic[1] != 'P' || magic[2] != 'B' || magic[3] != 'P')
   {
      log_cb(RETRO_LOG_ERROR, "Invalid PBP header: %s\n", path);
      return false;
   }

   // offsets of internal files
   fp->seek(0x8, SEEK_SET);
   for(int i = 0; i < PBP_NUM_FILES; i++)
      pbp_file_offsets[i] = fp->get_LE<uint32_t>();

   // only data.psar is relevant
   psisoimg_offset = pbp_file_offsets[DATA_PSAR];
   fp->seek(pbp_file_offsets[DATA_PSAR], SEEK_SET);

   fp->read(psar_sig, sizeof(psar_sig));
   if(strncmp(psar_sig, "PSTITLEIMG00", sizeof(psar_sig)) == 0)
   {
      // multidisk image
      uint8_t iso_map[0x2A0];
      uint32_t read_offset = 0;

      fp->seek(pbp_file_offsets[DATA_PSAR] + 0x200, SEEK_SET);
      fp->read(iso_map, sizeof(iso_map));

      // check for "\0PGD", should indicate whether TOC is encrypted or not?
      if(iso_map[0] == 0 && iso_map[1] == 'P' && iso_map[2] == 'G' && iso_map[3] == 'D')
      {
         log_cb(RETRO_LOG_DEBUG, "[PBP] decrypting multi-disc iso map...\n");
         int pdg_size = decrypt_pgd(iso_map, sizeof(iso_map));

         if(pdg_size < 1 || pdg_size > sizeof(iso_map))
         {
            log_cb(RETRO_LOG_ERROR, "[PBP] Failed to decrypt multi-disc iso map\n");
            return false;
         }

         is_official = true;
         read_offset += 0x90;
      }

      for(int i = 0; i < 5; i++)
      {
         memcpy(&discs_start_offset[i], iso_map+read_offset, sizeof(int32_t));
         if(discs_start_offset[i] == 0)
            break;

         PBP_DiscCount = i+1;
         read_offset += sizeof(int32_t);
         log_cb(RETRO_LOG_DEBUG, "[PBP] DISC[%i] offset = %#x\n", i, pbp_file_offsets[DATA_PSAR]+discs_start_offset[i]);
      }

      if(PBP_DiscCount == 0)
      {
         log_cb(RETRO_LOG_ERROR, "Multidisk eboot has 0 images?: %s\n", path);
         return false;
      }

      // default to first disc on loading
      psisoimg_offset += discs_start_offset[0];
      fp->seek(psisoimg_offset, SEEK_SET);

      fp->read(psar_sig, sizeof(psar_sig));
   }

   if(strncmp(psar_sig, "PSISOIMG0000", sizeof(psar_sig)) != 0)
   {
      log_cb(RETRO_LOG_ERROR, "Unexpected psar_sig: %s\n", psar_sig);
      return false;
   }

   // prepare sbi file path
   if(file_ext.length() == 4 && file_ext[0] == '.')
   {
      for(int i = 0; i < 3; i++)
      {
         if(file_ext[1 + i] >= 'A' && file_ext[1 + i] <= 'Z')
            sbi_ext[i] += 'A' - 'a';
      }
   }
   sbi_path = MDFN_EvalFIP(base_dir, file_base + std::string(".") + std::string(sbi_ext), true);

   // for multi-disc images change the sbi file syntax to [filename]_[disc_number].sbi instead of [filename].sbi
   if(PBP_DiscCount > 1)
   {
      // use a substitute char here, set the proper one in Read_TOC()
      sbi_path.insert(sbi_path.length()-4, "_x");
   }

   return true;
}

void CDAccess_PBP::Cleanup(void)
{
   if(fp != NULL)
   {
      fp->close();   // need to manually close for FileStreams?
      delete fp;
   }
   if(index_table != NULL)
      free(index_table);
}

CDAccess_PBP::CDAccess_PBP(const char *path, bool image_memcache) : NumTracks(0), FirstTrack(0), LastTrack(0), total_sectors(0)
{
   is_official = false;
   index_table = NULL;
   fp = NULL;
   kirk_init();
   if (!ImageOpen(path, image_memcache))
   {
   }
}

CDAccess_PBP::~CDAccess_PBP()
{
   Cleanup();
}

// Note: this function makes use of the current contents(as in |=) in SubPWBuf.
void CDAccess_PBP::MakeSubPQ(int32 lba, uint8 *SubPWBuf)
{
   unsigned i;
   uint8_t buf[0xC], adr, control;
   int32_t track;
   uint32_t lba_relative;
   uint32_t ma, sa, fa;
   uint32_t m, s, f;
   uint8_t pause_or = 0x00;
   bool track_found = false;

   for(track = FirstTrack; track < (FirstTrack + NumTracks); track++)
   {
      if(lba >= (Tracks[track].LBA - Tracks[track].pregap_dv - Tracks[track].pregap) && lba < (Tracks[track].LBA + Tracks[track].sectors + Tracks[track].postgap))
      {
         track_found = true;
         break;
      }
   }

   if(!track_found)
      track = FirstTrack;

   lba_relative = abs((int32)lba - Tracks[track].LBA);

   f            = (lba_relative % 75);
   s            = ((lba_relative / 75) % 60);
   m            = (lba_relative / 75 / 60);

   fa           = (lba + 150) % 75;
   sa           = ((lba + 150) / 75) % 60;
   ma           = ((lba + 150) / 75 / 60);

   adr          = 0x1; // Q channel data encodes position
   control      = Tracks[track].subq_control;

   // Handle pause(D7 of interleaved subchannel byte) bit, should be set to 1 when in pregap or postgap.
   if((lba < Tracks[track].LBA) || (lba >= Tracks[track].LBA + Tracks[track].sectors))
      pause_or = 0x80;

   // Handle pregap between audio->data track
   {
      int32_t pg_offset = (int32)lba - Tracks[track].LBA;

      // If we're more than 2 seconds(150 sectors) from the real "start" of the track/INDEX 01, and the track is a data track,
      // and the preceding track is an audio track, encode it as audio(by taking the SubQ control field from the preceding track).
      //
      // TODO: Look into how we're supposed to handle subq control field in the four combinations of track types(data/audio).
      //
      if(pg_offset < -150)
      {
         if((Tracks[track].subq_control & SUBQ_CTRLF_DATA) && (FirstTrack < track) && !(Tracks[track - 1].subq_control & SUBQ_CTRLF_DATA))
            control = Tracks[track - 1].subq_control;
      }
   }

   memset(buf, 0, 0xC);
   buf[0] = (adr << 0) | (control << 4);
   buf[1] = U8_to_BCD(track);

   if(lba < Tracks[track].LBA) // Index is 00 in pregap
      buf[2] = U8_to_BCD(0x00);
   else
      buf[2] = U8_to_BCD(0x01);

   /* Track relative MSF address */
   buf[3] = U8_to_BCD(m);
   buf[4] = U8_to_BCD(s);
   buf[5] = U8_to_BCD(f);
   buf[6] = 0;
   /* Absolute MSF address */
   buf[7] = U8_to_BCD(ma);
   buf[8] = U8_to_BCD(sa);
   buf[9] = U8_to_BCD(fa);

   subq_generate_checksum(buf);

   if(!SubQReplaceMap.empty())
   {
      std::map<uint32, cpp11_array_doodad>::const_iterator it = SubQReplaceMap.find(LBA_to_ABA(lba));

      if(it != SubQReplaceMap.end())
         memcpy(buf, it->second.data, 12);
   }

   for (i = 0; i < 96; i++)
      SubPWBuf[i] |= (((buf[i >> 3] >> (7 - (i & 0x7))) & 1) ? 0x40 : 0x00) | pause_or;
}

bool CDAccess_PBP::Read_Raw_PW(uint8_t *buf, int32_t lba)
{
   memset(buf, 0, 96);
   MakeSubPQ(lba, buf);
   return true;
}


int CDAccess_PBP::decompress2(void *out, uint32_t *out_size, void *in, uint32_t in_size)
{
   static z_stream z;
   int ret = 0;

   if (z.zalloc == NULL) {
      z.next_in = Z_NULL;
      z.avail_in = 0;
      z.zalloc = Z_NULL;
      z.zfree = Z_NULL;
      z.opaque = Z_NULL;
      ret = inflateInit2(&z, -15);
   }
   else
      ret = inflateReset(&z);

   if (ret != Z_OK)
      return ret;

   z.next_in = (Bytef*)in;
   z.avail_in = in_size;
   z.next_out = (Bytef*)out;
   z.avail_out = *out_size;

   ret = inflate(&z, Z_FINISH);

   *out_size -= z.avail_out;
   return ret == 1 ? 0 : ret;
}

bool CDAccess_PBP::Read_Raw_Sector(uint8 *buf, int32 lba)
{
   uint8_t SimuQ[0xC];

   int32_t block = lba >> 4;
   sector_in_blk = lba & 0xf;

   memset(buf + 2352, 0, 96);
   MakeSubPQ(lba, buf + 2352);
   subq_deinterleave(buf + 2352, SimuQ);

   if (block != current_block)
   {
      uint32_t start_byte = index_table[block];
      uint32_t size = index_table[block+1] - start_byte;
      bool is_compressed = true;

      if (lba >= index_len * 16)
      {
         log_cb(RETRO_LOG_ERROR, "[PBP] sector %d is past img end\n", lba);
         return false;
      }

      if (size > sizeof(buff_compressed))
      {
         log_cb(RETRO_LOG_ERROR, "[PBP] %u: block %d is too large (%u)\n", lba, block, size);
         return false;
      }
      else if(size == sizeof(buff_compressed))
         is_compressed = false;  // should be the case here?

      fp->seek(start_byte, SEEK_SET);
      fp->read(is_compressed ? buff_compressed : buff_raw[0], size);

//log_cb(RETRO_LOG_DEBUG, "lba = %d, block = %u, start_byte = %#x, index_table[%i] = %#x\n", lba, block, start_byte, block, index_table[block]);

      if (is_compressed)
      {
         if(is_official)
         {
            decompress(buff_raw[0], buff_compressed, sizeof(buff_compressed));
            fixed_sectors = 0;
         }
         else
         {
            uint32_t cdbuffer_size_expect = sizeof(buff_raw[0]) << 4;
            uint32_t cdbuffer_size = cdbuffer_size_expect;
            int ret = decompress2(buff_raw[0], &cdbuffer_size, buff_compressed, size);
            if (ret != 0)
            {
               log_cb(RETRO_LOG_ERROR, "[PBP] uncompress failed with %d for block %d, sector %d (%u)\n", ret, block, lba, size);
               return false;
            }
            if (cdbuffer_size != cdbuffer_size_expect)
            {
               log_cb(RETRO_LOG_WARN, "[PBP] cdbuffer_size: %lu != %lu, sector %d\n", cdbuffer_size, cdbuffer_size_expect, lba);
               return false;
            }
         }
      }
      current_block = block;
   }

   if(is_official)
   {
      // this will probably rarely get caught but better than trying to do it every time I guess...
      if(!(fixed_sectors & (0x1 << sector_in_blk)))
      {
         if(fix_sector(buff_raw[sector_in_blk], lba) != 0)
            log_cb(RETRO_LOG_WARN, "[PBP] Failed to fix sector %d\n", lba);
         else
            fixed_sectors |= (0x1 << sector_in_blk);
      }
   }

   memcpy(buf, buff_raw[sector_in_blk], 2352);

   return true;
}

bool CDAccess_PBP::Read_TOC(TOC *toc)
{
   struct {
      uint8_t type;
      uint8_t pad0;
      uint8_t track;
      uint8_t index0[3];
      uint8_t pad1;
      uint8_t index1[3];
   } toc_entry;

   struct {
      uint32_t offset;
      uint16_t size;
      uint16_t marker;
      uint8_t checksum[16];
      uint8_t padding[8];
   } index_entry;

   int i;
   int32_t sector_count = 0;

   uint32_t read_offset;
   uint32_t toc_offset = 0x400;
   uint32_t index_table_offset = 0x3C00;
   uint32_t cdimg_base = psisoimg_offset + 0x100000;

   uint8_t* iso_header = (uint8_t*)malloc(0xB6600);

   if(!iso_header)
   {
      log_cb(RETRO_LOG_ERROR, "[PBP] Read_TOC() - unable to allocate memory\n");
      return false;
   }
   
   TOC_Clear(toc);
   memset(Tracks, 0, sizeof(Tracks));

   fp->seek(psisoimg_offset + 0x400, SEEK_SET);
   fp->read(iso_header, 0xB6600);
   if(iso_header[0] == 0 && iso_header[1] == 'P' && iso_header[2] == 'G' && iso_header[3] == 'D')
   {
      log_cb(RETRO_LOG_DEBUG, "[PBP] decrypting iso header...\n");
      int pdg_size = decrypt_pgd(iso_header, 0xB6600);

      if(pdg_size < 1 || pdg_size > 0xB6600)
      {
         log_cb(RETRO_LOG_ERROR, "[PBP] Failed to decrypt multi-disc iso map\n");
         return false;
      }

      is_official = true;
      toc_offset += 0x90;
      index_table_offset += 0x90;
   }
   
   // initialize opposites
   FirstTrack = 99;
   LastTrack = 0;

   // read TOC, first three entries are special, skip first one
   read_offset = toc_offset + sizeof(toc_entry);

   // number of tracks
   memcpy(&toc_entry, iso_header+read_offset, sizeof(toc_entry));
   read_offset += sizeof(toc_entry);
   NumTracks = BCD_to_U8(toc_entry.index1[0]);

   // total length
   memcpy(&toc_entry, iso_header+read_offset, sizeof(toc_entry));
   read_offset += sizeof(toc_entry);
   total_sectors = (BCD_to_U8(toc_entry.index1[0])*60 + BCD_to_U8(toc_entry.index1[1])) * 75 + BCD_to_U8(toc_entry.index1[2]);

   log_cb(RETRO_LOG_DEBUG, "[PBP] psisoimg_offset = %#x, toc_offset = %#x, index_table_offset = %#x, Numtracks = %d, total_sectors = %d\n", psisoimg_offset, toc_offset, index_table_offset, NumTracks, total_sectors);

   // read track info
   for(i = 1; i <= NumTracks; i++)
   {
      memcpy(&toc_entry, iso_header+read_offset, sizeof(toc_entry));
      read_offset += sizeof(toc_entry);

      if(toc_entry.track < FirstTrack)
         FirstTrack = BCD_to_U8(toc_entry.track);
      if(toc_entry.track > LastTrack)
         LastTrack = BCD_to_U8(toc_entry.track);

      if(toc_entry.type == 1)
      {
         Tracks[i].DIFormat = DI_FORMAT_AUDIO;
         Tracks[i].subq_control &= ~SUBQ_CTRLF_DATA;
      }
      else  // TOCHECK: are there any psx games that have other formats than AUDIO and MODE2/2352?
      {
         Tracks[i].DIFormat = DI_FORMAT_MODE2_RAW;
         Tracks[i].subq_control |= SUBQ_CTRLF_DATA;
      }

      Tracks[i].index[0] = (BCD_to_U8(toc_entry.index0[0])*60 + BCD_to_U8(toc_entry.index0[1])) * 75 + BCD_to_U8(toc_entry.index0[2]);
      Tracks[i].index[1] = (BCD_to_U8(toc_entry.index1[0])*60 + BCD_to_U8(toc_entry.index1[1])) * 75 + BCD_to_U8(toc_entry.index1[2]);

      // HACK: force these values for converted files since conversion tools like PSX2PSP seem to specify an offset for data tracks at 00:02:00 which is not the actual location in the converted image data (but rather 00:00:00)
      if (!is_official && Tracks[i].DIFormat == DI_FORMAT_MODE2_RAW)
      {
         Tracks[i].index[0] = 0;
         Tracks[i].index[1] = 0;
      }

      Tracks[i].LBA = Tracks[i].index[1];

      // pre/postgaps shouldn't be required, dont know if the pbp format even supports it
      Tracks[i].pregap = 0;
      Tracks[i].postgap = 0;

      Tracks[i].pregap_dv = Tracks[i].index[1]-Tracks[i].index[0];
      if(Tracks[i].pregap_dv < 0)
         Tracks[i].pregap_dv = 0;

      if(i > 1)
         Tracks[i-1].sectors = Tracks[i].index[0] - Tracks[i-1].index[1];
      sector_count += Tracks[i-1].sectors;

      if(i == NumTracks)
      {
         Tracks[i].sectors = total_sectors - sector_count;
         sector_count += Tracks[i].sectors;
      }

      toc->tracks[i].control = Tracks[i].subq_control;
      toc->tracks[i].adr = ADR_CURPOS;
      toc->tracks[i].lba = Tracks[i].LBA;

      log_cb(RETRO_LOG_DEBUG, "[PBP] track[%i]: %s, lba = %i, adr = %i, control = %i, index[0] = %i, index[1] = %i\n", BCD_to_U8(toc_entry.track), DI_CUE_Strings[Tracks[i].DIFormat], toc->tracks[i].lba, toc->tracks[i].adr, toc->tracks[i].control, Tracks[i].index[0], Tracks[i].index[1]);

      if(BCD_to_U8(toc_entry.track) < i || BCD_to_U8(toc_entry.track) > i)
      {
         log_cb(RETRO_LOG_ERROR, "Tracks out of order\n");   // can this happen?
         return false;
      }
   }
   
   if(total_sectors != sector_count)
      log_cb(RETRO_LOG_WARN, "[PBP] sector counts dont match (%i != %i)\n", total_sectors, sector_count);

   // HACK: disable audio tracks for official pbp files for now until more is known about them
   if(is_official)
   {
      // causes a lot of "Attempt to read LBA..." messages to pop up in the log but cant think of an easier way to only omit the audio tracks for now
      FirstTrack = LastTrack = NumTracks = 1;
   }

   toc->first_track = FirstTrack;
   toc->last_track = LastTrack;
   toc->disc_type = DISC_TYPE_CD_XA;   // always?

   // now to ISO disc map table
   read_offset = index_table_offset;

   // set class variables
   fixed_sectors = 0;
   current_block = (uint32_t)-1;
   index_len = 0xAFC80 / sizeof(index_entry);   // disc map table has a fixed size of 0xAFC80 (22500 entries)?

   if(index_table != NULL)
      free(index_table);

   index_table = (unsigned int*)malloc((index_len + 1) * sizeof(*index_table));
   if (index_table == NULL)
   {
      log_cb(RETRO_LOG_ERROR, "Unable to allocate memory\n");
      return false;
   }

   for (i = 0; i < index_len; i++)
   {
      // TOCHECK: does struct reading (with entries that could be affected by endianness) work reliably between different platforms?
      memcpy(&index_entry, iso_header+read_offset, sizeof(index_entry));
      read_offset += sizeof(index_entry);

      // apparently indices with marker == 0 aren't part of the original image (official pbp files only), should they be skipped?

      if (index_entry.size == 0)
         break;

      index_table[i] = cdimg_base + index_entry.offset;
   }
   index_table[i] = cdimg_base + index_entry.offset + index_entry.size;

   toc->tracks[100].lba = total_sectors;
   toc->tracks[100].adr = ADR_CURPOS;
   toc->tracks[100].control = toc->tracks[toc->last_track].control & 0x4;

   // Convenience leadout track duplication.
   if(toc->last_track < 99)
      toc->tracks[toc->last_track + 1] = toc->tracks[100];

   free(iso_header);

   // sbi stuff
   if(PBP_DiscCount > 1 && PBP_DiscCount < 10)
      sbi_path[sbi_path.length()-5] = (CD_SelectedDisc+1) + '0';

   // Load SBI file, if present
   if (filestream_exists(sbi_path.c_str()))
   {
      if(!SubQReplaceMap.empty())
         SubQReplaceMap.clear();

      LoadSBI(sbi_path.c_str());
   }
   else if (!SubQReplaceMap.empty())
   {
      SubQReplaceMap.clear();

      // SBI should probably be loaded in this case but file path is invalid
      log_cb(RETRO_LOG_WARN, "[PBP] Invalid path/filename for SBI file %s\n", sbi_path.c_str());
   }

   return true;
}

int CDAccess_PBP::LoadSBI(const char* sbi_path)
{
   /* Loading SBI file */
   uint8 header[4];
   uint8 ed[4 + 10];
   uint8 tmpq[12];
   RFILE *sbis      = filestream_open(sbi_path,
         RETRO_VFS_FILE_ACCESS_READ,
         RETRO_VFS_FILE_ACCESS_HINT_NONE);

   if (!sbis)
      return -1;

   filestream_read(sbis, header, 4);

   if(memcmp(header, "SBI\0", 4))
      goto error;

   while(filestream_read(sbis, ed, sizeof(ed)) == sizeof(ed))
   {
      /* Bad BCD MSF offset in SBI file. */
      if(!BCD_is_valid(ed[0]) || !BCD_is_valid(ed[1]) || !BCD_is_valid(ed[2]))
         goto error;

      /* Unrecognized boogly oogly in SBI file */
      if(ed[3] != 0x01)
         goto error;

      memcpy(tmpq, &ed[4], 10);

      subq_generate_checksum(tmpq);
      tmpq[10] ^= 0xFF;
      tmpq[11] ^= 0xFF;

      uint32 aba = AMSF_to_ABA(BCD_to_U8(ed[0]), BCD_to_U8(ed[1]), BCD_to_U8(ed[2]));

      memcpy(SubQReplaceMap[aba].data, tmpq, 12);
   }

   log_cb(RETRO_LOG_INFO, "[PBP] Loaded SBI file %s\n", sbi_path);
   filestream_close(sbis);
   return 0;

error:
   if (sbis)
      filestream_close(sbis);
   return -1;
}

void CDAccess_PBP::Eject(bool eject_status)
{
   if(!eject_status && CD_SelectedDisc >= 0 
         && CD_SelectedDisc < PBP_DiscCount)
   {
      log_cb(RETRO_LOG_DEBUG, "[PBP] changing offset: old = %#x, new = %#x (%i of %i)\n",
            psisoimg_offset, pbp_file_offsets[DATA_PSAR]+discs_start_offset[CD_SelectedDisc],
            CD_SelectedDisc+1, PBP_DiscCount);
      psisoimg_offset = pbp_file_offsets[DATA_PSAR]+discs_start_offset[CD_SelectedDisc];
   }
}

int CDAccess_PBP::decrypt_pgd(unsigned char* pgd_data, int pgd_size)
{
   int result;
   PGD_HEADER PGD[1];
   MAC_KEY mkey;
   CIPHER_KEY ckey;

   // Read in the PGD header parameters.
   memset(PGD, 0, sizeof(PGD_HEADER));

   PGD->buf = pgd_data;
   PGD->key_index = *(u32*)(pgd_data + 4);
   PGD->drm_type  = *(u32*)(pgd_data + 8);
   PGD->open_flag = 0x2;

   // Set the hashing, crypto and open modes.
   if (PGD->drm_type == 1)
   {
      PGD->mac_type = 1;
      PGD->open_flag |= 4;

      if(PGD->key_index > 1)
      {
         PGD->mac_type = 3;
         PGD->open_flag |= 8;
      }
      PGD->cipher_type = 1;
   }
   else
   {
      PGD->mac_type = 2;
      PGD->cipher_type = 2;
   }

   // Test MAC hash at 0x80 (DNAS hash).
   sceDrmBBMacInit(&mkey, PGD->mac_type);
   sceDrmBBMacUpdate(&mkey, pgd_data, 0x80);
   result = sceDrmBBMacFinal2(&mkey, pgd_data + 0x80, dnas_key1A90);

   if (result)
   {
      log_cb(RETRO_LOG_WARN, "Invalid 0x80 MAC hash!\n");
      return -1;
   }

   // Test MAC hash at 0x70 (key hash).
   sceDrmBBMacInit(&mkey, PGD->mac_type);
   sceDrmBBMacUpdate(&mkey, pgd_data, 0x70);

   // Generate the key from MAC 0x70.
   bbmac_getkey(&mkey, pgd_data + 0x70, PGD->vkey);

   // Decrypt the PGD header block (0x30 bytes).
   sceDrmBBCipherInit(&ckey, PGD->cipher_type, 2, pgd_data + 0x10, PGD->vkey, 0);
   sceDrmBBCipherUpdate(&ckey, pgd_data + 0x30, 0x30);
   sceDrmBBCipherFinal(&ckey);

   // Get the decryption parameters from the decrypted header.
   PGD->data_size   = *(u32*)(pgd_data + 0x44);
   PGD->block_size  = *(u32*)(pgd_data + 0x48);
   PGD->data_offset = *(u32*)(pgd_data + 0x4c);

   // Additional size variables.
   PGD->align_size = (PGD->data_size + 15) &~ 15;
   PGD->table_offset = PGD->data_offset + PGD->align_size;
   PGD->block_nr = (PGD->align_size + PGD->block_size - 1) &~ (PGD->block_size - 1);
   PGD->block_nr = PGD->block_nr / PGD->block_size;

   if ((PGD->align_size + PGD->block_nr * 16) > pgd_size)
   {
      log_cb(RETRO_LOG_WARN, "Invalid data size!\n");
      return -1;
   }

   // Test MAC hash at 0x60 (table hash).
   sceDrmBBMacInit(&mkey, PGD->mac_type);
   sceDrmBBMacUpdate(&mkey, pgd_data + PGD->table_offset, PGD->block_nr * 16);
   result = sceDrmBBMacFinal2(&mkey, pgd_data + 0x60, PGD->vkey);

   if(result)
   {
      log_cb(RETRO_LOG_WARN, "Invalid 0x60 MAC hash!\n");
      return -1;
   }

   // Decrypt the data.
   sceDrmBBCipherInit(&ckey, PGD->cipher_type, 2, pgd_data + 0x30, PGD->vkey, 0);
   sceDrmBBCipherUpdate(&ckey, pgd_data + 0x90, PGD->align_size);
   sceDrmBBCipherFinal(&ckey);

   return PGD->data_size;
}

int CDAccess_PBP::decode_range(unsigned int *range, unsigned int *code, unsigned char **src)
{
   if (!((*range) >> 24))
   {
      (*range) <<= 8;
      *code = ((*code) << 8) + (*src)++[5];
      return 1;
   }

   return 0;
}

int CDAccess_PBP::decode_bit(unsigned int *range, unsigned int *code, 
      int *index, unsigned char **src, unsigned char *c)
{
   unsigned int val = *range;

   if (decode_range(range, code, src))
      val *= (*c);
   else
      val = (val >> 8) * (*c);

   *c -= ((*c) >> 3);
   if (index)
      (*index) <<= 1;

   if (*code < val)
   {
      *range = val;
      *c += 31;
      if (index) (*index)++;
      return 1;
   }

   *code -= val;
   *range -= val;
   return 0;
}

int CDAccess_PBP::decode_word(unsigned char *ptr, int index, 
      int *bit_flag, unsigned int *range, 
      unsigned int *code, unsigned char **src)
{
   int i = 1;
   index >>= 3;

   if (index >= 3)
   {
      decode_bit(range, code, &i, src, ptr);          // Offset 0x8A8
      if (index >= 4)
      {
         decode_bit(range, code, &i, src, ptr);      // Offset 0x8A8
         if (index >= 5)
         {
            decode_range(range, code, src);
            for (; index >= 5; index--)
            {
               i <<= 1;
               (*range) >>= 1;
               if (*code < *range)
                  i++;
               else
                  (*code) -= *range;
            }
         }
      }
   }

   *bit_flag = decode_bit(range, code, &i, src, ptr + 3);   // Offset 0x8A8 + 3

   if (index >= 1)
   {
      decode_bit(range, code, &i, src, ptr + 2);           // Offset 0x8A8 + 2
      if (index >= 2)
      {
         decode_bit(range, code, &i, src, ptr + 1);       // Offset 0x8A8 + 1
      }
   }

   return i;
}

int CDAccess_PBP::decode_number(unsigned char *ptr, int index, int *bit_flag, 
      unsigned int *range, unsigned int *code, unsigned char **src)
{
   int i = 1;

   if (index >= 3)
   {
      decode_bit(range, code, &i, src, ptr + 0x18);         // Offset 0x978
      if (index >= 4)
      {
         decode_bit(range, code, &i, src, ptr + 0x18);     // Offset 0x978
         if (index >= 5)
         {
            decode_range(range, code, src);
            for (; index >= 5; index--)
            {
               i <<= 1;
               (*range) >>= 1;
               if (*code < *range)
                  i++;
               else
                  (*code) -= *range;
            }
         }
      }
   }

   *bit_flag = decode_bit(range, code, &i, src, ptr);       // Offset 0x960

   if (index >= 1) 
   {
      decode_bit(range, code, &i, src, ptr + 0x8);         // Offset 0x968
      if (index >= 2)
      {
         decode_bit(range, code, &i, src, ptr + 0x10);    // Offset 0x970
      }
   }

   return i;
}

int CDAccess_PBP::decompress(unsigned char *out, unsigned char *in, unsigned int size)
{
   int result;

   unsigned char tmp[0xA70];

   int offset = 0;
   int bit_flag = 0;
   int data_length = 0;
   int data_offset = 0;

   unsigned char *tmp_sect1, *tmp_sect2, *tmp_sect3;
   unsigned char *buf_start, *buf_end;
   unsigned char prev = 0;

   unsigned char *start = out;
   unsigned char *end = (out + size);
   unsigned char head = in[0];

   unsigned int range = 0xFFFFFFFF;
   unsigned int code = (in[1] << 24) | (in[2] << 16) | (in[3] << 8) | in[4];

   if (head < 0) // Check if we have a valid starting byte.
   {
      // The dictionary header is invalid, the data is not compressed.
      result = -1;
      if (code <= size)
      {
         memcpy(out, (const void *)(in + 5), code);
         result = (start - out);
      }
   }
   else
   {
      // Set up a temporary buffer (sliding window).
      memset(tmp, 0x80, 0xA60);
      while (1)
      {
         // Start reading at 0x920.
         tmp_sect1 = tmp + offset + 0x920;
         if (!decode_bit(&range, &code, 0, &in, tmp_sect1))  // Raw char.
         {
            // Adjust offset and check for stream end.
            if (offset > 0) offset--;
            if (start == end) return (start - out);

            // Locate first section.
            int sect = (((((((int)(start - out)) & 7) << 8) + prev) >> head) & 7) * 0xFF - 1;
            tmp_sect1 = tmp + sect;
            int index = 1;

            // Read, decode and write back.
            do
            {
               decode_bit(&range, &code, &index, &in, tmp_sect1 + index);
            } while ((index >> 8) == 0);

            // Save index.
            *start++ = index;
         }
         else  // Compressed char stream.
         {
            int index = -1;

            // Identify the data length bit field.
            do {
               tmp_sect1 += 8;
               bit_flag = decode_bit(&range, &code, 0, &in, tmp_sect1);
               index += bit_flag;
            } while ((bit_flag != 0) && (index < 6));

            // Default block size is 0x40.
            int b_size = 0x40;
            tmp_sect2 = tmp + index + 0x7F1;

            // If the data length was found, parse it as a number.
            if ((index >= 0) || (bit_flag != 0)) 
            {
               // Locate next section.
               int sect = (index << 5) | (((((int)(start - out)) << index) & 3) << 3) | (offset & 7);
               tmp_sect1 = tmp + 0x960 + sect;

               // Decode the data length (8 bit fields).
               data_length = decode_number(tmp_sect1, index, &bit_flag, &range, &code, &in);

               // If we got valid parameters, seek to find data offset.
               if ((data_length != 3) && ((index > 0) || (bit_flag != 0))) {
                  tmp_sect2 += 0x38;
                  b_size = 0x80;  // Block size is now 0x80.
               }
            } else {
               // Assume one byte of advance.
               data_length = 1;
            }

            int diff = 0;
            int shift = 1;

            // Identify the data offset bit field.
            do {
               diff = (shift << 4) - b_size;
               bit_flag = decode_bit(&range, &code, &shift, &in, tmp_sect2 + (shift << 3));
            } while (diff < 0);

            // If the data offset was found, parse it as a number.
            if ((diff > 0) || (bit_flag != 0))
            {
               // Adjust diff if needed.
               if (bit_flag == 0) diff -= 8;

               // Locate section.
               tmp_sect3 = tmp + 0x8A8 + diff;

               // Decode the data offset (1 bit fields).
               data_offset = decode_word(tmp_sect3, diff, &bit_flag, &range, &code, &in);
            } else {
               // Assume one byte of advance.
               data_offset = 1;
            }

            // Set buffer start/end.
            buf_start = start - data_offset;
            buf_end = start + data_length + 1;

            // Underflow.
            if (buf_start < out)
               return -1;

            // Overflow.
            if (buf_end > end)
               return -1;

            // Update offset.
            offset = ((((int)(buf_end - out)) + 1) & 1) + 6;

            // Copy data.
            do {
               *start++ = *buf_start++;
            } while (start < buf_end);

         }
         prev = *(start - 1);
      }
      result = (start - out);
   }

   return result;
}

#include "edc_crc32.h"

// The following table is used for computing the error correction code (ECC).
static unsigned short RSPCTable[43][256] =
{
    /*          00      01      02      03      04      05      06      07      08      09      0A      0B      0C      0D      0E      0F      10      11      12      13      14      15      16      17      18      19      1A      1B      1C      1D      1E      1F      20      21      22      23      24      25      26      27      28      29      2A      2B      2C      2D      2E      2F      30      31      32      33      34      35      36      37      38      39      3A      3B      3C      3D      3E      3F      40      41      42      43      44      45      46      47      48      49      4A      4B      4C      4D      4E      4F      50      51      52      53      54      55      56      57      58      59      5A      5B      5C      5D      5E      5F      60      61      62      63      64      65      66      67      68      69      6A      6B      6C      6D      6E      6F      70      71      72      73      74      75      76      77      78      79      7A      7B      7C      7D      7E      7F      80      81      82      83      84      85      86      87      88      89      8A      8B      8C      8D      8E      8F      90      91      92      93      94      95      96      97      98      99      9A      9B      9C      9D      9E      9F      A0      A1      A2      A3      A4      A5      A6      A7      A8      A9      AA      AB      AC      AD      AE      AF      B0      B1      B2      B3      B4      B5      B6      B7      B8      B9      BA      BB      BC      BD      BE      BF      C0      C1      C2      C3      C4      C5      C6      C7      C8      C9      CA      CB      CC      CD      CE      CF      D0      D1      D2      D3      D4      D5      D6      D7      D8      D9      DA      DB      DC      DD      DE      DF      E0      E1      E2      E3      E4      E5      E6      E7      E8      E9      EA      EB      EC      ED      EE      EF      F0      F1      F2      F3      F4      F5      F6      F7      F8      F9      FA      FB      FC      FD      FE      FF           */
    /* 00 */ {0x0000, 0xAEAF, 0x4143, 0xEFEC, 0x8286, 0x2C29, 0xC3C5, 0x6D6A, 0x1911, 0xB7BE, 0x5852, 0xF6FD, 0x9B97, 0x3538, 0xDAD4, 0x747B, 0x3222, 0x9C8D, 0x7361, 0xDDCE, 0xB0A4, 0x1E0B, 0xF1E7, 0x5F48, 0x2B33, 0x859C, 0x6A70, 0xC4DF, 0xA9B5, 0x071A, 0xE8F6, 0x4659, 0x6444, 0xCAEB, 0x2507, 0x8BA8, 0xE6C2, 0x486D, 0xA781, 0x092E, 0x7D55, 0xD3FA, 0x3C16, 0x92B9, 0xFFD3, 0x517C, 0xBE90, 0x103F, 0x5666, 0xF8C9, 0x1725, 0xB98A, 0xD4E0, 0x7A4F, 0x95A3, 0x3B0C, 0x4F77, 0xE1D8, 0x0E34, 0xA09B, 0xCDF1, 0x635E, 0x8CB2, 0x221D, 0xC888, 0x6627, 0x89CB, 0x2764, 0x4A0E, 0xE4A1, 0x0B4D, 0xA5E2, 0xD199, 0x7F36, 0x90DA, 0x3E75, 0x531F, 0xFDB0, 0x125C, 0xBCF3, 0xFAAA, 0x5405, 0xBBE9, 0x1546, 0x782C, 0xD683, 0x396F, 0x97C0, 0xE3BB, 0x4D14, 0xA2F8, 0x0C57, 0x613D, 0xCF92, 0x207E, 0x8ED1, 0xACCC, 0x0263, 0xED8F, 0x4320, 0x2E4A, 0x80E5, 0x6F09, 0xC1A6, 0xB5DD, 0x1B72, 0xF49E, 0x5A31, 0x375B, 0x99F4, 0x7618, 0xD8B7, 0x9EEE, 0x3041, 0xDFAD, 0x7102, 0x1C68, 0xB2C7, 0x5D2B, 0xF384, 0x87FF, 0x2950, 0xC6BC, 0x6813, 0x0579, 0xABD6, 0x443A, 0xEA95, 0x8D0D, 0x23A2, 0xCC4E, 0x62E1, 0x0F8B, 0xA124, 0x4EC8, 0xE067, 0x941C, 0x3AB3, 0xD55F, 0x7BF0, 0x169A, 0xB835, 0x57D9, 0xF976, 0xBF2F, 0x1180, 0xFE6C, 0x50C3, 0x3DA9, 0x9306, 0x7CEA, 0xD245, 0xA63E, 0x0891, 0xE77D, 0x49D2, 0x24B8, 0x8A17, 0x65FB, 0xCB54, 0xE949, 0x47E6, 0xA80A, 0x06A5, 0x6BCF, 0xC560, 0x2A8C, 0x8423, 0xF058, 0x5EF7, 0xB11B, 0x1FB4, 0x72DE, 0xDC71, 0x339D, 0x9D32, 0xDB6B, 0x75C4, 0x9A28, 0x3487, 0x59ED, 0xF742, 0x18AE, 0xB601, 0xC27A, 0x6CD5, 0x8339, 0x2D96, 0x40FC, 0xEE53, 0x01BF, 0xAF10, 0x4585, 0xEB2A, 0x04C6, 0xAA69, 0xC703, 0x69AC, 0x8640, 0x28EF, 0x5C94, 0xF23B, 0x1DD7, 0xB378, 0xDE12, 0x70BD, 0x9F51, 0x31FE, 0x77A7, 0xD908, 0x36E4, 0x984B, 0xF521, 0x5B8E, 0xB462, 0x1ACD, 0x6EB6, 0xC019, 0x2FF5, 0x815A, 0xEC30, 0x429F, 0xAD73, 0x03DC, 0x21C1, 0x8F6E, 0x6082, 0xCE2D, 0xA347, 0x0DE8, 0xE204, 0x4CAB, 0x38D0, 0x967F, 0x7993, 0xD73C, 0xBA56, 0x14F9, 0xFB15, 0x55BA, 0x13E3, 0xBD4C, 0x52A0, 0xFC0F, 0x9165, 0x3FCA, 0xD026, 0x7E89, 0x0AF2, 0xA45D, 0x4BB1, 0xE51E, 0x8874, 0x26DB, 0xC937, 0x6798}, /* 00 */
    /* 01 */ {0x0000, 0xD9D8, 0xAFAD, 0x7675, 0x4347, 0x9A9F, 0xECEA, 0x3532, 0x868E, 0x5F56, 0x2923, 0xF0FB, 0xC5C9, 0x1C11, 0x6A64, 0xB3BC, 0x1101, 0xC8D9, 0xBEAC, 0x6774, 0x5246, 0x8B9E, 0xFDEB, 0x2433, 0x978F, 0x4E57, 0x3822, 0xE1FA, 0xD4C8, 0x0D10, 0x7B65, 0xA2BD, 0x2202, 0xFBDA, 0x8DAF, 0x5477, 0x6145, 0xB89D, 0xCEE8, 0x1730, 0xA48C, 0x7D54, 0x0B21, 0xD2F9, 0xE7CB, 0x3E13, 0x4866, 0x91BE, 0x3303, 0xEADB, 0x9CAE, 0x4576, 0x7044, 0xA99C, 0xDFE9, 0x0631, 0xB58D, 0x6C55, 0x1A20, 0xC3F8, 0xF6CA, 0x2F12, 0x5967, 0x80BF, 0x4404, 0x9DDC, 0xEBA9, 0x3271, 0x0743, 0xDE9B, 0xA8EE, 0x7136, 0xC28A, 0x1B52, 0x6D27, 0xB4FF, 0x81CD, 0x5815, 0x2E60, 0xF7B8, 0x5505, 0x8CDD, 0xFAA8, 0x2370, 0x1642, 0xCF9A, 0xB9EF, 0x6037, 0xD38B, 0x0A53, 0x7C26, 0xA5FE, 0x90CC, 0x4914, 0x3F61, 0xE6B9, 0x6606, 0xBFDE, 0xC9AB, 0x1073, 0x2541, 0xFC99, 0x8AEC, 0x5334, 0xE088, 0x3950, 0x4F25, 0x96FD, 0xA3CF, 0x7A17, 0x0C62, 0xD5BA, 0x7707, 0xAEDF, 0xD8AA, 0x0172, 0x3440, 0xED98, 0x9BED, 0x4235, 0xF189, 0x2851, 0x5E24, 0x87FC, 0xB2CE, 0x6B16, 0x1D63, 0xC4BB, 0x8808, 0x51D0, 0x27A5, 0xFE7D, 0xCB4F, 0x1297, 0x64E2, 0xBD3A, 0x0E86, 0xD75E, 0xA12B, 0x78F3, 0x4DC1, 0x9419, 0xE26C, 0x3BB4, 0x9909, 0x40D1, 0x36A4, 0xEF7C, 0xDA4E, 0x0396, 0x75E3, 0xAC3B, 0x1F87, 0xC65F, 0xB02A, 0x69F2, 0x5CC0, 0x8518, 0xF36D, 0x2AB5, 0xAA0A, 0x73D2, 0x05A7, 0xDC7F, 0xE94D, 0x3095, 0x46E0, 0x9F38, 0x2C84, 0xF55C, 0x8329, 0x5AF1, 0x6FC3, 0xB61B, 0xC06E, 0x19B6, 0xBB0B, 0x62D3, 0x14A6, 0xCD7E, 0xF84C, 0x2194, 0x57E1, 0x8E39, 0x3D85, 0xE45D, 0x9228, 0x4BF0, 0x7EC2, 0xA71A, 0xD16F, 0x08B7, 0xCC0C, 0x15D4, 0x63A1, 0xBA79, 0x8F4B, 0x5693, 0x20E6, 0xF93E, 0x4A82, 0x935A, 0xE52F, 0x3CF7, 0x09C5, 0xD01D, 0xA668, 0x7FB0, 0xDD0D, 0x04D5, 0x72A0, 0xAB78, 0x9E4A, 0x4792, 0x31E7, 0xE83F, 0x5B83, 0x825B, 0xF42E, 0x2DF6, 0x18C4, 0xC11C, 0xB769, 0x6EB1, 0xEE0E, 0x37D6, 0x41A3, 0x987B, 0xAD49, 0x7491, 0x02E4, 0xDB3C, 0x6880, 0xB158, 0xC72D, 0x1EF5, 0x2BC7, 0xF21F, 0x846A, 0x5DB2, 0xFF0F, 0x26D7, 0x50A2, 0x897A, 0xBC48, 0x6590, 0x13E5, 0xCA3D, 0x7981, 0xA059, 0xD62C, 0x0FF4, 0x3AC6, 0xE31E, 0x956B, 0x4CB3}, /* 01 */
    /* 02 */ {0x0000, 0x6C6D, 0xD8DA, 0xB4B7, 0xADA9, 0xC1C4, 0x7573, 0x191E, 0x474F, 0x2B22, 0x9F95, 0xF3F8, 0xEAE6, 0x868B, 0x323C, 0x5E51, 0x8E9E, 0xE2F3, 0x5644, 0x3A29, 0x2337, 0x4F5A, 0xFBED, 0x9780, 0xC9D1, 0xA5BC, 0x110B, 0x7D66, 0x6478, 0x0815, 0xBCA2, 0xD0CF, 0x0121, 0x6D4C, 0xD9FB, 0xB596, 0xAC88, 0xC0E5, 0x7452, 0x183F, 0x466E, 0x2A03, 0x9EB4, 0xF2D9, 0xEBC7, 0x87AA, 0x331D, 0x5F70, 0x8FBF, 0xE3D2, 0x5765, 0x3B08, 0x2216, 0x4E7B, 0xFACC, 0x96A1, 0xC8F0, 0xA49D, 0x102A, 0x7C47, 0x6559, 0x0934, 0xBD83, 0xD1EE, 0x0242, 0x6E2F, 0xDA98, 0xB6F5, 0xAFEB, 0xC386, 0x7731, 0x1B5C, 0x450D, 0x2960, 0x9DD7, 0xF1BA, 0xE8A4, 0x84C9, 0x307E, 0x5C13, 0x8CDC, 0xE0B1, 0x5406, 0x386B, 0x2175, 0x4D18, 0xF9AF, 0x95C2, 0xCB93, 0xA7FE, 0x1349, 0x7F24, 0x663A, 0x0A57, 0xBEE0, 0xD28D, 0x0363, 0x6F0E, 0xDBB9, 0xB7D4, 0xAECA, 0xC2A7, 0x7610, 0x1A7D, 0x442C, 0x2841, 0x9CF6, 0xF09B, 0xE985, 0x85E8, 0x315F, 0x5D32, 0x8DFD, 0xE190, 0x5527, 0x394A, 0x2054, 0x4C39, 0xF88E, 0x94E3, 0xCAB2, 0xA6DF, 0x1268, 0x7E05, 0x671B, 0x0B76, 0xBFC1, 0xD3AC, 0x0484, 0x68E9, 0xDC5E, 0xB033, 0xA92D, 0xC540, 0x71F7, 0x1D9A, 0x43CB, 0x2FA6, 0x9B11, 0xF77C, 0xEE62, 0x820F, 0x36B8, 0x5AD5, 0x8A1A, 0xE677, 0x52C0, 0x3EAD, 0x27B3, 0x4BDE, 0xFF69, 0x9304, 0xCD55, 0xA138, 0x158F, 0x79E2, 0x60FC, 0x0C91, 0xB826, 0xD44B, 0x05A5, 0x69C8, 0xDD7F, 0xB112, 0xA80C, 0xC461, 0x70D6, 0x1CBB, 0x42EA, 0x2E87, 0x9A30, 0xF65D, 0xEF43, 0x832E, 0x3799, 0x5BF4, 0x8B3B, 0xE756, 0x53E1, 0x3F8C, 0x2692, 0x4AFF, 0xFE48, 0x9225, 0xCC74, 0xA019, 0x14AE, 0x78C3, 0x61DD, 0x0DB0, 0xB907, 0xD56A, 0x06C6, 0x6AAB, 0xDE1C, 0xB271, 0xAB6F, 0xC702, 0x73B5, 0x1FD8, 0x4189, 0x2DE4, 0x9953, 0xF53E, 0xEC20, 0x804D, 0x34FA, 0x5897, 0x8858, 0xE435, 0x5082, 0x3CEF, 0x25F1, 0x499C, 0xFD2B, 0x9146, 0xCF17, 0xA37A, 0x17CD, 0x7BA0, 0x62BE, 0x0ED3, 0xBA64, 0xD609, 0x07E7, 0x6B8A, 0xDF3D, 0xB350, 0xAA4E, 0xC623, 0x7294, 0x1EF9, 0x40A8, 0x2CC5, 0x9872, 0xF41F, 0xED01, 0x816C, 0x35DB, 0x59B6, 0x8979, 0xE514, 0x51A3, 0x3DCE, 0x24D0, 0x48BD, 0xFC0A, 0x9067, 0xCE36, 0xA25B, 0x16EC, 0x7A81, 0x639F, 0x0FF2, 0xBB45, 0xD728}, /* 02 */
    /* 03 */ {0x0000, 0xB8B9, 0x6D6F, 0xD5D6, 0xDADE, 0x6267, 0xB7B1, 0x0F08, 0xA9A1, 0x1118, 0xC4CE, 0x7C77, 0x737F, 0xCBC6, 0x1E10, 0xA6A9, 0x4F5F, 0xF7E6, 0x2230, 0x9A89, 0x9581, 0x2D38, 0xF8EE, 0x4057, 0xE6FE, 0x5E47, 0x8B91, 0x3328, 0x3C20, 0x8499, 0x514F, 0xE9F6, 0x9EBE, 0x2607, 0xF3D1, 0x4B68, 0x4460, 0xFCD9, 0x290F, 0x91B6, 0x371F, 0x8FA6, 0x5A70, 0xE2C9, 0xEDC1, 0x5578, 0x80AE, 0x3817, 0xD1E1, 0x6958, 0xBC8E, 0x0437, 0x0B3F, 0xB386, 0x6650, 0xDEE9, 0x7840, 0xC0F9, 0x152F, 0xAD96, 0xA29E, 0x1A27, 0xCFF1, 0x7748, 0x2161, 0x99D8, 0x4C0E, 0xF4B7, 0xFBBF, 0x4306, 0x96D0, 0x2E69, 0x88C0, 0x3079, 0xE5AF, 0x5D16, 0x521E, 0xEAA7, 0x3F71, 0x87C8, 0x6E3E, 0xD687, 0x0351, 0xBBE8, 0xB4E0, 0x0C59, 0xD98F, 0x6136, 0xC79F, 0x7F26, 0xAAF0, 0x1249, 0x1D41, 0xA5F8, 0x702E, 0xC897, 0xBFDF, 0x0766, 0xD2B0, 0x6A09, 0x6501, 0xDDB8, 0x086E, 0xB0D7, 0x167E, 0xAEC7, 0x7B11, 0xC3A8, 0xCCA0, 0x7419, 0xA1CF, 0x1976, 0xF080, 0x4839, 0x9DEF, 0x2556, 0x2A5E, 0x92E7, 0x4731, 0xFF88, 0x5921, 0xE198, 0x344E, 0x8CF7, 0x83FF, 0x3B46, 0xEE90, 0x5629, 0x42C2, 0xFA7B, 0x2FAD, 0x9714, 0x981C, 0x20A5, 0xF573, 0x4DCA, 0xEB63, 0x53DA, 0x860C, 0x3EB5, 0x31BD, 0x8904, 0x5CD2, 0xE46B, 0x0D9D, 0xB524, 0x60F2, 0xD84B, 0xD743, 0x6FFA, 0xBA2C, 0x0295, 0xA43C, 0x1C85, 0xC953, 0x71EA, 0x7EE2, 0xC65B, 0x138D, 0xAB34, 0xDC7C, 0x64C5, 0xB113, 0x09AA, 0x06A2, 0xBE1B, 0x6BCD, 0xD374, 0x75DD, 0xCD64, 0x18B2, 0xA00B, 0xAF03, 0x17BA, 0xC26C, 0x7AD5, 0x9323, 0x2B9A, 0xFE4C, 0x46F5, 0x49FD, 0xF144, 0x2492, 0x9C2B, 0x3A82, 0x823B, 0x57ED, 0xEF54, 0xE05C, 0x58E5, 0x8D33, 0x358A, 0x63A3, 0xDB1A, 0x0ECC, 0xB675, 0xB97D, 0x01C4, 0xD412, 0x6CAB, 0xCA02, 0x72BB, 0xA76D, 0x1FD4, 0x10DC, 0xA865, 0x7DB3, 0xC50A, 0x2CFC, 0x9445, 0x4193, 0xF92A, 0xF622, 0x4E9B, 0x9B4D, 0x23F4, 0x855D, 0x3DE4, 0xE832, 0x508B, 0x5F83, 0xE73A, 0x32EC, 0x8A55, 0xFD1D, 0x45A4, 0x9072, 0x28CB, 0x27C3, 0x9F7A, 0x4AAC, 0xF215, 0x54BC, 0xEC05, 0x39D3, 0x816A, 0x8E62, 0x36DB, 0xE30D, 0x5BB4, 0xB242, 0x0AFB, 0xDF2D, 0x6794, 0x689C, 0xD025, 0x05F3, 0xBD4A, 0x1BE3, 0xA35A, 0x768C, 0xCE35, 0xC13D, 0x7984, 0xAC52, 0x14EB}, /* 03 */
    /* 04 */ {0x0000, 0xD2D3, 0xB9BB, 0x6B68, 0x6F6B, 0xBDB8, 0xD6D0, 0x0403, 0xDED6, 0x0C05, 0x676D, 0xB5BE, 0xB1BD, 0x636E, 0x0806, 0xDAD5, 0xA1B1, 0x7362, 0x180A, 0xCAD9, 0xCEDA, 0x1C09, 0x7761, 0xA5B2, 0x7F67, 0xADB4, 0xC6DC, 0x140F, 0x100C, 0xC2DF, 0xA9B7, 0x7B64, 0x5F7F, 0x8DAC, 0xE6C4, 0x3417, 0x3014, 0xE2C7, 0x89AF, 0x5B7C, 0x81A9, 0x537A, 0x3812, 0xEAC1, 0xEEC2, 0x3C11, 0x5779, 0x85AA, 0xFECE, 0x2C1D, 0x4775, 0x95A6, 0x91A5, 0x4376, 0x281E, 0xFACD, 0x2018, 0xF2CB, 0x99A3, 0x4B70, 0x4F73, 0x9DA0, 0xF6C8, 0x241B, 0xBEFE, 0x6C2D, 0x0745, 0xD596, 0xD195, 0x0346, 0x682E, 0xBAFD, 0x6028, 0xB2FB, 0xD993, 0x0B40, 0x0F43, 0xDD90, 0xB6F8, 0x642B, 0x1F4F, 0xCD9C, 0xA6F4, 0x7427, 0x7024, 0xA2F7, 0xC99F, 0x1B4C, 0xC199, 0x134A, 0x7822, 0xAAF1, 0xAEF2, 0x7C21, 0x1749, 0xC59A, 0xE181, 0x3352, 0x583A, 0x8AE9, 0x8EEA, 0x5C39, 0x3751, 0xE582, 0x3F57, 0xED84, 0x86EC, 0x543F, 0x503C, 0x82EF, 0xE987, 0x3B54, 0x4030, 0x92E3, 0xF98B, 0x2B58, 0x2F5B, 0xFD88, 0x96E0, 0x4433, 0x9EE6, 0x4C35, 0x275D, 0xF58E, 0xF18D, 0x235E, 0x4836, 0x9AE5, 0x61E1, 0xB332, 0xD85A, 0x0A89, 0x0E8A, 0xDC59, 0xB731, 0x65E2, 0xBF37, 0x6DE4, 0x068C, 0xD45F, 0xD05C, 0x028F, 0x69E7, 0xBB34, 0xC050, 0x1283, 0x79EB, 0xAB38, 0xAF3B, 0x7DE8, 0x1680, 0xC453, 0x1E86, 0xCC55, 0xA73D, 0x75EE, 0x71ED, 0xA33E, 0xC856, 0x1A85, 0x3E9E, 0xEC4D, 0x8725, 0x55F6, 0x51F5, 0x8326, 0xE84E, 0x3A9D, 0xE048, 0x329B, 0x59F3, 0x8B20, 0x8F23, 0x5DF0, 0x3698, 0xE44B, 0x9F2F, 0x4DFC, 0x2694, 0xF447, 0xF044, 0x2297, 0x49FF, 0x9B2C, 0x41F9, 0x932A, 0xF842, 0x2A91, 0x2E92, 0xFC41, 0x9729, 0x45FA, 0xDF1F, 0x0DCC, 0x66A4, 0xB477, 0xB074, 0x62A7, 0x09CF, 0xDB1C, 0x01C9, 0xD31A, 0xB872, 0x6AA1, 0x6EA2, 0xBC71, 0xD719, 0x05CA, 0x7EAE, 0xAC7D, 0xC715, 0x15C6, 0x11C5, 0xC316, 0xA87E, 0x7AAD, 0xA078, 0x72AB, 0x19C3, 0xCB10, 0xCF13, 0x1DC0, 0x76A8, 0xA47B, 0x8060, 0x52B3, 0x39DB, 0xEB08, 0xEF0B, 0x3DD8, 0x56B0, 0x8463, 0x5EB6, 0x8C65, 0xE70D, 0x35DE, 0x31DD, 0xE30E, 0x8866, 0x5AB5, 0x21D1, 0xF302, 0x986A, 0x4AB9, 0x4EBA, 0x9C69, 0xF701, 0x25D2, 0xFF07, 0x2DD4, 0x46BC, 0x946F, 0x906C, 0x42BF, 0x29D7, 0xFB04}, /* 04 */
    /* 05 */ {0x0000, 0xE7E6, 0xD3D1, 0x3437, 0xBBBF, 0x5C59, 0x686E, 0x8F88, 0x6B63, 0x8C85, 0xB8B2, 0x5F54, 0xD0DC, 0x373A, 0x030D, 0xE4EB, 0xD6C6, 0x3120, 0x0517, 0xE2F1, 0x6D79, 0x8A9F, 0xBEA8, 0x594E, 0xBDA5, 0x5A43, 0x6E74, 0x8992, 0x061A, 0xE1FC, 0xD5CB, 0x322D, 0xB191, 0x5677, 0x6240, 0x85A6, 0x0A2E, 0xEDC8, 0xD9FF, 0x3E19, 0xDAF2, 0x3D14, 0x0923, 0xEEC5, 0x614D, 0x86AB, 0xB29C, 0x557A, 0x6757, 0x80B1, 0xB486, 0x5360, 0xDCE8, 0x3B0E, 0x0F39, 0xE8DF, 0x0C34, 0xEBD2, 0xDFE5, 0x3803, 0xB78B, 0x506D, 0x645A, 0x83BC, 0x7F3F, 0x98D9, 0xACEE, 0x4B08, 0xC480, 0x2366, 0x1751, 0xF0B7, 0x145C, 0xF3BA, 0xC78D, 0x206B, 0xAFE3, 0x4805, 0x7C32, 0x9BD4, 0xA9F9, 0x4E1F, 0x7A28, 0x9DCE, 0x1246, 0xF5A0, 0xC197, 0x2671, 0xC29A, 0x257C, 0x114B, 0xF6AD, 0x7925, 0x9EC3, 0xAAF4, 0x4D12, 0xCEAE, 0x2948, 0x1D7F, 0xFA99, 0x7511, 0x92F7, 0xA6C0, 0x4126, 0xA5CD, 0x422B, 0x761C, 0x91FA, 0x1E72, 0xF994, 0xCDA3, 0x2A45, 0x1868, 0xFF8E, 0xCBB9, 0x2C5F, 0xA3D7, 0x4431, 0x7006, 0x97E0, 0x730B, 0x94ED, 0xA0DA, 0x473C, 0xC8B4, 0x2F52, 0x1B65, 0xFC83, 0xFE7E, 0x1998, 0x2DAF, 0xCA49, 0x45C1, 0xA227, 0x9610, 0x71F6, 0x951D, 0x72FB, 0x46CC, 0xA12A, 0x2EA2, 0xC944, 0xFD73, 0x1A95, 0x28B8, 0xCF5E, 0xFB69, 0x1C8F, 0x9307, 0x74E1, 0x40D6, 0xA730, 0x43DB, 0xA43D, 0x900A, 0x77EC, 0xF864, 0x1F82, 0x2BB5, 0xCC53, 0x4FEF, 0xA809, 0x9C3E, 0x7BD8, 0xF450, 0x13B6, 0x2781, 0xC067, 0x248C, 0xC36A, 0xF75D, 0x10BB, 0x9F33, 0x78D5, 0x4CE2, 0xAB04, 0x9929, 0x7ECF, 0x4AF8, 0xAD1E, 0x2296, 0xC570, 0xF147, 0x16A1, 0xF24A, 0x15AC, 0x219B, 0xC67D, 0x49F5, 0xAE13, 0x9A24, 0x7DC2, 0x8141, 0x66A7, 0x5290, 0xB576, 0x3AFE, 0xDD18, 0xE92F, 0x0EC9, 0xEA22, 0x0DC4, 0x39F3, 0xDE15, 0x519D, 0xB67B, 0x824C, 0x65AA, 0x5787, 0xB061, 0x8456, 0x63B0, 0xEC38, 0x0BDE, 0x3FE9, 0xD80F, 0x3CE4, 0xDB02, 0xEF35, 0x08D3, 0x875B, 0x60BD, 0x548A, 0xB36C, 0x30D0, 0xD736, 0xE301, 0x04E7, 0x8B6F, 0x6C89, 0x58BE, 0xBF58, 0x5BB3, 0xBC55, 0x8862, 0x6F84, 0xE00C, 0x07EA, 0x33DD, 0xD43B, 0xE616, 0x01F0, 0x35C7, 0xD221, 0x5DA9, 0xBA4F, 0x8E78, 0x699E, 0x8D75, 0x6A93, 0x5EA4, 0xB942, 0x36CA, 0xD12C, 0xE51B, 0x02FD}, /* 05 */
    /* 06 */ {0x0000, 0x7372, 0xE6E4, 0x9596, 0xD1D5, 0xA2A7, 0x3731, 0x4443, 0xBFB7, 0xCCC5, 0x5953, 0x2A21, 0x6E62, 0x1D10, 0x8886, 0xFBF4, 0x6373, 0x1001, 0x8597, 0xF6E5, 0xB2A6, 0xC1D4, 0x5442, 0x2730, 0xDCC4, 0xAFB6, 0x3A20, 0x4952, 0x0D11, 0x7E63, 0xEBF5, 0x9887, 0xC6E6, 0xB594, 0x2002, 0x5370, 0x1733, 0x6441, 0xF1D7, 0x82A5, 0x7951, 0x0A23, 0x9FB5, 0xECC7, 0xA884, 0xDBF6, 0x4E60, 0x3D12, 0xA595, 0xD6E7, 0x4371, 0x3003, 0x7440, 0x0732, 0x92A4, 0xE1D6, 0x1A22, 0x6950, 0xFCC6, 0x8FB4, 0xCBF7, 0xB885, 0x2D13, 0x5E61, 0x91D1, 0xE2A3, 0x7735, 0x0447, 0x4004, 0x3376, 0xA6E0, 0xD592, 0x2E66, 0x5D14, 0xC882, 0xBBF0, 0xFFB3, 0x8CC1, 0x1957, 0x6A25, 0xF2A2, 0x81D0, 0x1446, 0x6734, 0x2377, 0x5005, 0xC593, 0xB6E1, 0x4D15, 0x3E67, 0xABF1, 0xD883, 0x9CC0, 0xEFB2, 0x7A24, 0x0956, 0x5737, 0x2445, 0xB1D3, 0xC2A1, 0x86E2, 0xF590, 0x6006, 0x1374, 0xE880, 0x9BF2, 0x0E64, 0x7D16, 0x3955, 0x4A27, 0xDFB1, 0xACC3, 0x3444, 0x4736, 0xD2A0, 0xA1D2, 0xE591, 0x96E3, 0x0375, 0x7007, 0x8BF3, 0xF881, 0x6D17, 0x1E65, 0x5A26, 0x2954, 0xBCC2, 0xCFB0, 0x3FBF, 0x4CCD, 0xD95B, 0xAA29, 0xEE6A, 0x9D18, 0x088E, 0x7BFC, 0x8008, 0xF37A, 0x66EC, 0x159E, 0x51DD, 0x22AF, 0xB739, 0xC44B, 0x5CCC, 0x2FBE, 0xBA28, 0xC95A, 0x8D19, 0xFE6B, 0x6BFD, 0x188F, 0xE37B, 0x9009, 0x059F, 0x76ED, 0x32AE, 0x41DC, 0xD44A, 0xA738, 0xF959, 0x8A2B, 0x1FBD, 0x6CCF, 0x288C, 0x5BFE, 0xCE68, 0xBD1A, 0x46EE, 0x359C, 0xA00A, 0xD378, 0x973B, 0xE449, 0x71DF, 0x02AD, 0x9A2A, 0xE958, 0x7CCE, 0x0FBC, 0x4BFF, 0x388D, 0xAD1B, 0xDE69, 0x259D, 0x56EF, 0xC379, 0xB00B, 0xF448, 0x873A, 0x12AC, 0x61DE, 0xAE6E, 0xDD1C, 0x488A, 0x3BF8, 0x7FBB, 0x0CC9, 0x995F, 0xEA2D, 0x11D9, 0x62AB, 0xF73D, 0x844F, 0xC00C, 0xB37E, 0x26E8, 0x559A, 0xCD1D, 0xBE6F, 0x2BF9, 0x588B, 0x1CC8, 0x6FBA, 0xFA2C, 0x895E, 0x72AA, 0x01D8, 0x944E, 0xE73C, 0xA37F, 0xD00D, 0x459B, 0x36E9, 0x6888, 0x1BFA, 0x8E6C, 0xFD1E, 0xB95D, 0xCA2F, 0x5FB9, 0x2CCB, 0xD73F, 0xA44D, 0x31DB, 0x42A9, 0x06EA, 0x7598, 0xE00E, 0x937C, 0x0BFB, 0x7889, 0xED1F, 0x9E6D, 0xDA2E, 0xA95C, 0x3CCA, 0x4FB8, 0xB44C, 0xC73E, 0x52A8, 0x21DA, 0x6599, 0x16EB, 0x837D, 0xF00F}, /* 06 */
    /* 07 */ {0x0000, 0x3938, 0x7270, 0x4B48, 0xE4E0, 0xDDD8, 0x9690, 0xAFA8, 0xD5DD, 0xECE5, 0xA7AD, 0x9E95, 0x313D, 0x0805, 0x434D, 0x7A75, 0xB7A7, 0x8E9F, 0xC5D7, 0xFCEF, 0x5347, 0x6A7F, 0x2137, 0x180F, 0x627A, 0x5B42, 0x100A, 0x2932, 0x869A, 0xBFA2, 0xF4EA, 0xCDD2, 0x7353, 0x4A6B, 0x0123, 0x381B, 0x97B3, 0xAE8B, 0xE5C3, 0xDCFB, 0xA68E, 0x9FB6, 0xD4FE, 0xEDC6, 0x426E, 0x7B56, 0x301E, 0x0926, 0xC4F4, 0xFDCC, 0xB684, 0x8FBC, 0x2014, 0x192C, 0x5264, 0x6B5C, 0x1129, 0x2811, 0x6359, 0x5A61, 0xF5C9, 0xCCF1, 0x87B9, 0xBE81, 0xE6A6, 0xDF9E, 0x94D6, 0xADEE, 0x0246, 0x3B7E, 0x7036, 0x490E, 0x337B, 0x0A43, 0x410B, 0x7833, 0xD79B, 0xEEA3, 0xA5EB, 0x9CD3, 0x5101, 0x6839, 0x2371, 0x1A49, 0xB5E1, 0x8CD9, 0xC791, 0xFEA9, 0x84DC, 0xBDE4, 0xF6AC, 0xCF94, 0x603C, 0x5904, 0x124C, 0x2B74, 0x95F5, 0xACCD, 0xE785, 0xDEBD, 0x7115, 0x482D, 0x0365, 0x3A5D, 0x4028, 0x7910, 0x3258, 0x0B60, 0xA4C8, 0x9DF0, 0xD6B8, 0xEF80, 0x2252, 0x1B6A, 0x5022, 0x691A, 0xC6B2, 0xFF8A, 0xB4C2, 0x8DFA, 0xF78F, 0xCEB7, 0x85FF, 0xBCC7, 0x136F, 0x2A57, 0x611F, 0x5827, 0xD151, 0xE869, 0xA321, 0x9A19, 0x35B1, 0x0C89, 0x47C1, 0x7EF9, 0x048C, 0x3DB4, 0x76FC, 0x4FC4, 0xE06C, 0xD954, 0x921C, 0xAB24, 0x66F6, 0x5FCE, 0x1486, 0x2DBE, 0x8216, 0xBB2E, 0xF066, 0xC95E, 0xB32B, 0x8A13, 0xC15B, 0xF863, 0x57CB, 0x6EF3, 0x25BB, 0x1C83, 0xA202, 0x9B3A, 0xD072, 0xE94A, 0x46E2, 0x7FDA, 0x3492, 0x0DAA, 0x77DF, 0x4EE7, 0x05AF, 0x3C97, 0x933F, 0xAA07, 0xE14F, 0xD877, 0x15A5, 0x2C9D, 0x67D5, 0x5EED, 0xF145, 0xC87D, 0x8335, 0xBA0D, 0xC078, 0xF940, 0xB208, 0x8B30, 0x2498, 0x1DA0, 0x56E8, 0x6FD0, 0x37F7, 0x0ECF, 0x4587, 0x7CBF, 0xD317, 0xEA2F, 0xA167, 0x985F, 0xE22A, 0xDB12, 0x905A, 0xA962, 0x06CA, 0x3FF2, 0x74BA, 0x4D82, 0x8050, 0xB968, 0xF220, 0xCB18, 0x64B0, 0x5D88, 0x16C0, 0x2FF8, 0x558D, 0x6CB5, 0x27FD, 0x1EC5, 0xB16D, 0x8855, 0xC31D, 0xFA25, 0x44A4, 0x7D9C, 0x36D4, 0x0FEC, 0xA044, 0x997C, 0xD234, 0xEB0C, 0x9179, 0xA841, 0xE309, 0xDA31, 0x7599, 0x4CA1, 0x07E9, 0x3ED1, 0xF303, 0xCA3B, 0x8173, 0xB84B, 0x17E3, 0x2EDB, 0x6593, 0x5CAB, 0x26DE, 0x1FE6, 0x54AE, 0x6D96, 0xC23E, 0xFB06, 0xB04E, 0x8976}, /* 07 */
    /* 08 */ {0x0000, 0x1C1D, 0x383A, 0x2427, 0x7074, 0x6C69, 0x484E, 0x5453, 0xE0E8, 0xFCF5, 0xD8D2, 0xC4CF, 0x909C, 0x8C81, 0xA8A6, 0xB4BB, 0xDDCD, 0xC1D0, 0xE5F7, 0xF9EA, 0xADB9, 0xB1A4, 0x9583, 0x899E, 0x3D25, 0x2138, 0x051F, 0x1902, 0x4D51, 0x514C, 0x756B, 0x6976, 0xA787, 0xBB9A, 0x9FBD, 0x83A0, 0xD7F3, 0xCBEE, 0xEFC9, 0xF3D4, 0x476F, 0x5B72, 0x7F55, 0x6348, 0x371B, 0x2B06, 0x0F21, 0x133C, 0x7A4A, 0x6657, 0x4270, 0x5E6D, 0x0A3E, 0x1623, 0x3204, 0x2E19, 0x9AA2, 0x86BF, 0xA298, 0xBE85, 0xEAD6, 0xF6CB, 0xD2EC, 0xCEF1, 0x5313, 0x4F0E, 0x6B29, 0x7734, 0x2367, 0x3F7A, 0x1B5D, 0x0740, 0xB3FB, 0xAFE6, 0x8BC1, 0x97DC, 0xC38F, 0xDF92, 0xFBB5, 0xE7A8, 0x8EDE, 0x92C3, 0xB6E4, 0xAAF9, 0xFEAA, 0xE2B7, 0xC690, 0xDA8D, 0x6E36, 0x722B, 0x560C, 0x4A11, 0x1E42, 0x025F, 0x2678, 0x3A65, 0xF494, 0xE889, 0xCCAE, 0xD0B3, 0x84E0, 0x98FD, 0xBCDA, 0xA0C7, 0x147C, 0x0861, 0x2C46, 0x305B, 0x6408, 0x7815, 0x5C32, 0x402F, 0x2959, 0x3544, 0x1163, 0x0D7E, 0x592D, 0x4530, 0x6117, 0x7D0A, 0xC9B1, 0xD5AC, 0xF18B, 0xED96, 0xB9C5, 0xA5D8, 0x81FF, 0x9DE2, 0xA626, 0xBA3B, 0x9E1C, 0x8201, 0xD652, 0xCA4F, 0xEE68, 0xF275, 0x46CE, 0x5AD3, 0x7EF4, 0x62E9, 0x36BA, 0x2AA7, 0x0E80, 0x129D, 0x7BEB, 0x67F6, 0x43D1, 0x5FCC, 0x0B9F, 0x1782, 0x33A5, 0x2FB8, 0x9B03, 0x871E, 0xA339, 0xBF24, 0xEB77, 0xF76A, 0xD34D, 0xCF50, 0x01A1, 0x1DBC, 0x399B, 0x2586, 0x71D5, 0x6DC8, 0x49EF, 0x55F2, 0xE149, 0xFD54, 0xD973, 0xC56E, 0x913D, 0x8D20, 0xA907, 0xB51A, 0xDC6C, 0xC071, 0xE456, 0xF84B, 0xAC18, 0xB005, 0x9422, 0x883F, 0x3C84, 0x2099, 0x04BE, 0x18A3, 0x4CF0, 0x50ED, 0x74CA, 0x68D7, 0xF535, 0xE928, 0xCD0F, 0xD112, 0x8541, 0x995C, 0xBD7B, 0xA166, 0x15DD, 0x09C0, 0x2DE7, 0x31FA, 0x65A9, 0x79B4, 0x5D93, 0x418E, 0x28F8, 0x34E5, 0x10C2, 0x0CDF, 0x588C, 0x4491, 0x60B6, 0x7CAB, 0xC810, 0xD40D, 0xF02A, 0xEC37, 0xB864, 0xA479, 0x805E, 0x9C43, 0x52B2, 0x4EAF, 0x6A88, 0x7695, 0x22C6, 0x3EDB, 0x1AFC, 0x06E1, 0xB25A, 0xAE47, 0x8A60, 0x967D, 0xC22E, 0xDE33, 0xFA14, 0xE609, 0x8F7F, 0x9362, 0xB745, 0xAB58, 0xFF0B, 0xE316, 0xC731, 0xDB2C, 0x6F97, 0x738A, 0x57AD, 0x4BB0, 0x1FE3, 0x03FE, 0x27D9, 0x3BC4}, /* 08 */
    /* 09 */ {0x0000, 0x8081, 0x1D1F, 0x9D9E, 0x3A3E, 0xBABF, 0x2721, 0xA7A0, 0x747C, 0xF4FD, 0x6963, 0xE9E2, 0x4E42, 0xCEC3, 0x535D, 0xD3DC, 0xE8F8, 0x6879, 0xF5E7, 0x7566, 0xD2C6, 0x5247, 0xCFD9, 0x4F58, 0x9C84, 0x1C05, 0x819B, 0x011A, 0xA6BA, 0x263B, 0xBBA5, 0x3B24, 0xCDED, 0x4D6C, 0xD0F2, 0x5073, 0xF7D3, 0x7752, 0xEACC, 0x6A4D, 0xB991, 0x3910, 0xA48E, 0x240F, 0x83AF, 0x032E, 0x9EB0, 0x1E31, 0x2515, 0xA594, 0x380A, 0xB88B, 0x1F2B, 0x9FAA, 0x0234, 0x82B5, 0x5169, 0xD1E8, 0x4C76, 0xCCF7, 0x6B57, 0xEBD6, 0x7648, 0xF6C9, 0x87C7, 0x0746, 0x9AD8, 0x1A59, 0xBDF9, 0x3D78, 0xA0E6, 0x2067, 0xF3BB, 0x733A, 0xEEA4, 0x6E25, 0xC985, 0x4904, 0xD49A, 0x541B, 0x6F3F, 0xEFBE, 0x7220, 0xF2A1, 0x5501, 0xD580, 0x481E, 0xC89F, 0x1B43, 0x9BC2, 0x065C, 0x86DD, 0x217D, 0xA1FC, 0x3C62, 0xBCE3, 0x4A2A, 0xCAAB, 0x5735, 0xD7B4, 0x7014, 0xF095, 0x6D0B, 0xED8A, 0x3E56, 0xBED7, 0x2349, 0xA3C8, 0x0468, 0x84E9, 0x1977, 0x99F6, 0xA2D2, 0x2253, 0xBFCD, 0x3F4C, 0x98EC, 0x186D, 0x85F3, 0x0572, 0xD6AE, 0x562F, 0xCBB1, 0x4B30, 0xEC90, 0x6C11, 0xF18F, 0x710E, 0x1393, 0x9312, 0x0E8C, 0x8E0D, 0x29AD, 0xA92C, 0x34B2, 0xB433, 0x67EF, 0xE76E, 0x7AF0, 0xFA71, 0x5DD1, 0xDD50, 0x40CE, 0xC04F, 0xFB6B, 0x7BEA, 0xE674, 0x66F5, 0xC155, 0x41D4, 0xDC4A, 0x5CCB, 0x8F17, 0x0F96, 0x9208, 0x1289, 0xB529, 0x35A8, 0xA836, 0x28B7, 0xDE7E, 0x5EFF, 0xC361, 0x43E0, 0xE440, 0x64C1, 0xF95F, 0x79DE, 0xAA02, 0x2A83, 0xB71D, 0x379C, 0x903C, 0x10BD, 0x8D23, 0x0DA2, 0x3686, 0xB607, 0x2B99, 0xAB18, 0x0CB8, 0x8C39, 0x11A7, 0x9126, 0x42FA, 0xC27B, 0x5FE5, 0xDF64, 0x78C4, 0xF845, 0x65DB, 0xE55A, 0x9454, 0x14D5, 0x894B, 0x09CA, 0xAE6A, 0x2EEB, 0xB375, 0x33F4, 0xE028, 0x60A9, 0xFD37, 0x7DB6, 0xDA16, 0x5A97, 0xC709, 0x4788, 0x7CAC, 0xFC2D, 0x61B3, 0xE132, 0x4692, 0xC613, 0x5B8D, 0xDB0C, 0x08D0, 0x8851, 0x15CF, 0x954E, 0x32EE, 0xB26F, 0x2FF1, 0xAF70, 0x59B9, 0xD938, 0x44A6, 0xC427, 0x6387, 0xE306, 0x7E98, 0xFE19, 0x2DC5, 0xAD44, 0x30DA, 0xB05B, 0x17FB, 0x977A, 0x0AE4, 0x8A65, 0xB141, 0x31C0, 0xAC5E, 0x2CDF, 0x8B7F, 0x0BFE, 0x9660, 0x16E1, 0xC53D, 0x45BC, 0xD822, 0x58A3, 0xFF03, 0x7F82, 0xE21C, 0x629D}, /* 09 */
    /* 0A */ {0x0000, 0xCECF, 0x8183, 0x4F4C, 0x1F1B, 0xD1D4, 0x9E98, 0x5057, 0x3E36, 0xF0F9, 0xBFB5, 0x717A, 0x212D, 0xEFE2, 0xA0AE, 0x6E61, 0x7C6C, 0xB2A3, 0xFDEF, 0x3320, 0x6377, 0xADB8, 0xE2F4, 0x2C3B, 0x425A, 0x8C95, 0xC3D9, 0x0D16, 0x5D41, 0x938E, 0xDCC2, 0x120D, 0xF8D8, 0x3617, 0x795B, 0xB794, 0xE7C3, 0x290C, 0x6640, 0xA88F, 0xC6EE, 0x0821, 0x476D, 0x89A2, 0xD9F5, 0x173A, 0x5876, 0x96B9, 0x84B4, 0x4A7B, 0x0537, 0xCBF8, 0x9BAF, 0x5560, 0x1A2C, 0xD4E3, 0xBA82, 0x744D, 0x3B01, 0xF5CE, 0xA599, 0x6B56, 0x241A, 0xEAD5, 0xEDAD, 0x2362, 0x6C2E, 0xA2E1, 0xF2B6, 0x3C79, 0x7335, 0xBDFA, 0xD39B, 0x1D54, 0x5218, 0x9CD7, 0xCC80, 0x024F, 0x4D03, 0x83CC, 0x91C1, 0x5F0E, 0x1042, 0xDE8D, 0x8EDA, 0x4015, 0x0F59, 0xC196, 0xAFF7, 0x6138, 0x2E74, 0xE0BB, 0xB0EC, 0x7E23, 0x316F, 0xFFA0, 0x1575, 0xDBBA, 0x94F6, 0x5A39, 0x0A6E, 0xC4A1, 0x8BED, 0x4522, 0x2B43, 0xE58C, 0xAAC0, 0x640F, 0x3458, 0xFA97, 0xB5DB, 0x7B14, 0x6919, 0xA7D6, 0xE89A, 0x2655, 0x7602, 0xB8CD, 0xF781, 0x394E, 0x572F, 0x99E0, 0xD6AC, 0x1863, 0x4834, 0x86FB, 0xC9B7, 0x0778, 0xC747, 0x0988, 0x46C4, 0x880B, 0xD85C, 0x1693, 0x59DF, 0x9710, 0xF971, 0x37BE, 0x78F2, 0xB63D, 0xE66A, 0x28A5, 0x67E9, 0xA926, 0xBB2B, 0x75E4, 0x3AA8, 0xF467, 0xA430, 0x6AFF, 0x25B3, 0xEB7C, 0x851D, 0x4BD2, 0x049E, 0xCA51, 0x9A06, 0x54C9, 0x1B85, 0xD54A, 0x3F9F, 0xF150, 0xBE1C, 0x70D3, 0x2084, 0xEE4B, 0xA107, 0x6FC8, 0x01A9, 0xCF66, 0x802A, 0x4EE5, 0x1EB2, 0xD07D, 0x9F31, 0x51FE, 0x43F3, 0x8D3C, 0xC270, 0x0CBF, 0x5CE8, 0x9227, 0xDD6B, 0x13A4, 0x7DC5, 0xB30A, 0xFC46, 0x3289, 0x62DE, 0xAC11, 0xE35D, 0x2D92, 0x2AEA, 0xE425, 0xAB69, 0x65A6, 0x35F1, 0xFB3E, 0xB472, 0x7ABD, 0x14DC, 0xDA13, 0x955F, 0x5B90, 0x0BC7, 0xC508, 0x8A44, 0x448B, 0x5686, 0x9849, 0xD705, 0x19CA, 0x499D, 0x8752, 0xC81E, 0x06D1, 0x68B0, 0xA67F, 0xE933, 0x27FC, 0x77AB, 0xB964, 0xF628, 0x38E7, 0xD232, 0x1CFD, 0x53B1, 0x9D7E, 0xCD29, 0x03E6, 0x4CAA, 0x8265, 0xEC04, 0x22CB, 0x6D87, 0xA348, 0xF31F, 0x3DD0, 0x729C, 0xBC53, 0xAE5E, 0x6091, 0x2FDD, 0xE112, 0xB145, 0x7F8A, 0x30C6, 0xFE09, 0x9068, 0x5EA7, 0x11EB, 0xDF24, 0x8F73, 0x41BC, 0x0EF0, 0xC03F}, /* 0A */
    /* 0B */ {0x0000, 0xE9E8, 0xCFCD, 0x2625, 0x8387, 0x6A6F, 0x4C4A, 0xA5A2, 0x1B13, 0xF2FB, 0xD4DE, 0x3D36, 0x9894, 0x717C, 0x5759, 0xBEB1, 0x3626, 0xDFCE, 0xF9EB, 0x1003, 0xB5A1, 0x5C49, 0x7A6C, 0x9384, 0x2D35, 0xC4DD, 0xE2F8, 0x0B10, 0xAEB2, 0x475A, 0x617F, 0x8897, 0x6C4C, 0x85A4, 0xA381, 0x4A69, 0xEFCB, 0x0623, 0x2006, 0xC9EE, 0x775F, 0x9EB7, 0xB892, 0x517A, 0xF4D8, 0x1D30, 0x3B15, 0xD2FD, 0x5A6A, 0xB382, 0x95A7, 0x7C4F, 0xD9ED, 0x3005, 0x1620, 0xFFC8, 0x4179, 0xA891, 0x8EB4, 0x675C, 0xC2FE, 0x2B16, 0x0D33, 0xE4DB, 0xD898, 0x3170, 0x1755, 0xFEBD, 0x5B1F, 0xB2F7, 0x94D2, 0x7D3A, 0xC38B, 0x2A63, 0x0C46, 0xE5AE, 0x400C, 0xA9E4, 0x8FC1, 0x6629, 0xEEBE, 0x0756, 0x2173, 0xC89B, 0x6D39, 0x84D1, 0xA2F4, 0x4B1C, 0xF5AD, 0x1C45, 0x3A60, 0xD388, 0x762A, 0x9FC2, 0xB9E7, 0x500F, 0xB4D4, 0x5D3C, 0x7B19, 0x92F1, 0x3753, 0xDEBB, 0xF89E, 0x1176, 0xAFC7, 0x462F, 0x600A, 0x89E2, 0x2C40, 0xC5A8, 0xE38D, 0x0A65, 0x82F2, 0x6B1A, 0x4D3F, 0xA4D7, 0x0175, 0xE89D, 0xCEB8, 0x2750, 0x99E1, 0x7009, 0x562C, 0xBFC4, 0x1A66, 0xF38E, 0xD5AB, 0x3C43, 0xAD2D, 0x44C5, 0x62E0, 0x8B08, 0x2EAA, 0xC742, 0xE167, 0x088F, 0xB63E, 0x5FD6, 0x79F3, 0x901B, 0x35B9, 0xDC51, 0xFA74, 0x139C, 0x9B0B, 0x72E3, 0x54C6, 0xBD2E, 0x188C, 0xF164, 0xD741, 0x3EA9, 0x8018, 0x69F0, 0x4FD5, 0xA63D, 0x039F, 0xEA77, 0xCC52, 0x25BA, 0xC161, 0x2889, 0x0EAC, 0xE744, 0x42E6, 0xAB0E, 0x8D2B, 0x64C3, 0xDA72, 0x339A, 0x15BF, 0xFC57, 0x59F5, 0xB01D, 0x9638, 0x7FD0, 0xF747, 0x1EAF, 0x388A, 0xD162, 0x74C0, 0x9D28, 0xBB0D, 0x52E5, 0xEC54, 0x05BC, 0x2399, 0xCA71, 0x6FD3, 0x863B, 0xA01E, 0x49F6, 0x75B5, 0x9C5D, 0xBA78, 0x5390, 0xF632, 0x1FDA, 0x39FF, 0xD017, 0x6EA6, 0x874E, 0xA16B, 0x4883, 0xED21, 0x04C9, 0x22EC, 0xCB04, 0x4393, 0xAA7B, 0x8C5E, 0x65B6, 0xC014, 0x29FC, 0x0FD9, 0xE631, 0x5880, 0xB168, 0x974D, 0x7EA5, 0xDB07, 0x32EF, 0x14CA, 0xFD22, 0x19F9, 0xF011, 0xD634, 0x3FDC, 0x9A7E, 0x7396, 0x55B3, 0xBC5B, 0x02EA, 0xEB02, 0xCD27, 0x24CF, 0x816D, 0x6885, 0x4EA0, 0xA748, 0x2FDF, 0xC637, 0xE012, 0x09FA, 0xAC58, 0x45B0, 0x6395, 0x8A7D, 0x34CC, 0xDD24, 0xFB01, 0x12E9, 0xB74B, 0x5EA3, 0x7886, 0x916E}, /* 0B */
    /* 0C */ {0x0000, 0x7475, 0xE8EA, 0x9C9F, 0xCDC9, 0xB9BC, 0x2523, 0x5156, 0x878F, 0xF3FA, 0x6F65, 0x1B10, 0x4A46, 0x3E33, 0xA2AC, 0xD6D9, 0x1303, 0x6776, 0xFBE9, 0x8F9C, 0xDECA, 0xAABF, 0x3620, 0x4255, 0x948C, 0xE0F9, 0x7C66, 0x0813, 0x5945, 0x2D30, 0xB1AF, 0xC5DA, 0x2606, 0x5273, 0xCEEC, 0xBA99, 0xEBCF, 0x9FBA, 0x0325, 0x7750, 0xA189, 0xD5FC, 0x4963, 0x3D16, 0x6C40, 0x1835, 0x84AA, 0xF0DF, 0x3505, 0x4170, 0xDDEF, 0xA99A, 0xF8CC, 0x8CB9, 0x1026, 0x6453, 0xB28A, 0xC6FF, 0x5A60, 0x2E15, 0x7F43, 0x0B36, 0x97A9, 0xE3DC, 0x4C0C, 0x3879, 0xA4E6, 0xD093, 0x81C5, 0xF5B0, 0x692F, 0x1D5A, 0xCB83, 0xBFF6, 0x2369, 0x571C, 0x064A, 0x723F, 0xEEA0, 0x9AD5, 0x5F0F, 0x2B7A, 0xB7E5, 0xC390, 0x92C6, 0xE6B3, 0x7A2C, 0x0E59, 0xD880, 0xACF5, 0x306A, 0x441F, 0x1549, 0x613C, 0xFDA3, 0x89D6, 0x6A0A, 0x1E7F, 0x82E0, 0xF695, 0xA7C3, 0xD3B6, 0x4F29, 0x3B5C, 0xED85, 0x99F0, 0x056F, 0x711A, 0x204C, 0x5439, 0xC8A6, 0xBCD3, 0x7909, 0x0D7C, 0x91E3, 0xE596, 0xB4C0, 0xC0B5, 0x5C2A, 0x285F, 0xFE86, 0x8AF3, 0x166C, 0x6219, 0x334F, 0x473A, 0xDBA5, 0xAFD0, 0x9818, 0xEC6D, 0x70F2, 0x0487, 0x55D1, 0x21A4, 0xBD3B, 0xC94E, 0x1F97, 0x6BE2, 0xF77D, 0x8308, 0xD25E, 0xA62B, 0x3AB4, 0x4EC1, 0x8B1B, 0xFF6E, 0x63F1, 0x1784, 0x46D2, 0x32A7, 0xAE38, 0xDA4D, 0x0C94, 0x78E1, 0xE47E, 0x900B, 0xC15D, 0xB528, 0x29B7, 0x5DC2, 0xBE1E, 0xCA6B, 0x56F4, 0x2281, 0x73D7, 0x07A2, 0x9B3D, 0xEF48, 0x3991, 0x4DE4, 0xD17B, 0xA50E, 0xF458, 0x802D, 0x1CB2, 0x68C7, 0xAD1D, 0xD968, 0x45F7, 0x3182, 0x60D4, 0x14A1, 0x883E, 0xFC4B, 0x2A92, 0x5EE7, 0xC278, 0xB60D, 0xE75B, 0x932E, 0x0FB1, 0x7BC4, 0xD414, 0xA061, 0x3CFE, 0x488B, 0x19DD, 0x6DA8, 0xF137, 0x8542, 0x539B, 0x27EE, 0xBB71, 0xCF04, 0x9E52, 0xEA27, 0x76B8, 0x02CD, 0xC717, 0xB362, 0x2FFD, 0x5B88, 0x0ADE, 0x7EAB, 0xE234, 0x9641, 0x4098, 0x34ED, 0xA872, 0xDC07, 0x8D51, 0xF924, 0x65BB, 0x11CE, 0xF212, 0x8667, 0x1AF8, 0x6E8D, 0x3FDB, 0x4BAE, 0xD731, 0xA344, 0x759D, 0x01E8, 0x9D77, 0xE902, 0xB854, 0xCC21, 0x50BE, 0x24CB, 0xE111, 0x9564, 0x09FB, 0x7D8E, 0x2CD8, 0x58AD, 0xC432, 0xB047, 0x669E, 0x12EB, 0x8E74, 0xFA01, 0xAB57, 0xDF22, 0x43BD, 0x37C8}, /* 0C */
    /* 0D */ {0x0000, 0xB4B5, 0x7577, 0xC1C2, 0xEAEE, 0x5E5B, 0x9F99, 0x2B2C, 0xC9C1, 0x7D74, 0xBCB6, 0x0803, 0x232F, 0x979A, 0x5658, 0xE2ED, 0x8F9F, 0x3B2A, 0xFAE8, 0x4E5D, 0x6571, 0xD1C4, 0x1006, 0xA4B3, 0x465E, 0xF2EB, 0x3329, 0x879C, 0xACB0, 0x1805, 0xD9C7, 0x6D72, 0x0323, 0xB796, 0x7654, 0xC2E1, 0xE9CD, 0x5D78, 0x9CBA, 0x280F, 0xCAE2, 0x7E57, 0xBF95, 0x0B20, 0x200C, 0x94B9, 0x557B, 0xE1CE, 0x8CBC, 0x3809, 0xF9CB, 0x4D7E, 0x6652, 0xD2E7, 0x1325, 0xA790, 0x457D, 0xF1C8, 0x300A, 0x84BF, 0xAF93, 0x1B26, 0xDAE4, 0x6E51, 0x0646, 0xB2F3, 0x7331, 0xC784, 0xECA8, 0x581D, 0x99DF, 0x2D6A, 0xCF87, 0x7B32, 0xBAF0, 0x0E45, 0x2569, 0x91DC, 0x501E, 0xE4AB, 0x89D9, 0x3D6C, 0xFCAE, 0x481B, 0x6337, 0xD782, 0x1640, 0xA2F5, 0x4018, 0xF4AD, 0x356F, 0x81DA, 0xAAF6, 0x1E43, 0xDF81, 0x6B34, 0x0565, 0xB1D0, 0x7012, 0xC4A7, 0xEF8B, 0x5B3E, 0x9AFC, 0x2E49, 0xCCA4, 0x7811, 0xB9D3, 0x0D66, 0x264A, 0x92FF, 0x533D, 0xE788, 0x8AFA, 0x3E4F, 0xFF8D, 0x4B38, 0x6014, 0xD4A1, 0x1563, 0xA1D6, 0x433B, 0xF78E, 0x364C, 0x82F9, 0xA9D5, 0x1D60, 0xDCA2, 0x6817, 0x0C8C, 0xB839, 0x79FB, 0xCD4E, 0xE662, 0x52D7, 0x9315, 0x27A0, 0xC54D, 0x71F8, 0xB03A, 0x048F, 0x2FA3, 0x9B16, 0x5AD4, 0xEE61, 0x8313, 0x37A6, 0xF664, 0x42D1, 0x69FD, 0xDD48, 0x1C8A, 0xA83F, 0x4AD2, 0xFE67, 0x3FA5, 0x8B10, 0xA03C, 0x1489, 0xD54B, 0x61FE, 0x0FAF, 0xBB1A, 0x7AD8, 0xCE6D, 0xE541, 0x51F4, 0x9036, 0x2483, 0xC66E, 0x72DB, 0xB319, 0x07AC, 0x2C80, 0x9835, 0x59F7, 0xED42, 0x8030, 0x3485, 0xF547, 0x41F2, 0x6ADE, 0xDE6B, 0x1FA9, 0xAB1C, 0x49F1, 0xFD44, 0x3C86, 0x8833, 0xA31F, 0x17AA, 0xD668, 0x62DD, 0x0ACA, 0xBE7F, 0x7FBD, 0xCB08, 0xE024, 0x5491, 0x9553, 0x21E6, 0xC30B, 0x77BE, 0xB67C, 0x02C9, 0x29E5, 0x9D50, 0x5C92, 0xE827, 0x8555, 0x31E0, 0xF022, 0x4497, 0x6FBB, 0xDB0E, 0x1ACC, 0xAE79, 0x4C94, 0xF821, 0x39E3, 0x8D56, 0xA67A, 0x12CF, 0xD30D, 0x67B8, 0x09E9, 0xBD5C, 0x7C9E, 0xC82B, 0xE307, 0x57B2, 0x9670, 0x22C5, 0xC028, 0x749D, 0xB55F, 0x01EA, 0x2AC6, 0x9E73, 0x5FB1, 0xEB04, 0x8676, 0x32C3, 0xF301, 0x47B4, 0x6C98, 0xD82D, 0x19EF, 0xAD5A, 0x4FB7, 0xFB02, 0x3AC0, 0x8E75, 0xA559, 0x11EC, 0xD02E, 0x649B}, /* 0D */
    /* 0E */ {0x0000, 0xD4D5, 0xB5B7, 0x6162, 0x7773, 0xA3A6, 0xC2C4, 0x1611, 0xEEE6, 0x3A33, 0x5B51, 0x8F84, 0x9995, 0x4D40, 0x2C22, 0xF8F7, 0xC1D1, 0x1504, 0x7466, 0xA0B3, 0xB6A2, 0x6277, 0x0315, 0xD7C0, 0x2F37, 0xFBE2, 0x9A80, 0x4E55, 0x5844, 0x8C91, 0xEDF3, 0x3926, 0x9FBF, 0x4B6A, 0x2A08, 0xFEDD, 0xE8CC, 0x3C19, 0x5D7B, 0x89AE, 0x7159, 0xA58C, 0xC4EE, 0x103B, 0x062A, 0xD2FF, 0xB39D, 0x6748, 0x5E6E, 0x8ABB, 0xEBD9, 0x3F0C, 0x291D, 0xFDC8, 0x9CAA, 0x487F, 0xB088, 0x645D, 0x053F, 0xD1EA, 0xC7FB, 0x132E, 0x724C, 0xA699, 0x2363, 0xF7B6, 0x96D4, 0x4201, 0x5410, 0x80C5, 0xE1A7, 0x3572, 0xCD85, 0x1950, 0x7832, 0xACE7, 0xBAF6, 0x6E23, 0x0F41, 0xDB94, 0xE2B2, 0x3667, 0x5705, 0x83D0, 0x95C1, 0x4114, 0x2076, 0xF4A3, 0x0C54, 0xD881, 0xB9E3, 0x6D36, 0x7B27, 0xAFF2, 0xCE90, 0x1A45, 0xBCDC, 0x6809, 0x096B, 0xDDBE, 0xCBAF, 0x1F7A, 0x7E18, 0xAACD, 0x523A, 0x86EF, 0xE78D, 0x3358, 0x2549, 0xF19C, 0x90FE, 0x442B, 0x7D0D, 0xA9D8, 0xC8BA, 0x1C6F, 0x0A7E, 0xDEAB, 0xBFC9, 0x6B1C, 0x93EB, 0x473E, 0x265C, 0xF289, 0xE498, 0x304D, 0x512F, 0x85FA, 0x46C6, 0x9213, 0xF371, 0x27A4, 0x31B5, 0xE560, 0x8402, 0x50D7, 0xA820, 0x7CF5, 0x1D97, 0xC942, 0xDF53, 0x0B86, 0x6AE4, 0xBE31, 0x8717, 0x53C2, 0x32A0, 0xE675, 0xF064, 0x24B1, 0x45D3, 0x9106, 0x69F1, 0xBD24, 0xDC46, 0x0893, 0x1E82, 0xCA57, 0xAB35, 0x7FE0, 0xD979, 0x0DAC, 0x6CCE, 0xB81B, 0xAE0A, 0x7ADF, 0x1BBD, 0xCF68, 0x379F, 0xE34A, 0x8228, 0x56FD, 0x40EC, 0x9439, 0xF55B, 0x218E, 0x18A8, 0xCC7D, 0xAD1F, 0x79CA, 0x6FDB, 0xBB0E, 0xDA6C, 0x0EB9, 0xF64E, 0x229B, 0x43F9, 0x972C, 0x813D, 0x55E8, 0x348A, 0xE05F, 0x65A5, 0xB170, 0xD012, 0x04C7, 0x12D6, 0xC603, 0xA761, 0x73B4, 0x8B43, 0x5F96, 0x3EF4, 0xEA21, 0xFC30, 0x28E5, 0x4987, 0x9D52, 0xA474, 0x70A1, 0x11C3, 0xC516, 0xD307, 0x07D2, 0x66B0, 0xB265, 0x4A92, 0x9E47, 0xFF25, 0x2BF0, 0x3DE1, 0xE934, 0x8856, 0x5C83, 0xFA1A, 0x2ECF, 0x4FAD, 0x9B78, 0x8D69, 0x59BC, 0x38DE, 0xEC0B, 0x14FC, 0xC029, 0xA14B, 0x759E, 0x638F, 0xB75A, 0xD638, 0x02ED, 0x3BCB, 0xEF1E, 0x8E7C, 0x5AA9, 0x4CB8, 0x986D, 0xF90F, 0x2DDA, 0xD52D, 0x01F8, 0x609A, 0xB44F, 0xA25E, 0x768B, 0x17E9, 0xC33C}, /* 0E */
    /* 0F */ {0x0000, 0xE4E5, 0xD5D7, 0x3132, 0xB7B3, 0x5356, 0x6264, 0x8681, 0x737B, 0x979E, 0xA6AC, 0x4249, 0xC4C8, 0x202D, 0x111F, 0xF5FA, 0xE6F6, 0x0213, 0x3321, 0xD7C4, 0x5145, 0xB5A0, 0x8492, 0x6077, 0x958D, 0x7168, 0x405A, 0xA4BF, 0x223E, 0xC6DB, 0xF7E9, 0x130C, 0xD1F1, 0x3514, 0x0426, 0xE0C3, 0x6642, 0x82A7, 0xB395, 0x5770, 0xA28A, 0x466F, 0x775D, 0x93B8, 0x1539, 0xF1DC, 0xC0EE, 0x240B, 0x3707, 0xD3E2, 0xE2D0, 0x0635, 0x80B4, 0x6451, 0x5563, 0xB186, 0x447C, 0xA099, 0x91AB, 0x754E, 0xF3CF, 0x172A, 0x2618, 0xC2FD, 0xBFFF, 0x5B1A, 0x6A28, 0x8ECD, 0x084C, 0xECA9, 0xDD9B, 0x397E, 0xCC84, 0x2861, 0x1953, 0xFDB6, 0x7B37, 0x9FD2, 0xAEE0, 0x4A05, 0x5909, 0xBDEC, 0x8CDE, 0x683B, 0xEEBA, 0x0A5F, 0x3B6D, 0xDF88, 0x2A72, 0xCE97, 0xFFA5, 0x1B40, 0x9DC1, 0x7924, 0x4816, 0xACF3, 0x6E0E, 0x8AEB, 0xBBD9, 0x5F3C, 0xD9BD, 0x3D58, 0x0C6A, 0xE88F, 0x1D75, 0xF990, 0xC8A2, 0x2C47, 0xAAC6, 0x4E23, 0x7F11, 0x9BF4, 0x88F8, 0x6C1D, 0x5D2F, 0xB9CA, 0x3F4B, 0xDBAE, 0xEA9C, 0x0E79, 0xFB83, 0x1F66, 0x2E54, 0xCAB1, 0x4C30, 0xA8D5, 0x99E7, 0x7D02, 0x63E3, 0x8706, 0xB634, 0x52D1, 0xD450, 0x30B5, 0x0187, 0xE562, 0x1098, 0xF47D, 0xC54F, 0x21AA, 0xA72B, 0x43CE, 0x72FC, 0x9619, 0x8515, 0x61F0, 0x50C2, 0xB427, 0x32A6, 0xD643, 0xE771, 0x0394, 0xF66E, 0x128B, 0x23B9, 0xC75C, 0x41DD, 0xA538, 0x940A, 0x70EF, 0xB212, 0x56F7, 0x67C5, 0x8320, 0x05A1, 0xE144, 0xD076, 0x3493, 0xC169, 0x258C, 0x14BE, 0xF05B, 0x76DA, 0x923F, 0xA30D, 0x47E8, 0x54E4, 0xB001, 0x8133, 0x65D6, 0xE357, 0x07B2, 0x3680, 0xD265, 0x279F, 0xC37A, 0xF248, 0x16AD, 0x902C, 0x74C9, 0x45FB, 0xA11E, 0xDC1C, 0x38F9, 0x09CB, 0xED2E, 0x6BAF, 0x8F4A, 0xBE78, 0x5A9D, 0xAF67, 0x4B82, 0x7AB0, 0x9E55, 0x18D4, 0xFC31, 0xCD03, 0x29E6, 0x3AEA, 0xDE0F, 0xEF3D, 0x0BD8, 0x8D59, 0x69BC, 0x588E, 0xBC6B, 0x4991, 0xAD74, 0x9C46, 0x78A3, 0xFE22, 0x1AC7, 0x2BF5, 0xCF10, 0x0DED, 0xE908, 0xD83A, 0x3CDF, 0xBA5E, 0x5EBB, 0x6F89, 0x8B6C, 0x7E96, 0x9A73, 0xAB41, 0x4FA4, 0xC925, 0x2DC0, 0x1CF2, 0xF817, 0xEB1B, 0x0FFE, 0x3ECC, 0xDA29, 0x5CA8, 0xB84D, 0x897F, 0x6D9A, 0x9860, 0x7C85, 0x4DB7, 0xA952, 0x2FD3, 0xCB36, 0xFA04, 0x1EE1}, /* 0F */
    /* 10 */ {0x0000, 0xFCFD, 0xE5E7, 0x191A, 0xD7D3, 0x2B2E, 0x3234, 0xCEC9, 0xB3BB, 0x4F46, 0x565C, 0xAAA1, 0x6468, 0x9895, 0x818F, 0x7D72, 0x7B6B, 0x8796, 0x9E8C, 0x6271, 0xACB8, 0x5045, 0x495F, 0xB5A2, 0xC8D0, 0x342D, 0x2D37, 0xD1CA, 0x1F03, 0xE3FE, 0xFAE4, 0x0619, 0xF6D6, 0x0A2B, 0x1331, 0xEFCC, 0x2105, 0xDDF8, 0xC4E2, 0x381F, 0x456D, 0xB990, 0xA08A, 0x5C77, 0x92BE, 0x6E43, 0x7759, 0x8BA4, 0x8DBD, 0x7140, 0x685A, 0x94A7, 0x5A6E, 0xA693, 0xBF89, 0x4374, 0x3E06, 0xC2FB, 0xDBE1, 0x271C, 0xE9D5, 0x1528, 0x0C32, 0xF0CF, 0xF1B1, 0x0D4C, 0x1456, 0xE8AB, 0x2662, 0xDA9F, 0xC385, 0x3F78, 0x420A, 0xBEF7, 0xA7ED, 0x5B10, 0x95D9, 0x6924, 0x703E, 0x8CC3, 0x8ADA, 0x7627, 0x6F3D, 0x93C0, 0x5D09, 0xA1F4, 0xB8EE, 0x4413, 0x3961, 0xC59C, 0xDC86, 0x207B, 0xEEB2, 0x124F, 0x0B55, 0xF7A8, 0x0767, 0xFB9A, 0xE280, 0x1E7D, 0xD0B4, 0x2C49, 0x3553, 0xC9AE, 0xB4DC, 0x4821, 0x513B, 0xADC6, 0x630F, 0x9FF2, 0x86E8, 0x7A15, 0x7C0C, 0x80F1, 0x99EB, 0x6516, 0xABDF, 0x5722, 0x4E38, 0xB2C5, 0xCFB7, 0x334A, 0x2A50, 0xD6AD, 0x1864, 0xE499, 0xFD83, 0x017E, 0xFF7F, 0x0382, 0x1A98, 0xE665, 0x28AC, 0xD451, 0xCD4B, 0x31B6, 0x4CC4, 0xB039, 0xA923, 0x55DE, 0x9B17, 0x67EA, 0x7EF0, 0x820D, 0x8414, 0x78E9, 0x61F3, 0x9D0E, 0x53C7, 0xAF3A, 0xB620, 0x4ADD, 0x37AF, 0xCB52, 0xD248, 0x2EB5, 0xE07C, 0x1C81, 0x059B, 0xF966, 0x09A9, 0xF554, 0xEC4E, 0x10B3, 0xDE7A, 0x2287, 0x3B9D, 0xC760, 0xBA12, 0x46EF, 0x5FF5, 0xA308, 0x6DC1, 0x913C, 0x8826, 0x74DB, 0x72C2, 0x8E3F, 0x9725, 0x6BD8, 0xA511, 0x59EC, 0x40F6, 0xBC0B, 0xC179, 0x3D84, 0x249E, 0xD863, 0x16AA, 0xEA57, 0xF34D, 0x0FB0, 0x0ECE, 0xF233, 0xEB29, 0x17D4, 0xD91D, 0x25E0, 0x3CFA, 0xC007, 0xBD75, 0x4188, 0x5892, 0xA46F, 0x6AA6, 0x965B, 0x8F41, 0x73BC, 0x75A5, 0x8958, 0x9042, 0x6CBF, 0xA276, 0x5E8B, 0x4791, 0xBB6C, 0xC61E, 0x3AE3, 0x23F9, 0xDF04, 0x11CD, 0xED30, 0xF42A, 0x08D7, 0xF818, 0x04E5, 0x1DFF, 0xE102, 0x2FCB, 0xD336, 0xCA2C, 0x36D1, 0x4BA3, 0xB75E, 0xAE44, 0x52B9, 0x9C70, 0x608D, 0x7997, 0x856A, 0x8373, 0x7F8E, 0x6694, 0x9A69, 0x54A0, 0xA85D, 0xB147, 0x4DBA, 0x30C8, 0xCC35, 0xD52F, 0x29D2, 0xE71B, 0x1BE6, 0x02FC, 0xFE01}, /* 10 */
    /* 11 */ {0x0000, 0xF0F1, 0xFDFF, 0x0D0E, 0xE7E3, 0x1712, 0x1A1C, 0xEAED, 0xD3DB, 0x232A, 0x2E24, 0xDED5, 0x3438, 0xC4C9, 0xC9C7, 0x3936, 0xBBAB, 0x4B5A, 0x4654, 0xB6A5, 0x5C48, 0xACB9, 0xA1B7, 0x5146, 0x6870, 0x9881, 0x958F, 0x657E, 0x8F93, 0x7F62, 0x726C, 0x829D, 0x6B4B, 0x9BBA, 0x96B4, 0x6645, 0x8CA8, 0x7C59, 0x7157, 0x81A6, 0xB890, 0x4861, 0x456F, 0xB59E, 0x5F73, 0xAF82, 0xA28C, 0x527D, 0xD0E0, 0x2011, 0x2D1F, 0xDDEE, 0x3703, 0xC7F2, 0xCAFC, 0x3A0D, 0x033B, 0xF3CA, 0xFEC4, 0x0E35, 0xE4D8, 0x1429, 0x1927, 0xE9D6, 0xD696, 0x2667, 0x2B69, 0xDB98, 0x3175, 0xC184, 0xCC8A, 0x3C7B, 0x054D, 0xF5BC, 0xF8B2, 0x0843, 0xE2AE, 0x125F, 0x1F51, 0xEFA0, 0x6D3D, 0x9DCC, 0x90C2, 0x6033, 0x8ADE, 0x7A2F, 0x7721, 0x87D0, 0xBEE6, 0x4E17, 0x4319, 0xB3E8, 0x5905, 0xA9F4, 0xA4FA, 0x540B, 0xBDDD, 0x4D2C, 0x4022, 0xB0D3, 0x5A3E, 0xAACF, 0xA7C1, 0x5730, 0x6E06, 0x9EF7, 0x93F9, 0x6308, 0x89E5, 0x7914, 0x741A, 0x84EB, 0x0676, 0xF687, 0xFB89, 0x0B78, 0xE195, 0x1164, 0x1C6A, 0xEC9B, 0xD5AD, 0x255C, 0x2852, 0xD8A3, 0x324E, 0xC2BF, 0xCFB1, 0x3F40, 0xB131, 0x41C0, 0x4CCE, 0xBC3F, 0x56D2, 0xA623, 0xAB2D, 0x5BDC, 0x62EA, 0x921B, 0x9F15, 0x6FE4, 0x8509, 0x75F8, 0x78F6, 0x8807, 0x0A9A, 0xFA6B, 0xF765, 0x0794, 0xED79, 0x1D88, 0x1086, 0xE077, 0xD941, 0x29B0, 0x24BE, 0xD44F, 0x3EA2, 0xCE53, 0xC35D, 0x33AC, 0xDA7A, 0x2A8B, 0x2785, 0xD774, 0x3D99, 0xCD68, 0xC066, 0x3097, 0x09A1, 0xF950, 0xF45E, 0x04AF, 0xEE42, 0x1EB3, 0x13BD, 0xE34C, 0x61D1, 0x9120, 0x9C2E, 0x6CDF, 0x8632, 0x76C3, 0x7BCD, 0x8B3C, 0xB20A, 0x42FB, 0x4FF5, 0xBF04, 0x55E9, 0xA518, 0xA816, 0x58E7, 0x67A7, 0x9756, 0x9A58, 0x6AA9, 0x8044, 0x70B5, 0x7DBB, 0x8D4A, 0xB47C, 0x448D, 0x4983, 0xB972, 0x539F, 0xA36E, 0xAE60, 0x5E91, 0xDC0C, 0x2CFD, 0x21F3, 0xD102, 0x3BEF, 0xCB1E, 0xC610, 0x36E1, 0x0FD7, 0xFF26, 0xF228, 0x02D9, 0xE834, 0x18C5, 0x15CB, 0xE53A, 0x0CEC, 0xFC1D, 0xF113, 0x01E2, 0xEB0F, 0x1BFE, 0x16F0, 0xE601, 0xDF37, 0x2FC6, 0x22C8, 0xD239, 0x38D4, 0xC825, 0xC52B, 0x35DA, 0xB747, 0x47B6, 0x4AB8, 0xBA49, 0x50A4, 0xA055, 0xAD5B, 0x5DAA, 0x649C, 0x946D, 0x9963, 0x6992, 0x837F, 0x738E, 0x7E80, 0x8E71}, /* 11 */
    /* 12 */ {0x0000, 0xF6F7, 0xF1F3, 0x0704, 0xFFFB, 0x090C, 0x0E08, 0xF8FF, 0xE3EB, 0x151C, 0x1218, 0xE4EF, 0x1C10, 0xEAE7, 0xEDE3, 0x1B14, 0xDBCB, 0x2D3C, 0x2A38, 0xDCCF, 0x2430, 0xD2C7, 0xD5C3, 0x2334, 0x3820, 0xCED7, 0xC9D3, 0x3F24, 0xC7DB, 0x312C, 0x3628, 0xC0DF, 0xAB8B, 0x5D7C, 0x5A78, 0xAC8F, 0x5470, 0xA287, 0xA583, 0x5374, 0x4860, 0xBE97, 0xB993, 0x4F64, 0xB79B, 0x416C, 0x4668, 0xB09F, 0x7040, 0x86B7, 0x81B3, 0x7744, 0x8FBB, 0x794C, 0x7E48, 0x88BF, 0x93AB, 0x655C, 0x6258, 0x94AF, 0x6C50, 0x9AA7, 0x9DA3, 0x6B54, 0x4B0B, 0xBDFC, 0xBAF8, 0x4C0F, 0xB4F0, 0x4207, 0x4503, 0xB3F4, 0xA8E0, 0x5E17, 0x5913, 0xAFE4, 0x571B, 0xA1EC, 0xA6E8, 0x501F, 0x90C0, 0x6637, 0x6133, 0x97C4, 0x6F3B, 0x99CC, 0x9EC8, 0x683F, 0x732B, 0x85DC, 0x82D8, 0x742F, 0x8CD0, 0x7A27, 0x7D23, 0x8BD4, 0xE080, 0x1677, 0x1173, 0xE784, 0x1F7B, 0xE98C, 0xEE88, 0x187F, 0x036B, 0xF59C, 0xF298, 0x046F, 0xFC90, 0x0A67, 0x0D63, 0xFB94, 0x3B4B, 0xCDBC, 0xCAB8, 0x3C4F, 0xC4B0, 0x3247, 0x3543, 0xC3B4, 0xD8A0, 0x2E57, 0x2953, 0xDFA4, 0x275B, 0xD1AC, 0xD6A8, 0x205F, 0x9616, 0x60E1, 0x67E5, 0x9112, 0x69ED, 0x9F1A, 0x981E, 0x6EE9, 0x75FD, 0x830A, 0x840E, 0x72F9, 0x8A06, 0x7CF1, 0x7BF5, 0x8D02, 0x4DDD, 0xBB2A, 0xBC2E, 0x4AD9, 0xB226, 0x44D1, 0x43D5, 0xB522, 0xAE36, 0x58C1, 0x5FC5, 0xA932, 0x51CD, 0xA73A, 0xA03E, 0x56C9, 0x3D9D, 0xCB6A, 0xCC6E, 0x3A99, 0xC266, 0x3491, 0x3395, 0xC562, 0xDE76, 0x2881, 0x2F85, 0xD972, 0x218D, 0xD77A, 0xD07E, 0x2689, 0xE656, 0x10A1, 0x17A5, 0xE152, 0x19AD, 0xEF5A, 0xE85E, 0x1EA9, 0x05BD, 0xF34A, 0xF44E, 0x02B9, 0xFA46, 0x0CB1, 0x0BB5, 0xFD42, 0xDD1D, 0x2BEA, 0x2CEE, 0xDA19, 0x22E6, 0xD411, 0xD315, 0x25E2, 0x3EF6, 0xC801, 0xCF05, 0x39F2, 0xC10D, 0x37FA, 0x30FE, 0xC609, 0x06D6, 0xF021, 0xF725, 0x01D2, 0xF92D, 0x0FDA, 0x08DE, 0xFE29, 0xE53D, 0x13CA, 0x14CE, 0xE239, 0x1AC6, 0xEC31, 0xEB35, 0x1DC2, 0x7696, 0x8061, 0x8765, 0x7192, 0x896D, 0x7F9A, 0x789E, 0x8E69, 0x957D, 0x638A, 0x648E, 0x9279, 0x6A86, 0x9C71, 0x9B75, 0x6D82, 0xAD5D, 0x5BAA, 0x5CAE, 0xAA59, 0x52A6, 0xA451, 0xA355, 0x55A2, 0x4EB6, 0xB841, 0xBF45, 0x49B2, 0xB14D, 0x47BA, 0x40BE, 0xB649}, /* 12 */
    /* 13 */ {0x0000, 0xF5F4, 0xF7F5, 0x0201, 0xF3F7, 0x0603, 0x0402, 0xF1F6, 0xFBF3, 0x0E07, 0x0C06, 0xF9F2, 0x0804, 0xFDF0, 0xFFF1, 0x0A05, 0xEBFB, 0x1E0F, 0x1C0E, 0xE9FA, 0x180C, 0xEDF8, 0xEFF9, 0x1A0D, 0x1008, 0xE5FC, 0xE7FD, 0x1209, 0xE3FF, 0x160B, 0x140A, 0xE1FE, 0xCBEB, 0x3E1F, 0x3C1E, 0xC9EA, 0x381C, 0xCDE8, 0xCFE9, 0x3A1D, 0x3018, 0xC5EC, 0xC7ED, 0x3219, 0xC3EF, 0x361B, 0x341A, 0xC1EE, 0x2010, 0xD5E4, 0xD7E5, 0x2211, 0xD3E7, 0x2613, 0x2412, 0xD1E6, 0xDBE3, 0x2E17, 0x2C16, 0xD9E2, 0x2814, 0xDDE0, 0xDFE1, 0x2A15, 0x8BCB, 0x7E3F, 0x7C3E, 0x89CA, 0x783C, 0x8DC8, 0x8FC9, 0x7A3D, 0x7038, 0x85CC, 0x87CD, 0x7239, 0x83CF, 0x763B, 0x743A, 0x81CE, 0x6030, 0x95C4, 0x97C5, 0x6231, 0x93C7, 0x6633, 0x6432, 0x91C6, 0x9BC3, 0x6E37, 0x6C36, 0x99C2, 0x6834, 0x9DC0, 0x9FC1, 0x6A35, 0x4020, 0xB5D4, 0xB7D5, 0x4221, 0xB3D7, 0x4623, 0x4422, 0xB1D6, 0xBBD3, 0x4E27, 0x4C26, 0xB9D2, 0x4824, 0xBDD0, 0xBFD1, 0x4A25, 0xABDB, 0x5E2F, 0x5C2E, 0xA9DA, 0x582C, 0xADD8, 0xAFD9, 0x5A2D, 0x5028, 0xA5DC, 0xA7DD, 0x5229, 0xA3DF, 0x562B, 0x542A, 0xA1DE, 0x0B8B, 0xFE7F, 0xFC7E, 0x098A, 0xF87C, 0x0D88, 0x0F89, 0xFA7D, 0xF078, 0x058C, 0x078D, 0xF279, 0x038F, 0xF67B, 0xF47A, 0x018E, 0xE070, 0x1584, 0x1785, 0xE271, 0x1387, 0xE673, 0xE472, 0x1186, 0x1B83, 0xEE77, 0xEC76, 0x1982, 0xE874, 0x1D80, 0x1F81, 0xEA75, 0xC060, 0x3594, 0x3795, 0xC261, 0x3397, 0xC663, 0xC462, 0x3196, 0x3B93, 0xCE67, 0xCC66, 0x3992, 0xC864, 0x3D90, 0x3F91, 0xCA65, 0x2B9B, 0xDE6F, 0xDC6E, 0x299A, 0xD86C, 0x2D98, 0x2F99, 0xDA6D, 0xD068, 0x259C, 0x279D, 0xD269, 0x239F, 0xD66B, 0xD46A, 0x219E, 0x8040, 0x75B4, 0x77B5, 0x8241, 0x73B7, 0x8643, 0x8442, 0x71B6, 0x7BB3, 0x8E47, 0x8C46, 0x79B2, 0x8844, 0x7DB0, 0x7FB1, 0x8A45, 0x6BBB, 0x9E4F, 0x9C4E, 0x69BA, 0x984C, 0x6DB8, 0x6FB9, 0x9A4D, 0x9048, 0x65BC, 0x67BD, 0x9249, 0x63BF, 0x964B, 0x944A, 0x61BE, 0x4BAB, 0xBE5F, 0xBC5E, 0x49AA, 0xB85C, 0x4DA8, 0x4FA9, 0xBA5D, 0xB058, 0x45AC, 0x47AD, 0xB259, 0x43AF, 0xB65B, 0xB45A, 0x41AE, 0xA050, 0x55A4, 0x57A5, 0xA251, 0x53A7, 0xA653, 0xA452, 0x51A6, 0x5BA3, 0xAE57, 0xAC56, 0x59A2, 0xA854, 0x5DA0, 0x5FA1, 0xAA55}, /* 13 */
    /* 14 */ {0x0000, 0x7A7B, 0xF4F6, 0x8E8D, 0xF5F1, 0x8F8A, 0x0107, 0x7B7C, 0xF7FF, 0x8D84, 0x0309, 0x7972, 0x020E, 0x7875, 0xF6F8, 0x8C83, 0xF3E3, 0x8998, 0x0715, 0x7D6E, 0x0612, 0x7C69, 0xF2E4, 0x889F, 0x041C, 0x7E67, 0xF0EA, 0x8A91, 0xF1ED, 0x8B96, 0x051B, 0x7F60, 0xFBDB, 0x81A0, 0x0F2D, 0x7556, 0x0E2A, 0x7451, 0xFADC, 0x80A7, 0x0C24, 0x765F, 0xF8D2, 0x82A9, 0xF9D5, 0x83AE, 0x0D23, 0x7758, 0x0838, 0x7243, 0xFCCE, 0x86B5, 0xFDC9, 0x87B2, 0x093F, 0x7344, 0xFFC7, 0x85BC, 0x0B31, 0x714A, 0x0A36, 0x704D, 0xFEC0, 0x84BB, 0xEBAB, 0x91D0, 0x1F5D, 0x6526, 0x1E5A, 0x6421, 0xEAAC, 0x90D7, 0x1C54, 0x662F, 0xE8A2, 0x92D9, 0xE9A5, 0x93DE, 0x1D53, 0x6728, 0x1848, 0x6233, 0xECBE, 0x96C5, 0xEDB9, 0x97C2, 0x194F, 0x6334, 0xEFB7, 0x95CC, 0x1B41, 0x613A, 0x1A46, 0x603D, 0xEEB0, 0x94CB, 0x1070, 0x6A0B, 0xE486, 0x9EFD, 0xE581, 0x9FFA, 0x1177, 0x6B0C, 0xE78F, 0x9DF4, 0x1379, 0x6902, 0x127E, 0x6805, 0xE688, 0x9CF3, 0xE393, 0x99E8, 0x1765, 0x6D1E, 0x1662, 0x6C19, 0xE294, 0x98EF, 0x146C, 0x6E17, 0xE09A, 0x9AE1, 0xE19D, 0x9BE6, 0x156B, 0x6F10, 0xCB4B, 0xB130, 0x3FBD, 0x45C6, 0x3EBA, 0x44C1, 0xCA4C, 0xB037, 0x3CB4, 0x46CF, 0xC842, 0xB239, 0xC945, 0xB33E, 0x3DB3, 0x47C8, 0x38A8, 0x42D3, 0xCC5E, 0xB625, 0xCD59, 0xB722, 0x39AF, 0x43D4, 0xCF57, 0xB52C, 0x3BA1, 0x41DA, 0x3AA6, 0x40DD, 0xCE50, 0xB42B, 0x3090, 0x4AEB, 0xC466, 0xBE1D, 0xC561, 0xBF1A, 0x3197, 0x4BEC, 0xC76F, 0xBD14, 0x3399, 0x49E2, 0x329E, 0x48E5, 0xC668, 0xBC13, 0xC373, 0xB908, 0x3785, 0x4DFE, 0x3682, 0x4CF9, 0xC274, 0xB80F, 0x348C, 0x4EF7, 0xC07A, 0xBA01, 0xC17D, 0xBB06, 0x358B, 0x4FF0, 0x20E0, 0x5A9B, 0xD416, 0xAE6D, 0xD511, 0xAF6A, 0x21E7, 0x5B9C, 0xD71F, 0xAD64, 0x23E9, 0x5992, 0x22EE, 0x5895, 0xD618, 0xAC63, 0xD303, 0xA978, 0x27F5, 0x5D8E, 0x26F2, 0x5C89, 0xD204, 0xA87F, 0x24FC, 0x5E87, 0xD00A, 0xAA71, 0xD10D, 0xAB76, 0x25FB, 0x5F80, 0xDB3B, 0xA140, 0x2FCD, 0x55B6, 0x2ECA, 0x54B1, 0xDA3C, 0xA047, 0x2CC4, 0x56BF, 0xD832, 0xA249, 0xD935, 0xA34E, 0x2DC3, 0x57B8, 0x28D8, 0x52A3, 0xDC2E, 0xA655, 0xDD29, 0xA752, 0x29DF, 0x53A4, 0xDF27, 0xA55C, 0x2BD1, 0x51AA, 0x2AD6, 0x50AD, 0xDE20, 0xA45B}, /* 14 */
    /* 15 */ {0x0000, 0xB3B2, 0x7B79, 0xC8CB, 0xF6F2, 0x4540, 0x8D8B, 0x3E39, 0xF1F9, 0x424B, 0x8A80, 0x3932, 0x070B, 0xB4B9, 0x7C72, 0xCFC0, 0xFFEF, 0x4C5D, 0x8496, 0x3724, 0x091D, 0xBAAF, 0x7264, 0xC1D6, 0x0E16, 0xBDA4, 0x756F, 0xC6DD, 0xF8E4, 0x4B56, 0x839D, 0x302F, 0xE3C3, 0x5071, 0x98BA, 0x2B08, 0x1531, 0xA683, 0x6E48, 0xDDFA, 0x123A, 0xA188, 0x6943, 0xDAF1, 0xE4C8, 0x577A, 0x9FB1, 0x2C03, 0x1C2C, 0xAF9E, 0x6755, 0xD4E7, 0xEADE, 0x596C, 0x91A7, 0x2215, 0xEDD5, 0x5E67, 0x96AC, 0x251E, 0x1B27, 0xA895, 0x605E, 0xD3EC, 0xDB9B, 0x6829, 0xA0E2, 0x1350, 0x2D69, 0x9EDB, 0x5610, 0xE5A2, 0x2A62, 0x99D0, 0x511B, 0xE2A9, 0xDC90, 0x6F22, 0xA7E9, 0x145B, 0x2474, 0x97C6, 0x5F0D, 0xECBF, 0xD286, 0x6134, 0xA9FF, 0x1A4D, 0xD58D, 0x663F, 0xAEF4, 0x1D46, 0x237F, 0x90CD, 0x5806, 0xEBB4, 0x3858, 0x8BEA, 0x4321, 0xF093, 0xCEAA, 0x7D18, 0xB5D3, 0x0661, 0xC9A1, 0x7A13, 0xB2D8, 0x016A, 0x3F53, 0x8CE1, 0x442A, 0xF798, 0xC7B7, 0x7405, 0xBCCE, 0x0F7C, 0x3145, 0x82F7, 0x4A3C, 0xF98E, 0x364E, 0x85FC, 0x4D37, 0xFE85, 0xC0BC, 0x730E, 0xBBC5, 0x0877, 0xAB2B, 0x1899, 0xD052, 0x63E0, 0x5DD9, 0xEE6B, 0x26A0, 0x9512, 0x5AD2, 0xE960, 0x21AB, 0x9219, 0xAC20, 0x1F92, 0xD759, 0x64EB, 0x54C4, 0xE776, 0x2FBD, 0x9C0F, 0xA236, 0x1184, 0xD94F, 0x6AFD, 0xA53D, 0x168F, 0xDE44, 0x6DF6, 0x53CF, 0xE07D, 0x28B6, 0x9B04, 0x48E8, 0xFB5A, 0x3391, 0x8023, 0xBE1A, 0x0DA8, 0xC563, 0x76D1, 0xB911, 0x0AA3, 0xC268, 0x71DA, 0x4FE3, 0xFC51, 0x349A, 0x8728, 0xB707, 0x04B5, 0xCC7E, 0x7FCC, 0x41F5, 0xF247, 0x3A8C, 0x893E, 0x46FE, 0xF54C, 0x3D87, 0x8E35, 0xB00C, 0x03BE, 0xCB75, 0x78C7, 0x70B0, 0xC302, 0x0BC9, 0xB87B, 0x8642, 0x35F0, 0xFD3B, 0x4E89, 0x8149, 0x32FB, 0xFA30, 0x4982, 0x77BB, 0xC409, 0x0CC2, 0xBF70, 0x8F5F, 0x3CED, 0xF426, 0x4794, 0x79AD, 0xCA1F, 0x02D4, 0xB166, 0x7EA6, 0xCD14, 0x05DF, 0xB66D, 0x8854, 0x3BE6, 0xF32D, 0x409F, 0x9373, 0x20C1, 0xE80A, 0x5BB8, 0x6581, 0xD633, 0x1EF8, 0xAD4A, 0x628A, 0xD138, 0x19F3, 0xAA41, 0x9478, 0x27CA, 0xEF01, 0x5CB3, 0x6C9C, 0xDF2E, 0x17E5, 0xA457, 0x9A6E, 0x29DC, 0xE117, 0x52A5, 0x9D65, 0x2ED7, 0xE61C, 0x55AE, 0x6B97, 0xD825, 0x10EE, 0xA35C}, /* 15 */
    /* 16 */ {0x0000, 0x5958, 0xB2B0, 0xEBE8, 0x797D, 0x2025, 0xCBCD, 0x9295, 0xF2FA, 0xABA2, 0x404A, 0x1912, 0x8B87, 0xD2DF, 0x3937, 0x606F, 0xF9E9, 0xA0B1, 0x4B59, 0x1201, 0x8094, 0xD9CC, 0x3224, 0x6B7C, 0x0B13, 0x524B, 0xB9A3, 0xE0FB, 0x726E, 0x2B36, 0xC0DE, 0x9986, 0xEFCF, 0xB697, 0x5D7F, 0x0427, 0x96B2, 0xCFEA, 0x2402, 0x7D5A, 0x1D35, 0x446D, 0xAF85, 0xF6DD, 0x6448, 0x3D10, 0xD6F8, 0x8FA0, 0x1626, 0x4F7E, 0xA496, 0xFDCE, 0x6F5B, 0x3603, 0xDDEB, 0x84B3, 0xE4DC, 0xBD84, 0x566C, 0x0F34, 0x9DA1, 0xC4F9, 0x2F11, 0x7649, 0xC383, 0x9ADB, 0x7133, 0x286B, 0xBAFE, 0xE3A6, 0x084E, 0x5116, 0x3179, 0x6821, 0x83C9, 0xDA91, 0x4804, 0x115C, 0xFAB4, 0xA3EC, 0x3A6A, 0x6332, 0x88DA, 0xD182, 0x4317, 0x1A4F, 0xF1A7, 0xA8FF, 0xC890, 0x91C8, 0x7A20, 0x2378, 0xB1ED, 0xE8B5, 0x035D, 0x5A05, 0x2C4C, 0x7514, 0x9EFC, 0xC7A4, 0x5531, 0x0C69, 0xE781, 0xBED9, 0xDEB6, 0x87EE, 0x6C06, 0x355E, 0xA7CB, 0xFE93, 0x157B, 0x4C23, 0xD5A5, 0x8CFD, 0x6715, 0x3E4D, 0xACD8, 0xF580, 0x1E68, 0x4730, 0x275F, 0x7E07, 0x95EF, 0xCCB7, 0x5E22, 0x077A, 0xEC92, 0xB5CA, 0x9B1B, 0xC243, 0x29AB, 0x70F3, 0xE266, 0xBB3E, 0x50D6, 0x098E, 0x69E1, 0x30B9, 0xDB51, 0x8209, 0x109C, 0x49C4, 0xA22C, 0xFB74, 0x62F2, 0x3BAA, 0xD042, 0x891A, 0x1B8F, 0x42D7, 0xA93F, 0xF067, 0x9008, 0xC950, 0x22B8, 0x7BE0, 0xE975, 0xB02D, 0x5BC5, 0x029D, 0x74D4, 0x2D8C, 0xC664, 0x9F3C, 0x0DA9, 0x54F1, 0xBF19, 0xE641, 0x862E, 0xDF76, 0x349E, 0x6DC6, 0xFF53, 0xA60B, 0x4DE3, 0x14BB, 0x8D3D, 0xD465, 0x3F8D, 0x66D5, 0xF440, 0xAD18, 0x46F0, 0x1FA8, 0x7FC7, 0x269F, 0xCD77, 0x942F, 0x06BA, 0x5FE2, 0xB40A, 0xED52, 0x5898, 0x01C0, 0xEA28, 0xB370, 0x21E5, 0x78BD, 0x9355, 0xCA0D, 0xAA62, 0xF33A, 0x18D2, 0x418A, 0xD31F, 0x8A47, 0x61AF, 0x38F7, 0xA171, 0xF829, 0x13C1, 0x4A99, 0xD80C, 0x8154, 0x6ABC, 0x33E4, 0x538B, 0x0AD3, 0xE13B, 0xB863, 0x2AF6, 0x73AE, 0x9846, 0xC11E, 0xB757, 0xEE0F, 0x05E7, 0x5CBF, 0xCE2A, 0x9772, 0x7C9A, 0x25C2, 0x45AD, 0x1CF5, 0xF71D, 0xAE45, 0x3CD0, 0x6588, 0x8E60, 0xD738, 0x4EBE, 0x17E6, 0xFC0E, 0xA556, 0x37C3, 0x6E9B, 0x8573, 0xDC2B, 0xBC44, 0xE51C, 0x0EF4, 0x57AC, 0xC539, 0x9C61, 0x7789, 0x2ED1}, /* 16 */
    /* 17 */ {0x0000, 0x2C2D, 0x585A, 0x7477, 0xB0B4, 0x9C99, 0xE8EE, 0xC4C3, 0x7D75, 0x5158, 0x252F, 0x0902, 0xCDC1, 0xE1EC, 0x959B, 0xB9B6, 0xFAEA, 0xD6C7, 0xA2B0, 0x8E9D, 0x4A5E, 0x6673, 0x1204, 0x3E29, 0x879F, 0xABB2, 0xDFC5, 0xF3E8, 0x372B, 0x1B06, 0x6F71, 0x435C, 0xE9C9, 0xC5E4, 0xB193, 0x9DBE, 0x597D, 0x7550, 0x0127, 0x2D0A, 0x94BC, 0xB891, 0xCCE6, 0xE0CB, 0x2408, 0x0825, 0x7C52, 0x507F, 0x1323, 0x3F0E, 0x4B79, 0x6754, 0xA397, 0x8FBA, 0xFBCD, 0xD7E0, 0x6E56, 0x427B, 0x360C, 0x1A21, 0xDEE2, 0xF2CF, 0x86B8, 0xAA95, 0xCF8F, 0xE3A2, 0x97D5, 0xBBF8, 0x7F3B, 0x5316, 0x2761, 0x0B4C, 0xB2FA, 0x9ED7, 0xEAA0, 0xC68D, 0x024E, 0x2E63, 0x5A14, 0x7639, 0x3565, 0x1948, 0x6D3F, 0x4112, 0x85D1, 0xA9FC, 0xDD8B, 0xF1A6, 0x4810, 0x643D, 0x104A, 0x3C67, 0xF8A4, 0xD489, 0xA0FE, 0x8CD3, 0x2646, 0x0A6B, 0x7E1C, 0x5231, 0x96F2, 0xBADF, 0xCEA8, 0xE285, 0x5B33, 0x771E, 0x0369, 0x2F44, 0xEB87, 0xC7AA, 0xB3DD, 0x9FF0, 0xDCAC, 0xF081, 0x84F6, 0xA8DB, 0x6C18, 0x4035, 0x3442, 0x186F, 0xA1D9, 0x8DF4, 0xF983, 0xD5AE, 0x116D, 0x3D40, 0x4937, 0x651A, 0x8303, 0xAF2E, 0xDB59, 0xF774, 0x33B7, 0x1F9A, 0x6BED, 0x47C0, 0xFE76, 0xD25B, 0xA62C, 0x8A01, 0x4EC2, 0x62EF, 0x1698, 0x3AB5, 0x79E9, 0x55C4, 0x21B3, 0x0D9E, 0xC95D, 0xE570, 0x9107, 0xBD2A, 0x049C, 0x28B1, 0x5CC6, 0x70EB, 0xB428, 0x9805, 0xEC72, 0xC05F, 0x6ACA, 0x46E7, 0x3290, 0x1EBD, 0xDA7E, 0xF653, 0x8224, 0xAE09, 0x17BF, 0x3B92, 0x4FE5, 0x63C8, 0xA70B, 0x8B26, 0xFF51, 0xD37C, 0x9020, 0xBC0D, 0xC87A, 0xE457, 0x2094, 0x0CB9, 0x78CE, 0x54E3, 0xED55, 0xC178, 0xB50F, 0x9922, 0x5DE1, 0x71CC, 0x05BB, 0x2996, 0x4C8C, 0x60A1, 0x14D6, 0x38FB, 0xFC38, 0xD015, 0xA462, 0x884F, 0x31F9, 0x1DD4, 0x69A3, 0x458E, 0x814D, 0xAD60, 0xD917, 0xF53A, 0xB666, 0x9A4B, 0xEE3C, 0xC211, 0x06D2, 0x2AFF, 0x5E88, 0x72A5, 0xCB13, 0xE73E, 0x9349, 0xBF64, 0x7BA7, 0x578A, 0x23FD, 0x0FD0, 0xA545, 0x8968, 0xFD1F, 0xD132, 0x15F1, 0x39DC, 0x4DAB, 0x6186, 0xD830, 0xF41D, 0x806A, 0xAC47, 0x6884, 0x44A9, 0x30DE, 0x1CF3, 0x5FAF, 0x7382, 0x07F5, 0x2BD8, 0xEF1B, 0xC336, 0xB741, 0x9B6C, 0x22DA, 0x0EF7, 0x7A80, 0x56AD, 0x926E, 0xBE43, 0xCA34, 0xE619}, /* 17 */
    /* 18 */ {0x0000, 0x9899, 0x2D2F, 0xB5B6, 0x5A5E, 0xC2C7, 0x7771, 0xEFE8, 0xB4BC, 0x2C25, 0x9993, 0x010A, 0xEEE2, 0x767B, 0xC3CD, 0x5B54, 0x7565, 0xEDFC, 0x584A, 0xC0D3, 0x2F3B, 0xB7A2, 0x0214, 0x9A8D, 0xC1D9, 0x5940, 0xECF6, 0x746F, 0x9B87, 0x031E, 0xB6A8, 0x2E31, 0xEACA, 0x7253, 0xC7E5, 0x5F7C, 0xB094, 0x280D, 0x9DBB, 0x0522, 0x5E76, 0xC6EF, 0x7359, 0xEBC0, 0x0428, 0x9CB1, 0x2907, 0xB19E, 0x9FAF, 0x0736, 0xB280, 0x2A19, 0xC5F1, 0x5D68, 0xE8DE, 0x7047, 0x2B13, 0xB38A, 0x063C, 0x9EA5, 0x714D, 0xE9D4, 0x5C62, 0xC4FB, 0xC989, 0x5110, 0xE4A6, 0x7C3F, 0x93D7, 0x0B4E, 0xBEF8, 0x2661, 0x7D35, 0xE5AC, 0x501A, 0xC883, 0x276B, 0xBFF2, 0x0A44, 0x92DD, 0xBCEC, 0x2475, 0x91C3, 0x095A, 0xE6B2, 0x7E2B, 0xCB9D, 0x5304, 0x0850, 0x90C9, 0x257F, 0xBDE6, 0x520E, 0xCA97, 0x7F21, 0xE7B8, 0x2343, 0xBBDA, 0x0E6C, 0x96F5, 0x791D, 0xE184, 0x5432, 0xCCAB, 0x97FF, 0x0F66, 0xBAD0, 0x2249, 0xCDA1, 0x5538, 0xE08E, 0x7817, 0x5626, 0xCEBF, 0x7B09, 0xE390, 0x0C78, 0x94E1, 0x2157, 0xB9CE, 0xE29A, 0x7A03, 0xCFB5, 0x572C, 0xB8C4, 0x205D, 0x95EB, 0x0D72, 0x8F0F, 0x1796, 0xA220, 0x3AB9, 0xD551, 0x4DC8, 0xF87E, 0x60E7, 0x3BB3, 0xA32A, 0x169C, 0x8E05, 0x61ED, 0xF974, 0x4CC2, 0xD45B, 0xFA6A, 0x62F3, 0xD745, 0x4FDC, 0xA034, 0x38AD, 0x8D1B, 0x1582, 0x4ED6, 0xD64F, 0x63F9, 0xFB60, 0x1488, 0x8C11, 0x39A7, 0xA13E, 0x65C5, 0xFD5C, 0x48EA, 0xD073, 0x3F9B, 0xA702, 0x12B4, 0x8A2D, 0xD179, 0x49E0, 0xFC56, 0x64CF, 0x8B27, 0x13BE, 0xA608, 0x3E91, 0x10A0, 0x8839, 0x3D8F, 0xA516, 0x4AFE, 0xD267, 0x67D1, 0xFF48, 0xA41C, 0x3C85, 0x8933, 0x11AA, 0xFE42, 0x66DB, 0xD36D, 0x4BF4, 0x4686, 0xDE1F, 0x6BA9, 0xF330, 0x1CD8, 0x8441, 0x31F7, 0xA96E, 0xF23A, 0x6AA3, 0xDF15, 0x478C, 0xA864, 0x30FD, 0x854B, 0x1DD2, 0x33E3, 0xAB7A, 0x1ECC, 0x8655, 0x69BD, 0xF124, 0x4492, 0xDC0B, 0x875F, 0x1FC6, 0xAA70, 0x32E9, 0xDD01, 0x4598, 0xF02E, 0x68B7, 0xAC4C, 0x34D5, 0x8163, 0x19FA, 0xF612, 0x6E8B, 0xDB3D, 0x43A4, 0x18F0, 0x8069, 0x35DF, 0xAD46, 0x42AE, 0xDA37, 0x6F81, 0xF718, 0xD929, 0x41B0, 0xF406, 0x6C9F, 0x8377, 0x1BEE, 0xAE58, 0x36C1, 0x6D95, 0xF50C, 0x40BA, 0xD823, 0x37CB, 0xAF52, 0x1AE4, 0x827D}, /* 18 */
    /* 19 */ {0x0000, 0xC2C3, 0x999B, 0x5B58, 0x2F2B, 0xEDE8, 0xB6B0, 0x7473, 0x5E56, 0x9C95, 0xC7CD, 0x050E, 0x717D, 0xB3BE, 0xE8E6, 0x2A25, 0xBCAC, 0x7E6F, 0x2537, 0xE7F4, 0x9387, 0x5144, 0x0A1C, 0xC8DF, 0xE2FA, 0x2039, 0x7B61, 0xB9A2, 0xCDD1, 0x0F12, 0x544A, 0x9689, 0x6545, 0xA786, 0xFCDE, 0x3E1D, 0x4A6E, 0x88AD, 0xD3F5, 0x1136, 0x3B13, 0xF9D0, 0xA288, 0x604B, 0x1438, 0xD6FB, 0x8DA3, 0x4F60, 0xD9E9, 0x1B2A, 0x4072, 0x82B1, 0xF6C2, 0x3401, 0x6F59, 0xAD9A, 0x87BF, 0x457C, 0x1E24, 0xDCE7, 0xA894, 0x6A57, 0x310F, 0xF3CC, 0xCA8A, 0x0849, 0x5311, 0x91D2, 0xE5A1, 0x2762, 0x7C3A, 0xBEF9, 0x94DC, 0x561F, 0x0D47, 0xCF84, 0xBBF7, 0x7934, 0x226C, 0xE0AF, 0x7626, 0xB4E5, 0xEFBD, 0x2D7E, 0x590D, 0x9BCE, 0xC096, 0x0255, 0x2870, 0xEAB3, 0xB1EB, 0x7328, 0x075B, 0xC598, 0x9EC0, 0x5C03, 0xAFCF, 0x6D0C, 0x3654, 0xF497, 0x80E4, 0x4227, 0x197F, 0xDBBC, 0xF199, 0x335A, 0x6802, 0xAAC1, 0xDEB2, 0x1C71, 0x4729, 0x85EA, 0x1363, 0xD1A0, 0x8AF8, 0x483B, 0x3C48, 0xFE8B, 0xA5D3, 0x6710, 0x4D35, 0x8FF6, 0xD4AE, 0x166D, 0x621E, 0xA0DD, 0xFB85, 0x3946, 0x8909, 0x4BCA, 0x1092, 0xD251, 0xA622, 0x64E1, 0x3FB9, 0xFD7A, 0xD75F, 0x159C, 0x4EC4, 0x8C07, 0xF874, 0x3AB7, 0x61EF, 0xA32C, 0x35A5, 0xF766, 0xAC3E, 0x6EFD, 0x1A8E, 0xD84D, 0x8315, 0x41D6, 0x6BF3, 0xA930, 0xF268, 0x30AB, 0x44D8, 0x861B, 0xDD43, 0x1F80, 0xEC4C, 0x2E8F, 0x75D7, 0xB714, 0xC367, 0x01A4, 0x5AFC, 0x983F, 0xB21A, 0x70D9, 0x2B81, 0xE942, 0x9D31, 0x5FF2, 0x04AA, 0xC669, 0x50E0, 0x9223, 0xC97B, 0x0BB8, 0x7FCB, 0xBD08, 0xE650, 0x2493, 0x0EB6, 0xCC75, 0x972D, 0x55EE, 0x219D, 0xE35E, 0xB806, 0x7AC5, 0x4383, 0x8140, 0xDA18, 0x18DB, 0x6CA8, 0xAE6B, 0xF533, 0x37F0, 0x1DD5, 0xDF16, 0x844E, 0x468D, 0x32FE, 0xF03D, 0xAB65, 0x69A6, 0xFF2F, 0x3DEC, 0x66B4, 0xA477, 0xD004, 0x12C7, 0x499F, 0x8B5C, 0xA179, 0x63BA, 0x38E2, 0xFA21, 0x8E52, 0x4C91, 0x17C9, 0xD50A, 0x26C6, 0xE405, 0xBF5D, 0x7D9E, 0x09ED, 0xCB2E, 0x9076, 0x52B5, 0x7890, 0xBA53, 0xE10B, 0x23C8, 0x57BB, 0x9578, 0xCE20, 0x0CE3, 0x9A6A, 0x58A9, 0x03F1, 0xC132, 0xB541, 0x7782, 0x2CDA, 0xEE19, 0xC43C, 0x06FF, 0x5DA7, 0x9F64, 0xEB17, 0x29D4, 0x728C, 0xB04F}, /* 19 */
    /* 1A */ {0x0000, 0xEFEE, 0xC3C1, 0x2C2F, 0x9B9F, 0x7471, 0x585E, 0xB7B0, 0x2B23, 0xC4CD, 0xE8E2, 0x070C, 0xB0BC, 0x5F52, 0x737D, 0x9C93, 0x5646, 0xB9A8, 0x9587, 0x7A69, 0xCDD9, 0x2237, 0x0E18, 0xE1F6, 0x7D65, 0x928B, 0xBEA4, 0x514A, 0xE6FA, 0x0914, 0x253B, 0xCAD5, 0xAC8C, 0x4362, 0x6F4D, 0x80A3, 0x3713, 0xD8FD, 0xF4D2, 0x1B3C, 0x87AF, 0x6841, 0x446E, 0xAB80, 0x1C30, 0xF3DE, 0xDFF1, 0x301F, 0xFACA, 0x1524, 0x390B, 0xD6E5, 0x6155, 0x8EBB, 0xA294, 0x4D7A, 0xD1E9, 0x3E07, 0x1228, 0xFDC6, 0x4A76, 0xA598, 0x89B7, 0x6659, 0x4505, 0xAAEB, 0x86C4, 0x692A, 0xDE9A, 0x3174, 0x1D5B, 0xF2B5, 0x6E26, 0x81C8, 0xADE7, 0x4209, 0xF5B9, 0x1A57, 0x3678, 0xD996, 0x1343, 0xFCAD, 0xD082, 0x3F6C, 0x88DC, 0x6732, 0x4B1D, 0xA4F3, 0x3860, 0xD78E, 0xFBA1, 0x144F, 0xA3FF, 0x4C11, 0x603E, 0x8FD0, 0xE989, 0x0667, 0x2A48, 0xC5A6, 0x7216, 0x9DF8, 0xB1D7, 0x5E39, 0xC2AA, 0x2D44, 0x016B, 0xEE85, 0x5935, 0xB6DB, 0x9AF4, 0x751A, 0xBFCF, 0x5021, 0x7C0E, 0x93E0, 0x2450, 0xCBBE, 0xE791, 0x087F, 0x94EC, 0x7B02, 0x572D, 0xB8C3, 0x0F73, 0xE09D, 0xCCB2, 0x235C, 0x8A0A, 0x65E4, 0x49CB, 0xA625, 0x1195, 0xFE7B, 0xD254, 0x3DBA, 0xA129, 0x4EC7, 0x62E8, 0x8D06, 0x3AB6, 0xD558, 0xF977, 0x1699, 0xDC4C, 0x33A2, 0x1F8D, 0xF063, 0x47D3, 0xA83D, 0x8412, 0x6BFC, 0xF76F, 0x1881, 0x34AE, 0xDB40, 0x6CF0, 0x831E, 0xAF31, 0x40DF, 0x2686, 0xC968, 0xE547, 0x0AA9, 0xBD19, 0x52F7, 0x7ED8, 0x9136, 0x0DA5, 0xE24B, 0xCE64, 0x218A, 0x963A, 0x79D4, 0x55FB, 0xBA15, 0x70C0, 0x9F2E, 0xB301, 0x5CEF, 0xEB5F, 0x04B1, 0x289E, 0xC770, 0x5BE3, 0xB40D, 0x9822, 0x77CC, 0xC07C, 0x2F92, 0x03BD, 0xEC53, 0xCF0F, 0x20E1, 0x0CCE, 0xE320, 0x5490, 0xBB7E, 0x9751, 0x78BF, 0xE42C, 0x0BC2, 0x27ED, 0xC803, 0x7FB3, 0x905D, 0xBC72, 0x539C, 0x9949, 0x76A7, 0x5A88, 0xB566, 0x02D6, 0xED38, 0xC117, 0x2EF9, 0xB26A, 0x5D84, 0x71AB, 0x9E45, 0x29F5, 0xC61B, 0xEA34, 0x05DA, 0x6383, 0x8C6D, 0xA042, 0x4FAC, 0xF81C, 0x17F2, 0x3BDD, 0xD433, 0x48A0, 0xA74E, 0x8B61, 0x648F, 0xD33F, 0x3CD1, 0x10FE, 0xFF10, 0x35C5, 0xDA2B, 0xF604, 0x19EA, 0xAE5A, 0x41B4, 0x6D9B, 0x8275, 0x1EE6, 0xF108, 0xDD27, 0x32C9, 0x8579, 0x6A97, 0x46B8, 0xA956}, /* 1A */
    /* 1B */ {0x0000, 0x7776, 0xEEEC, 0x999A, 0xC1C5, 0xB6B3, 0x2F29, 0x585F, 0x9F97, 0xE8E1, 0x717B, 0x060D, 0x5E52, 0x2924, 0xB0BE, 0xC7C8, 0x2333, 0x5445, 0xCDDF, 0xBAA9, 0xE2F6, 0x9580, 0x0C1A, 0x7B6C, 0xBCA4, 0xCBD2, 0x5248, 0x253E, 0x7D61, 0x0A17, 0x938D, 0xE4FB, 0x4666, 0x3110, 0xA88A, 0xDFFC, 0x87A3, 0xF0D5, 0x694F, 0x1E39, 0xD9F1, 0xAE87, 0x371D, 0x406B, 0x1834, 0x6F42, 0xF6D8, 0x81AE, 0x6555, 0x1223, 0x8BB9, 0xFCCF, 0xA490, 0xD3E6, 0x4A7C, 0x3D0A, 0xFAC2, 0x8DB4, 0x142E, 0x6358, 0x3B07, 0x4C71, 0xD5EB, 0xA29D, 0x8CCC, 0xFBBA, 0x6220, 0x1556, 0x4D09, 0x3A7F, 0xA3E5, 0xD493, 0x135B, 0x642D, 0xFDB7, 0x8AC1, 0xD29E, 0xA5E8, 0x3C72, 0x4B04, 0xAFFF, 0xD889, 0x4113, 0x3665, 0x6E3A, 0x194C, 0x80D6, 0xF7A0, 0x3068, 0x471E, 0xDE84, 0xA9F2, 0xF1AD, 0x86DB, 0x1F41, 0x6837, 0xCAAA, 0xBDDC, 0x2446, 0x5330, 0x0B6F, 0x7C19, 0xE583, 0x92F5, 0x553D, 0x224B, 0xBBD1, 0xCCA7, 0x94F8, 0xE38E, 0x7A14, 0x0D62, 0xE999, 0x9EEF, 0x0775, 0x7003, 0x285C, 0x5F2A, 0xC6B0, 0xB1C6, 0x760E, 0x0178, 0x98E2, 0xEF94, 0xB7CB, 0xC0BD, 0x5927, 0x2E51, 0x0585, 0x72F3, 0xEB69, 0x9C1F, 0xC440, 0xB336, 0x2AAC, 0x5DDA, 0x9A12, 0xED64, 0x74FE, 0x0388, 0x5BD7, 0x2CA1, 0xB53B, 0xC24D, 0x26B6, 0x51C0, 0xC85A, 0xBF2C, 0xE773, 0x9005, 0x099F, 0x7EE9, 0xB921, 0xCE57, 0x57CD, 0x20BB, 0x78E4, 0x0F92, 0x9608, 0xE17E, 0x43E3, 0x3495, 0xAD0F, 0xDA79, 0x8226, 0xF550, 0x6CCA, 0x1BBC, 0xDC74, 0xAB02, 0x3298, 0x45EE, 0x1DB1, 0x6AC7, 0xF35D, 0x842B, 0x60D0, 0x17A6, 0x8E3C, 0xF94A, 0xA115, 0xD663, 0x4FF9, 0x388F, 0xFF47, 0x8831, 0x11AB, 0x66DD, 0x3E82, 0x49F4, 0xD06E, 0xA718, 0x8949, 0xFE3F, 0x67A5, 0x10D3, 0x488C, 0x3FFA, 0xA660, 0xD116, 0x16DE, 0x61A8, 0xF832, 0x8F44, 0xD71B, 0xA06D, 0x39F7, 0x4E81, 0xAA7A, 0xDD0C, 0x4496, 0x33E0, 0x6BBF, 0x1CC9, 0x8553, 0xF225, 0x35ED, 0x429B, 0xDB01, 0xAC77, 0xF428, 0x835E, 0x1AC4, 0x6DB2, 0xCF2F, 0xB859, 0x21C3, 0x56B5, 0x0EEA, 0x799C, 0xE006, 0x9770, 0x50B8, 0x27CE, 0xBE54, 0xC922, 0x917D, 0xE60B, 0x7F91, 0x08E7, 0xEC1C, 0x9B6A, 0x02F0, 0x7586, 0x2DD9, 0x5AAF, 0xC335, 0xB443, 0x738B, 0x04FD, 0x9D67, 0xEA11, 0xB24E, 0xC538, 0x5CA2, 0x2BD4}, /* 1B */
    /* 1C */ {0x0000, 0x3B3A, 0x7674, 0x4D4E, 0xECE8, 0xD7D2, 0x9A9C, 0xA1A6, 0xC5CD, 0xFEF7, 0xB3B9, 0x8883, 0x2925, 0x121F, 0x5F51, 0x646B, 0x9787, 0xACBD, 0xE1F3, 0xDAC9, 0x7B6F, 0x4055, 0x0D1B, 0x3621, 0x524A, 0x6970, 0x243E, 0x1F04, 0xBEA2, 0x8598, 0xC8D6, 0xF3EC, 0x3313, 0x0829, 0x4567, 0x7E5D, 0xDFFB, 0xE4C1, 0xA98F, 0x92B5, 0xF6DE, 0xCDE4, 0x80AA, 0xBB90, 0x1A36, 0x210C, 0x6C42, 0x5778, 0xA494, 0x9FAE, 0xD2E0, 0xE9DA, 0x487C, 0x7346, 0x3E08, 0x0532, 0x6159, 0x5A63, 0x172D, 0x2C17, 0x8DB1, 0xB68B, 0xFBC5, 0xC0FF, 0x6626, 0x5D1C, 0x1052, 0x2B68, 0x8ACE, 0xB1F4, 0xFCBA, 0xC780, 0xA3EB, 0x98D1, 0xD59F, 0xEEA5, 0x4F03, 0x7439, 0x3977, 0x024D, 0xF1A1, 0xCA9B, 0x87D5, 0xBCEF, 0x1D49, 0x2673, 0x6B3D, 0x5007, 0x346C, 0x0F56, 0x4218, 0x7922, 0xD884, 0xE3BE, 0xAEF0, 0x95CA, 0x5535, 0x6E0F, 0x2341, 0x187B, 0xB9DD, 0x82E7, 0xCFA9, 0xF493, 0x90F8, 0xABC2, 0xE68C, 0xDDB6, 0x7C10, 0x472A, 0x0A64, 0x315E, 0xC2B2, 0xF988, 0xB4C6, 0x8FFC, 0x2E5A, 0x1560, 0x582E, 0x6314, 0x077F, 0x3C45, 0x710B, 0x4A31, 0xEB97, 0xD0AD, 0x9DE3, 0xA6D9, 0xCC4C, 0xF776, 0xBA38, 0x8102, 0x20A4, 0x1B9E, 0x56D0, 0x6DEA, 0x0981, 0x32BB, 0x7FF5, 0x44CF, 0xE569, 0xDE53, 0x931D, 0xA827, 0x5BCB, 0x60F1, 0x2DBF, 0x1685, 0xB723, 0x8C19, 0xC157, 0xFA6D, 0x9E06, 0xA53C, 0xE872, 0xD348, 0x72EE, 0x49D4, 0x049A, 0x3FA0, 0xFF5F, 0xC465, 0x892B, 0xB211, 0x13B7, 0x288D, 0x65C3, 0x5EF9, 0x3A92, 0x01A8, 0x4CE6, 0x77DC, 0xD67A, 0xED40, 0xA00E, 0x9B34, 0x68D8, 0x53E2, 0x1EAC, 0x2596, 0x8430, 0xBF0A, 0xF244, 0xC97E, 0xAD15, 0x962F, 0xDB61, 0xE05B, 0x41FD, 0x7AC7, 0x3789, 0x0CB3, 0xAA6A, 0x9150, 0xDC1E, 0xE724, 0x4682, 0x7DB8, 0x30F6, 0x0BCC, 0x6FA7, 0x549D, 0x19D3, 0x22E9, 0x834F, 0xB875, 0xF53B, 0xCE01, 0x3DED, 0x06D7, 0x4B99, 0x70A3, 0xD105, 0xEA3F, 0xA771, 0x9C4B, 0xF820, 0xC31A, 0x8E54, 0xB56E, 0x14C8, 0x2FF2, 0x62BC, 0x5986, 0x9979, 0xA243, 0xEF0D, 0xD437, 0x7591, 0x4EAB, 0x03E5, 0x38DF, 0x5CB4, 0x678E, 0x2AC0, 0x11FA, 0xB05C, 0x8B66, 0xC628, 0xFD12, 0x0EFE, 0x35C4, 0x788A, 0x43B0, 0xE216, 0xD92C, 0x9462, 0xAF58, 0xCB33, 0xF009, 0xBD47, 0x867D, 0x27DB, 0x1CE1, 0x51AF, 0x6A95}, /* 1C */
    /* 1D */ {0x0000, 0x1D1C, 0x3A38, 0x2724, 0x7470, 0x696C, 0x4E48, 0x5354, 0xE8E0, 0xF5FC, 0xD2D8, 0xCFC4, 0x9C90, 0x818C, 0xA6A8, 0xBBB4, 0xCDDD, 0xD0C1, 0xF7E5, 0xEAF9, 0xB9AD, 0xA4B1, 0x8395, 0x9E89, 0x253D, 0x3821, 0x1F05, 0x0219, 0x514D, 0x4C51, 0x6B75, 0x7669, 0x87A7, 0x9ABB, 0xBD9F, 0xA083, 0xF3D7, 0xEECB, 0xC9EF, 0xD4F3, 0x6F47, 0x725B, 0x557F, 0x4863, 0x1B37, 0x062B, 0x210F, 0x3C13, 0x4A7A, 0x5766, 0x7042, 0x6D5E, 0x3E0A, 0x2316, 0x0432, 0x192E, 0xA29A, 0xBF86, 0x98A2, 0x85BE, 0xD6EA, 0xCBF6, 0xECD2, 0xF1CE, 0x1353, 0x0E4F, 0x296B, 0x3477, 0x6723, 0x7A3F, 0x5D1B, 0x4007, 0xFBB3, 0xE6AF, 0xC18B, 0xDC97, 0x8FC3, 0x92DF, 0xB5FB, 0xA8E7, 0xDE8E, 0xC392, 0xE4B6, 0xF9AA, 0xAAFE, 0xB7E2, 0x90C6, 0x8DDA, 0x366E, 0x2B72, 0x0C56, 0x114A, 0x421E, 0x5F02, 0x7826, 0x653A, 0x94F4, 0x89E8, 0xAECC, 0xB3D0, 0xE084, 0xFD98, 0xDABC, 0xC7A0, 0x7C14, 0x6108, 0x462C, 0x5B30, 0x0864, 0x1578, 0x325C, 0x2F40, 0x5929, 0x4435, 0x6311, 0x7E0D, 0x2D59, 0x3045, 0x1761, 0x0A7D, 0xB1C9, 0xACD5, 0x8BF1, 0x96ED, 0xC5B9, 0xD8A5, 0xFF81, 0xE29D, 0x26A6, 0x3BBA, 0x1C9E, 0x0182, 0x52D6, 0x4FCA, 0x68EE, 0x75F2, 0xCE46, 0xD35A, 0xF47E, 0xE962, 0xBA36, 0xA72A, 0x800E, 0x9D12, 0xEB7B, 0xF667, 0xD143, 0xCC5F, 0x9F0B, 0x8217, 0xA533, 0xB82F, 0x039B, 0x1E87, 0x39A3, 0x24BF, 0x77EB, 0x6AF7, 0x4DD3, 0x50CF, 0xA101, 0xBC1D, 0x9B39, 0x8625, 0xD571, 0xC86D, 0xEF49, 0xF255, 0x49E1, 0x54FD, 0x73D9, 0x6EC5, 0x3D91, 0x208D, 0x07A9, 0x1AB5, 0x6CDC, 0x71C0, 0x56E4, 0x4BF8, 0x18AC, 0x05B0, 0x2294, 0x3F88, 0x843C, 0x9920, 0xBE04, 0xA318, 0xF04C, 0xED50, 0xCA74, 0xD768, 0x35F5, 0x28E9, 0x0FCD, 0x12D1, 0x4185, 0x5C99, 0x7BBD, 0x66A1, 0xDD15, 0xC009, 0xE72D, 0xFA31, 0xA965, 0xB479, 0x935D, 0x8E41, 0xF828, 0xE534, 0xC210, 0xDF0C, 0x8C58, 0x9144, 0xB660, 0xAB7C, 0x10C8, 0x0DD4, 0x2AF0, 0x37EC, 0x64B8, 0x79A4, 0x5E80, 0x439C, 0xB252, 0xAF4E, 0x886A, 0x9576, 0xC622, 0xDB3E, 0xFC1A, 0xE106, 0x5AB2, 0x47AE, 0x608A, 0x7D96, 0x2EC2, 0x33DE, 0x14FA, 0x09E6, 0x7F8F, 0x6293, 0x45B7, 0x58AB, 0x0BFF, 0x16E3, 0x31C7, 0x2CDB, 0x976F, 0x8A73, 0xAD57, 0xB04B, 0xE31F, 0xFE03, 0xD927, 0xC43B}, /* 1D */
    /* 1E */ {0x0000, 0x0E0F, 0x1C1E, 0x1211, 0x383C, 0x3633, 0x2422, 0x2A2D, 0x7078, 0x7E77, 0x6C66, 0x6269, 0x4844, 0x464B, 0x545A, 0x5A55, 0xE0F0, 0xEEFF, 0xFCEE, 0xF2E1, 0xD8CC, 0xD6C3, 0xC4D2, 0xCADD, 0x9088, 0x9E87, 0x8C96, 0x8299, 0xA8B4, 0xA6BB, 0xB4AA, 0xBAA5, 0xDDFD, 0xD3F2, 0xC1E3, 0xCFEC, 0xE5C1, 0xEBCE, 0xF9DF, 0xF7D0, 0xAD85, 0xA38A, 0xB19B, 0xBF94, 0x95B9, 0x9BB6, 0x89A7, 0x87A8, 0x3D0D, 0x3302, 0x2113, 0x2F1C, 0x0531, 0x0B3E, 0x192F, 0x1720, 0x4D75, 0x437A, 0x516B, 0x5F64, 0x7549, 0x7B46, 0x6957, 0x6758, 0xA7E7, 0xA9E8, 0xBBF9, 0xB5F6, 0x9FDB, 0x91D4, 0x83C5, 0x8DCA, 0xD79F, 0xD990, 0xCB81, 0xC58E, 0xEFA3, 0xE1AC, 0xF3BD, 0xFDB2, 0x4717, 0x4918, 0x5B09, 0x5506, 0x7F2B, 0x7124, 0x6335, 0x6D3A, 0x376F, 0x3960, 0x2B71, 0x257E, 0x0F53, 0x015C, 0x134D, 0x1D42, 0x7A1A, 0x7415, 0x6604, 0x680B, 0x4226, 0x4C29, 0x5E38, 0x5037, 0x0A62, 0x046D, 0x167C, 0x1873, 0x325E, 0x3C51, 0x2E40, 0x204F, 0x9AEA, 0x94E5, 0x86F4, 0x88FB, 0xA2D6, 0xACD9, 0xBEC8, 0xB0C7, 0xEA92, 0xE49D, 0xF68C, 0xF883, 0xD2AE, 0xDCA1, 0xCEB0, 0xC0BF, 0x53D3, 0x5DDC, 0x4FCD, 0x41C2, 0x6BEF, 0x65E0, 0x77F1, 0x79FE, 0x23AB, 0x2DA4, 0x3FB5, 0x31BA, 0x1B97, 0x1598, 0x0789, 0x0986, 0xB323, 0xBD2C, 0xAF3D, 0xA132, 0x8B1F, 0x8510, 0x9701, 0x990E, 0xC35B, 0xCD54, 0xDF45, 0xD14A, 0xFB67, 0xF568, 0xE779, 0xE976, 0x8E2E, 0x8021, 0x9230, 0x9C3F, 0xB612, 0xB81D, 0xAA0C, 0xA403, 0xFE56, 0xF059, 0xE248, 0xEC47, 0xC66A, 0xC865, 0xDA74, 0xD47B, 0x6EDE, 0x60D1, 0x72C0, 0x7CCF, 0x56E2, 0x58ED, 0x4AFC, 0x44F3, 0x1EA6, 0x10A9, 0x02B8, 0x0CB7, 0x269A, 0x2895, 0x3A84, 0x348B, 0xF434, 0xFA3B, 0xE82A, 0xE625, 0xCC08, 0xC207, 0xD016, 0xDE19, 0x844C, 0x8A43, 0x9852, 0x965D, 0xBC70, 0xB27F, 0xA06E, 0xAE61, 0x14C4, 0x1ACB, 0x08DA, 0x06D5, 0x2CF8, 0x22F7, 0x30E6, 0x3EE9, 0x64BC, 0x6AB3, 0x78A2, 0x76AD, 0x5C80, 0x528F, 0x409E, 0x4E91, 0x29C9, 0x27C6, 0x35D7, 0x3BD8, 0x11F5, 0x1FFA, 0x0DEB, 0x03E4, 0x59B1, 0x57BE, 0x45AF, 0x4BA0, 0x618D, 0x6F82, 0x7D93, 0x739C, 0xC939, 0xC736, 0xD527, 0xDB28, 0xF105, 0xFF0A, 0xED1B, 0xE314, 0xB941, 0xB74E, 0xA55F, 0xAB50, 0x817D, 0x8F72, 0x9D63, 0x936C}, /* 1E */
    /* 1F */ {0x0000, 0x8988, 0x0F0D, 0x8685, 0x1E1A, 0x9792, 0x1117, 0x989F, 0x3C34, 0xB5BC, 0x3339, 0xBAB1, 0x222E, 0xABA6, 0x2D23, 0xA4AB, 0x7868, 0xF1E0, 0x7765, 0xFEED, 0x6672, 0xEFFA, 0x697F, 0xE0F7, 0x445C, 0xCDD4, 0x4B51, 0xC2D9, 0x5A46, 0xD3CE, 0x554B, 0xDCC3, 0xF0D0, 0x7958, 0xFFDD, 0x7655, 0xEECA, 0x6742, 0xE1C7, 0x684F, 0xCCE4, 0x456C, 0xC3E9, 0x4A61, 0xD2FE, 0x5B76, 0xDDF3, 0x547B, 0x88B8, 0x0130, 0x87B5, 0x0E3D, 0x96A2, 0x1F2A, 0x99AF, 0x1027, 0xB48C, 0x3D04, 0xBB81, 0x3209, 0xAA96, 0x231E, 0xA59B, 0x2C13, 0xFDBD, 0x7435, 0xF2B0, 0x7B38, 0xE3A7, 0x6A2F, 0xECAA, 0x6522, 0xC189, 0x4801, 0xCE84, 0x470C, 0xDF93, 0x561B, 0xD09E, 0x5916, 0x85D5, 0x0C5D, 0x8AD8, 0x0350, 0x9BCF, 0x1247, 0x94C2, 0x1D4A, 0xB9E1, 0x3069, 0xB6EC, 0x3F64, 0xA7FB, 0x2E73, 0xA8F6, 0x217E, 0x0D6D, 0x84E5, 0x0260, 0x8BE8, 0x1377, 0x9AFF, 0x1C7A, 0x95F2, 0x3159, 0xB8D1, 0x3E54, 0xB7DC, 0x2F43, 0xA6CB, 0x204E, 0xA9C6, 0x7505, 0xFC8D, 0x7A08, 0xF380, 0x6B1F, 0xE297, 0x6412, 0xED9A, 0x4931, 0xC0B9, 0x463C, 0xCFB4, 0x572B, 0xDEA3, 0x5826, 0xD1AE, 0xE767, 0x6EEF, 0xE86A, 0x61E2, 0xF97D, 0x70F5, 0xF670, 0x7FF8, 0xDB53, 0x52DB, 0xD45E, 0x5DD6, 0xC549, 0x4CC1, 0xCA44, 0x43CC, 0x9F0F, 0x1687, 0x9002, 0x198A, 0x8115, 0x089D, 0x8E18, 0x0790, 0xA33B, 0x2AB3, 0xAC36, 0x25BE, 0xBD21, 0x34A9, 0xB22C, 0x3BA4, 0x17B7, 0x9E3F, 0x18BA, 0x9132, 0x09AD, 0x8025, 0x06A0, 0x8F28, 0x2B83, 0xA20B, 0x248E, 0xAD06, 0x3599, 0xBC11, 0x3A94, 0xB31C, 0x6FDF, 0xE657, 0x60D2, 0xE95A, 0x71C5, 0xF84D, 0x7EC8, 0xF740, 0x53EB, 0xDA63, 0x5CE6, 0xD56E, 0x4DF1, 0xC479, 0x42FC, 0xCB74, 0x1ADA, 0x9352, 0x15D7, 0x9C5F, 0x04C0, 0x8D48, 0x0BCD, 0x8245, 0x26EE, 0xAF66, 0x29E3, 0xA06B, 0x38F4, 0xB17C, 0x37F9, 0xBE71, 0x62B2, 0xEB3A, 0x6DBF, 0xE437, 0x7CA8, 0xF520, 0x73A5, 0xFA2D, 0x5E86, 0xD70E, 0x518B, 0xD803, 0x409C, 0xC914, 0x4F91, 0xC619, 0xEA0A, 0x6382, 0xE507, 0x6C8F, 0xF410, 0x7D98, 0xFB1D, 0x7295, 0xD63E, 0x5FB6, 0xD933, 0x50BB, 0xC824, 0x41AC, 0xC729, 0x4EA1, 0x9262, 0x1BEA, 0x9D6F, 0x14E7, 0x8C78, 0x05F0, 0x8375, 0x0AFD, 0xAE56, 0x27DE, 0xA15B, 0x28D3, 0xB04C, 0x39C4, 0xBF41, 0x36C9}, /* 1F */
    /* 20 */ {0x0000, 0x4445, 0x888A, 0xCCCF, 0x0D09, 0x494C, 0x8583, 0xC1C6, 0x1A12, 0x5E57, 0x9298, 0xD6DD, 0x171B, 0x535E, 0x9F91, 0xDBD4, 0x3424, 0x7061, 0xBCAE, 0xF8EB, 0x392D, 0x7D68, 0xB1A7, 0xF5E2, 0x2E36, 0x6A73, 0xA6BC, 0xE2F9, 0x233F, 0x677A, 0xABB5, 0xEFF0, 0x6848, 0x2C0D, 0xE0C2, 0xA487, 0x6541, 0x2104, 0xEDCB, 0xA98E, 0x725A, 0x361F, 0xFAD0, 0xBE95, 0x7F53, 0x3B16, 0xF7D9, 0xB39C, 0x5C6C, 0x1829, 0xD4E6, 0x90A3, 0x5165, 0x1520, 0xD9EF, 0x9DAA, 0x467E, 0x023B, 0xCEF4, 0x8AB1, 0x4B77, 0x0F32, 0xC3FD, 0x87B8, 0xD090, 0x94D5, 0x581A, 0x1C5F, 0xDD99, 0x99DC, 0x5513, 0x1156, 0xCA82, 0x8EC7, 0x4208, 0x064D, 0xC78B, 0x83CE, 0x4F01, 0x0B44, 0xE4B4, 0xA0F1, 0x6C3E, 0x287B, 0xE9BD, 0xADF8, 0x6137, 0x2572, 0xFEA6, 0xBAE3, 0x762C, 0x3269, 0xF3AF, 0xB7EA, 0x7B25, 0x3F60, 0xB8D8, 0xFC9D, 0x3052, 0x7417, 0xB5D1, 0xF194, 0x3D5B, 0x791E, 0xA2CA, 0xE68F, 0x2A40, 0x6E05, 0xAFC3, 0xEB86, 0x2749, 0x630C, 0x8CFC, 0xC8B9, 0x0476, 0x4033, 0x81F5, 0xC5B0, 0x097F, 0x4D3A, 0x96EE, 0xD2AB, 0x1E64, 0x5A21, 0x9BE7, 0xDFA2, 0x136D, 0x5728, 0xBD3D, 0xF978, 0x35B7, 0x71F2, 0xB034, 0xF471, 0x38BE, 0x7CFB, 0xA72F, 0xE36A, 0x2FA5, 0x6BE0, 0xAA26, 0xEE63, 0x22AC, 0x66E9, 0x8919, 0xCD5C, 0x0193, 0x45D6, 0x8410, 0xC055, 0x0C9A, 0x48DF, 0x930B, 0xD74E, 0x1B81, 0x5FC4, 0x9E02, 0xDA47, 0x1688, 0x52CD, 0xD575, 0x9130, 0x5DFF, 0x19BA, 0xD87C, 0x9C39, 0x50F6, 0x14B3, 0xCF67, 0x8B22, 0x47ED, 0x03A8, 0xC26E, 0x862B, 0x4AE4, 0x0EA1, 0xE151, 0xA514, 0x69DB, 0x2D9E, 0xEC58, 0xA81D, 0x64D2, 0x2097, 0xFB43, 0xBF06, 0x73C9, 0x378C, 0xF64A, 0xB20F, 0x7EC0, 0x3A85, 0x6DAD, 0x29E8, 0xE527, 0xA162, 0x60A4, 0x24E1, 0xE82E, 0xAC6B, 0x77BF, 0x33FA, 0xFF35, 0xBB70, 0x7AB6, 0x3EF3, 0xF23C, 0xB679, 0x5989, 0x1DCC, 0xD103, 0x9546, 0x5480, 0x10C5, 0xDC0A, 0x984F, 0x439B, 0x07DE, 0xCB11, 0x8F54, 0x4E92, 0x0AD7, 0xC618, 0x825D, 0x05E5, 0x41A0, 0x8D6F, 0xC92A, 0x08EC, 0x4CA9, 0x8066, 0xC423, 0x1FF7, 0x5BB2, 0x977D, 0xD338, 0x12FE, 0x56BB, 0x9A74, 0xDE31, 0x31C1, 0x7584, 0xB94B, 0xFD0E, 0x3CC8, 0x788D, 0xB442, 0xF007, 0x2BD3, 0x6F96, 0xA359, 0xE71C, 0x26DA, 0x629F, 0xAE50, 0xEA15}, /* 20 */
    /* 21 */ {0x0000, 0xACAD, 0x4547, 0xE9EA, 0x8A8E, 0x2623, 0xCFC9, 0x6364, 0x0901, 0xA5AC, 0x4C46, 0xE0EB, 0x838F, 0x2F22, 0xC6C8, 0x6A65, 0x1202, 0xBEAF, 0x5745, 0xFBE8, 0x988C, 0x3421, 0xDDCB, 0x7166, 0x1B03, 0xB7AE, 0x5E44, 0xF2E9, 0x918D, 0x3D20, 0xD4CA, 0x7867, 0x2404, 0x88A9, 0x6143, 0xCDEE, 0xAE8A, 0x0227, 0xEBCD, 0x4760, 0x2D05, 0x81A8, 0x6842, 0xC4EF, 0xA78B, 0x0B26, 0xE2CC, 0x4E61, 0x3606, 0x9AAB, 0x7341, 0xDFEC, 0xBC88, 0x1025, 0xF9CF, 0x5562, 0x3F07, 0x93AA, 0x7A40, 0xD6ED, 0xB589, 0x1924, 0xF0CE, 0x5C63, 0x4808, 0xE4A5, 0x0D4F, 0xA1E2, 0xC286, 0x6E2B, 0x87C1, 0x2B6C, 0x4109, 0xEDA4, 0x044E, 0xA8E3, 0xCB87, 0x672A, 0x8EC0, 0x226D, 0x5A0A, 0xF6A7, 0x1F4D, 0xB3E0, 0xD084, 0x7C29, 0x95C3, 0x396E, 0x530B, 0xFFA6, 0x164C, 0xBAE1, 0xD985, 0x7528, 0x9CC2, 0x306F, 0x6C0C, 0xC0A1, 0x294B, 0x85E6, 0xE682, 0x4A2F, 0xA3C5, 0x0F68, 0x650D, 0xC9A0, 0x204A, 0x8CE7, 0xEF83, 0x432E, 0xAAC4, 0x0669, 0x7E0E, 0xD2A3, 0x3B49, 0x97E4, 0xF480, 0x582D, 0xB1C7, 0x1D6A, 0x770F, 0xDBA2, 0x3248, 0x9EE5, 0xFD81, 0x512C, 0xB8C6, 0x146B, 0x9010, 0x3CBD, 0xD557, 0x79FA, 0x1A9E, 0xB633, 0x5FD9, 0xF374, 0x9911, 0x35BC, 0xDC56, 0x70FB, 0x139F, 0xBF32, 0x56D8, 0xFA75, 0x8212, 0x2EBF, 0xC755, 0x6BF8, 0x089C, 0xA431, 0x4DDB, 0xE176, 0x8B13, 0x27BE, 0xCE54, 0x62F9, 0x019D, 0xAD30, 0x44DA, 0xE877, 0xB414, 0x18B9, 0xF153, 0x5DFE, 0x3E9A, 0x9237, 0x7BDD, 0xD770, 0xBD15, 0x11B8, 0xF852, 0x54FF, 0x379B, 0x9B36, 0x72DC, 0xDE71, 0xA616, 0x0ABB, 0xE351, 0x4FFC, 0x2C98, 0x8035, 0x69DF, 0xC572, 0xAF17, 0x03BA, 0xEA50, 0x46FD, 0x2599, 0x8934, 0x60DE, 0xCC73, 0xD818, 0x74B5, 0x9D5F, 0x31F2, 0x5296, 0xFE3B, 0x17D1, 0xBB7C, 0xD119, 0x7DB4, 0x945E, 0x38F3, 0x5B97, 0xF73A, 0x1ED0, 0xB27D, 0xCA1A, 0x66B7, 0x8F5D, 0x23F0, 0x4094, 0xEC39, 0x05D3, 0xA97E, 0xC31B, 0x6FB6, 0x865C, 0x2AF1, 0x4995, 0xE538, 0x0CD2, 0xA07F, 0xFC1C, 0x50B1, 0xB95B, 0x15F6, 0x7692, 0xDA3F, 0x33D5, 0x9F78, 0xF51D, 0x59B0, 0xB05A, 0x1CF7, 0x7F93, 0xD33E, 0x3AD4, 0x9679, 0xEE1E, 0x42B3, 0xAB59, 0x07F4, 0x6490, 0xC83D, 0x21D7, 0x8D7A, 0xE71F, 0x4BB2, 0xA258, 0x0EF5, 0x6D91, 0xC13C, 0x28D6, 0x847B}, /* 21 */
    /* 22 */ {0x0000, 0xD8D9, 0xADAF, 0x7576, 0x4743, 0x9F9A, 0xEAEC, 0x3235, 0x8E86, 0x565F, 0x2329, 0xFBF0, 0xC9C5, 0x111C, 0x646A, 0xBCB3, 0x0111, 0xD9C8, 0xACBE, 0x7467, 0x4652, 0x9E8B, 0xEBFD, 0x3324, 0x8F97, 0x574E, 0x2238, 0xFAE1, 0xC8D4, 0x100D, 0x657B, 0xBDA2, 0x0222, 0xDAFB, 0xAF8D, 0x7754, 0x4561, 0x9DB8, 0xE8CE, 0x3017, 0x8CA4, 0x547D, 0x210B, 0xF9D2, 0xCBE7, 0x133E, 0x6648, 0xBE91, 0x0333, 0xDBEA, 0xAE9C, 0x7645, 0x4470, 0x9CA9, 0xE9DF, 0x3106, 0x8DB5, 0x556C, 0x201A, 0xF8C3, 0xCAF6, 0x122F, 0x6759, 0xBF80, 0x0444, 0xDC9D, 0xA9EB, 0x7132, 0x4307, 0x9BDE, 0xEEA8, 0x3671, 0x8AC2, 0x521B, 0x276D, 0xFFB4, 0xCD81, 0x1558, 0x602E, 0xB8F7, 0x0555, 0xDD8C, 0xA8FA, 0x7023, 0x4216, 0x9ACF, 0xEFB9, 0x3760, 0x8BD3, 0x530A, 0x267C, 0xFEA5, 0xCC90, 0x1449, 0x613F, 0xB9E6, 0x0666, 0xDEBF, 0xABC9, 0x7310, 0x4125, 0x99FC, 0xEC8A, 0x3453, 0x88E0, 0x5039, 0x254F, 0xFD96, 0xCFA3, 0x177A, 0x620C, 0xBAD5, 0x0777, 0xDFAE, 0xAAD8, 0x7201, 0x4034, 0x98ED, 0xED9B, 0x3542, 0x89F1, 0x5128, 0x245E, 0xFC87, 0xCEB2, 0x166B, 0x631D, 0xBBC4, 0x0888, 0xD051, 0xA527, 0x7DFE, 0x4FCB, 0x9712, 0xE264, 0x3ABD, 0x860E, 0x5ED7, 0x2BA1, 0xF378, 0xC14D, 0x1994, 0x6CE2, 0xB43B, 0x0999, 0xD140, 0xA436, 0x7CEF, 0x4EDA, 0x9603, 0xE375, 0x3BAC, 0x871F, 0x5FC6, 0x2AB0, 0xF269, 0xC05C, 0x1885, 0x6DF3, 0xB52A, 0x0AAA, 0xD273, 0xA705, 0x7FDC, 0x4DE9, 0x9530, 0xE046, 0x389F, 0x842C, 0x5CF5, 0x2983, 0xF15A, 0xC36F, 0x1BB6, 0x6EC0, 0xB619, 0x0BBB, 0xD362, 0xA614, 0x7ECD, 0x4CF8, 0x9421, 0xE157, 0x398E, 0x853D, 0x5DE4, 0x2892, 0xF04B, 0xC27E, 0x1AA7, 0x6FD1, 0xB708, 0x0CCC, 0xD415, 0xA163, 0x79BA, 0x4B8F, 0x9356, 0xE620, 0x3EF9, 0x824A, 0x5A93, 0x2FE5, 0xF73C, 0xC509, 0x1DD0, 0x68A6, 0xB07F, 0x0DDD, 0xD504, 0xA072, 0x78AB, 0x4A9E, 0x9247, 0xE731, 0x3FE8, 0x835B, 0x5B82, 0x2EF4, 0xF62D, 0xC418, 0x1CC1, 0x69B7, 0xB16E, 0x0EEE, 0xD637, 0xA341, 0x7B98, 0x49AD, 0x9174, 0xE402, 0x3CDB, 0x8068, 0x58B1, 0x2DC7, 0xF51E, 0xC72B, 0x1FF2, 0x6A84, 0xB25D, 0x0FFF, 0xD726, 0xA250, 0x7A89, 0x48BC, 0x9065, 0xE513, 0x3DCA, 0x8179, 0x59A0, 0x2CD6, 0xF40F, 0xC63A, 0x1EE3, 0x6B95, 0xB34C}, /* 22 */
    /* 23 */ {0x0000, 0xE2E3, 0xD9DB, 0x3B38, 0xAFAB, 0x4D48, 0x7670, 0x9493, 0x434B, 0xA1A8, 0x9A90, 0x7873, 0xECE0, 0x0E03, 0x353B, 0xD7D8, 0x8696, 0x6475, 0x5F4D, 0xBDAE, 0x293D, 0xCBDE, 0xF0E6, 0x1205, 0xC5DD, 0x273E, 0x1C06, 0xFEE5, 0x6A76, 0x8895, 0xB3AD, 0x514E, 0x1131, 0xF3D2, 0xC8EA, 0x2A09, 0xBE9A, 0x5C79, 0x6741, 0x85A2, 0x527A, 0xB099, 0x8BA1, 0x6942, 0xFDD1, 0x1F32, 0x240A, 0xC6E9, 0x97A7, 0x7544, 0x4E7C, 0xAC9F, 0x380C, 0xDAEF, 0xE1D7, 0x0334, 0xD4EC, 0x360F, 0x0D37, 0xEFD4, 0x7B47, 0x99A4, 0xA29C, 0x407F, 0x2262, 0xC081, 0xFBB9, 0x195A, 0x8DC9, 0x6F2A, 0x5412, 0xB6F1, 0x6129, 0x83CA, 0xB8F2, 0x5A11, 0xCE82, 0x2C61, 0x1759, 0xF5BA, 0xA4F4, 0x4617, 0x7D2F, 0x9FCC, 0x0B5F, 0xE9BC, 0xD284, 0x3067, 0xE7BF, 0x055C, 0x3E64, 0xDC87, 0x4814, 0xAAF7, 0x91CF, 0x732C, 0x3353, 0xD1B0, 0xEA88, 0x086B, 0x9CF8, 0x7E1B, 0x4523, 0xA7C0, 0x7018, 0x92FB, 0xA9C3, 0x4B20, 0xDFB3, 0x3D50, 0x0668, 0xE48B, 0xB5C5, 0x5726, 0x6C1E, 0x8EFD, 0x1A6E, 0xF88D, 0xC3B5, 0x2156, 0xF68E, 0x146D, 0x2F55, 0xCDB6, 0x5925, 0xBBC6, 0x80FE, 0x621D, 0x44C4, 0xA627, 0x9D1F, 0x7FFC, 0xEB6F, 0x098C, 0x32B4, 0xD057, 0x078F, 0xE56C, 0xDE54, 0x3CB7, 0xA824, 0x4AC7, 0x71FF, 0x931C, 0xC252, 0x20B1, 0x1B89, 0xF96A, 0x6DF9, 0x8F1A, 0xB422, 0x56C1, 0x8119, 0x63FA, 0x58C2, 0xBA21, 0x2EB2, 0xCC51, 0xF769, 0x158A, 0x55F5, 0xB716, 0x8C2E, 0x6ECD, 0xFA5E, 0x18BD, 0x2385, 0xC166, 0x16BE, 0xF45D, 0xCF65, 0x2D86, 0xB915, 0x5BF6, 0x60CE, 0x822D, 0xD363, 0x3180, 0x0AB8, 0xE85B, 0x7CC8, 0x9E2B, 0xA513, 0x47F0, 0x9028, 0x72CB, 0x49F3, 0xAB10, 0x3F83, 0xDD60, 0xE658, 0x04BB, 0x66A6, 0x8445, 0xBF7D, 0x5D9E, 0xC90D, 0x2BEE, 0x10D6, 0xF235, 0x25ED, 0xC70E, 0xFC36, 0x1ED5, 0x8A46, 0x68A5, 0x539D, 0xB17E, 0xE030, 0x02D3, 0x39EB, 0xDB08, 0x4F9B, 0xAD78, 0x9640, 0x74A3, 0xA37B, 0x4198, 0x7AA0, 0x9843, 0x0CD0, 0xEE33, 0xD50B, 0x37E8, 0x7797, 0x9574, 0xAE4C, 0x4CAF, 0xD83C, 0x3ADF, 0x01E7, 0xE304, 0x34DC, 0xD63F, 0xED07, 0x0FE4, 0x9B77, 0x7994, 0x42AC, 0xA04F, 0xF101, 0x13E2, 0x28DA, 0xCA39, 0x5EAA, 0xBC49, 0x8771, 0x6592, 0xB24A, 0x50A9, 0x6B91, 0x8972, 0x1DE1, 0xFF02, 0xC43A, 0x26D9}, /* 23 */
    /* 24 */ {0x0000, 0xFFFE, 0xE3E1, 0x1C1F, 0xDBDF, 0x2421, 0x383E, 0xC7C0, 0xABA3, 0x545D, 0x4842, 0xB7BC, 0x707C, 0x8F82, 0x939D, 0x6C63, 0x4B5B, 0xB4A5, 0xA8BA, 0x5744, 0x9084, 0x6F7A, 0x7365, 0x8C9B, 0xE0F8, 0x1F06, 0x0319, 0xFCE7, 0x3B27, 0xC4D9, 0xD8C6, 0x2738, 0x96B6, 0x6948, 0x7557, 0x8AA9, 0x4D69, 0xB297, 0xAE88, 0x5176, 0x3D15, 0xC2EB, 0xDEF4, 0x210A, 0xE6CA, 0x1934, 0x052B, 0xFAD5, 0xDDED, 0x2213, 0x3E0C, 0xC1F2, 0x0632, 0xF9CC, 0xE5D3, 0x1A2D, 0x764E, 0x89B0, 0x95AF, 0x6A51, 0xAD91, 0x526F, 0x4E70, 0xB18E, 0x3171, 0xCE8F, 0xD290, 0x2D6E, 0xEAAE, 0x1550, 0x094F, 0xF6B1, 0x9AD2, 0x652C, 0x7933, 0x86CD, 0x410D, 0xBEF3, 0xA2EC, 0x5D12, 0x7A2A, 0x85D4, 0x99CB, 0x6635, 0xA1F5, 0x5E0B, 0x4214, 0xBDEA, 0xD189, 0x2E77, 0x3268, 0xCD96, 0x0A56, 0xF5A8, 0xE9B7, 0x1649, 0xA7C7, 0x5839, 0x4426, 0xBBD8, 0x7C18, 0x83E6, 0x9FF9, 0x6007, 0x0C64, 0xF39A, 0xEF85, 0x107B, 0xD7BB, 0x2845, 0x345A, 0xCBA4, 0xEC9C, 0x1362, 0x0F7D, 0xF083, 0x3743, 0xC8BD, 0xD4A2, 0x2B5C, 0x473F, 0xB8C1, 0xA4DE, 0x5B20, 0x9CE0, 0x631E, 0x7F01, 0x80FF, 0x62E2, 0x9D1C, 0x8103, 0x7EFD, 0xB93D, 0x46C3, 0x5ADC, 0xA522, 0xC941, 0x36BF, 0x2AA0, 0xD55E, 0x129E, 0xED60, 0xF17F, 0x0E81, 0x29B9, 0xD647, 0xCA58, 0x35A6, 0xF266, 0x0D98, 0x1187, 0xEE79, 0x821A, 0x7DE4, 0x61FB, 0x9E05, 0x59C5, 0xA63B, 0xBA24, 0x45DA, 0xF454, 0x0BAA, 0x17B5, 0xE84B, 0x2F8B, 0xD075, 0xCC6A, 0x3394, 0x5FF7, 0xA009, 0xBC16, 0x43E8, 0x8428, 0x7BD6, 0x67C9, 0x9837, 0xBF0F, 0x40F1, 0x5CEE, 0xA310, 0x64D0, 0x9B2E, 0x8731, 0x78CF, 0x14AC, 0xEB52, 0xF74D, 0x08B3, 0xCF73, 0x308D, 0x2C92, 0xD36C, 0x5393, 0xAC6D, 0xB072, 0x4F8C, 0x884C, 0x77B2, 0x6BAD, 0x9453, 0xF830, 0x07CE, 0x1BD1, 0xE42F, 0x23EF, 0xDC11, 0xC00E, 0x3FF0, 0x18C8, 0xE736, 0xFB29, 0x04D7, 0xC317, 0x3CE9, 0x20F6, 0xDF08, 0xB36B, 0x4C95, 0x508A, 0xAF74, 0x68B4, 0x974A, 0x8B55, 0x74AB, 0xC525, 0x3ADB, 0x26C4, 0xD93A, 0x1EFA, 0xE104, 0xFD1B, 0x02E5, 0x6E86, 0x9178, 0x8D67, 0x7299, 0xB559, 0x4AA7, 0x56B8, 0xA946, 0x8E7E, 0x7180, 0x6D9F, 0x9261, 0x55A1, 0xAA5F, 0xB640, 0x49BE, 0x25DD, 0xDA23, 0xC63C, 0x39C2, 0xFE02, 0x01FC, 0x1DE3, 0xE21D}, /* 24 */
    /* 25 */ {0x0000, 0x7F7E, 0xFEFC, 0x8182, 0xE1E5, 0x9E9B, 0x1F19, 0x6067, 0xDFD7, 0xA0A9, 0x212B, 0x5E55, 0x3E32, 0x414C, 0xC0CE, 0xBFB0, 0xA3B3, 0xDCCD, 0x5D4F, 0x2231, 0x4256, 0x3D28, 0xBCAA, 0xC3D4, 0x7C64, 0x031A, 0x8298, 0xFDE6, 0x9D81, 0xE2FF, 0x637D, 0x1C03, 0x5B7B, 0x2405, 0xA587, 0xDAF9, 0xBA9E, 0xC5E0, 0x4462, 0x3B1C, 0x84AC, 0xFBD2, 0x7A50, 0x052E, 0x6549, 0x1A37, 0x9BB5, 0xE4CB, 0xF8C8, 0x87B6, 0x0634, 0x794A, 0x192D, 0x6653, 0xE7D1, 0x98AF, 0x271F, 0x5861, 0xD9E3, 0xA69D, 0xC6FA, 0xB984, 0x3806, 0x4778, 0xB6F6, 0xC988, 0x480A, 0x3774, 0x5713, 0x286D, 0xA9EF, 0xD691, 0x6921, 0x165F, 0x97DD, 0xE8A3, 0x88C4, 0xF7BA, 0x7638, 0x0946, 0x1545, 0x6A3B, 0xEBB9, 0x94C7, 0xF4A0, 0x8BDE, 0x0A5C, 0x7522, 0xCA92, 0xB5EC, 0x346E, 0x4B10, 0x2B77, 0x5409, 0xD58B, 0xAAF5, 0xED8D, 0x92F3, 0x1371, 0x6C0F, 0x0C68, 0x7316, 0xF294, 0x8DEA, 0x325A, 0x4D24, 0xCCA6, 0xB3D8, 0xD3BF, 0xACC1, 0x2D43, 0x523D, 0x4E3E, 0x3140, 0xB0C2, 0xCFBC, 0xAFDB, 0xD0A5, 0x5127, 0x2E59, 0x91E9, 0xEE97, 0x6F15, 0x106B, 0x700C, 0x0F72, 0x8EF0, 0xF18E, 0x71F1, 0x0E8F, 0x8F0D, 0xF073, 0x9014, 0xEF6A, 0x6EE8, 0x1196, 0xAE26, 0xD158, 0x50DA, 0x2FA4, 0x4FC3, 0x30BD, 0xB13F, 0xCE41, 0xD242, 0xAD3C, 0x2CBE, 0x53C0, 0x33A7, 0x4CD9, 0xCD5B, 0xB225, 0x0D95, 0x72EB, 0xF369, 0x8C17, 0xEC70, 0x930E, 0x128C, 0x6DF2, 0x2A8A, 0x55F4, 0xD476, 0xAB08, 0xCB6F, 0xB411, 0x3593, 0x4AED, 0xF55D, 0x8A23, 0x0BA1, 0x74DF, 0x14B8, 0x6BC6, 0xEA44, 0x953A, 0x8939, 0xF647, 0x77C5, 0x08BB, 0x68DC, 0x17A2, 0x9620, 0xE95E, 0x56EE, 0x2990, 0xA812, 0xD76C, 0xB70B, 0xC875, 0x49F7, 0x3689, 0xC707, 0xB879, 0x39FB, 0x4685, 0x26E2, 0x599C, 0xD81E, 0xA760, 0x18D0, 0x67AE, 0xE62C, 0x9952, 0xF935, 0x864B, 0x07C9, 0x78B7, 0x64B4, 0x1BCA, 0x9A48, 0xE536, 0x8551, 0xFA2F, 0x7BAD, 0x04D3, 0xBB63, 0xC41D, 0x459F, 0x3AE1, 0x5A86, 0x25F8, 0xA47A, 0xDB04, 0x9C7C, 0xE302, 0x6280, 0x1DFE, 0x7D99, 0x02E7, 0x8365, 0xFC1B, 0x43AB, 0x3CD5, 0xBD57, 0xC229, 0xA24E, 0xDD30, 0x5CB2, 0x23CC, 0x3FCF, 0x40B1, 0xC133, 0xBE4D, 0xDE2A, 0xA154, 0x20D6, 0x5FA8, 0xE018, 0x9F66, 0x1EE4, 0x619A, 0x01FD, 0x7E83, 0xFF01, 0x807F}, /* 25 */
    /* 26 */ {0x0000, 0x3F3E, 0x7E7C, 0x4142, 0xFCF8, 0xC3C6, 0x8284, 0xBDBA, 0xE5ED, 0xDAD3, 0x9B91, 0xA4AF, 0x1915, 0x262B, 0x6769, 0x5857, 0xD7C7, 0xE8F9, 0xA9BB, 0x9685, 0x2B3F, 0x1401, 0x5543, 0x6A7D, 0x322A, 0x0D14, 0x4C56, 0x7368, 0xCED2, 0xF1EC, 0xB0AE, 0x8F90, 0xB393, 0x8CAD, 0xCDEF, 0xF2D1, 0x4F6B, 0x7055, 0x3117, 0x0E29, 0x567E, 0x6940, 0x2802, 0x173C, 0xAA86, 0x95B8, 0xD4FA, 0xEBC4, 0x6454, 0x5B6A, 0x1A28, 0x2516, 0x98AC, 0xA792, 0xE6D0, 0xD9EE, 0x81B9, 0xBE87, 0xFFC5, 0xC0FB, 0x7D41, 0x427F, 0x033D, 0x3C03, 0x7B3B, 0x4405, 0x0547, 0x3A79, 0x87C3, 0xB8FD, 0xF9BF, 0xC681, 0x9ED6, 0xA1E8, 0xE0AA, 0xDF94, 0x622E, 0x5D10, 0x1C52, 0x236C, 0xACFC, 0x93C2, 0xD280, 0xEDBE, 0x5004, 0x6F3A, 0x2E78, 0x1146, 0x4911, 0x762F, 0x376D, 0x0853, 0xB5E9, 0x8AD7, 0xCB95, 0xF4AB, 0xC8A8, 0xF796, 0xB6D4, 0x89EA, 0x3450, 0x0B6E, 0x4A2C, 0x7512, 0x2D45, 0x127B, 0x5339, 0x6C07, 0xD1BD, 0xEE83, 0xAFC1, 0x90FF, 0x1F6F, 0x2051, 0x6113, 0x5E2D, 0xE397, 0xDCA9, 0x9DEB, 0xA2D5, 0xFA82, 0xC5BC, 0x84FE, 0xBBC0, 0x067A, 0x3944, 0x7806, 0x4738, 0xF676, 0xC948, 0x880A, 0xB734, 0x0A8E, 0x35B0, 0x74F2, 0x4BCC, 0x139B, 0x2CA5, 0x6DE7, 0x52D9, 0xEF63, 0xD05D, 0x911F, 0xAE21, 0x21B1, 0x1E8F, 0x5FCD, 0x60F3, 0xDD49, 0xE277, 0xA335, 0x9C0B, 0xC45C, 0xFB62, 0xBA20, 0x851E, 0x38A4, 0x079A, 0x46D8, 0x79E6, 0x45E5, 0x7ADB, 0x3B99, 0x04A7, 0xB91D, 0x8623, 0xC761, 0xF85F, 0xA008, 0x9F36, 0xDE74, 0xE14A, 0x5CF0, 0x63CE, 0x228C, 0x1DB2, 0x9222, 0xAD1C, 0xEC5E, 0xD360, 0x6EDA, 0x51E4, 0x10A6, 0x2F98, 0x77CF, 0x48F1, 0x09B3, 0x368D, 0x8B37, 0xB409, 0xF54B, 0xCA75, 0x8D4D, 0xB273, 0xF331, 0xCC0F, 0x71B5, 0x4E8B, 0x0FC9, 0x30F7, 0x68A0, 0x579E, 0x16DC, 0x29E2, 0x9458, 0xAB66, 0xEA24, 0xD51A, 0x5A8A, 0x65B4, 0x24F6, 0x1BC8, 0xA672, 0x994C, 0xD80E, 0xE730, 0xBF67, 0x8059, 0xC11B, 0xFE25, 0x439F, 0x7CA1, 0x3DE3, 0x02DD, 0x3EDE, 0x01E0, 0x40A2, 0x7F9C, 0xC226, 0xFD18, 0xBC5A, 0x8364, 0xDB33, 0xE40D, 0xA54F, 0x9A71, 0x27CB, 0x18F5, 0x59B7, 0x6689, 0xE919, 0xD627, 0x9765, 0xA85B, 0x15E1, 0x2ADF, 0x6B9D, 0x54A3, 0x0CF4, 0x33CA, 0x7288, 0x4DB6, 0xF00C, 0xCF32, 0x8E70, 0xB14E}, /* 26 */
    /* 27 */ {0x0000, 0x1F1E, 0x3E3C, 0x2122, 0x7C78, 0x6366, 0x4244, 0x5D5A, 0xF8F0, 0xE7EE, 0xC6CC, 0xD9D2, 0x8488, 0x9B96, 0xBAB4, 0xA5AA, 0xEDFD, 0xF2E3, 0xD3C1, 0xCCDF, 0x9185, 0x8E9B, 0xAFB9, 0xB0A7, 0x150D, 0x0A13, 0x2B31, 0x342F, 0x6975, 0x766B, 0x5749, 0x4857, 0xC7E7, 0xD8F9, 0xF9DB, 0xE6C5, 0xBB9F, 0xA481, 0x85A3, 0x9ABD, 0x3F17, 0x2009, 0x012B, 0x1E35, 0x436F, 0x5C71, 0x7D53, 0x624D, 0x2A1A, 0x3504, 0x1426, 0x0B38, 0x5662, 0x497C, 0x685E, 0x7740, 0xD2EA, 0xCDF4, 0xECD6, 0xF3C8, 0xAE92, 0xB18C, 0x90AE, 0x8FB0, 0x93D3, 0x8CCD, 0xADEF, 0xB2F1, 0xEFAB, 0xF0B5, 0xD197, 0xCE89, 0x6B23, 0x743D, 0x551F, 0x4A01, 0x175B, 0x0845, 0x2967, 0x3679, 0x7E2E, 0x6130, 0x4012, 0x5F0C, 0x0256, 0x1D48, 0x3C6A, 0x2374, 0x86DE, 0x99C0, 0xB8E2, 0xA7FC, 0xFAA6, 0xE5B8, 0xC49A, 0xDB84, 0x5434, 0x4B2A, 0x6A08, 0x7516, 0x284C, 0x3752, 0x1670, 0x096E, 0xACC4, 0xB3DA, 0x92F8, 0x8DE6, 0xD0BC, 0xCFA2, 0xEE80, 0xF19E, 0xB9C9, 0xA6D7, 0x87F5, 0x98EB, 0xC5B1, 0xDAAF, 0xFB8D, 0xE493, 0x4139, 0x5E27, 0x7F05, 0x601B, 0x3D41, 0x225F, 0x037D, 0x1C63, 0x3BBB, 0x24A5, 0x0587, 0x1A99, 0x47C3, 0x58DD, 0x79FF, 0x66E1, 0xC34B, 0xDC55, 0xFD77, 0xE269, 0xBF33, 0xA02D, 0x810F, 0x9E11, 0xD646, 0xC958, 0xE87A, 0xF764, 0xAA3E, 0xB520, 0x9402, 0x8B1C, 0x2EB6, 0x31A8, 0x108A, 0x0F94, 0x52CE, 0x4DD0, 0x6CF2, 0x73EC, 0xFC5C, 0xE342, 0xC260, 0xDD7E, 0x8024, 0x9F3A, 0xBE18, 0xA106, 0x04AC, 0x1BB2, 0x3A90, 0x258E, 0x78D4, 0x67CA, 0x46E8, 0x59F6, 0x11A1, 0x0EBF, 0x2F9D, 0x3083, 0x6DD9, 0x72C7, 0x53E5, 0x4CFB, 0xE951, 0xF64F, 0xD76D, 0xC873, 0x9529, 0x8A37, 0xAB15, 0xB40B, 0xA868, 0xB776, 0x9654, 0x894A, 0xD410, 0xCB0E, 0xEA2C, 0xF532, 0x5098, 0x4F86, 0x6EA4, 0x71BA, 0x2CE0, 0x33FE, 0x12DC, 0x0DC2, 0x4595, 0x5A8B, 0x7BA9, 0x64B7, 0x39ED, 0x26F3, 0x07D1, 0x18CF, 0xBD65, 0xA27B, 0x8359, 0x9C47, 0xC11D, 0xDE03, 0xFF21, 0xE03F, 0x6F8F, 0x7091, 0x51B3, 0x4EAD, 0x13F7, 0x0CE9, 0x2DCB, 0x32D5, 0x977F, 0x8861, 0xA943, 0xB65D, 0xEB07, 0xF419, 0xD53B, 0xCA25, 0x8272, 0x9D6C, 0xBC4E, 0xA350, 0xFE0A, 0xE114, 0xC036, 0xDF28, 0x7A82, 0x659C, 0x44BE, 0x5BA0, 0x06FA, 0x19E4, 0x38C6, 0x27D8}, /* 27 */
    /* 28 */ {0x0000, 0x0F0E, 0x1E1C, 0x1112, 0x3C38, 0x3336, 0x2224, 0x2D2A, 0x7870, 0x777E, 0x666C, 0x6962, 0x4448, 0x4B46, 0x5A54, 0x555A, 0xF0E0, 0xFFEE, 0xEEFC, 0xE1F2, 0xCCD8, 0xC3D6, 0xD2C4, 0xDDCA, 0x8890, 0x879E, 0x968C, 0x9982, 0xB4A8, 0xBBA6, 0xAAB4, 0xA5BA, 0xFDDD, 0xF2D3, 0xE3C1, 0xECCF, 0xC1E5, 0xCEEB, 0xDFF9, 0xD0F7, 0x85AD, 0x8AA3, 0x9BB1, 0x94BF, 0xB995, 0xB69B, 0xA789, 0xA887, 0x0D3D, 0x0233, 0x1321, 0x1C2F, 0x3105, 0x3E0B, 0x2F19, 0x2017, 0x754D, 0x7A43, 0x6B51, 0x645F, 0x4975, 0x467B, 0x5769, 0x5867, 0xE7A7, 0xE8A9, 0xF9BB, 0xF6B5, 0xDB9F, 0xD491, 0xC583, 0xCA8D, 0x9FD7, 0x90D9, 0x81CB, 0x8EC5, 0xA3EF, 0xACE1, 0xBDF3, 0xB2FD, 0x1747, 0x1849, 0x095B, 0x0655, 0x2B7F, 0x2471, 0x3563, 0x3A6D, 0x6F37, 0x6039, 0x712B, 0x7E25, 0x530F, 0x5C01, 0x4D13, 0x421D, 0x1A7A, 0x1574, 0x0466, 0x0B68, 0x2642, 0x294C, 0x385E, 0x3750, 0x620A, 0x6D04, 0x7C16, 0x7318, 0x5E32, 0x513C, 0x402E, 0x4F20, 0xEA9A, 0xE594, 0xF486, 0xFB88, 0xD6A2, 0xD9AC, 0xC8BE, 0xC7B0, 0x92EA, 0x9DE4, 0x8CF6, 0x83F8, 0xAED2, 0xA1DC, 0xB0CE, 0xBFC0, 0xD353, 0xDC5D, 0xCD4F, 0xC241, 0xEF6B, 0xE065, 0xF177, 0xFE79, 0xAB23, 0xA42D, 0xB53F, 0xBA31, 0x971B, 0x9815, 0x8907, 0x8609, 0x23B3, 0x2CBD, 0x3DAF, 0x32A1, 0x1F8B, 0x1085, 0x0197, 0x0E99, 0x5BC3, 0x54CD, 0x45DF, 0x4AD1, 0x67FB, 0x68F5, 0x79E7, 0x76E9, 0x2E8E, 0x2180, 0x3092, 0x3F9C, 0x12B6, 0x1DB8, 0x0CAA, 0x03A4, 0x56FE, 0x59F0, 0x48E2, 0x47EC, 0x6AC6, 0x65C8, 0x74DA, 0x7BD4, 0xDE6E, 0xD160, 0xC072, 0xCF7C, 0xE256, 0xED58, 0xFC4A, 0xF344, 0xA61E, 0xA910, 0xB802, 0xB70C, 0x9A26, 0x9528, 0x843A, 0x8B34, 0x34F4, 0x3BFA, 0x2AE8, 0x25E6, 0x08CC, 0x07C2, 0x16D0, 0x19DE, 0x4C84, 0x438A, 0x5298, 0x5D96, 0x70BC, 0x7FB2, 0x6EA0, 0x61AE, 0xC414, 0xCB1A, 0xDA08, 0xD506, 0xF82C, 0xF722, 0xE630, 0xE93E, 0xBC64, 0xB36A, 0xA278, 0xAD76, 0x805C, 0x8F52, 0x9E40, 0x914E, 0xC929, 0xC627, 0xD735, 0xD83B, 0xF511, 0xFA1F, 0xEB0D, 0xE403, 0xB159, 0xBE57, 0xAF45, 0xA04B, 0x8D61, 0x826F, 0x937D, 0x9C73, 0x39C9, 0x36C7, 0x27D5, 0x28DB, 0x05F1, 0x0AFF, 0x1BED, 0x14E3, 0x41B9, 0x4EB7, 0x5FA5, 0x50AB, 0x7D81, 0x728F, 0x639D, 0x6C93}, /* 28 */
    /* 29 */ {0x0000, 0x0706, 0x0E0C, 0x090A, 0x1C18, 0x1B1E, 0x1214, 0x1512, 0x3830, 0x3F36, 0x363C, 0x313A, 0x2428, 0x232E, 0x2A24, 0x2D22, 0x7060, 0x7766, 0x7E6C, 0x796A, 0x6C78, 0x6B7E, 0x6274, 0x6572, 0x4850, 0x4F56, 0x465C, 0x415A, 0x5448, 0x534E, 0x5A44, 0x5D42, 0xE0C0, 0xE7C6, 0xEECC, 0xE9CA, 0xFCD8, 0xFBDE, 0xF2D4, 0xF5D2, 0xD8F0, 0xDFF6, 0xD6FC, 0xD1FA, 0xC4E8, 0xC3EE, 0xCAE4, 0xCDE2, 0x90A0, 0x97A6, 0x9EAC, 0x99AA, 0x8CB8, 0x8BBE, 0x82B4, 0x85B2, 0xA890, 0xAF96, 0xA69C, 0xA19A, 0xB488, 0xB38E, 0xBA84, 0xBD82, 0xDD9D, 0xDA9B, 0xD391, 0xD497, 0xC185, 0xC683, 0xCF89, 0xC88F, 0xE5AD, 0xE2AB, 0xEBA1, 0xECA7, 0xF9B5, 0xFEB3, 0xF7B9, 0xF0BF, 0xADFD, 0xAAFB, 0xA3F1, 0xA4F7, 0xB1E5, 0xB6E3, 0xBFE9, 0xB8EF, 0x95CD, 0x92CB, 0x9BC1, 0x9CC7, 0x89D5, 0x8ED3, 0x87D9, 0x80DF, 0x3D5D, 0x3A5B, 0x3351, 0x3457, 0x2145, 0x2643, 0x2F49, 0x284F, 0x056D, 0x026B, 0x0B61, 0x0C67, 0x1975, 0x1E73, 0x1779, 0x107F, 0x4D3D, 0x4A3B, 0x4331, 0x4437, 0x5125, 0x5623, 0x5F29, 0x582F, 0x750D, 0x720B, 0x7B01, 0x7C07, 0x6915, 0x6E13, 0x6719, 0x601F, 0xA727, 0xA021, 0xA92B, 0xAE2D, 0xBB3F, 0xBC39, 0xB533, 0xB235, 0x9F17, 0x9811, 0x911B, 0x961D, 0x830F, 0x8409, 0x8D03, 0x8A05, 0xD747, 0xD041, 0xD94B, 0xDE4D, 0xCB5F, 0xCC59, 0xC553, 0xC255, 0xEF77, 0xE871, 0xE17B, 0xE67D, 0xF36F, 0xF469, 0xFD63, 0xFA65, 0x47E7, 0x40E1, 0x49EB, 0x4EED, 0x5BFF, 0x5CF9, 0x55F3, 0x52F5, 0x7FD7, 0x78D1, 0x71DB, 0x76DD, 0x63CF, 0x64C9, 0x6DC3, 0x6AC5, 0x3787, 0x3081, 0x398B, 0x3E8D, 0x2B9F, 0x2C99, 0x2593, 0x2295, 0x0FB7, 0x08B1, 0x01BB, 0x06BD, 0x13AF, 0x14A9, 0x1DA3, 0x1AA5, 0x7ABA, 0x7DBC, 0x74B6, 0x73B0, 0x66A2, 0x61A4, 0x68AE, 0x6FA8, 0x428A, 0x458C, 0x4C86, 0x4B80, 0x5E92, 0x5994, 0x509E, 0x5798, 0x0ADA, 0x0DDC, 0x04D6, 0x03D0, 0x16C2, 0x11C4, 0x18CE, 0x1FC8, 0x32EA, 0x35EC, 0x3CE6, 0x3BE0, 0x2EF2, 0x29F4, 0x20FE, 0x27F8, 0x9A7A, 0x9D7C, 0x9476, 0x9370, 0x8662, 0x8164, 0x886E, 0x8F68, 0xA24A, 0xA54C, 0xAC46, 0xAB40, 0xBE52, 0xB954, 0xB05E, 0xB758, 0xEA1A, 0xED1C, 0xE416, 0xE310, 0xF602, 0xF104, 0xF80E, 0xFF08, 0xD22A, 0xD52C, 0xDC26, 0xDB20, 0xCE32, 0xC934, 0xC03E, 0xC738}, /* 29 */
    /* 2A */ {0x0000, 0x0302, 0x0604, 0x0506, 0x0C08, 0x0F0A, 0x0A0C, 0x090E, 0x1810, 0x1B12, 0x1E14, 0x1D16, 0x1418, 0x171A, 0x121C, 0x111E, 0x3020, 0x3322, 0x3624, 0x3526, 0x3C28, 0x3F2A, 0x3A2C, 0x392E, 0x2830, 0x2B32, 0x2E34, 0x2D36, 0x2438, 0x273A, 0x223C, 0x213E, 0x6040, 0x6342, 0x6644, 0x6546, 0x6C48, 0x6F4A, 0x6A4C, 0x694E, 0x7850, 0x7B52, 0x7E54, 0x7D56, 0x7458, 0x775A, 0x725C, 0x715E, 0x5060, 0x5362, 0x5664, 0x5566, 0x5C68, 0x5F6A, 0x5A6C, 0x596E, 0x4870, 0x4B72, 0x4E74, 0x4D76, 0x4478, 0x477A, 0x427C, 0x417E, 0xC080, 0xC382, 0xC684, 0xC586, 0xCC88, 0xCF8A, 0xCA8C, 0xC98E, 0xD890, 0xDB92, 0xDE94, 0xDD96, 0xD498, 0xD79A, 0xD29C, 0xD19E, 0xF0A0, 0xF3A2, 0xF6A4, 0xF5A6, 0xFCA8, 0xFFAA, 0xFAAC, 0xF9AE, 0xE8B0, 0xEBB2, 0xEEB4, 0xEDB6, 0xE4B8, 0xE7BA, 0xE2BC, 0xE1BE, 0xA0C0, 0xA3C2, 0xA6C4, 0xA5C6, 0xACC8, 0xAFCA, 0xAACC, 0xA9CE, 0xB8D0, 0xBBD2, 0xBED4, 0xBDD6, 0xB4D8, 0xB7DA, 0xB2DC, 0xB1DE, 0x90E0, 0x93E2, 0x96E4, 0x95E6, 0x9CE8, 0x9FEA, 0x9AEC, 0x99EE, 0x88F0, 0x8BF2, 0x8EF4, 0x8DF6, 0x84F8, 0x87FA, 0x82FC, 0x81FE, 0x9D1D, 0x9E1F, 0x9B19, 0x981B, 0x9115, 0x9217, 0x9711, 0x9413, 0x850D, 0x860F, 0x8309, 0x800B, 0x8905, 0x8A07, 0x8F01, 0x8C03, 0xAD3D, 0xAE3F, 0xAB39, 0xA83B, 0xA135, 0xA237, 0xA731, 0xA433, 0xB52D, 0xB62F, 0xB329, 0xB02B, 0xB925, 0xBA27, 0xBF21, 0xBC23, 0xFD5D, 0xFE5F, 0xFB59, 0xF85B, 0xF155, 0xF257, 0xF751, 0xF453, 0xE54D, 0xE64F, 0xE349, 0xE04B, 0xE945, 0xEA47, 0xEF41, 0xEC43, 0xCD7D, 0xCE7F, 0xCB79, 0xC87B, 0xC175, 0xC277, 0xC771, 0xC473, 0xD56D, 0xD66F, 0xD369, 0xD06B, 0xD965, 0xDA67, 0xDF61, 0xDC63, 0x5D9D, 0x5E9F, 0x5B99, 0x589B, 0x5195, 0x5297, 0x5791, 0x5493, 0x458D, 0x468F, 0x4389, 0x408B, 0x4985, 0x4A87, 0x4F81, 0x4C83, 0x6DBD, 0x6EBF, 0x6BB9, 0x68BB, 0x61B5, 0x62B7, 0x67B1, 0x64B3, 0x75AD, 0x76AF, 0x73A9, 0x70AB, 0x79A5, 0x7AA7, 0x7FA1, 0x7CA3, 0x3DDD, 0x3EDF, 0x3BD9, 0x38DB, 0x31D5, 0x32D7, 0x37D1, 0x34D3, 0x25CD, 0x26CF, 0x23C9, 0x20CB, 0x29C5, 0x2AC7, 0x2FC1, 0x2CC3, 0x0DFD, 0x0EFF, 0x0BF9, 0x08FB, 0x01F5, 0x02F7, 0x07F1, 0x04F3, 0x15ED, 0x16EF, 0x13E9, 0x10EB, 0x19E5, 0x1AE7, 0x1FE1, 0x1CE3}  /* 2A */
    /*          00      01      02      03      04      05      06      07      08      09      0A      0B      0C      0D      0E      0F      10      11      12      13      14      15      16      17      18      19      1A      1B      1C      1D      1E      1F      20      21      22      23      24      25      26      27      28      29      2A      2B      2C      2D      2E      2F      30      31      32      33      34      35      36      37      38      39      3A      3B      3C      3D      3E      3F      40      41      42      43      44      45      46      47      48      49      4A      4B      4C      4D      4E      4F      50      51      52      53      54      55      56      57      58      59      5A      5B      5C      5D      5E      5F      60      61      62      63      64      65      66      67      68      69      6A      6B      6C      6D      6E      6F      70      71      72      73      74      75      76      77      78      79      7A      7B      7C      7D      7E      7F      80      81      82      83      84      85      86      87      88      89      8A      8B      8C      8D      8E      8F      90      91      92      93      94      95      96      97      98      99      9A      9B      9C      9D      9E      9F      A0      A1      A2      A3      A4      A5      A6      A7      A8      A9      AA      AB      AC      AD      AE      AF      B0      B1      B2      B3      B4      B5      B6      B7      B8      B9      BA      BB      BC      BD      BE      BF      C0      C1      C2      C3      C4      C5      C6      C7      C8      C9      CA      CB      CC      CD      CE      CF      D0      D1      D2      D3      D4      D5      D6      D7      D8      D9      DA      DB      DC      DD      DE      DF      E0      E1      E2      E3      E4      E5      E6      E7      E8      E9      EA      EB      EC      ED      EE      EF      F0      F1      F2      F3      F4      F5      F6      F7      F8      F9      FA      FB      FC      FD      FE      FF           */
};

#define HEADER_OFFSET                  12
#define HEADER_SIZE                    4
#define CDROMXA_SUBHEADER_OFFSET       (HEADER_OFFSET + HEADER_SIZE)
#define CDROMXA_SUBHEADER_SIZE         8
#define CDROMXA_FORM2_EDC_OFFSET       ((CDROMXA_SUBHEADER_OFFSET+CDROMXA_SUBHEADER_SIZE) + 2324)

#define CDROMXA_FORM1_EDC_OFFSET       ((CDROMXA_SUBHEADER_OFFSET+CDROMXA_SUBHEADER_SIZE) + 2048)
#define CDROMXA_FORM1_PARITY_P_OFFSET  (CDROMXA_FORM1_EDC_OFFSET + 4)
#define CDROMXA_FORM1_PARITY_Q_OFFSET  (CDROMXA_FORM1_PARITY_P_OFFSET + 172)

#define EDC_SIZE                       4
#define CDROMXA_FORM1_USER_DATA_SIZE   2048
#define CDROMXA_FORM1_PARITY_P_SIZE    (43 * 2 * 2)

int CDAccess_PBP::fix_sector(uint8_t* sector, int32_t lba)
{
   // Find mode.
   unsigned char mode = sector[HEADER_OFFSET + 3];

   // Process sector based on mode.
   if(mode == 0)
   {
      // Check that the sector is really all-zero.
      for(int j = HEADER_OFFSET + HEADER_SIZE; j < 2352; ++j)
      {
         if(sector[j] != 0x00)
            return -1;
      }
      // probably reached the beginning of the zero-padding - skip check
   }
   else if(mode == 2)
   {
      // Sync pattern
      unsigned char sync[] = {0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00};

      // calculate current offset in data based on sector
      unsigned char amsf[3];
      LBA_to_AMSF(lba, &amsf[0], &amsf[1], &amsf[2]);

      // Write sync
      memcpy(sector, sync, sizeof(sync));

      // Check that the two copies of the subheader data are equivalent.
      if(memcmp(sector+CDROMXA_SUBHEADER_OFFSET, sector+CDROMXA_SUBHEADER_OFFSET+4, 4) != 0)
         return -1;

      // Determine CD ROM XA Mode 2 form.
      if((sector[CDROMXA_SUBHEADER_OFFSET + 2] & 0x20))
      {
         // Compute form 2 EDC.
         uint32_t EDC = EDCCrc32(sector+CDROMXA_SUBHEADER_OFFSET, CDROMXA_FORM2_EDC_OFFSET-CDROMXA_SUBHEADER_OFFSET);
         // Write EDC.
         memcpy(sector+CDROMXA_FORM2_EDC_OFFSET, &EDC, sizeof(EDC));
      }
      else
      {
         // Compute form 1 EDC.
         uint32_t EDC = EDCCrc32(sector+CDROMXA_SUBHEADER_OFFSET, CDROMXA_FORM1_EDC_OFFSET-CDROMXA_SUBHEADER_OFFSET);
         // Write EDC.
         memcpy(sector+CDROMXA_FORM1_EDC_OFFSET, &EDC, sizeof(EDC));
         // clear header
         memset(sector+HEADER_OFFSET, 0, 4);

         // Write error-correction data.
         unsigned char* src = sector + HEADER_OFFSET;
         unsigned char* dst = sector + CDROMXA_FORM1_PARITY_P_OFFSET;
         for(int i = 0; i < 43; ++i)
         {
            unsigned short x = 0x0000;
            unsigned short y = 0x0000;
            for(int j = 19; j < 43; ++j)
            {
               x ^= RSPCTable[j][src[0]]; // LSB
               y ^= RSPCTable[j][src[1]]; // MSB
               src += 2 * 43;
            }
            dst[         0] = x >> 8;
            dst[2 * 43 + 0] = x & 0xFF;
            dst[         1] = y >> 8;
            dst[2 * 43 + 1] = y & 0xFF;
            dst += 2;
            src -= (43 - 19) * 2 * 43; // Restore src to the state before the inner loop.
            src += 2;
         }

         // Calculate Q parity.
         src = sector + HEADER_OFFSET;
         dst = sector + CDROMXA_FORM1_PARITY_Q_OFFSET;
         unsigned char* src_end = sector + CDROMXA_FORM1_PARITY_Q_OFFSET;
         for(int i = 0; i < 26; ++i)
         {
            unsigned char* src_backup = src;
            unsigned short x = 0x0000;
            unsigned short y = 0x0000;
            for(int j = 0; j < 43; ++j)
            {
               x ^= RSPCTable[j][src[0]]; // LSB
               y ^= RSPCTable[j][src[1]]; // MSB
               src += 2 * 44;
               if(src >= src_end)
               {
                  src = src - (HEADER_SIZE + CDROMXA_SUBHEADER_SIZE + CDROMXA_FORM1_USER_DATA_SIZE + EDC_SIZE + CDROMXA_FORM1_PARITY_P_SIZE);
               }
            }

            dst[         0] = x >> 8;
            dst[2 * 26 + 0] = x & 0xFF;
            dst[         1] = y >> 8;
            dst[2 * 26 + 1] = y & 0xFF;
            dst += 2;
            src = src_backup;
            src += 2 * 43;
         }
      }

      // Write header
      sector[HEADER_OFFSET + 0] = U8_to_BCD(amsf[0]);
      sector[HEADER_OFFSET + 1] = U8_to_BCD(amsf[1]);
      sector[HEADER_OFFSET + 2] = U8_to_BCD(amsf[2]);
      sector[HEADER_OFFSET + 3] = mode;
   }
   else  // mode 1 unsupported
      return -1;

   return 0;
}
