// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A0__F231

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_48_0_T_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xd11340feu, 0xa85c8b2au, 23, 29 }, // d11340fea85c8b2a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x9ef1abe5u, 0xc8eab7bfu, 23, 6 }, // 9ef1abe5c8eab7bf = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xa8fb52ceu, 0x14c0000bu, 23, 46 }, // a8fb52ce14c0000b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x0221674cu, 0x1389bacbu, 23, 86 }, // 0221674c1389bacb = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__32_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x70ddb531u, 0xc6582742u, 23, 63 }, // 70ddb531c6582742 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__32_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x98a772e7u, 0x2566b51cu, 103, 109 }, // 98a772e72566b51c = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__64_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xfd7964c4u, 0xca0f74d3u, 103, 126 }, // fd7964c4ca0f74d3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xb93aa9a8u, 0x2e7b885du, 103, 86 }, // b93aa9a82e7b885d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x93e17780u, 0x787068a1u, 103, 143 }, // 93e17780787068a1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__64_16__CO__wave4_warp4_stg1--Arch_gfx950
    { 0x013275bbu, 0x81ad5bd6u, 160, 29 }, // 013275bb81ad5bd6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xb912eb91u, 0x206f7edau, 160, 6 }, // b912eb91206f7eda = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_0_T_T_1___gfx950__P__64_32__CO__wave2_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 5, 7, 3, 1, 0, 4, 1, 1, 1, 1},
 { 7, 5, 6, 7, 0, 2, 1, 2, 2, 2},
 { 6, 6, 5, 5, 0, 2, 2, 2, 2, 2},
 { 7, 7, 7, 7, 0, 1, 2, 2, 2, 2},
 { 7, 7, 7, 7,10, 2,10,10,10,10},
 { 6, 7, 6, 7,10,10,10,10,10,10},
 { 7, 6, 6, 7,10,10,10,10,10,10},
 { 7, 6, 7, 7,10,10,10,10,10,10},
 { 6, 6, 6, 6,10,10,10,10,10,10},
 { 7, 7, 8, 7,10,10,10,10, 9,10}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A0__F231 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 48
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 48 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

