// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A1__F334

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_192_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1100)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x3a8fd3dcu, 0x68f11da1u, 40, 46 }, // 3a8fd3dc68f11da1 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1100
    { 0x5aa0cf56u, 0x916d837du, 40, 138 }, // 5aa0cf56916d837d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_16__CO__wave1_warp4_stg1--Arch_gfx1100
    { 0x5f080fecu, 0x065f5644u, 40, 6 }, // 5f080fec065f5644 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1100
    { 0xf5f8b7aau, 0x96a29b77u, 40, 155 }, // f5f8b7aa96a29b77 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1100
    { 0x9ef2e14fu, 0x252f9336u, 40, 103 }, // 9ef2e14f252f9336 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1100
    { 0x26161d42u, 0xf941f48cu, 97, 138 }, // 26161d42f941f48c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1100
    { 0x0bc0b9c6u, 0x22cc4ffcu, 97, 80 }, // 0bc0b9c622cc4ffc = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1100
    { 0xa49e6b23u, 0xa7b30bffu, 97, 155 }, // a49e6b23a7b30bff = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1100
    { 0x90eeefaau, 0x37e09060u, 120, 138 }, // 90eeefaa37e09060 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1100
    { 0x91023a14u, 0x509afe4au, 120, 80 }, // 91023a14509afe4a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1100
    { 0xe434b495u, 0xc735a2f0u, 120, 155 }, // e434b495c735a2f0 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1100
    { 0xab4aedabu, 0xb443e7fdu, 120, 103 }, // ab4aedabb443e7fd = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1100
    { 0x37aaede9u, 0xa1e266dau, 126, 138 }, // 37aaede9a1e266da = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__64_32__CO__wave1_warp4_stg1--Arch_gfx1100
    { 0x3b1bcbadu, 0xad1dfb12u, 126, 80 }, // 3b1bcbadad1dfb12 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1100
    { 0xaa8c96ceu, 0xcdc9a0f7u, 126, 103 }, // aa8c96cecdc9a0f7 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_192_3_T_T_0___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1100
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1100_mod0
{{ 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
 { 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
 { 1, 1, 1, 1, 4, 4, 1, 1, 1, 3},
 { 1, 5,10, 1, 1, 1, 1, 1, 1, 4},
 { 1, 5,10,10,10, 2, 0, 0, 5, 5},
 { 1,12,11,12, 0, 0, 0, 1, 1, 1},
 { 8,12,12,11, 8, 8, 6, 7, 7, 7},
 { 9,12,12,10,11,10,10,11,11, 9},
 { 0,14,12,11,14,13, 0, 2, 8, 9},
 {10,14,12,12,12,13,14,11,10,10}}
// End of GPU gfx1100_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A1__F334 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 192
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 192 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

