// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A2__F273

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_96_0_F_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1101)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x3276d126u, 0xdb52fa08u, 0, 46 }, // 3276d126db52fa08 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1101
    { 0x68165130u, 0x4daede28u, 0, 63 }, // 681651304daede28 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1101
    { 0xb45807bau, 0xdd19c5dcu, 0, 80 }, // b45807badd19c5dc = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1101
    { 0x0ba4ba62u, 0xe9a9ece8u, 40, 138 }, // 0ba4ba62e9a9ece8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__32_16__CO__wave1_warp4_stg1--Arch_gfx1101
    { 0x9a874d8au, 0x8fbb6775u, 40, 80 }, // 9a874d8a8fbb6775 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1101
    { 0x870a4103u, 0x713b9b48u, 40, 155 }, // 870a4103713b9b48 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1101
    { 0x900a12b2u, 0xff176679u, 40, 103 }, // 900a12b2ff176679 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1101
    { 0x7833b8bfu, 0x3b14bb4eu, 97, 138 }, // 7833b8bf3b14bb4e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1101
    { 0x8f7e194fu, 0xd8a4eeedu, 97, 80 }, // 8f7e194fd8a4eeed = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1101
    { 0xd0cbba52u, 0xdfe1c37eu, 120, 138 }, // d0cbba52dfe1c37e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1101
    { 0xee1585f4u, 0xac8be801u, 120, 80 }, // ee1585f4ac8be801 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1101
    { 0x53cd1efdu, 0xeacb40e3u, 120, 103 }, // 53cd1efdeacb40e3 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1101
    { 0x2c2d4483u, 0xc847829du, 126, 138 }, // 2c2d4483c847829d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__64_32__CO__wave1_warp4_stg1--Arch_gfx1101
    { 0x06215fecu, 0x278a5b97u, 126, 80 }, // 06215fec278a5b97 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1101
    { 0xbba8d9a6u, 0xbe59621eu, 126, 155 }, // bba8d9a6be59621e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1101
    { 0x4e742bbfu, 0x27b613e9u, 126, 103 }, // 4e742bbf27b613e9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_96_0_F_F_1___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1101
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1101_mod0
{{ 6, 6, 5, 5, 0, 0, 0, 0, 0, 0},
 { 7, 5, 6, 5, 5, 3, 1, 6, 5, 5},
 { 7,11,11,11,11,11,11,15,15,13},
 { 7, 3, 5, 5, 5, 0, 1, 0, 0, 4},
 { 7, 5, 5, 4,11, 5, 0, 0, 0, 2},
 { 8, 5, 5,11,11,11, 5,14, 4, 3},
 {12,11,10, 9,11,11, 5, 1, 6, 6},
 {12,10, 9,11,11,11,15,15, 1,15},
 {12,10, 5, 5,11, 4, 1, 1, 1, 3},
 {15, 9, 6,11,11,11, 1,15,15,14}}
// End of GPU gfx1101_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A2__F273 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 96
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 96 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

