// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F149

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_224_0_T_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xa6fb08e4u, 0xa08f1e37u, 0, 46 }, // a6fb08e4a08f1e37 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x0c016579u, 0xbb42a8bau, 0, 63 }, // 0c016579bb42a8ba = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xeb79f248u, 0x6659ad00u, 0, 80 }, // eb79f2486659ad00 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x022943d9u, 0xd96bc6e0u, 40, 63 }, // 022943d9d96bc6e0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x91f92d56u, 0x9779b269u, 97, 46 }, // 91f92d569779b269 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x83ee611cu, 0x4fdb95e8u, 97, 138 }, // 83ee611c4fdb95e8 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__32_32__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xd85aa00du, 0x6f2da862u, 120, 80 }, // d85aa00d6f2da862 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x4caf2044u, 0xe3843b6au, 126, 46 }, // 4caf2044e3843b6a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x7beb5c8cu, 0xee525358u, 132, 138 }, // 7beb5c8cee525358 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_0_T_F_1___gfx950__P__64_64__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{6,5,4,4,4,4,4,4,0,0},
 {2,2,2,2,2,0,4,4,4,4},
 {2,2,2,0,5,4,4,4,4,4},
 {1,1,1,1,4,4,8,8,4,8},
 {3,3,3,3,3,7,7,7,7,7},
 {1,6,6,3,7,7,7,7,7,7},
 {6,6,3,3,7,7,7,7,7,8},
 {6,6,3,3,7,7,7,7,4,4},
 {6,3,3,3,7,7,7,8,4,8},
 {6,3,3,3,7,7,7,7,3,3}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F149 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 224
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 224 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

