/*
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (C) 2025 Raspberry Pi Ltd
 */

#ifndef FILE_OPERATIONS_LINUX_H_
#define FILE_OPERATIONS_LINUX_H_

#include "../file_operations.h"

namespace rpi_imager {

// Linux implementation using POSIX file operations
class LinuxFileOperations : public FileOperations {
 public:
  LinuxFileOperations();
  ~LinuxFileOperations() override;

  // Non-copyable, movable
  LinuxFileOperations(const LinuxFileOperations&) = delete;
  LinuxFileOperations& operator=(const LinuxFileOperations&) = delete;
  LinuxFileOperations(LinuxFileOperations&&) = default;
  LinuxFileOperations& operator=(LinuxFileOperations&&) = default;

  FileError OpenDevice(const std::string& path) override;
  FileError CreateTestFile(const std::string& path, std::uint64_t size) override;
  FileError WriteAtOffset(
      std::uint64_t offset,
      const std::uint8_t* data,
      std::size_t size) override;
  FileError GetSize(std::uint64_t& size) override;
  FileError Close() override;
  bool IsOpen() const override;

  // Streaming I/O operations
  FileError WriteSequential(const std::uint8_t* data, std::size_t size) override;
  FileError ReadSequential(std::uint8_t* data, std::size_t size, std::size_t& bytes_read) override;
  
  // File positioning
  FileError Seek(std::uint64_t position) override;
  std::uint64_t Tell() const override;
  
  // Sync operations
  FileError ForceSync() override;
  FileError Flush() override;
  
  // Sequential read optimization
  void PrepareForSequentialRead(std::uint64_t offset, std::uint64_t length) override;
  
  // Handle access
  int GetHandle() const override;

  // Get the last errno error code
  int GetLastErrorCode() const override;

  // Check if direct I/O is enabled
  bool IsDirectIOEnabled() const override { return using_direct_io_; }
  
  // Get direct I/O attempt details (Linux: not yet implemented)
  DirectIOInfo GetDirectIOInfo() const override { 
      DirectIOInfo info;
      info.currently_enabled = using_direct_io_;
      return info;
  }

 private:
  int fd_;
  std::string current_path_;
  int last_error_code_;
  bool using_direct_io_;  // Track if we opened with O_DIRECT
  
  FileError OpenInternal(const char* path, int flags, mode_t mode = 0);
  
  // Helper to determine if path is a block device
  static bool IsBlockDevicePath(const std::string& path);
};

} // namespace rpi_imager

#endif // FILE_OPERATIONS_LINUX_H_ 