/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#include "ArgumentCoders.h"
#include "Connection.h"
#include "MessageNames.h"
#include <WebCore/ShareableBitmapHandle.h>
#include <wtf/Forward.h>
#include <wtf/RuntimeApplicationChecks.h>
#include <wtf/ThreadSafeRefCounted.h>
#include <wtf/Vector.h>

namespace WebCore {
class IntSize;
}

namespace Messages {
namespace AcceleratedBackingStore {

static inline IPC::ReceiverName messageReceiverName()
{
    return IPC::ReceiverName::AcceleratedBackingStore;
}

class DidCreateDMABufBuffer {
public:
    using Arguments = std::tuple<uint64_t, WebCore::IntSize, uint32_t, Vector<UnixFileDescriptor>, Vector<uint32_t>, Vector<uint32_t>, uint64_t, WebKit::RendererBufferFormat::Usage>;

    static IPC::MessageName name() { return IPC::MessageName::AcceleratedBackingStore_DidCreateDMABufBuffer; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;

    DidCreateDMABufBuffer(uint64_t id, const WebCore::IntSize& size, uint32_t format, const Vector<UnixFileDescriptor>& fds, const Vector<uint32_t>& offsets, const Vector<uint32_t>& strides, uint64_t modifier, const WebKit::RendererBufferFormat::Usage& usage)
        : m_id(id)
        , m_size(size)
        , m_format(format)
        , m_fds(fds)
        , m_offsets(offsets)
        , m_strides(strides)
        , m_modifier(modifier)
        , m_usage(usage)
    {
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        encoder << m_id;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_size;
        encoder << m_format;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_fds;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_offsets;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_strides;
        encoder << m_modifier;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_usage;
    }

private:
    uint64_t m_id;
    SUPPRESS_FORWARD_DECL_MEMBER const WebCore::IntSize& m_size;
    uint32_t m_format;
    SUPPRESS_FORWARD_DECL_MEMBER const Vector<UnixFileDescriptor>& m_fds;
    SUPPRESS_FORWARD_DECL_MEMBER const Vector<uint32_t>& m_offsets;
    SUPPRESS_FORWARD_DECL_MEMBER const Vector<uint32_t>& m_strides;
    uint64_t m_modifier;
    SUPPRESS_FORWARD_DECL_MEMBER const WebKit::RendererBufferFormat::Usage& m_usage;
};

class DidCreateSHMBuffer {
public:
    using Arguments = std::tuple<uint64_t, WebCore::ShareableBitmapHandle>;

    static IPC::MessageName name() { return IPC::MessageName::AcceleratedBackingStore_DidCreateSHMBuffer; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;

    DidCreateSHMBuffer(uint64_t id, WebCore::ShareableBitmapHandle&& handle)
        : m_id(id)
        , m_handle(WTFMove(handle))
    {
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        encoder << m_id;
        encoder << WTFMove(m_handle);
    }

private:
    uint64_t m_id;
    WebCore::ShareableBitmapHandle&& m_handle;
};

class DidDestroyBuffer {
public:
    using Arguments = std::tuple<uint64_t>;

    static IPC::MessageName name() { return IPC::MessageName::AcceleratedBackingStore_DidDestroyBuffer; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;

    explicit DidDestroyBuffer(uint64_t id)
        : m_id(id)
    {
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        encoder << m_id;
    }

private:
    uint64_t m_id;
};

class Frame {
public:
    using Arguments = std::tuple<uint64_t, Vector<WebCore::IntRect, 1>, UnixFileDescriptor>;

    static IPC::MessageName name() { return IPC::MessageName::AcceleratedBackingStore_Frame; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;

    Frame(uint64_t id, const Vector<WebCore::IntRect, 1>& damage, const UnixFileDescriptor& syncFD)
        : m_id(id)
        , m_damage(damage)
        , m_syncFD(syncFD)
    {
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        encoder << m_id;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_damage;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_syncFD;
    }

private:
    uint64_t m_id;
    SUPPRESS_FORWARD_DECL_MEMBER const Vector<WebCore::IntRect, 1>& m_damage;
    SUPPRESS_FORWARD_DECL_MEMBER const UnixFileDescriptor& m_syncFD;
};

} // namespace AcceleratedBackingStore
} // namespace Messages
