/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(MEDIA_STREAM)

#include "ArgumentCoders.h"
#include "Connection.h"
#include "MessageNames.h"
#include <wtf/Expected.h>
#include <wtf/Forward.h>
#include <wtf/RuntimeApplicationChecks.h>
#include <wtf/ThreadSafeRefCounted.h>
#include <wtf/Vector.h>

namespace WebCore {
struct MediaDeviceHashSalts;
struct MediaStreamRequest;
}

namespace Messages {
namespace UserMediaCaptureManager {

static inline IPC::ReceiverName messageReceiverName()
{
    return IPC::ReceiverName::UserMediaCaptureManager;
}

class ValidateUserMediaRequestConstraints {
public:
    using Arguments = std::tuple<WebCore::MediaStreamRequest, WebCore::MediaDeviceHashSalts>;

    static IPC::MessageName name() { return IPC::MessageName::UserMediaCaptureManager_ValidateUserMediaRequestConstraints; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::UserMediaCaptureManager_ValidateUserMediaRequestConstraintsReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Expected<WebCore::RealtimeMediaSourceCenter::ValidDevices, WebCore::MediaConstraintType>>;
    using Reply = CompletionHandler<void(Expected<WebCore::RealtimeMediaSourceCenter::ValidDevices, WebCore::MediaConstraintType>&&)>;
    using Promise = WTF::NativePromise<Expected<WebCore::RealtimeMediaSourceCenter::ValidDevices, WebCore::MediaConstraintType>, IPC::Error>;
    ValidateUserMediaRequestConstraints(const WebCore::MediaStreamRequest& request, const WebCore::MediaDeviceHashSalts& mediaDeviceIdentifierHashSalts)
        : m_request(request)
        , m_mediaDeviceIdentifierHashSalts(mediaDeviceIdentifierHashSalts)
    {
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        SUPPRESS_FORWARD_DECL_ARG encoder << m_request;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_mediaDeviceIdentifierHashSalts;
    }

private:
    SUPPRESS_FORWARD_DECL_MEMBER const WebCore::MediaStreamRequest& m_request;
    SUPPRESS_FORWARD_DECL_MEMBER const WebCore::MediaDeviceHashSalts& m_mediaDeviceIdentifierHashSalts;
};

class GetMediaStreamDevices {
public:
    using Arguments = std::tuple<bool>;

    static IPC::MessageName name() { return IPC::MessageName::UserMediaCaptureManager_GetMediaStreamDevices; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;

    static IPC::MessageName asyncMessageReplyName() { return IPC::MessageName::UserMediaCaptureManager_GetMediaStreamDevicesReply; }
    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Vector<WebCore::CaptureDeviceWithCapabilities>>;
    using Reply = CompletionHandler<void(Vector<WebCore::CaptureDeviceWithCapabilities>&&)>;
    using Promise = WTF::NativePromise<Vector<WebCore::CaptureDeviceWithCapabilities>, IPC::Error>;
    explicit GetMediaStreamDevices(bool revealIdsAndLabels)
        : m_revealIdsAndLabels(revealIdsAndLabels)
    {
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        encoder << m_revealIdsAndLabels;
    }

private:
    bool m_revealIdsAndLabels;
};

} // namespace UserMediaCaptureManager
} // namespace Messages

#endif // ENABLE(MEDIA_STREAM)
